package csbase.client.csdk.v1_0.filesystem;

import java.awt.Window;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;

import tecgraf.javautils.core.io.FileUtils;
import csbase.client.ClientLocalFile;
import csbase.client.project.ClientFileLock;
import csbase.logic.ClientFile;
import csbase.logic.ClientFileType;
import csbase.logic.ClientProjectFile;
import csdk.v1_0.api.filesystem.FileLocationType;
import csdk.v1_0.api.filesystem.FileLockedException;
import csdk.v1_0.api.filesystem.IFile;
import csdk.v1_0.api.filesystem.IFileLock;

/**
 * Encapsula um arquivo (da rea de projetos ou do sistema local) em um arquivo
 * acessvel pelas aplicaes baseadas no CSDK.
 */
public class CSDKFile implements IFile {

  /**
   * Arquivo encapsulado.
   */
  private final ClientFile file;

  /**
   * Construtor
   * 
   * @param file arquivo encapsulado.
   */
  public CSDKFile(ClientFile file) {
    if (file == null) {
      throw new IllegalArgumentException("Parmetro file no pode ser nulo");
    }
    this.file = file;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void close(boolean force) throws IOException {
    file.close(force);
  }

  /**
   * Fecha o arquivo.
   * 
   * @throws IOException em caso de erro de I/O.
   */
  public void close() throws IOException {
    file.close(true);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile[] getChildren() throws Exception {
    ClientFile[] children = file.getChildren();
    List<IFile> files = new ArrayList<IFile>();
    for (ClientFile clientProjectFile : children) {
      files.add(new CSDKFile(clientProjectFile));
    }
    return files.toArray(new IFile[files.size()]);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public InputStream getInputStream() throws IOException {
    return file.getInputStream();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long getModificationDate() {
    return file.getModificationDate();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getName() {
    return file.getName();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public OutputStream getOutputStream() throws IOException {
    return file.getOutputStream();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String[] getPath() {
    return file.getPath();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getStringPath() {
    return file.getStringPath();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getType() {
    return file.getType();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isDirectory() {
    return file.isDirectory();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void open(boolean readOnly) throws Exception {
    file.open(readOnly);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long position() throws IOException {
    return file.position();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void position(long newPosition) throws IOException {
    file.position(newPosition);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int read(byte[] dst, int off, int len, long position) throws Exception {
    return file.read(dst, off, len, position);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int read(byte[] dst, long position) throws Exception {
    return file.read(dst, position);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long size() {
    return file.size();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void write(byte[] src, int off, int len, long position)
    throws IOException, FileLockedException {
    file.write(src, off, len, position);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void write(byte[] src, long position) throws IOException,
    FileLockedException {
    file.write(src, position);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public FileLocationType getFileLocationType() {
    switch (file.getClientFileType()) {
      case LOCAL:
        return FileLocationType.LOCAL;
      case REMOTE:
        return FileLocationType.REMOTE;
      default:
        return null;
    }
  }

  /**
   * Obtm o arquivo encapsulado.
   * 
   * @return file o arquivo.
   */
  public ClientFile getFile() {
    return file;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFile getParent() {
    return new CSDKFile(file.getParent());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean exists() throws IOException {
    return file.exists();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canExecute() {
    return file.canExecute();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canRead() {
    return file.canRead();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canWrite() {
    return file.canWrite();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFileLock acquireExclusiveLock(Window window) throws Exception {
    if (file.getClientFileType() == ClientFileType.REMOTE) {
      ClientProjectFile projectFile = (ClientProjectFile) file;
      return new CSDKProjectFileLock(ClientFileLock.acquireExclusiveLock(
        window, projectFile));
    }
    else {
      ClientLocalFile clientlocalFile = (ClientLocalFile) file;
      String[] path = clientlocalFile.getPath();
      File localFile = new File(FileUtils.joinPath(path));
      return new CSDKLocalFileLock(localFile, false, window);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IFileLock acquireSharedLock(Window window) throws Exception {
    if (file.getClientFileType() == ClientFileType.REMOTE) {
      ClientProjectFile projectFile = (ClientProjectFile) file;
      return new CSDKProjectFileLock(ClientFileLock.acquireSharedLock(window,
        projectFile));
    }
    else {
      ClientLocalFile clientlocalFile = (ClientLocalFile) file;
      String[] path = clientlocalFile.getPath();
      File localFile = new File(FileUtils.joinPath(path));
      return new CSDKLocalFileLock(localFile, true, window);
    }
  }
}
