/*
 * CommandNotificationHandler.java @author Cristina Ururahy
 */
package csbase.client.desktop;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.SwingThreadDispatcher;
import csbase.logic.CommandNotification;
import csbase.logic.CommonClientProject;
import csbase.logic.Notification;
import csbase.logic.NotificationHandler;

/**
 * Handler para as notificaes de trmino de comando.
 */
public class CommandNotificationHandler implements NotificationHandler {

  /**
   * Responsvel por entregar os eventos aos ouvintes.
   */
  private Executor executor;

  /** Listeners interessados nas notificaes dos comandos */
  private final List<CommandListener> listeners;

  /** Instncia nica do handler. */
  private static CommandNotificationHandler instance =
    new CommandNotificationHandler();

  /** Indica se deve ser feita a atualizao da rvore de projetos. */
  private boolean updateProjectTree;

  /** Indica se uma RemoteTask est atualizando a rvore de projetos. */
  private boolean runningTask;

  /**
   * Construtor
   */
  private CommandNotificationHandler() {
    this.listeners =
      Collections.synchronizedList(new ArrayList<CommandListener>());
    executor = Executors.newSingleThreadExecutor();
  }

  /**
   * Verifica o interesse do handler nessa notificao. Retorna true quando a
   * <code>NotificationData</code>  uma instncia de
   * <code>CommandNotificationData</code>.
   * 
   * @param data a notificao.
   * @return se o handler tem interesse na notificao.
   */
  private boolean isInterested(final Notification data) {
    return (data instanceof CommandNotification);
  }

  /**
   * Obtm a instncia nica do <code>CommandNotificationHandler</code>.
   * 
   * @return a instncia nica do <code>CommandNotificationHandler</code>.
   */
  public static CommandNotificationHandler getInstance() {
    return instance;
  }

  /**
   * Indica se a RemoteTask deve fazer uma nova atualizao na rvore de
   * projetos (devido a uma nova notificao).
   * 
   * @return true se deve fazer uma nova atualizao.
   */
  private synchronized boolean hasToUpdate() {
    if (updateProjectTree) {
      updateProjectTree = false;
      return true;
    }
    // Se no precisa mais atualizar a rvore, termina a RemoteTask.
    runningTask = false;
    return false;
  }

  /**
   * Indica que dever ser feita uma nova atualizao na rvore de projetos e
   * informa se uma nova RemoteTask deve ser criada para fazer essa atualizao.
   * 
   * @return true se uma nova RemoteTask deve ser criada.
   */
  private synchronized boolean createNewTask() {
    updateProjectTree = true;
    if (!runningTask) {
      runningTask = true;
      return true;
    }
    return false;
  }

  /**
   * Trata o evento da chegada de uma <code>NotificationData</code>.
   * 
   * @param data informaes sobre o comando
   */
  @Override
  public void gotNotification(Notification data) {
    if (!isInterested(data)) {
      return;
    }

    final CommandNotification notification = (CommandNotification) data;
    updateProjectTree(notification);
    List<CommandListener> toRemove = new ArrayList<CommandListener>();
    synchronized (listeners) {
      for (final CommandListener listener : listeners) {
        executor.execute(new Runnable() {
          @Override
          public void run() {
            //Tratamento especfico da aplicao
            listener.notifyCommand(notification);
          }
        });

        if (listener.shouldRemove()) {
          toRemove.add(listener);
        }
      }

      // Remove os listeners que foram notificados e querem ser removidos
      // depois da notificao.
      for (CommandListener listener : toRemove) {
        listeners.remove(listener);
      }
    }
  }

  /**
   * Atualiza a rvore de projetos quando o projeto em que o comando foi
   * executado est aberto.
   * 
   * @param notification dados sobre a notificao de fim de comando
   */
  private void updateProjectTree(CommandNotification notification) {
    final DesktopFrame desktop = DesktopFrame.getInstance();
    final CommonClientProject prj = desktop.getProject();
    if (prj != null && prj.getId().equals(notification.getProjectId())) {
      if (!desktop.mustUpdateProjectTree()) {
        SwingThreadDispatcher.invokeLater(new Runnable() {
          @Override
          public void run() {
            desktop.getTree().setOutOfDate();
          }
        });
        return;
      }
      if (createNewTask()) {
        while (hasToUpdate()) {
          RemoteTask<Void> task = new RemoteTask<Void>() {
            @Override
            protected void performTask() throws Exception {
              prj.refreshTree();
            }
          };
          String prjName = prj.getName();
          task.execute(DesktopComponentFrame.getFocusedWindow(), String.format(
            LNG.get("CommandNotificationHandler.title.refresh.tree"), prjName),
            String.format(
              LNG.get("CommandNotificationHandler.msg.refresh.tree"), prjName));
        }
      }
    }
  }

  /**
   * Cadastra um novo listener  lista de listeners cadastrados.
   * 
   * @param listener listener
   */
  public void addListener(CommandListener listener) {
    this.listeners.add(listener);
  }

  /**
   * Remove um listener da lista de listeners cadastrados.
   * 
   * @param listener listener
   */
  public void removeListener(CommandListener listener) {
    this.listeners.remove(listener);
  }
}
