package csbase.client.util.filechooser.filetablepanel;

import java.awt.Color;
import java.awt.Component;
import java.awt.Font;
import java.util.Date;

import javax.swing.ImageIcon;
import javax.swing.JLabel;
import javax.swing.JTable;
import javax.swing.SwingConstants;
import javax.swing.table.DefaultTableCellRenderer;

import tecgraf.javautils.core.lng.FormatUtils;
import csbase.client.project.ClientProjectFileType;
import csbase.client.util.filechooser.images.ChooserImages;
import csbase.client.util.filechooser.util.ClientFileChooserUtils;
import csbase.logic.ClientFile;

/**
 * Renderizador da tabela de requisies
 * 
 * @author Tecgraf/PUC-Rio
 */
class ClientFileTableRenderer extends DefaultTableCellRenderer {

  /**
   * Fonte padro.
   */
  final private static Font DEFAULT_FONT = new JLabel().getFont().deriveFont(
    Font.PLAIN);

  /**
   * Modelo da tabela a ser renderizada.
   */
  private final ClientFileTableModel model;

  /**
   * Construtor.
   * @param model modelo da tabela a ser renderizada.
   */
  public ClientFileTableRenderer(ClientFileTableModel model) {
    super();
    this.model = model;
  }

  /**
   * Ajusta o label de data de modificao.
   * 
   * @param label o label
   * @param file o arquivo
   */
  private void adjustAsModifiedDate(final JLabel label, final ClientFile file) {
    setLabelAttrBasedOnHiddenOrLocked(label, file);
    if (!file.canRead()) {
      label.setText("-");
      return;
    }
    final long date = file.getModificationDate();
    if (date == 0L) {
      label.setText("-");
      return;
    }
    final String text = dateToString(date);
    label.setText(text);
    label.setHorizontalAlignment(SwingConstants.RIGHT);
  }

  /**
   * Ajusta o label de nome
   * 
   * @param label o label
   * @param file o arquivo
   */
  private void adjustAsName(final JLabel label, final ClientFile file) {
    setLabelAttrBasedOnHiddenOrLocked(label, file);
    label.setHorizontalAlignment(SwingConstants.LEFT);

    final String fileName;
    if (file.equals(model.getDirectory())) {
      fileName = ".";
    }
    else {
      fileName = file.getName();
    }
    label.setText(fileName);

    if (!file.canRead()) {
      label.setHorizontalTextPosition(SwingConstants.LEFT);
      label.setVerticalTextPosition(SwingConstants.BOTTOM);
      label.setIcon(ChooserImages.LOCKED_FILE_ICON);
      return;
    }

    label.setHorizontalTextPosition(SwingConstants.LEFT);
    label.setVerticalTextPosition(SwingConstants.CENTER);
    label.setToolTipText(null);
  }

  /**
   * Ajusta atributos de label do renderer com base nos atributos do arquivo.
   * 
   * @param label label
   * @param file arquivo envolvido
   */
  private void setLabelAttrBasedOnHiddenOrLocked(final JLabel label,
    final ClientFile file) {
    final boolean isHidden = ClientFileChooserUtils.isHiddenFile(file);
    final boolean isLocked = !file.canRead();
    Color color = Color.black;
    Font font = ClientFileTableRenderer.DEFAULT_FONT;
    if (isHidden) {
      color = Color.gray;
      font = ClientFileTableRenderer.DEFAULT_FONT.deriveFont(Font.ITALIC);
    }

    if (isLocked) {
      color = Color.gray;
    }

    label.setForeground(color);
    label.setFont(font);
  }

  /**
   * Ajusta o label de tipo
   * 
   * @param label o label
   * @param file o arquivo
   */
  private void adjustAsSize(final JLabel label, final ClientFile file) {
    setLabelAttrBasedOnHiddenOrLocked(label, file);
    if (file.isDirectory() || file.size() < 0) {
      label.setText("-");
      label.setHorizontalAlignment(SwingConstants.CENTER);
    }
    else {
      final long size = file.size();
      label.setText(FormatUtils.formatSize(size, 2));
      label.setHorizontalAlignment(SwingConstants.RIGHT);
    }

  }

  /**
   * Ajusta o label de tipo
   * 
   * @param label o label
   * @param cpf o arquivo
   */
  private void adjustAsType(final JLabel label, final ClientFile cpf) {
    final String type = cpf.getType();
    ClientProjectFileType fileType = ClientProjectFileType.getFileType(type);
    label.setToolTipText(fileType.getDescription());
    ImageIcon icon;
    if (cpf.isDirectory()) {
      icon = ClientFileChooserUtils.getDirectoryImageIcon(cpf);
    }
    else {
      if (cpf.canRead()) {
        icon = fileType.getBaseIcon();
      }
      else {
        icon = fileType.getCutIcon();
      }
    }
    label.setIcon(icon);
    label.setHorizontalAlignment(SwingConstants.CENTER);
    label.setText(null);
  }

  /**
   * Monta string formatada de data.
   * 
   * @param time data
   * @return string
   */
  private String dateToString(final long time) {
    final Date date = new Date(time);
    final String fmt = "%1$td/%1$tm/%1$ty - %1$tH:%1$tM:%1$tS";
    final String text = String.format(fmt, date);
    return text;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Component getTableCellRendererComponent(final JTable table,
    final Object value, final boolean isSelected, final boolean hasFocus,
    final int row, final int column) {

    final Component component =
      super.getTableCellRendererComponent(table, value, isSelected, hasFocus,
        row, column);

    final ClientFile clientFile = (ClientFile) value;
    if (clientFile == null) {
      return component;
    }

    final JLabel label = (JLabel) component;
    label.setForeground(Color.black);
    label.setFont(ClientFileTableRenderer.DEFAULT_FONT);
    label.setText(null);
    label.setToolTipText(null);
    label.setHorizontalAlignment(SwingConstants.CENTER);
    label.setIcon(null);

    if (column == 0) {
      adjustAsType(label, clientFile);
    }
    else if (column == 1) {
      adjustAsName(label, clientFile);
    }
    else if (column == 2) {
      adjustAsSize(label, clientFile);
    }
    else if (column == 3) {
      adjustAsModifiedDate(label, clientFile);
    }

    return label;
  }
}
