/*
 * $Id: ClientFileChooserUtils.java 176168 2016-09-22 21:12:51Z fpina $
 */
package csbase.client.util.filechooser.util;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import javax.swing.ImageIcon;

import csbase.client.ClientLocalFile;
import csbase.client.desktop.DesktopFrame;
import csbase.client.project.ClientProjectFileProperties;
import csbase.client.project.ClientProjectFileType;
import csbase.client.util.filechooser.images.ChooserImages;
import csbase.logic.ClientFile;
import csbase.logic.ClientFileType;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;

/**
 * Utilitrio
 *
 * @author Tecgraf/PUC-Rio
 */
public class ClientFileChooserUtils {

  /**
   * Consulta o home local do usurio.
   *
   * @return o home
   */
  public static final ClientLocalFile getLocalHomeDirectory() {
    final String prop = System.getProperty("user.home");
    if (prop == null) {
      return null;
    }
    final File home = new File(prop);
    if (!home.exists()) {
      return null;
    }
    final ClientLocalFile localHomeFile = new ClientLocalFile(home);
    return localHomeFile;
  }

  /**
   * Consulta o root local.
   *
   * @return o root.
   */
  public static final List<ClientLocalFile> getLocalRootDirectories() {
    List<ClientLocalFile> rootFiles = new ArrayList<ClientLocalFile>();
    File[] roots = File.listRoots();
    for (File file : roots) {
      rootFiles.add(new ClientLocalFile(file));
    }
    Collections.sort(rootFiles, new Comparator<ClientLocalFile>() {
      @Override
      public int compare(ClientLocalFile f1, ClientLocalFile f2) {
        return f1.getStringPath().compareTo(f2.getStringPath());
      }
    });
    return rootFiles;
  }

  /**
   * Consulta o root local.
   *
   * @param currentDir o diretrio corrente.
   *
   * @return o root.
   */
  public static final ClientLocalFile getLocalRootDirectory(
    ClientFile currentDir) {
    if (currentDir == null || currentDir
      .getClientFileType() != ClientFileType.LOCAL) {
      return getLocalRootDirectories().iterator().next();
    }
    ClientLocalFile elem = (ClientLocalFile) currentDir;
    while (elem.getParent() != null) {
      elem = elem.getParent();
    }
    return elem;
  }

  /**
   * Consulta o home de projeto do usurio.
   *
   * @return o home
   */
  public static final ClientFile getProjectHomeDirectory() {
    final DesktopFrame desktop = DesktopFrame.getInstance();
    final CommonClientProject project = desktop.getProject();
    if (project == null) {
      return null;
    }
    final ClientFile directory = project.getRoot();
    return directory;
  }

  /**
   * Indica se arquivo  o home local do usurio.
   *
   * @param file o diretrio.
   * @return indicaitvo
   */
  final public static boolean isLocalHomeDirectory(final ClientFile file) {
    final ClientLocalFile home = ClientFileChooserUtils.getLocalHomeDirectory();
    if (home == null) {
      return false;
    }
    if (!(file instanceof ClientLocalFile)) {
      return false;
    }
    ClientLocalFile localFile = (ClientLocalFile) file;
    return home.equals(localFile);
  }

  /**
   * Indica se arquivo  o home local do usurio.
   *
   * @param file o diretrio.
   * @return indicaitvo
   */
  final public static boolean isProjectHomeDirectory(final ClientFile file) {
    final ClientFile home = ClientFileChooserUtils.getProjectHomeDirectory();
    if (home == null) {
      return false;
    }
    return home.equals(file);
  }

  /**
   * Consulta indica se arquivo  root local.
   *
   * @param file o diretrio.
   * @return indicativo
   */
  final public static boolean isLocalRootDirectory(final ClientFile file) {
    if (file.getClientFileType() == ClientFileType.LOCAL) {
      for (ClientFile root : ClientFileChooserUtils.getLocalRootDirectories()) {
        if (root.equals(file)) {
          return true;
        }
      }
    }
    return false;
  }

  /**
   * Retorna uma imagem associada ao diretrio.
   *
   * Convenes:<br>
   * - Diretrios ocultos so cinza; <br>
   * - Diretrios sem permisso de leitura so cinza; <br>
   * - Diretrios sem permisso de execuo so cinza com cadeado amarelo; <br>
   * - Diretrios sem nenhuma permisso so cinza com cadeado vermelho.
   *
   * @param file o file
   * @return a imagem.
   */
  final public static ImageIcon getDirectoryImageIcon(final ClientFile file) {
    if (isLocalHomeDirectory(file)) {
      return ChooserImages.LOCAL_HOME_ICON;
    }
    if (isProjectHomeDirectory(file)) {
      return ChooserImages.PROJECT_HOME_ICON;
    }
    if (isLocalRootDirectory(file)) {
      return ChooserImages.LOCAL_ROOT_ICON;
    }

    if (file.getClientFileType() == ClientFileType.REMOTE) {
      ClientProjectFile clientProjectFile = (ClientProjectFile) file;
      return ClientProjectFileProperties.getImageIcon(clientProjectFile);
    }

    final String type = file.getType();
    ClientProjectFileType fileType = ClientProjectFileType.getFileType(type);
    if (file.canRead()) {
      return fileType.getBaseIcon();
    }

    return fileType.getCutIcon();
  }

  /**
   * Consulta o texto a ser exibido com base em um diretrio.
   *
   * @param directory o diretrio
   * @return o texto
   */
  public static String getDirectoryName(ClientFile directory) {
    if (isProjectHomeDirectory(directory)) {
      return directory.getName() + "/";
    }
    if (isLocalRootDirectory(directory)) {
      return directory.getStringPath();
    }
    return directory.getName();
  }

  /**
   * Indica se arquivo  considerado oculto.
   *
   * @param file arquivo
   * @return indica se arquivo  oculto (pelo nome).
   */
  public static boolean isHiddenFile(ClientFile file) {
    final String name = file.getName();
    return name.startsWith(".");
  }
}
