package csbase.client.applications.fileexchanger.panels.detailpanel;

import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.awt.geom.Point2D;
import java.util.ArrayList;

import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingConstants;

import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.NumberAxis;
import org.jfree.chart.plot.PlotOrientation;
import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.renderer.xy.XYLineAndShapeRenderer;
import org.jfree.data.xy.XYSeries;
import org.jfree.data.xy.XYSeriesCollection;

import tecgraf.javautils.core.lng.FormatUtils;
import tecgraf.javautils.gui.BorderUtil;
import tecgraf.javautils.gui.GBC;
import csbase.client.applications.fileexchanger.FileExchanger;
import csbase.client.applications.fileexchanger.logic.Exchange;
import csbase.client.applications.fileexchanger.panels.core.FileExchangerPanel;

/**
 * Painel de operaes
 * 
 * @author Tecgraf/PUC-Rio
 */
public class DetailPanel extends FileExchangerPanel {

  /**
   * Texto de tempo transcorrido
   */
  final private JTextField rateText = new JTextField();

  /**
   * Texto de tempo restante estimado
   */
  final private JTextField sizeText = new JTextField();

  /**
   * grfico JFreeChart
   */
  final private JFreeChart jchart;

  /**
   * Painel do grfico.
   */
  final private ChartPanel chartPanel;

  /**
   * Coleo de sries geral exibida no grfico
   */
  final private XYSeriesCollection internalSeriesCollection =
    new XYSeriesCollection();

  /**
   * Atualizao.
   * 
   * @param exchanges listas das operaes.
   */
  public void updateData(final ArrayList<Exchange> exchanges) {
    updateDataTexts(exchanges);
    updateDataGraphs(exchanges);
  }

  /**
   * Atualizao.
   * 
   * @param exchanges listas das operaes.
   */
  private void updateDataGraphs(final ArrayList<Exchange> exchanges) {
    internalSeriesCollection.removeAllSeries();
    for (Exchange exchange : exchanges) {
      final String label = exchange.getLabel();
      final XYSeries series = new XYSeries(label, false, true);
      final ArrayList<Point2D.Double> points = exchange.getPoints();
      if (points == null) {
        continue;
      }
      final int size = points.size();
      for (int idx = 0; idx < size - 1; idx++) {
        final Point2D.Double pt1 = points.get(idx);
        final Point2D.Double pt2 = points.get(idx + 1);
        final double dt = pt2.x - pt1.x;
        final double ds = pt2.y - pt1.y;
        final double rate = ds / dt;
        final double t = (pt1.x + pt2.x) / 2.0;
        series.add(t, rate);
      }
      internalSeriesCollection.addSeries(series);
    }
  }

  /**
   * Atualizao.
   * 
   * @param exchanges listas das operaes.
   */
  private void updateDataTexts(final ArrayList<Exchange> exchanges) {
    if (exchanges == null || exchanges.size() == 0) {
      rateText.setText(null);
      sizeText.setText(null);
      return;
    }

    double rateAvg = 0.0;
    long sizeTotal = 0;
    for (Exchange exchange : exchanges) {
      rateAvg += exchange.getGlobalTransferRateMbSec();
      sizeTotal += exchange.getTotalTransferSize();
    }
    rateAvg /= exchanges.size();

    rateText.setText(String.format("%.2f", rateAvg) + " Mb/s");
    sizeText.setText(FormatUtils.formatSize(sizeTotal, 2));
  }

  /**
   * Monta o painel de tempos.
   * 
   * @return o painel
   */
  private JPanel buildTextPanel() {
    rateText.setEditable(false);
    sizeText.setEditable(false);

    final Dimension preferredSize = new Dimension(170, 22);
    rateText.setHorizontalAlignment(SwingConstants.RIGHT);
    sizeText.setHorizontalAlignment(SwingConstants.RIGHT);
    rateText.setPreferredSize(preferredSize);
    sizeText.setPreferredSize(preferredSize);

    final JLabel rateLabel =
      new JLabel(getString("DetailPanel.avg.rate.label"));
    final JLabel sizeLabel =
      new JLabel(getString("DetailPanel.total.size.label"));

    final int M = 7;
    final JPanel panel = new JPanel();

    panel.setLayout(new GridBagLayout());
    panel.add(rateLabel, new GBC(0, 0).none().west().insets(M, M, 0, M));
    panel.add(rateText, new GBC(0, 1).none().west().insets(0, M, 0, M));

    panel.add(sizeLabel, new GBC(0, 2).none().west().insets(M, M, 0, M));
    panel.add(sizeText, new GBC(0, 3).none().west().insets(0, M, M, M));
    return panel;
  }

  /**
   * Construtor
   * 
   * @param fileExchanger a aplicao
   */
  public DetailPanel(final FileExchanger fileExchanger) {
    super(fileExchanger);

    final NumberAxis xAxis = new NumberAxis();
    final NumberAxis yAxis = new NumberAxis();
    xAxis.setAutoRange(true);
    yAxis.setAutoRange(true);
    xAxis.setAutoRangeIncludesZero(false);
    yAxis.setAutoRangeIncludesZero(true);
    xAxis.setAutoRangeStickyZero(true);
    yAxis.setAutoRangeStickyZero(true);

    final XYPlot plot =
      new XYPlot(internalSeriesCollection, xAxis, yAxis,
        new XYLineAndShapeRenderer(true, true));
    plot.setOrientation(PlotOrientation.VERTICAL);
    plot.setDataset(internalSeriesCollection);
    jchart = new JFreeChart(null, JFreeChart.DEFAULT_TITLE_FONT, plot, true);
    chartPanel = new ChartPanel(jchart);
    chartPanel.setPreferredSize(new Dimension(200, 100));

    setLayout(new GridBagLayout());
    add(chartPanel, new GBC(0, 0).both().weightx(7.0));
    add(buildTextPanel(), new GBC(1, 0).vertical().weightx(1.0));
    BorderUtil.setTitledBorder(this, getString("DetailPanel.frame"));
  }

}
