/*
 * $Id: SearchPanel.java 110580 2010-09-27 20:27:37Z clinio $
 */
package csbase.client.applications.notepad;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.KeyEvent;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.swing.AbstractAction;
import javax.swing.ActionMap;
import javax.swing.BorderFactory;
import javax.swing.InputMap;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.KeyStroke;
import javax.swing.UIManager;
import javax.swing.text.BadLocationException;
import javax.swing.text.Caret;
import javax.swing.text.DefaultHighlighter;
import javax.swing.text.Highlighter.Highlight;
import javax.swing.text.Highlighter.HighlightPainter;
import javax.swing.text.JTextComponent;

import tecgraf.javautils.gui.GBC;

/**
 * Painel simples para procura de um trecho de texto em uma rea de texto.
 * 
 * @author Tecgraf/PUC-Rio
 */
final class SearchPanel extends JPanel {

  /**
   * boto para fechar o painel de procura.
   */
  private class CloseButton extends JButton implements ActionListener {
    /**
     * {@inheritDoc}
     */
    @Override
    public void actionPerformed(final ActionEvent e) {
      SearchPanel.this.setVisible(false);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isFocusable() {
      return false;
    }

    /**
     * Construtor
     */
    CloseButton() {
      super(UIManager.getIcon("InternalFrame.closeIcon"));
      setBorder(BorderFactory.createEmptyBorder());
      setFocusPainted(false);
      addActionListener(this);
    }
  }

  /**
   * Atualiza altura preferencial de um componente para 'height'. A largura no
   *  afetada.
   * 
   * @param component o componente que ter sua altura preferida modificada.
   * @param height a altura
   */
  private static void setPreferredHeight(final Component component,
    final int height) {
    final Dimension prefSize = component.getPreferredSize();
    prefSize.height = height;
    component.setPreferredSize(prefSize);
  }

  /**
   * a aplicao notepad
   */
  private final Notepad application;

  /**
   * O componente com todo o contedo de texto a ser analisado.
   */
  private final JTextComponent component;

  /**
   * O campo com o texto a ser procurado.
   */
  private final JTextField searchTextField;

  /**
   * A checkbox que indica se queremos dif. maisculas de minsculas
   */
  private final JCheckBox caseSensitiveCheck;

  /**
   * Boto de procura: next
   */
  private final JButton next;

  /**
   * Boto de procura: prev
   */
  private final JButton prev;

  /**
   * indica se ltima procura foi para frente [TRUE] ou para trs [FALSE]
   */
  private boolean lastDirectionIsForward = true;

  /**
   * pintor de highlight
   */
  private final HighlightPainter highlightPainter =
    new DefaultHighlighter.DefaultHighlightPainter(
      UIManager.getColor("TextArea.selectionBackground"));

  /**
   * o highlight atual - temos essa ref. para poder remover o highlight
   */
  private Highlight highlight;

  /**
   * Verifica se a seleo atual est correta, com o 'marcador' maior que a
   * 'posio' do cursor, ou seja, com o cursor no incio da seleo, e no no
   * fim dela, visando facilitar novas procuras. Alm disso, esse mtodo fora
   * com que a rea de texto receba o foco antes de receber uma chamada para uma
   * nova 'seleo' [trecho encontrado pela busca]. Isso  necessrio pois temos
   * outro componente de texto na mesma janela [o de busca], que quando est com
   * foco impede a seleo na rea de texto.
   */
  private void checkSelection() {
    final Caret caret = component.getCaret();
    final int dot = caret.getDot();
    final int mark = caret.getMark();

    if (dot > mark) {
      // re-selecionar de 'trs pra frente', para as prximas procuras:
      select(mark, dot);
    }
  }

  /**
   * Traz o foco para o campo com o texto a ser procurado.
   */
  void editSearchString() {
    searchTextField.requestFocusInWindow();
    searchTextField.selectAll();
  }

  /**
   * Procura um texto e retorna o ndice inicial, ou '-1' para no encontrado.
   * 
   * @param contentToSearch - contedo onde procurar o texto.
   * @param pattern - trecho de texto a ser encontrado.
   * @param startIndex - indica a partir de onde procurar. Se houver um 'match'
   *        exatamente em startIndex, ele ser escolhido, ento deve-se chamar
   *        com 'caretPosition' +1 ou -1 para que a procura 'se mova'.
   * @param backwards - indica para procurarmos em posies anteriores  posio
   *        do cursor [TRUE], ou em posies posteriores [FALSE].
   * @param caseSensitive - indica que devemos diferenciar maisculas de
   *        minsculas.
   * @return ndice do incio do trecho encontrado, ou -1 se nada foi
   *         encontrado.
   */
  private int find(final String contentToSearch, final String pattern,
    final int startIndex, final boolean backwards, final boolean caseSensitive) {

    if (caseSensitive) {
      if (backwards) {
        return contentToSearch.lastIndexOf(pattern, startIndex);
      }
      return contentToSearch.indexOf(pattern, startIndex);
    }

    // ignore case !!!
    if (backwards) {
      return findBackwardsIgnoreCase(contentToSearch, pattern, startIndex);
    }
    return findForwardIgnoreCase(contentToSearch, pattern, startIndex);
  }

  /**
   * Procura um texto e o seleciona, caso seja encontrado.
   * 
   * @param contentToSearch - contedo onde procurar o texto.
   * @param pattern - trecho de texto a ser encontrado.
   * @param startIndex - indica a partir de onde procurar. Se houver um 'match'
   *        exatamente em startIndex, ele ser escolhido, ento deve-se chamar
   *        com 'caretPosition' +1 ou -1 para que a procura 'se mova'.
   * @param backwards - indica para procurarmos em posies anteriores  posio
   *        do cursor [TRUE], ou em posies posteriores [FALSE].
   * @param caseSensitive indicativo de case sensitive
   */
  private void findAndSelect(final String contentToSearch,
    final String pattern, final int startIndex, final boolean backwards,
    final boolean caseSensitive) {

    // se o trecho de texto a ser encontrado for 'ruim', retornar ...
    if (pattern == null || pattern.length() == 0) {
      application.getApplicationFrame().getStatusBar().clearStatus();
      return;
    }

    // se o contedo for menor que o trecho de procura, no temos o que fazer !
    if (contentToSearch.length() < pattern.length()) {
      application.getApplicationFrame().getStatusBar().clearStatus();
      return;
    }

    int index =
      find(contentToSearch, pattern, startIndex, backwards, caseSensitive);

    if (index >= 0) {
      // texto encontrado ... selecionar !
      select(index, index + pattern.length());
      final String info =
        "\"" + pattern + "\" " + application.getString("SearchPanel.msg.found");
      application.getApplicationFrame().getStatusBar().setText(info);
      return;
    }

    // texto no encontrado ... fazer 'wrap' e tentar do incio [ou do fim]
    if (!backwards && startIndex == 0 || backwards
      && startIndex == contentToSearch.length() - 1) {
      // no adianta wrap, j estava no comeo ou no fim ...
      final String info = application.getString("SearchPanel.msg.notFound");
      application.getApplicationFrame().getStatusBar().setText(info);
      return;
    }

    int initIdx;
    // wrap !!!
    if (!backwards) {
      initIdx = 0;
    }
    else {
      initIdx = contentToSearch.length() - 1;
    }

    index = find(contentToSearch, pattern, initIdx, backwards, caseSensitive);

    if (index >= 0) {
      // texto encontrado depois do wrap ... selecionar e indicar wrap !
      select(index, index + pattern.length());
      final String info =
        String.format(
          "\"%s\" %s - %s",
          pattern,
          application.getString("SearchPanel.msg.found"),
          application.getString("SearchPanel.msg.wrapped"
            + (backwards ? ".backwards" : "")));

      application.getApplicationFrame().getStatusBar().setText(info);
    }
    else {
      final String info =
        String.format(
          "%s - %s",
          application.getString("SearchPanel.msg.notFound"),
          application.getString("SearchPanel.msg.wrapped"
            + (backwards ? ".backwards" : "")));

      application.getApplicationFrame().getStatusBar().setText(info);
    }
  }

  /**
   * Procura o trecho de cdigo 'pattern' no texto 'contentToSearch' e retorna
   * seu ndice em tal texto, ou -1 caso 'pattern' no tenha sido encontrado.
   * Por ser procura 'para trs', o trecho encontrado  o mais prximo de
   * 'startIndex' que seja anterior ao mesmo.
   * 
   * @param contentToSearch - o texto onde faremos a busca.
   * @param pattern - o trecho a ser encontrado.
   * @param startIndex - a partir de onde comear a procura.
   * @return o ndice no string, ou -1 caso 'pattern' no tenha sido encontrado.
   */
  private int findBackwardsIgnoreCase(final String contentToSearch,
    final String pattern, final int startIndex) {

    final char lower = Character.toLowerCase(pattern.charAt(0));
    final char upper = Character.toUpperCase(pattern.charAt(0));

    final int pattLen = pattern.length();
    char charAt;
    for (int i = startIndex; i >= 0; i--) {
      charAt = contentToSearch.charAt(i);
      if (charAt == lower || charAt == upper) {
        if (contentToSearch.regionMatches(true, i, pattern, 0, pattLen)) {
          return i;
        }
      }
    }

    return -1;
  }

  /**
   * Procura o trecho de cdigo 'pattern' no texto 'contentToSearch' e retorna
   * seu ndice em tal texto, ou -1 caso 'pattern' no tenha sido encontrado.
   * 
   * @param contentToSearch - o texto onde faremos a busca.
   * @param pattern - o trecho a ser encontrado.
   * @param startIndex - a partir de onde comear a procura.
   * @return o ndice no string, ou -1 caso 'pattern' no tenha sido encontrado.
   */
  private int findForwardIgnoreCase(final String contentToSearch,
    final String pattern, final int startIndex) {
    final Pattern patt =
      Pattern.compile(pattern, Pattern.CASE_INSENSITIVE | Pattern.LITERAL);
    final Matcher matcher = patt.matcher(contentToSearch);
    return matcher.find(startIndex) ? matcher.start() : -1;
  }

  /**
   * Encontra e marca o prximo 'match'.
   */
  void findNext() {
    lastDirectionIsForward = true;
    checkSelection();
    int caretPosition = component.getCaretPosition();
    final String text = component.getText();
    if (caretPosition > text.length() - 1) {
      caretPosition = text.length() - 1;
    }
    findAndSelect(text, searchTextField.getText(), caretPosition + 1, false,
      caseSensitiveCheck.isSelected());
  }

  /**
   * Encontra e marca o 'match' anterior.
   */
  void findPrevious() {
    lastDirectionIsForward = false;
    checkSelection();
    final int caretPosition = component.getCaretPosition();
    // se vier zero, vamos tentar uma procura iniciando de '-1',
    // mas isso no  um problema ... logo vai rolar o wrap !
    findAndSelect(component.getText(), searchTextField.getText(),
      caretPosition - 1, true, caseSensitiveCheck.isSelected());
  }

  /**
   * Pinta um trecho de texto, destacando-o.
   * 
   * @param start - ndice de incio.
   * @param end - ndice final, no incluso.
   */
  private void highlight(final int start, final int end) {
    removeHighlight(); // removendo o anterior
    try {
      highlight =
        (Highlight) component.getHighlighter().addHighlight(start, end,
          this.highlightPainter);
    }
    catch (final BadLocationException e) {
    }
  }

  /**
   * Remove o destaque do trecho de texto.
   */
  private void removeHighlight() {
    if (highlight != null) {
      component.getHighlighter().removeHighlight(this.highlight);
      highlight = null;
    }
  }

  /**
   * Repete a procura recm realizada, para frente ou para trs.
   */
  void searchAgain() {
    if (lastDirectionIsForward) {
      findNext();
    }
    else {
      findPrevious();
    }
  }

  /**
   * Marca o texto de 'index' at 'endIndex', porm com a seleo de trs para
   * frente [cursor fica no primeiro caractere do texto marcado - 'index'].
   * 
   * @param startIndex - incio da marcao.
   * @param endIndex - fim da marcao.
   */
  private void select(final int startIndex, final int endIndex) {
    component.setCaretPosition(endIndex);
    component.moveCaretPosition(startIndex); // dot [cursor] fica em startIndex
    highlight(startIndex, endIndex);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setVisible(final boolean visible) {
    super.setVisible(visible);
    if (visible) {
      // painel apareceu ... ver seleo atual da rea de texto e assumir como
      // trecho a ser procurado, mas s se for um trecho de apenas uma linha:
      final String selectedText = component.getSelectedText();
      if (selectedText != null) {
        if (selectedText.indexOf('\n') < 0) {
          // Ok, temos APENAS UMA linha selecionada ...
          // indicamos o texto selecionado no campo de busca:
          searchTextField.setText(selectedText);
        }
        else {
          // oops ... temos mais de uma linha ... remover seleo para procurar
          component.select(0, 0);
        }
      }

      // j que abrimos o painel de busca, o foco deve ir para ele:
      searchTextField.selectAll();
      searchTextField.requestFocusInWindow();
    }
    else {
      removeHighlight();
    }
  }

  /**
   * Construtor.
   * 
   * @param app - aplicao notepad.
   * @param comp - componente de texto com o contedo a ser analisado.
   */
  SearchPanel(final Notepad app, final JTextComponent comp) {
    super(new GridBagLayout());
    this.component = comp;
    this.application = app;

    // se o foco for para o componente de edio de texto, remover o highlight
    // para no atrapalhar
    comp.addFocusListener(new FocusAdapter() {
      @Override
      public void focusGained(final FocusEvent e) {
        removeHighlight();
      }
    });

    // criando componentes:
    final JLabel findLabel =
      new JLabel(application.getString("SearchPanel.find"));
    searchTextField = new JTextField(3);
    caseSensitiveCheck =
      new JCheckBox(application.getString("SearchPanel.caseSensitive"));

    next = new JButton(application.getString("SearchPanel.next"));
    next.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        findNext();
      }
    });

    prev = new JButton(application.getString("SearchPanel.previous"));
    prev.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        findPrevious();
      }
    });

    // todos os componentes tero a altura do JTextField de busca:
    final Dimension prefSize = searchTextField.getPreferredSize();
    final int prefHeight = prefSize.height;
    setPreferredHeight(next, prefHeight);
    setPreferredHeight(prev, prefHeight);
    setPreferredHeight(findLabel, prefHeight);
    setPreferredHeight(caseSensitiveCheck, prefHeight);

    final int top = 5, bottom = 5, left = 5;
    final Insets insets = new Insets(top, left, bottom, 0);

    add(new CloseButton(), new GBC().insets(top, left, bottom, 5));
    add(findLabel, new GBC(1, 0).insets(insets));
    add(searchTextField, new GBC(2, 0).insets(insets).horizontal());
    add(next, new GBC(3, 0).insets(insets));
    add(prev, new GBC(4, 0).insets(insets));
    add(caseSensitiveCheck, new GBC(5, 0).insets(top, 2 * left, bottom, left));

    // adicionar teclas no mapa de eventos
    final InputMap inputMap =
      getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);
    final ActionMap aMap = getActionMap();

    // inserir ao para o ENTER: procurar novamente
    final KeyStroke enterKey = KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0);
    final AbstractAction enterAction = new AbstractAction() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        searchAgain();
      }
    };
    inputMap.put(enterKey, enterKey.toString());
    aMap.put(enterKey.toString(), enterAction);

    // inserir ao para o ESC: fechar [esconder] painel de busca
    final KeyStroke escKey = KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0);
    final AbstractAction escAction = new AbstractAction() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        setVisible(false);
      }
    };
    inputMap.put(escKey, escKey.toString());
    aMap.put(escKey.toString(), escAction);
  }

}
