package csbase.client.applications.projectsmanager.actions;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import javax.swing.ImageIcon;

import csbase.client.applications.ApplicationImages;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.actions.core.ProjectsManagerAction;
import csbase.client.applications.projectsmanager.dialogs.ProjectCommentsDialog;
import csbase.client.applications.projectsmanager.dialogs.StatusDialog;
import csbase.client.applications.projectsmanager.models.ProjectSpaceAllocation;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.logic.ProjectPermissions.SharingType;
import csbase.logic.User;

/**
 * Ao que abre dilogo de incluso de comentrio no histrico de projetos.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class OpenProjectCommentsDialogAction extends ProjectsManagerAction {

  /**
   * Construtor
   * 
   * @param projectsManager A aplicao
   */
  public OpenProjectCommentsDialogAction(ProjectsManager projectsManager) {
    super(projectsManager);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void actionDone() {
    ProjectsManager projectsManager = getProjectsManager();
    List<ProjectsManagerData> projectList =
      projectsManager.getSelectedProjects();
    if (projectList == null || projectList.size() == 0) {
      return;
    }

    List<ProjectsManagerData> projects = validateProjects(projectList);
    if (projects == null) {
      return;
    }

    ProjectCommentsDialog phd =
      new ProjectCommentsDialog(projectsManager, projects);
    phd.pack();
    //    phd.center();
    phd.setVisible(true);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ImageIcon getStandardImageIcon() {
    return ApplicationImages.ICON_HISTORY_16;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected List<ProjectsManagerData> validateProjects(
    List<ProjectsManagerData> projectList) {

    List<ProjectsManagerData> prjList = new ArrayList<ProjectsManagerData>();
    prjList.addAll(projectList);

    List<ProjectsManagerData> failedProjects =
      new ArrayList<ProjectsManagerData>();

    User loggedUser = User.getLoggedUser();
    Object loggedUserId = loggedUser.getId();

    int i = 0;
    while (i < prjList.size()) {
      ProjectsManagerData pmd = prjList.get(i);
      ProjectSpaceAllocation spaceAllocation = pmd.getProjectSpaceAllocation();

      SharingType sharingType = pmd.getSharingType();
      Set<Object> usersRW = pmd.getUsersRW();

      boolean isAdmin = loggedUser.isAdmin();
      boolean isOwner = pmd.getOwnerId().equals(loggedUserId);
      boolean hasWriteAccess =
        sharingType == SharingType.ALL_RW
          || (sharingType == SharingType.PARTIAL && usersRW
            .contains(loggedUserId));

      /*
       * Para que um usurio possa escrever algo no histrico do projeto, 
       * preciso que: - O projeto possa ser aberto (no esteja aguardando
       * alocao / desalocao). - O usurio seja o administrador, ou o dono do
       * projeto, ou que o projeto seja pblico ou compartilhado e o usurio
       * tenha acesso de escrita.
       */
      if (spaceAllocation.isOpenable()
        && (isAdmin || isOwner || hasWriteAccess)) {
        i++;
      }
      else {
        failedProjects.add(pmd);
        prjList.remove(i);
      }
    }

    ProjectsManager projectsManager = getProjectsManager();
    if (prjList.size() == 0) {
      String deniedMessage =
        getString("OpenProjectCommentsDialogAction.project.selection.denied.message")
          + getString("OpenProjectCommentsDialogAction.project.requirements.message");
      StatusDialog
        .showNoneOkDialog(
          projectsManager,
          deniedMessage,
          failedProjects,
          getString("OpenProjectCommentsDialogAction.project.selection.failure.message"));
      return null;
    }
    else if (failedProjects.size() > 0) {
      String someDeniedMessage =
        getString("OpenProjectCommentsDialogAction.project.selection.some.denied.message")
          + getString("OpenProjectCommentsDialogAction.project.requirements.message");
      int res =
        StatusDialog
          .showSomeOkDialog(
            projectsManager,
            someDeniedMessage,
            prjList,
            getString("OpenProjectCommentsDialogAction.project.selection.some.failure.message"));
      if (res == 0) {
        return null;
      }
    }

    return prjList;
  }
}
