/**
 * $Id: ServerStatisticsPanel.java 152996 2014-05-30 13:26:14Z mjulia $
 */
package csbase.client.applications.serverdiagnostic.statistics;

import java.awt.Color;
import java.awt.Component;
import java.awt.GridBagLayout;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.text.DateFormatSymbols;
import java.text.MessageFormat;
import java.util.Calendar;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.swing.BorderFactory;
import javax.swing.DefaultComboBoxModel;
import javax.swing.DefaultListModel;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTable;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.plaf.basic.BasicComboBoxRenderer;

import tecgraf.javautils.gui.GBC;
import csbase.client.applications.ApplicationComponentPanel;
import csbase.client.applications.serverdiagnostic.ServerDiagnostic;
import csbase.logic.CommandFinalizationType;
import csbase.logic.algorithms.ExecutionType;
import csbase.logic.diagnosticservice.ServerStatisticsInfo;

/**
 * Painel com as estatsticas do servidor.
 * 
 * @author Tecgraf
 */
public class ServerStatisticsPanel extends
  ApplicationComponentPanel<ServerDiagnostic> {

  /**
   * Os dados estatsticos do servidor
   */
  private ServerStatisticsInfo serverStatisticsInfo;
  /**
   * Construtor de tabelas de dados estatsticos de login
   */
  private StatisticalTable<StatisticDataEntry<Integer>> loginStatisticalTable;
  /**
   * Construtor de tabelas de dados estatsticos de execuco de comandos
   * (agrupados por tipo de execuo)
   */
  private StatisticalTable<StatisticDataEntry<Integer>> executionTypeCommandStatisticalTable;
  /**
   * Construtor de tabelas de dados estatsticos de execuco de comandos
   * (agrupados por algoritmos)
   */
  private StatisticalTable<StatisticDataEntry<Integer>> algorithmCommandStatisticalTable;
  /**
   * Construtor de tabelas de dados estatsticos de execuco de comandos
   * (agrupados por usurios)
   */

  private StatisticalTable<StatisticDataEntry<Integer>> userCommandStatisticalTable;
  /**
   * Construtor de tabelas de dados estatsticos de execuco de comandos
   * (agrupados por mquina)
   */
  private StatisticalTable<StatisticDataEntry<Integer>> machineCommandStatisticalTable;
  /**
   * Construtor de tabelas de dados estatsticos de execuco de comandos
   * (agrupados pelo resultado de execuo)
   */
  private StatisticalTable<StatisticDataEntry<Integer>> resultCommandStatisticalTable;

  /**
   * Painel principal.
   */
  JSplitPane mainPanel;

  /**
   * A lista com as categorias das informaes estatsticas.
   */
  JList categotyList;

  /**
   * Construtor.
   * 
   * @param application a aplicao
   * @param serverStatisticsInfo os dados estatsticos do servidor
   */
  public ServerStatisticsPanel(final ServerDiagnostic application,
    ServerStatisticsInfo serverStatisticsInfo) {
    super(application);
    this.serverStatisticsInfo = serverStatisticsInfo;
    this.loginStatisticalTable =
      new StatisticalTable<StatisticDataEntry<Integer>>(
        getString("login.table.entity.column.name"),
        getString("login.table.value.column.name"));

    this.executionTypeCommandStatisticalTable =
      new StatisticalTable<StatisticDataEntry<Integer>>(
        getString("execution.type.table.entity.column.name"),
        getString("execution.type.table.value.column.name"));

    this.algorithmCommandStatisticalTable =
      new StatisticalTable<StatisticDataEntry<Integer>>(
        getString("algorithm.table.entity.column.name"),
        getString("algorithm.table.value.column.name"));

    this.userCommandStatisticalTable =
      new StatisticalTable<StatisticDataEntry<Integer>>(
        getString("user.table.entity.column.name"),
        getString("user.table.value.column.name"));

    this.machineCommandStatisticalTable =
      new StatisticalTable<StatisticDataEntry<Integer>>(
        getString("machine.table.entity.column.name"),
        getString("machine.table.value.column.name"));

    this.resultCommandStatisticalTable =
      new StatisticalTable<StatisticDataEntry<Integer>>(
        getString("result.table.entity.column.name"),
        getString("result.table.value.column.name"));

    buildInterface();
  }

  /**
   * Constri a interface.
   */
  private void buildInterface() {
    JPanel listPanel = buildCategoryListPanel();
    JPanel contentPanel = new JPanel();

    mainPanel =
      new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, listPanel, contentPanel);
    mainPanel.setOneTouchExpandable(true);
    mainPanel.setDividerLocation(200);
    mainPanel.setBorder(BorderFactory.createEmptyBorder());

    setLayout(new GridBagLayout());
    this.add(mainPanel, new GBC(0, 0).both().insets(5, 5, 5, 5));
    categotyList.setSelectedIndex(0);
  }

  /**
   * Constri o painel com a lista de categorias.
   * 
   * @return o painel
   */
  private JPanel buildCategoryListPanel() {
    DefaultListModel categoryListModel = new DefaultListModel();

    categoryListModel.addElement(getString("login.category"));
    categoryListModel.addElement(getString("command.execution.category"));

    this.categotyList = new JList(categoryListModel);

    categotyList.addListSelectionListener(new ListSelectionListener() {
      @Override
      public void valueChanged(ListSelectionEvent event) {
        if (!event.getValueIsAdjusting()) {
          if (((JList) event.getSource()).getSelectedValue().equals(
            getString("login.category"))) {
            mainPanel.setRightComponent(buildUserStatisticsPanel());
          }
          else if (((JList) event.getSource()).getSelectedValue().equals(
            getString("command.execution.category"))) {
            mainPanel.setRightComponent(buildCommandExecutionStatisticsPanel());
          }
          mainPanel.setDividerLocation(mainPanel.getDividerLocation());
        }
      }
    });
    JPanel categoryListPanel = new JPanel();
    categoryListPanel.setLayout(new GridBagLayout());
    categoryListPanel.add(new JScrollPane(categotyList),
      new GBC(0, 0).insets(0, 0, 0, 5).both());
    return categoryListPanel;
  }

  /**
   * Constri o painel com os dados sobre usurios.
   * 
   * @return o painel
   */
  private JPanel buildUserStatisticsPanel() {
    JPanel userStatsPanel = new JPanel();
    userStatsPanel.setLayout(new GridBagLayout());

    userStatsPanel.add(buildTimeStampPanel(), new GBC(0, 0).insets(0, 5, 5, 0)
      .horizontal().northwest().gridwidth(2));

    JLabel connectedUserLabel =
      new JLabel(MessageFormat.format(getString("connected.users.label"),
        this.serverStatisticsInfo.usersStatisticsInfo.connectedUsers.length,
        this.serverStatisticsInfo.usersStatisticsInfo.numRegisteredUsers));
    userStatsPanel.add(connectedUserLabel, new GBC(0, 1).insets(5, 5, 5, 0)
      .northwest().gridwidth(2));

    JLabel loginTryNumberLabel =
      new JLabel(
        MessageFormat
          .format(
            getString("login.try.number.label"),
            countLogin(this.serverStatisticsInfo.loginStatisticsInfo.succeededLogins)
              + countLogin(this.serverStatisticsInfo.loginStatisticsInfo.failedLogins)));

    userStatsPanel.add(loginTryNumberLabel, new GBC(0, 2).insets(5, 5, 5, 0)
      .northwest().gridwidth(2));

    JLabel succeededLoginLabel =
      new JLabel(
        MessageFormat
          .format(
            getString("succeeded.login.number.label"),
            countLogin(this.serverStatisticsInfo.loginStatisticsInfo.succeededLogins)));
    userStatsPanel.add(succeededLoginLabel, new GBC(0, 3).insets(5, 5, 5, 5)
      .northwest());
    JLabel faledLoginLabel =
      new JLabel(MessageFormat.format(getString("faled.login.label"),
        countLogin(this.serverStatisticsInfo.loginStatisticsInfo.failedLogins)));
    userStatsPanel.add(faledLoginLabel, new GBC(1, 3).insets(5, 5, 5, 0)
      .northwest());
    userStatsPanel
      .add(
        new JScrollPane(
          buildLoginStatsTable(this.serverStatisticsInfo.loginStatisticsInfo.succeededLogins)),
        new GBC(0, 4).insets(5, 5, 0, 5).both().weightx(0.5));
    userStatsPanel
      .add(
        new JScrollPane(
          buildLoginStatsTable(this.serverStatisticsInfo.loginStatisticsInfo.failedLogins)),
        new GBC(1, 4).insets(5, 5, 0, 0).both().weightx(0.5));

    return userStatsPanel;
  }

  /**
   * Constri o painel com os dados estatsicos sobre execuo de comandos.
   * 
   * @return o painel
   */
  private JPanel buildCommandExecutionStatisticsPanel() {
    final JPanel statisticsPanel = new JPanel();
    statisticsPanel.setLayout(new GridBagLayout());

    DefaultComboBoxModel groupByComboBoxModel = new DefaultComboBoxModel();
    groupByComboBoxModel
      .addElement(new FilterItem(
        getString("execution.type.filter.item.name"),
        buildExecutionTypeStatsTable(this.serverStatisticsInfo.commandExecutionStatisticsInfo.exeStats)));
    groupByComboBoxModel
      .addElement(new FilterItem(
        getString("algorithm.filter.item.name"),
        buildAlgorithmStatsTable(
          this.serverStatisticsInfo.commandExecutionStatisticsInfo.simpleAlgoExecutionStats,
          this.serverStatisticsInfo.commandExecutionStatisticsInfo.flowAlgoExecutionStats)));
    groupByComboBoxModel
      .addElement(new FilterItem(
        getString("user.filter.item.name"),
        buildUserStatsTable(this.serverStatisticsInfo.commandExecutionStatisticsInfo.userExecutionStats)));
    groupByComboBoxModel
      .addElement(new FilterItem(
        getString("machine.filter.item.name"),
        buildMachineStatsTable(this.serverStatisticsInfo.commandExecutionStatisticsInfo.sgasExecutionStats)));
    groupByComboBoxModel
      .addElement(new FilterItem(
        getString("result.filter.item.name"),
        buildResultStatsTable(
          this.serverStatisticsInfo.commandExecutionStatisticsInfo.simpleAlgoResultsStats,
          this.serverStatisticsInfo.commandExecutionStatisticsInfo.flowAlgoResultsStats)));
    JComboBox groupByComboBox = new JComboBox(groupByComboBoxModel);
    groupByComboBox.setSelectedIndex(-1);

    groupByComboBox.setRenderer(new BasicComboBoxRenderer() {

      @Override
      public Component getListCellRendererComponent(JList list, Object value,
        int index, boolean isSelected, boolean cellHasFocus) {
        Object itemText = value == null ? value : ((FilterItem) value).itemName;

        return super.getListCellRendererComponent(list, itemText, index,
          isSelected, cellHasFocus);
      }
    });

    groupByComboBox.addItemListener(new ItemListener() {
      @Override
      public void itemStateChanged(ItemEvent event) {
        if (event.getStateChange() == ItemEvent.SELECTED) {
          FilterItem selectedItem = (FilterItem) event.getItem();
          statisticsPanel.removeAll();
          statisticsPanel.add(new JScrollPane(selectedItem.table),
            new GBC(0, 0).northwest().both());
          statisticsPanel.repaint();
          statisticsPanel.revalidate();
        }
      }
    });

    JPanel commandExecutionStatisticsPanel = new JPanel();
    commandExecutionStatisticsPanel.setLayout(new GridBagLayout());
    commandExecutionStatisticsPanel.add(buildTimeStampPanel(), new GBC(0, 0)
      .insets(0, 5, 5, 0).horizontal().northwest().gridwidth(2));

    int totalSubmissions =
      countTotalSubmissions(this.serverStatisticsInfo.commandExecutionStatisticsInfo.exeStats);

    commandExecutionStatisticsPanel.add(
      new JLabel(MessageFormat.format(getString("total.submission.label"),
        totalSubmissions)), new GBC(0, 1).insets(5, 5, 5, 0).horizontal()
        .northwest().gridwidth(2));

    commandExecutionStatisticsPanel.add(
      new JLabel(getString("criteria.label")), new GBC(0, 2).insets(5, 5, 5, 5)
        .center());

    commandExecutionStatisticsPanel.add(groupByComboBox,
      new GBC(1, 2).insets(5, 0, 5, 0).northwest());
    commandExecutionStatisticsPanel.add(statisticsPanel,
      new GBC(1, 3).insets(5, 5, 0, 0).both().gridwidth(2));

    return commandExecutionStatisticsPanel;
  }

  /**
   * Conta a quantidade de execuo de comandos.
   * 
   * @param submissionStats o mapa com as informaes de execuo de comandos
   * 
   * @return a quantidade de execues
   */
  private int countTotalSubmissions(Map<?, Integer> submissionStats) {
    int total = 0;
    for (int itemCount : submissionStats.values()) {
      total += itemCount;
    }
    return total;
  }

  /**
   * Conta a quantidade de logins.
   * 
   * @param loginMap o mapa com as informaes do login
   * 
   * @return a quantidade de logins
   */
  private int countLogin(Map<String, Integer> loginMap) {
    int count = 0;
    for (Integer userLoginCount : loginMap.values()) {
      count += userLoginCount;
    }

    return count;
  }

  /**
   * Converta um mapa de dados estatsticos obtidos do servidor para uma lista
   * de {@link StatisticDataEntry}.
   * 
   * @param data o mapa
   * 
   * @return a lista
   */
  private <E> List<StatisticDataEntry<E>> convertToStatisticDataEntry(
    Map<String, E> data) {
    List<StatisticDataEntry<E>> entryList =
      new LinkedList<StatisticDataEntry<E>>();
    for (String login : data.keySet()) {
      entryList.add(new StatisticDataEntry<E>(login, data.get(login)));
    }

    return entryList;
  }

  /**
   * Constri uma tabela com dados estatsticos de login.
   * 
   * @param data o mapa com os dados de login
   * 
   * @return a tabela com os dados de login
   */
  private JTable buildLoginStatsTable(Map<String, Integer> data) {
    List<StatisticDataEntry<Integer>> entryList =
      convertToStatisticDataEntry(data);

    return this.loginStatisticalTable.builtTable(entryList);
  }

  /**
   * Cria a tabela com as estatsticas sobre o resultado da execuo dos
   * comandos.
   * 
   * @param simpleData os dados estatsticos de algoritmos simples
   * @param flowData os dados estatsticos de fluxos
   * 
   * @return a tabela
   */
  private JTable buildResultStatsTable(
    Map<CommandFinalizationType, Integer> simpleData,
    Map<CommandFinalizationType, Integer> flowData) {
    Map<String, Integer> newSimpleData = new HashMap<String, Integer>();
    for (CommandFinalizationType finalizationType : simpleData.keySet()) {
      newSimpleData.put(finalizationType.toString(),
        simpleData.get(finalizationType));
    }
    List<StatisticDataEntry<Integer>> simpleEntryList =
      convertToStatisticDataEntry(newSimpleData);

    Map<String, Integer> newFlowData = new HashMap<String, Integer>();
    for (CommandFinalizationType finalizationType : flowData.keySet()) {
      newFlowData.put(finalizationType.toString(),
        flowData.get(finalizationType));
    }
    List<StatisticDataEntry<Integer>> flowEntryList =
      convertToStatisticDataEntry(newFlowData);

    simpleEntryList.addAll(flowEntryList);

    return this.resultCommandStatisticalTable.builtTable(simpleEntryList);
  }

  /**
   * Cria a tabela com as estatsticas sobre a mquina de execuo dos comandos.
   * 
   * @param data os dados estatsticos
   * 
   * @return a tabela
   */
  private JTable buildMachineStatsTable(Map<String, Integer> data) {
    List<StatisticDataEntry<Integer>> entryList =
      convertToStatisticDataEntry(data);

    return this.machineCommandStatisticalTable.builtTable(entryList);
  }

  /**
   * Cria a tabela com as estatsticas sobre os usurio de execuo dos
   * comandos.
   * 
   * @param data os dados estatsticos
   * 
   * @return a tabela
   */
  private JTable buildUserStatsTable(Map<String, Integer> data) {
    List<StatisticDataEntry<Integer>> entryList =
      convertToStatisticDataEntry(data);

    return this.userCommandStatisticalTable.builtTable(entryList);
  }

  /**
   * Cria a tabela com as estatsticas sobre o algoritmos dos comandos.
   * 
   * @param simpleData os dados estatsticos de algoritmos simples
   * @param flowData os dados estatsticos de fluxos
   * 
   * @return a tabela
   */
  private JTable buildAlgorithmStatsTable(Map<String, Integer> simpleData,
    Map<String, Integer> flowData) {
    List<StatisticDataEntry<Integer>> simpleEntryList =
      convertToStatisticDataEntry(simpleData);
    List<StatisticDataEntry<Integer>> flowEntryList =
      convertToStatisticDataEntry(flowData);
    simpleEntryList.addAll(flowEntryList);

    return this.algorithmCommandStatisticalTable.builtTable(simpleEntryList);
  }

  /**
   * Cria a tabela com as estatsticas sobre o tipo de execuo dos comandos.
   * 
   * @param data os dados estatsticos
   * 
   * @return a tabela
   */
  private JTable buildExecutionTypeStatsTable(Map<ExecutionType, Integer> data) {
    Map<String, Integer> newData = new HashMap<String, Integer>();
    for (ExecutionType execType : data.keySet()) {
      newData.put(execType.toString(), data.get(execType));
    }
    List<StatisticDataEntry<Integer>> entryList =
      convertToStatisticDataEntry(newData);

    return this.executionTypeCommandStatisticalTable.builtTable(entryList);
  }

  /**
   * Cria o painel com a mensagem que informa o momento no qual a informao
   * exibida foi obtida no servidor.
   * 
   * @return o painel
   */
  private JPanel buildTimeStampPanel() {
    JLabel updateTimeLable = new JLabel();

    String message =
      "<html>" + getString("timestamp.message.label") + "</html>";

    Calendar startTime = Calendar.getInstance();
    startTime.setTime(serverStatisticsInfo.time);

    updateTimeLable.setText(MessageFormat.format(message,
      startTime.get(Calendar.DAY_OF_MONTH),
      getMonth(startTime.get(Calendar.MONTH)),
      String.format("%02d", startTime.get(Calendar.HOUR_OF_DAY)),
      String.format("%02d", startTime.get(Calendar.MINUTE)),
      String.format("%02d", startTime.get(Calendar.SECOND))));

    JPanel timeStampPanel = new JPanel();
    timeStampPanel.setLayout(new GridBagLayout());

    timeStampPanel.add(updateTimeLable, new GBC(0, 0).insets(0, 0, 20, 0)
      .both().northwest());
    timeStampPanel.setBorder(BorderFactory.createMatteBorder(0, 0, 1, 0,
      Color.BLACK));

    return timeStampPanel;
  }

  /**
   * Obtm o nome do ms a partir do nmero usando o locale definido na
   * aplicao.
   * 
   * @param month o nmero do ms
   * 
   * @return o nome do ms
   */
  private String getMonth(int month) {
    Locale locale = getLocale();
    if (locale != null) {
      locale = getDefaultLocale();
    }
    return DateFormatSymbols.getInstance(locale).getMonths()[month];
  }
}
