/*
 * Detalhes da ltima alterao:
 * 
 * $Id$
 */
package csbase.client.project.dialogs;

import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.rmi.RemoteException;
import java.text.MessageFormat;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.StandardDialogs;
import tecgraf.javautils.gui.StatusBar;
import csbase.client.desktop.DesktopComponentDialog;
import csbase.client.desktop.RemoteTask;
import csbase.client.openbus.OpenBusAccessPoint;
import csbase.exception.CSBaseException;
import csbase.logic.ClientProjectFile;
import csbase.logic.User;
import csbase.remote.ClientRemoteLocator;

/**
 * Representa um dilogo de agendamento de atualizaes de um arquivo.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class UpdateFileDialog extends DesktopComponentDialog {
  /**
   * O arquivo objeto da atualizao.
   */
  private ClientProjectFile file;
  /**
   * Campo texto utilizado para se obter o intervalo da atualizao.
   */
  private JTextField intervalField;
  /**
   * Boto responsvel por parar as atualizaes.
   */
  private JButton stopButton;
  /**
   * Responsvel por indicar se o usurio deseja receber notificaes sobre a
   * atualizao do arquivo.
   */
  private JCheckBox notificationBox;

  /**
   * Cria o dilogo.
   * 
   * @param owner A janela dona do dilogo.
   * @param file O arquivo cuja atualizao ser agendada.
   */
  public UpdateFileDialog(JFrame owner, ClientProjectFile file) {
    super(owner, MessageFormat.format(getString("title"), new Object[] { file
      .getName() }));
    this.file = file;
    this.createGui();
  }

  /**
   * Cria a interface grfica.
   */
  private void createGui() {
    this.setModal(true);
    this.setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
    this.addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(WindowEvent e) {
        close();
      }
    });

    // Elementos
    JLabel intervalLabel = new JLabel(getString("interval"));
    long interval = this.file.getUpdateInterval();
    String intervalText = (interval > 0) ? String.valueOf(interval) : "";
    intervalField = new JTextField(intervalText, 5);
    JLabel secondsLabel = new JLabel(getString("seconds"));
    notificationBox = new JCheckBox(getString("notification"));
    String text =
      (interval > 0) ? MessageFormat.format(getString("update.status.user"),
        file.getUpdateUserLogin()) : getString("update.status.none");
    StatusBar statusBar = new StatusBar();
    statusBar.setText(text);

      // Composio
      JPanel mainPanel = new JPanel(new GridBagLayout());
      JPanel intervalPanel = new JPanel(new GridBagLayout());
      intervalPanel.add(intervalLabel, new GBC(0, 0).none().insets(0, 0, 0, 6)
        .west());
      intervalPanel.add(intervalField, new GBC(1, 0).horizontal().insets(0, 6, 0,
        6).west());
      intervalPanel.add(secondsLabel, new GBC(2, 0).none().insets(0, 6, 0, 0)
        .west());

      mainPanel.add(intervalPanel, new GBC(0, 0).west().horizontal().insets(11,
        11, 6, 11));
      mainPanel.add(notificationBox, new GBC(0, 1).west().horizontal().insets(6,
        11, 6, 11));
      mainPanel.add(makeButtonPanel(), new GBC(0, 2).center().horizontal()
        .insets(6, 11, 6, 11));

    JPanel statusPanel = new JPanel(new GridBagLayout());
      statusPanel.add(statusBar, new GBC(0, 0).horizontal().insets(1, 1, 1, 1)
        .west());
      statusPanel.setBorder(BorderFactory.createLoweredBevelBorder());
      mainPanel.add(statusPanel, new GBC(0, 3).center().horizontal().insets(6, 0,
      0, 0));

      this.getContentPane().add(mainPanel);
      this.setResizable(false);
      this.pack();
      this.center(this.getOwner());
  }

  /**
   * Cria o painel de botes.
   * 
   * @return O painel de botes.
   */
  private JPanel makeButtonPanel() {
    JPanel panel = new JPanel();
    String startButtonTitle;
    if (this.file.getUpdateInterval() == 0) {
      startButtonTitle = getString("start");
    }
    else {
      startButtonTitle = getString("update");
    }
    JButton startButton = (JButton) panel.add(new JButton(startButtonTitle));
    startButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        if (handleStartAction()) {
          close();
        }
      }
    });

    this.stopButton = (JButton) panel.add(new JButton(getString("stop")));
    this.stopButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        if (handleStopAction()) {
          close();
        }
      }
    });

    JButton cancelButton = (JButton) panel.add(new JButton(getString("close")));
    cancelButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        close();
      }
    });

    GUIUtils.matchPreferredSizes(new JComponent[] { startButton,
      this.stopButton, cancelButton });
    this.getRootPane().setDefaultButton(startButton);
    if (this.file.getUpdateInterval() == 0) {
      this.stopButton.setEnabled(false);
    }
    else {
      this.stopButton.setEnabled(true);
    }
    return panel;
  }

  /**
   * Executa a ao do boto iniciar.
   * 
   * @return {@code true} caso a aao tenha sido executada, ou {@code false},
   *         caso contrrio.
   */
  private boolean handleStartAction() {
    String text = this.intervalField.getText();
    int interval;
    try {
      interval = Integer.parseInt(text);
    }
    catch (NumberFormatException e) {
      interval = -1;
    }
    if (interval <= 0) {
      StandardDialogs.showErrorDialog(this, MessageFormat.format(
        getString("invalid.interval.title"), new Object[] { this.getTitle() }),
        getString("invalid.interval.message"));
      return false;
    }
    final int updateInterval = interval;
    RemoteTask<Boolean> task = new RemoteTask<Boolean>() {
      @Override
      protected void performTask() throws RemoteException, CSBaseException {
        OpenBusAccessPoint oap = OpenBusAccessPoint.getInstance();
        oap.init();
        byte[] chain = oap.makeChain();

        ClientRemoteLocator.projectService.startUpdate(file.getProjectId(),
          file.getPath(), updateInterval, notificationBox.isSelected(), chain);
        file.setUpdateInfo(User.getLoggedUser().getLogin(), updateInterval);
      }
    };
    if (task.execute(this.getOwner(), getString("task.title"), MessageFormat
      .format(getString("task.message"), new Object[] { this.file.getName() }))) {
      this.stopButton.setEnabled(true);
      StandardDialogs.showInfoDialog(this, MessageFormat.format(
        getString("update.start.title"), this.file.getName()),
        getString("update.start.success.message"));
      return true;
    }
    StandardDialogs.showInfoDialog(this, MessageFormat.format(
      getString("update.start.title"), this.file.getName()),
      getString("update.start.error.message"));
    return false;
  }

  /**
   * Executa a ao do boto parar.
   * 
   * @return {@code true} caso a aao tenha sido executada, ou {@code false},
   *         caso contrrio.
   */
  private boolean handleStopAction() {
    RemoteTask<Boolean> task = new RemoteTask<Boolean>() {
      @Override
      protected void performTask() throws RemoteException {
        ClientRemoteLocator.projectService.stopUpdate(file.getProjectId(), file
          .getPath());
        file.setUpdateInfo(null, 0);
      }
    };
    if (task.execute(this.getOwner(), getString("task.title"), MessageFormat
      .format(getString("task.message"), new Object[] { this.file.getName() }))) {
      StandardDialogs.showInfoDialog(this, MessageFormat.format(
        getString("update.stop.title"), this.file.getName()),
        getString("update.stop.success.message"));
      this.stopButton.setEnabled(false);
      return true;
    }
    StandardDialogs.showInfoDialog(this, MessageFormat.format(
      getString("update.stop.title"), this.file.getName()),
      getString("update.stop.error.message"));
    return false;
  }

  /**
   * Obtm um texto internacionalizado
   * 
   * @param key A chave
   * @return o texto internacioalizado
   */
  private static String getString(String key) {
    // No  this.getClass().getSimpleName() pois precisa ser esttico
    return LNG.get("UpdateFileDialog." + key);
  }
}
