package csbase.logic.algorithms.flows.configurator;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import tecgraf.javautils.core.io.FileUtils;
import csbase.exception.BugException;
import csbase.exception.ParseException;
import csbase.exception.algorithms.ParameterNotFoundException;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import csbase.logic.algorithms.CommandLineBuilder;
import csbase.logic.algorithms.CommandLineContext;
import csbase.logic.algorithms.CommandScript;
import csbase.logic.algorithms.EnvironmentVariable;
import csbase.logic.algorithms.ExecutionLocation;
import csbase.logic.algorithms.ExecutionType;
import csbase.logic.algorithms.FlowCommandLineContext;
import csbase.logic.algorithms.flows.Flow;
import csbase.logic.algorithms.flows.FlowLink;
import csbase.logic.algorithms.flows.FlowNode;
import csbase.logic.algorithms.flows.LinkParameter;
import csbase.logic.algorithms.flows.NodeParameter;
import csbase.logic.algorithms.parameters.InputFileParameter;
import csbase.logic.algorithms.parameters.InputURLParameter;
import csbase.logic.algorithms.parameters.OutputFileParameter;
import csbase.logic.algorithms.parameters.OutputURLParameter;
import csbase.logic.algorithms.parameters.FileURLValue;
import csbase.logic.algorithms.validation.FlowNodeValidationResult;
import csbase.logic.algorithms.validation.Validation;
import csbase.logic.algorithms.validation.ValidationContext;
import csbase.logic.algorithms.validation.ValidationSuccess;

/**
 * <p>
 * Configurador de Fluxo de Algoritmos:
 * </p>
 *
 * <p>
 * Um fluxo de algoritmos  um multigrafo orientado cujos os vrtices so os ns
 * de algoritmo e as aretas so conexes entre parmetro do tipo arquivo destes
 * algoritmos.
 * </p>
 *
 * @author lmoreira
 */
public final class FlowAlgorithmConfigurator extends AlgorithmConfigurator {
  /**
   * Caractere de aspas
   */
  private static final String QUOTE = "\"";
  /**
   * O ndice da captura da expresso {@link #PARAMETER_NAME_REGEX} que se
   * refere ao identificador do n.
   */
  private static final int NODE_ID_INDEX = 1;
  /**
   * A expresso regular que  utilizada para entender os nomes codificados para
   * parmetros nos mtodos: {@link #getParameterNames()},
   * {@link #getParameterLabel(String)}, {@link #getParameterType(String)},
   * {@link #getParameterValue(String)},
   * {@link #setParameterValue(String, String)}.
   */
  private static final String PARAMETER_NAME_REGEX = "([0-9]*)\\.(.*)";

  /**
   * O ndice da captura da expresso {@link #PARAMETER_NAME_REGEX} que se
   * refere ao nome do parmetro.
   */
  private static final int REAL_PARAMETER_NAME_INDEX = 2;

  /**
   * Uma estrutura de dados que descreve o fluxo.
   */
  private Flow flow;

  /**
   * Arquivo de log que armazena o identificador do n responsvel por erro na
   * execuo do fluxo (se houver).
   */
  private FileURLValue guiltyNodeLog;

  /**
   * Os ns indexados pelo identificador.
   */
  private Map<Integer, Node> nodesById;

  /**
   * Determina se  possvel executar com a configurao atual (levando em
   * consideraco possveis desvios feitos no fluxo)
   */
  private boolean canBeExecuted;

  /**
   * Cria um configurador.
   *
   * @param algorithmVersion A verso do algoritmo (No aceita {@code null}).
   * @param flow A estrutura que descreve o fluxo (No aceita {@code null}).
   */
  public FlowAlgorithmConfigurator(AlgorithmVersionInfo algorithmVersion,
    Flow flow) {
    super(ConfiguratorType.FLOW, algorithmVersion, null, ExecutionType.SIMPLE,
      ExecutionLocation.BINARY_DIR, "FLOWABBR", false, null);
    setFlow(flow);
  }

  /**
   * Cria um configurador.
   *
   * @param filePath O caminho para o arquivo que descreve o fluxo (No aceita
   *        {@code null}).
   *
   * @param flow A estrutura que descreve o fluxo (No aceita {@code null}).
   */
  public FlowAlgorithmConfigurator(String filePath, Flow flow) {
    super(ConfiguratorType.FLOW, filePath, null, ExecutionType.SIMPLE,
      ExecutionLocation.BINARY_DIR, "FLOWABBR", false, null);
    setFlow(flow);
  }

  /**
   * Cria um configurador.
   *
   * @param flow A estrutura que descreve o fluxo (No aceita {@code null}).
   */
  public FlowAlgorithmConfigurator(Flow flow) {
    super(ConfiguratorType.FLOW, null, ExecutionType.SIMPLE,
      ExecutionLocation.BINARY_DIR, "FLOWABBR", false, null);
    setFlow(flow);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<Object, Object> exportValues() {
    Map<Object, Object> values = new HashMap<Object, Object>();
    for (Node node : this.nodesById.values()) {
      values.put(node.getId(), node.getConfigurator().exportValues());
    }
    return values;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<String> getRequirements() {
    Set<String> requirementsSet = new HashSet<String>();
    for (Node node : this.nodesById.values()) {
      Set<String> requirements = node.getConfigurator().getRequirements();
      requirementsSet.addAll(requirements);
    }
    return requirementsSet;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<String[]> getBinaryDirectoriesAsArray(String platformId) {
    Set<String[]> binaryDirectories = new HashSet<String[]>();
    for (Node node : this.nodesById.values()) {
      AlgorithmConfigurator configurator = node.getConfigurator();
      binaryDirectories.addAll(configurator
        .getBinaryDirectoriesAsArray(platformId));
    }
    return binaryDirectories;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<String> getBinaryDirectories(String platformId, char fileSeparator) {
    Set<String> binaryDirectories = new HashSet<String>();
    for (Node node : this.nodesById.values()) {
      AlgorithmConfigurator configurator = node.getConfigurator();
      binaryDirectories.addAll(configurator.getBinaryDirectories(platformId,
        fileSeparator));
    }
    return binaryDirectories;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<FileURLValue> getInputDirectories() {
    Set<FileURLValue> directories = new HashSet<FileURLValue>();
    for (Node node : this.nodesById.values()) {
      AlgorithmConfigurator configurator = node.getConfigurator();
      Set<FileURLValue> currentDirectories = configurator.getInputDirectories();
      directories.addAll(currentDirectories);
    }
    return Collections.unmodifiableSet(directories);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<FileURLValue> getOutputDirectories() {
    Set<FileURLValue> directories = new HashSet<FileURLValue>();
    for (Node node : this.nodesById.values()) {
      AlgorithmConfigurator configurator = node.getConfigurator();
      Set<FileURLValue> currentDirectories = configurator.getOutputDirectories();
      directories.addAll(currentDirectories);
    }
    return Collections.unmodifiableSet(directories);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<InputFileParameter> getInputFileParameters() {
    List<InputFileParameter> inputFileParameters =
      new LinkedList<InputFileParameter>();
    for (Node node : getNodes()) {
      AlgorithmConfigurator configurator = node.getConfigurator();
      inputFileParameters.addAll(configurator.getInputFileParameters());
    }
    return Collections.unmodifiableList(inputFileParameters);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<FileURLValue> getInputFiles() {
    Set<FileURLValue> inputFiles = new HashSet<FileURLValue>();
    for (Node node : this.nodesById.values()) {
      AlgorithmConfigurator configurator = node.getConfigurator();
      Set<FileURLValue> currentInputFiles = configurator.getInputFiles();
      inputFiles.addAll(currentInputFiles);
    }
    return Collections.unmodifiableSet(inputFiles);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<InputURLParameter> getInputURLParameters() {
    List<InputURLParameter> inputURLParameters =
      new LinkedList<InputURLParameter>();
    for (Node node : getNodes()) {
      AlgorithmConfigurator configurator = node.getConfigurator();
      inputURLParameters.addAll(configurator.getInputURLParameters());
    }
    return Collections.unmodifiableList(inputURLParameters);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<FileURLValue> getInputURLs() {
    Set<FileURLValue> inputURLs = new HashSet<FileURLValue>();
    for (Node node : this.nodesById.values()) {
      AlgorithmConfigurator configurator = node.getConfigurator();
      Set<FileURLValue> currentInputURLs = configurator.getInputURLs();
      inputURLs.addAll(currentInputURLs);
    }
    return Collections.unmodifiableSet(inputURLs);
  }

  /**
   * Obtm a estrutura que descreve o fluxo.
   *
   * @return a estrutura que descreve o fluxo.
   */
  public Flow getFlow() {
    return this.flow;
  }

  /**
   * Retorna o arquivo de log que armazena o identificador do n responsvel por
   * erro na execuo do fluxo (se houver).
   *
   * @return o arquivo de log.
   */
  public FileURLValue getGuiltyNodeLog() {
    return guiltyNodeLog;
  }

  /**
   * Atribui o arquivo de log que armazena o identificador do n responsvel por
   * erro na execuo do fluxo (se houver).
   *
   * @param guiltyNodeLog o arquivo de log.
   */
  public void setGuiltyNodeLog(FileURLValue guiltyNodeLog) {
    this.guiltyNodeLog = guiltyNodeLog;
  }

  /**
   * <p>
   * Obtm todos os arquivos de log.
   * </p>
   *
   * <p>
   * O conjunto retornado  imutvel (Collections.unmodifiableSet(Set)).
   * </p>
   *
   * @return o conjunto de arquivos de log (se no existir arquivo de log,
   *         retornar um conjunto vazio).
   */
  @Override
  public Set<FileURLValue> getLogFiles() {
    Set<FileURLValue> logFiles = new HashSet<FileURLValue>();
    for (Node node : getNodes()) {
      AlgorithmConfigurator configurator = node.getConfigurator();
      if (configurator.getConfiguratorType() == ConfiguratorType.FLOW) {
        FlowAlgorithmConfigurator flowConfigurator =
          (FlowAlgorithmConfigurator) configurator;
        logFiles.addAll(flowConfigurator.getLogFiles());
      }
      else {
        FileURLValue logFile = configurator.getLogFile();
        if (logFile != null) {
          logFiles.add(logFile);
        }
      }
    }
    return Collections.unmodifiableSet(logFiles);
  }

  /**
   * Obtm o nome do fluxo.
   *
   * @return O nome do fluxo ou {@code null} se ele no tiver um nome.
   */
  public String getName() {
    return this.flow.getName();
  }

  /**
   * Obtm um n de algoritmo atravs do seu identificador.
   *
   * @param id O identificador.
   *
   * @return O n ou {@code null} se ele no existir.
   */
  public Node getNode(int id) {
    return this.nodesById.get(id);
  }

  /**
   * <p>
   * Obtm o conjunto de ns deste fluxo.
   * </p>
   *
   * <p>
   * O conjunto retornado  imutvel (veja
   * {@link Collections#unmodifiableSet(Set)}).
   * </p>
   *
   * @return O conjunto (Retorna um conjunto vazio se no houver ns).
   */
  public Set<Node> getNodes() {
    Set<Node> nodes = new HashSet<Node>();
    nodes.addAll(nodesById.values());
    return Collections.unmodifiableSet(nodes);
  }

  /**
   * Determina se o fluxo contm algum n que est sendo desviado.
   *
   * @return verdadeiro se o fluxo contm pelo menos um n desviado ou falso
   *         caso contrrio.
   */
  public boolean hasBypassedNodes() {
    for (FlowNode node : flow.getNodes()) {
      if (node.isBypassed()) {
        return true;
      }
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<OutputFileParameter> getOutputFileParameters() {
    List<OutputFileParameter> outputFileParameters =
      new LinkedList<OutputFileParameter>();
    for (Node node : getNodes()) {
      AlgorithmConfigurator configurator = node.getConfigurator();
      outputFileParameters.addAll(configurator.getOutputFileParameters());
    }
    return Collections.unmodifiableList(outputFileParameters);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<FileURLValue> getOutputFiles() {
    Set<FileURLValue> outputFiles = new HashSet<FileURLValue>();
    for (Node node : this.nodesById.values()) {
      AlgorithmConfigurator configurator = node.getConfigurator();
      Set<FileURLValue> currentOutputFiles = configurator.getOutputFiles();
      outputFiles.addAll(currentOutputFiles);
    }
    return Collections.unmodifiableSet(outputFiles);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<OutputURLParameter> getOutputURLParameters() {
    List<OutputURLParameter> outputURLParameters =
      new LinkedList<OutputURLParameter>();
    for (Node node : getNodes()) {
      AlgorithmConfigurator configurator = node.getConfigurator();
      outputURLParameters.addAll(configurator.getOutputURLParameters());
    }
    return Collections.unmodifiableList(outputURLParameters);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<FileURLValue> getOutputURLs() {
    Set<FileURLValue> outputFiles = new HashSet<FileURLValue>();
    for (Node node : this.nodesById.values()) {
      AlgorithmConfigurator configurator = node.getConfigurator();
      Set<FileURLValue> currentOutputFiles = configurator.getOutputURLs();
      outputFiles.addAll(currentOutputFiles);
    }
    return Collections.unmodifiableSet(outputFiles);
  }

  /**
   * @param encodedParameterName Nome codificado do parmetro. Para uma
   *        explicao sobre o nome codificado veja {@link #getParameterNames()}
   *        .
   */
  @Override
  public String getParameterLabel(String encodedParameterName)
    throws ParameterNotFoundException {
    Integer nodeId = getNodeIdFromEncodedParameterName(encodedParameterName);
    if (nodeId == null) {
      throw new ParameterNotFoundException(encodedParameterName);
    }
    Node node = nodesById.get(nodeId);
    if (node == null) {
      throw new ParameterNotFoundException(encodedParameterName);
    }
    String realParameterName =
      getRealParameterNameFromEncodedParameterName(encodedParameterName);
    if (realParameterName == null) {
      return null;
    }
    return node.getConfigurator().getParameterLabel(realParameterName);
  }

  /**
   * <p>
   * Os nomes dos parmetros so codificados utilizando com a seguinte
   * gramtica:
   *
   * <pre>
   * &lt;nome_codificado&gt; := &lt;id_do_no&gt;&lt;nome_codificado&gt;
   * ou
   * &lt;nome_codificado&gt; := &lt;nome_do_parametro&gt;
   * </pre>
   *
   * <ul>
   * <li>nome_codificado  o nome do codificado do parmetro.</li>
   * <li>id_do_no  o identificador do n.</li>
   * <li>nome_do_parametro  o nome real do parmetro.</li>
   * </ul>
   * </p>
   */
  @Override
  public Set<String> getParameterNames() {
    Set<String> allParameterNames = new HashSet<String>();
    for (Node node : nodesById.values()) {
      AlgorithmConfigurator configurator = node.getConfigurator();
      Set<String> thisNodeParameterNames = configurator.getParameterNames();
      for (String parameterName : thisNodeParameterNames) {
        allParameterNames.add(node.getId() + "." + parameterName);
      }
    }
    return Collections.unmodifiableSet(allParameterNames);
  }

  /**
   * @param encodedParameterName Nome codificado do parmetro. Para uma
   *        explicao sobre o nome codificado veja {@link #getParameterNames()}
   *        .
   */
  @Override
  public String getParameterType(String encodedParameterName)
    throws ParameterNotFoundException {
    Integer nodeId = getNodeIdFromEncodedParameterName(encodedParameterName);
    if (nodeId == null) {
      throw new ParameterNotFoundException(encodedParameterName);
    }
    Node node = nodesById.get(nodeId);
    if (node == null) {
      throw new ParameterNotFoundException(encodedParameterName);
    }
    String realParameterName =
      getRealParameterNameFromEncodedParameterName(encodedParameterName);
    if (realParameterName == null) {
      return null;
    }
    return node.getConfigurator().getParameterType(realParameterName);
  }

  /**
   * @param encodedParameterName Nome codificado do parmetro. Para uma
   *        explicao sobre o nome codificado veja {@link #getParameterNames()}
   *        .
   */
  @Override
  public String getParameterValue(String encodedParameterName)
    throws ParameterNotFoundException {
    Integer nodeId = getNodeIdFromEncodedParameterName(encodedParameterName);
    if (nodeId == null) {
      throw new ParameterNotFoundException(encodedParameterName);
    }
    Node node = nodesById.get(nodeId);
    if (node == null) {
      throw new ParameterNotFoundException(encodedParameterName);
    }
    String realParameterName =
      getRealParameterNameFromEncodedParameterName(encodedParameterName);
    if (realParameterName == null) {
      return null;
    }
    return node.getConfigurator().getParameterValue(realParameterName);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<String> getPlatforms() {
    Set<String> platforms = new HashSet<String>();
    if (!this.nodesById.isEmpty()) {
      Iterator<Node> nodeIterator = this.nodesById.values().iterator();
      Node node = nodeIterator.next();
      AlgorithmConfigurator configurator = node.getConfigurator();
      platforms.addAll(configurator.getPlatforms());
      while (nodeIterator.hasNext()) {
        node = nodeIterator.next();
        configurator = node.getConfigurator();
        platforms.retainAll(configurator.getPlatforms());
      }
    }
    return Collections.unmodifiableSet(platforms);
  }

  /**
   * {@inheritDoc}
   */
  @SuppressWarnings("unchecked")
  @Override
  public void importValues(Map<Object, Object> values) throws ParseException {
    for (Node node : this.nodesById.values()) {
      Map<Object, Object> nodeValues =
        (Map<Object, Object>) values.get(node.getId());
      if (nodeValues != null) {
        node.getConfigurator().importValues(nodeValues);
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isEnabled() {
    for (Node node : nodesById.values()) {
      AlgorithmConfigurator configurator = node.getConfigurator();
      if (configurator != null) {
        if (configurator.isEnabled()) {
          return true;
        }
      }
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isSetDefaultValues() {
    for (Node node : this.nodesById.values()) {
      if (!node.getConfigurator().isSetDefaultValues()) {
        return false;
      }
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public CommandScript[] makeCommandLineAsScript(CommandLineContext context) {
    if (canBeRunAsSimpleCommand()) {
      String simpleCommandLine = makeSimpleCommandLine(context);
      Node node = this.nodesById.values().iterator().next();
      CommandScript simpleCommandScript =
        new CommandScript(context, simpleCommandLine, CommandScript
          .makeFileName(String.valueOf(node.getId())));
      return new CommandScript[] { simpleCommandScript };
    }

    FlowCommandLineContext flowContext = (FlowCommandLineContext) context;
    Map<Integer, String> commands = makeCommands(true, flowContext);
    List<CommandScript> scripts = new ArrayList<CommandScript>();
    Map<Integer, String> scriptCommands = new HashMap<Integer, String>();
    for (Entry<Integer, String> command : commands.entrySet()) {
      CommandLineContext nodeContext =
        flowContext.getFlowNodeContext(command.getKey());
      CommandScript script =
        new CommandScript(nodeContext, command.getValue(), CommandScript
          .makeFileName(String.valueOf(command.getKey())));
      scripts.add(script);
      scriptCommands.put(command.getKey(), script.makeCommandLine());
    }

    // Adiciona a prpria linha de comando do fluxo na primeira posio da lista.
    String flowCommandLine = makeFlowCommandLine(flowContext, scriptCommands);
    scripts.add(0, new CommandScript(flowContext, flowCommandLine));

    return scripts.toArray(new CommandScript[scripts.size()]);
  }

  /**
   * Cria a linha de comando como se este fosse um configurador simples. Deve
   * ser utilizada somente quando o fluxo possui um nico n.
   *
   * @param context O contexto para gerao da linha de comando.
   * @return A linha de comando.
   */
  private String makeSimpleCommandLine(CommandLineContext context) {
    if (nodesById.size() != 1) {
      return null;
    }
    Collection<Node> values = this.nodesById.values();
    Iterator<Node> nodeIterator = values.iterator();
    Node node = nodeIterator.next();
    AlgorithmConfigurator configurator = node.getConfigurator();
    return configurator.makeCommandLine(context);
  }

  /**
   * Determina se o fluxo pode ser rodado como um comando simples. Para isso, o
   * fluxo no pode ter mais de um n. Fluxos sem nenhum n tambm podem ser
   * considerados como comandos simples (mesmo que no possam ser executados na
   * prtica).
   *
   * @return verdadeiro se o fluxo puder ser rodado como um comando simples ou
   *         falso, caso contrrio.
   */
  public boolean canBeRunAsSimpleCommand() {
    return this.nodesById.size() <= 1;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String makeCommandLine(CommandLineContext context) {
    if (canBeRunAsSimpleCommand()) {
      return makeSimpleCommandLine(context);
    }

    FlowCommandLineContext flowContext = (FlowCommandLineContext) context;
    Map<Integer, String> commands = makeCommands(false, flowContext);
    return makeFlowCommandLine(flowContext, commands);
  }

  /**
   * Cria a parte da linha de comando que declara as variveis de ambiente
   * utilizadas e disponibilizadas pelo comando.
   *
   * @param context Contexto de execuo do comando.
   * @return A declarao de variveis de ambiente para a linha de comando.
   */
  protected String makeEnvironmentVariablesDeclaration(
    FlowCommandLineContext context) {

    CommandLineBuilder commandLineBuilder =
      new CommandLineBuilder(context.isScript());

    if (hasExitCode()) {
      String flowErrorNodeLog =
        FileUtils.joinPath(context.getFileSeparator(), context
          .getProjectRootDirectory(), context.getProjectDirectory(),
          getGuiltyNodeLog().getPath());

      commandLineBuilder.appendEnvironmentVariableDeclaration(
        EnvironmentVariable.FLOW_GUILTY_NODE_ID_LOG, flowErrorNodeLog);
    }

    String sandboxDir =
      FileUtils.joinPath(context.getFileSeparator(), context
        .getSandboxRootDirectory(), context.getSandboxDirectory());
    commandLineBuilder.appendEnvironmentVariableDeclaration(
      EnvironmentVariable.FLOW_SANDBOX_DIR, sandboxDir);
    return commandLineBuilder.toString();
  }

  /**
   * Gera a linha de comando do fluxo. O comando gerado usa named-pipes para
   * representar as conexes e permitir comunicar os dados entre os algoritmos.
   * Ele tambm solicita a execuo simultnea de todos os comandos gerados por
   * cada um dos ns de algoritmos.
   *
   * @param context O contexto com informaes para a gerao da linha de
   *        comando.
   * @param commands A lista de comandos gerados por cada um dos ns de
   *        algoritmos.
   * @return A linha de comando.
   */
  protected String makeFlowCommandLine(FlowCommandLineContext context,
    Map<Integer, String> commands) {
    Map<Integer, Map<String, String>> fromPipes = context.getFromPipes();
    Map<Integer, Map<String, String>> toPipes = context.getToPipes();
    Collection<String> tees = getTees(fromPipes, toPipes, context.isScript());

    CommandLineBuilder builder = new CommandLineBuilder(context.isScript());
    builder.append(makeEnvironmentVariablesDeclaration(context));
    builder.append(CommandLineBuilder.SHELL).append(" ./flowmonitor ");

    builder.append(QUOTE);
    for (Map<String, String> pipes : fromPipes.values()) {
      for (String pipe : pipes.values()) {
        builder.append(pipe).append(" ");
      }
    }
    for (Map<String, String> pipes : toPipes.values()) {
      for (String pipe : pipes.values()) {
        builder.append(pipe).append(" ");
      }
    }
    builder.append(QUOTE);

    builder.append(" ").append(QUOTE);
    for (int i = 0; i < tees.size(); i++) {
      builder.append(String.valueOf(-(i + 1))).append(" ");
    }
    for (Node node : this.nodesById.values()) {
      builder.append(String.valueOf(node.getId())).append(" ");
    }
    builder.append(QUOTE);

    for (String tee : tees) {
      builder.append(" ").append(QUOTE).append(tee).append(QUOTE);
    }
    for (String cmd : commands.values()) {
      builder.append(" ").append(QUOTE).append(cmd).append(QUOTE);
    }
    builder.appendEmptyLine();
    String cmd = builder.toString();
    if (!context.isScript()) {
      /*
       * Para uso no shell diretamente,  necessrio "escapar" o caracter '$'
       * utilizado nas referncias s variveis de ambiente.
       */
      cmd = CommandLineBuilder.escapeReferenceChar(cmd);
    }
    return cmd;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void resetValues() {
    for (Node node : this.nodesById.values()) {
      node.getConfigurator().resetValues();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean setDefaultInputFile(FileURLValue inputFile) {
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean setEnabled(boolean isEnabled) {
    if (isEnabled() == isEnabled) {
      return false;
    }
    for (Node node : nodesById.values()) {
      AlgorithmConfigurator configurator = node.getConfigurator();
      if (configurator != null) {
        configurator.setEnabled(isEnabled);
      }
    }
    fireWasSetEnabled();
    return true;
  }

  /**
   * @param encodedParameterName Nome codificado do parmetro. Para uma
   *        explicao sobre o nome codificado veja {@link #getParameterNames()}
   *        .
   */
  @Override
  public void setParameterValue(String encodedParameterName,
    String parameterValue) throws ParameterNotFoundException, ParseException {
    Integer nodeId = getNodeIdFromEncodedParameterName(encodedParameterName);
    if (nodeId == null) {
      throw new ParameterNotFoundException(encodedParameterName);
    }
    Node node = nodesById.get(nodeId);
    if (node == null) {
      throw new ParameterNotFoundException(encodedParameterName);
    }
    String realParameterName =
      getRealParameterNameFromEncodedParameterName(encodedParameterName);
    if (realParameterName == null) {
      throw new ParameterNotFoundException(encodedParameterName);
    }
    AlgorithmConfigurator configurator = node.getConfigurator();
    configurator.setParameterValue(realParameterName, parameterValue);

    // Altera o valor no parmetro armazenad o no Flow.
    Flow flx = getFlow();
    if (null != flx) {
      FlowNode flowNode = flx.getNode(node.getId());
      if (null != flowNode) {
        NodeParameter flowNodeParameter =
          flowNode.getParameter(realParameterName);
        if (null != flowNodeParameter) {
          flowNodeParameter.setValue(parameterValue);
        }
      }
    }
  }

  /**
   * <p>
   * Atribui o valor do arquivo que deve receber a sada padro da execuo de
   * um algoritmo representado por este configurador.
   * </p>
   * <p>
   * Este valor ser repassado para todos os ns deste fluxo.
   * </p>
   *
   * @param standardOutputFile o valor do arquivo que deve receber a sada
   *        padro.
   */
  @Override
  public void setStandardOutputFile(FileURLValue standardOutputFile) {
    flow.setStandardOutputFile(standardOutputFile);
  }

  /**
   * Obtm um conjunto imutvel contendo os arquivos de sada padro de cada n
   * que sero utilizadas durante a execuo de um fluxo.
   *
   * @return um conjunto imutvel contendo os arquivos de sada padro de cada
   *         n que sero utilizadas durante a execuo de um fluxo.
   */
  @Override
  public Set<FileURLValue> getStandardOutputFiles() {
    return flow.getStandardOutputFiles();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setExitCodeLogFile(FileURLValue exitCodeLogFile) {
    flow.setExitCodeLogFile(exitCodeLogFile);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<FileURLValue> getExitCodeLogFiles() {
    return flow.getExitCodeLogFiles();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean hasExitCode() {
    return flow.hasExitCode();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getCurrentParameterFileVersion() {
    return "1.0";
  }

  /**
   * Obtm o identificador do n a partir do nome do parmetro codificado.
   *
   * @param encodedParameterName O nome de um parmetro codificado (No pode ser
   *        {@code null}).
   *
   * @return O identificador do n ou {@code null} o texto informado no estiver
   *         no formato esperado.
   */
  private Integer getNodeIdFromEncodedParameterName(String encodedParameterName) {
    Pattern pattern = Pattern.compile(PARAMETER_NAME_REGEX);
    Matcher matcher = pattern.matcher(encodedParameterName);
    if (!matcher.matches()) {
      return null;
    }
    String nodeIdAsText = matcher.group(NODE_ID_INDEX);
    return Integer.parseInt(nodeIdAsText);
  }

  /**
   * Obtm o nome do parmetro a partir do nome do parmetro codificado.
   *
   * @param encodedParameterName O nome de um parmetro codificado (No pode ser
   *        {@code null}).
   *
   * @return O nome do parmetro ou {@code null} o texto informado no estiver
   *         no formato esperado.
   */
  private String getRealParameterNameFromEncodedParameterName(
    String encodedParameterName) {
    Pattern pattern = Pattern.compile(PARAMETER_NAME_REGEX);
    Matcher matcher = pattern.matcher(encodedParameterName);
    if (!matcher.matches()) {
      return null;
    }
    return matcher.group(REAL_PARAMETER_NAME_INDEX);
  }

  /**
   * Obtm os comandos <i>tee</i> que devem ser utilizados na linha de comando
   * gerada por este configurador de fluxo de algoritmos.
   *
   * @param fromPipes Os pipes de entrada (No pode ser {@code null}).
   * @param toPipes Os pipes de sada (No pode ser {@code null}).
   * @param asScript indica se comando deve ser gerado como script.
   *
   * @return Uma coleo no-modificvel com os comandos (a coleo estar
   *         vazia, caso no haja necessidade de criar tees).
   */
  private Collection<String> getTees(
    Map<Integer, Map<String, String>> fromPipes,
    Map<Integer, Map<String, String>> toPipes, boolean asScript) {
    Collection<String> tees = new LinkedList<String>();
    Collection<Node> nodes = this.nodesById.values();
    for (Node node : nodes) {
      for (Output output : node.getOutputs()) {
        if (!output.isDir()) {
          Collection<Input> inputs = output.getInputs();
          if (!inputs.isEmpty()) {
            CommandLineBuilder teeCommand = new CommandLineBuilder(asScript);
            teeCommand.append("tee ");
            for (Input input : inputs) {
              int nodeId = input.getNodeId();
              String parameterName = input.getParameterName();
              String toPipe = toPipes.get(nodeId).get(parameterName);
              teeCommand.append(toPipe).append(" ");
            }
            int nodeId = output.getNodeId();
            String parameterName = output.getParameterName();
            String fromPipe = fromPipes.get(nodeId).get(parameterName);
            teeCommand.append("< ").append(fromPipe).discardOutput();
            tees.add(teeCommand.toString());
          }
        }
      }
    }
    return tees;
  }

  /**
   * Cria a linha de comando de um n (algoritmo) do fluxo.
   *
   * @param asScript Indica se o comando dever ser gerado como um script.
   * @param nodeId Identificador do n.
   * @param context O contexto para gerao da linha de comando do n (
   *        diferente do contexto do comando do fluxo).
   * @param fromPipes Named-pipes de origem.
   * @param toPipes Named-pipes de destino.
   * @param linkDirectories Diretrios utilizados em ligaes.
   * @return a linha de comando do algoritmo do n.
   */
  private String makeNodeCommand(boolean asScript, Integer nodeId,
    CommandLineContext context, Map<String, String> fromPipes,
    Map<String, String> toPipes, Map<String, String> linkDirectories) {
    final Node node = this.nodesById.get(nodeId);
    AlgorithmConfigurator configurator = node.getConfigurator();
    Map<String, String> parameterValuesByName = new HashMap<String, String>();
    for (String parameterName : configurator.getParameterNames()) {
      try {
        String parameterValue = configurator.getParameterValue(parameterName);
        parameterValuesByName.put(parameterName, parameterValue);
      }
      catch (ParameterNotFoundException e) {
        throw new BugException(e);
      }
    }
    if (toPipes != null) {
      parameterValuesByName.putAll(toPipes);
    }
    if (fromPipes != null) {
      parameterValuesByName.putAll(fromPipes);
    }
    if (linkDirectories != null) {
      parameterValuesByName.putAll(linkDirectories);
    }

    try {
      configurator.setParameterValuesByName(parameterValuesByName);
    }
    catch (ParseException e) {
      throw new BugException(e);
    }
    catch (ParameterNotFoundException e) {
      throw new BugException(e);
    }
    String command = configurator.makeCommandLine(context);
    return command;
  }

  /**
   * Obtm a lista com todos os comandos gerados por cada um dos ns do fluxo.
   *
   * @param asScript Indica se o comando deve ser gerado como um script.
   * @param flowContext O contexto para a criao da linha de comando.
   *
   * @return A lista de comandos (se no houver ns, a lista estar vazia).
   */
  private Map<Integer, String> makeCommands(boolean asScript,
    FlowCommandLineContext flowContext) {
    Map<Integer, String> commands = new HashMap<Integer, String>();
    Map<Integer, Map<String, String>> linkDirectories =
      flowContext.getLinkDirectories();
    for (Integer nodeId : this.nodesById.keySet()) {
      String command =
        makeNodeCommand(asScript, nodeId, flowContext
          .getFlowNodeContext(nodeId), flowContext.getToPipes().get(nodeId),
          flowContext.getFromPipes().get(nodeId), linkDirectories.get(nodeId));
      if (command != null) {
        commands.put(nodeId, command);
      }
    }
    return commands;
  }

  /**
   * Atribui a estrutura que descreve o fluxo de algoritmos a este configurador.
   *
   * @param flow A estrutura (No pode ser {@code null}).
   */
  private void setFlow(Flow flow) {
    if (flow == null) {
      throw new IllegalArgumentException("O parmetro flow est nulo.");
    }
    this.flow = flow;
    this.canBeExecuted = false;
    Map<Integer, Node> nodesMap = createNodesMap(flow);
    if (nodesMap != null) {
      this.nodesById = nodesMap;
      this.canBeExecuted = true;
    }
    else {
      this.nodesById = Collections.emptyMap();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setCommandDescription(String commandDescription) {
    this.flow.setDescription(commandDescription);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getCommandDescription() {
    return flow.getDescription();
  }

  /**
   * Retorna se  possvel executar o fluxo com a configurao atual (levando em
   * consideraco possveis desvios feitos no fluxo)
   *
   * @return verdadeiro se for possvel executar o fluxo com a configurao
   *         atual ou falso caso contrrio
   */
  public boolean canBeExecuted() {
    return canBeExecuted;
  }

  /**
   * Extrai os ns {@link FlowNode} e ligaes {@link FlowLink} de um fluxo
   * {@link Flow} e o coloca num mapa de ns {@link Node} indexados pelos seus
   * identificadores.
   *
   * @param flx Fluxo a ser utilizado para a criao do mapa.
   *
   * @return mapa com os ns (no desviados) do fluxo indexados por seus
   *         identificadores.
   */
  protected Map<Integer, Node> createNodesMap(Flow flx) {
    Map<Integer, Node> nodesMap = new HashMap<Integer, Node>();
    if (isConfigured(flx)) {
      addNodes(nodesMap, flx);
      if (addLinks(nodesMap, flx)) {
        return nodesMap;
      }
    }
    return null;
  }

  /**
   * Determina se um fluxo est configurado (todos os algoritmos do fluxo tm
   * configurador)
   *
   * @param flx O fluxo que deve ser testado.
   * @return verdadeiro se o fluxo estiver pronto para execuo ou falso caso
   *         contrrio.
   */
  protected boolean isConfigured(Flow flx) {
    for (FlowNode flowNode : flx.getNodes()) {
      AlgorithmConfigurator algorithmConfigurator =
        flowNode.getAlgorithmConfigurator();
      if (algorithmConfigurator == null) {
        return false;
      }
    }
    return true;
  }

  /**
   * Adiciona os ns do fluxo no mapa indexado por seus identificadores. Ns que
   * esto sendo marcados como desviados no so adicionados porque no devero
   * ser executados.
   *
   * @param nodesMap mapa com os ns (no desviados) do fluxo indexados por seus
   *        identificadores.
   * @param flx O fluxo a ser utilizado.
   */
  protected void addNodes(Map<Integer, Node> nodesMap, Flow flx) {
    for (FlowNode flowNode : flx.getNodes()) {
      if (!flowNode.isBypassed()) {
        AlgorithmConfigurator algorithmConfigurator =
          flowNode.getAlgorithmConfigurator();
        Node node = new Node(flowNode.getId(), algorithmConfigurator);
        nodesMap.put(node.getId(), node);
      }
    }
  }

  /**
   * Adiciona as ligaes do fluxo nos ns que esto no mapa. Caso hajam ns
   * desviados no fluxo, pode no ser possvel determinar automaticamente como
   * as ligaes devem ser feitas nos ns restantes. Nesse caso, o mtodo
   * retorna falso. Se for possvel determinar, sem ambiguidades, como as
   * ligaes devem ser feitas, o mtodo retorna verdadeiro.
   *
   * @param nodesMap mapa com os ns (no desviados) do fluxo indexados por seus
   *        identificadores.
   * @param flx O fluxo a ser utilizado.
   * @return verdadeiro se foi possvel adicionar todos os links sem
   *         ambiguidade.
   */
  protected boolean addLinks(Map<Integer, Node> nodesMap, Flow flx) {
    /*
     * Lista de ligaes cujos outputs esto conectadas a ns desviados. Cada
     * uma dessas ligaes vai ser analisada para se tentar determinar a qual n
     * ela seu output dever ser ligado.
     */
    List<FlowLink> pendingLinks = new ArrayList<FlowLink>();
    /*
     * Lista de ligaes cujos inputs esto conectados a ns desviados. Essas
     * ligaes vo ser ignoradas na execuo do fluxo.
     */
    List<FlowLink> bypassedLinks = new ArrayList<FlowLink>();

    for (FlowLink flowLink : flx.getLinks()) {
      LinkParameter inputLinkParameter = flowLink.getInput();
      int inputNodeId = inputLinkParameter.getNodeId();
      Node inputNode = nodesMap.get(inputNodeId);
      /*
       * Se inputNode estiver nulo,  porque o n no foi adicionado ao mapa, o
       * que quer dizer que foi feito desvio desse n e, portanto, essa aresta
       * vai ser ignorada durante a execuo.
       */
      if (inputNode == null) {
        bypassedLinks.add(flowLink);
      }
      else {
        LinkParameter outputLinkParameter = flowLink.getOutput();
        int outputNodeId = outputLinkParameter.getNodeId();
        Node outputNode = nodesMap.get(outputNodeId);
        /*
         * Se outputNode estiver nulo,  porque o n no foi adicionado ao mapa,
         * o que quer dizer que foi feito desvio desse n e, portanto, essa
         * aresta vai ter que ser analisada para ver se pode ser ligada a outro
         * n.
         */
        if (outputNode == null) {
          pendingLinks.add(flowLink);
        }
        else {
          /*
           * Se a ligao no estiver conectada a nenhum n desviado, pode ser
           * adiciona sem alteraes
           */
          addLink(outputNode, outputLinkParameter, inputNode,
            inputLinkParameter);
        }
      }
    }
    if (pendingLinks.size() > 0) {
      /*
       * Adicionar ligaes que ficaram pendentes
       */
      return addPendingLinks(nodesMap, pendingLinks, bypassedLinks);
    }
    return true;
  }

  /**
   * Adiciona ligao ao n de sada atravs dos parmetros de entrada e sada
   * especificados.
   *
   * @param outputNode N ao qual ser adicionado a ligao como sada.
   * @param outputLinkParameter Parmetro de sada da ligao.
   * @param inputNode N ao qual ser adicionado a ligao como entrada.
   * @param inputLinkParameter Parmetro de entrada da ligao.
   */
  protected void addLink(Node outputNode, LinkParameter outputLinkParameter,
    Node inputNode, LinkParameter inputLinkParameter) {
    Input input =
      new Input(inputLinkParameter.getNodeId(), inputLinkParameter.getName());
    Output output = outputNode.getOutput(outputLinkParameter.getName());
    if (output != null) {
      output.addInput(input);
      AlgorithmConfigurator outputConfigurator = outputNode.getConfigurator();
      /* File */
      List<OutputFileParameter> outputFileParameters =
        outputConfigurator.getOutputFileParameters();
      for (OutputFileParameter parameter : outputFileParameters) {
        if (parameter.getName().equals(output.getParameterName())) {
          parameter.setHasLink(true);
          break;
        }
      }
      /* URL */
      List<OutputURLParameter> outputURLParameters =
        outputConfigurator.getOutputURLParameters();
      for (OutputURLParameter parameter : outputURLParameters) {
        if (parameter.getName().equals(output.getParameterName())) {
          parameter.setHasLink(true);
          break;
        }
      }

      AlgorithmConfigurator inputConfigurator = inputNode.getConfigurator();
      /* File */
      List<InputFileParameter> inputFileParameters =
        inputConfigurator.getInputFileParameters();
      for (InputFileParameter parameter : inputFileParameters) {
        if (parameter.getName().equals(input.getParameterName())) {
          parameter.setHasLink(true);
          break;
        }
      }
      /* URL */
      List<InputURLParameter> inputURLParameters =
        inputConfigurator.getInputURLParameters();
      for (InputURLParameter parameter : inputURLParameters) {
        if (parameter.getName().equals(input.getParameterName())) {
          parameter.setHasLink(true);
          break;
        }
      }
    }
  }

  /**
   * Adiciona ligaes que ficaram pendentes, se for possvel determinar sem
   * ambiguidade como as ligaes devem ser reconectadas de acordo com os
   * desvios feitos no fluxo. Para cada ligao pendente  procurada uma nica
   * ligao desviada que ela vai substituir. Se no for possvel encontrar
   * nenhuma ligao equivalente ou existirem diversas ligaes que podem ser
   * equivalentes, o fluxo no poder ser executado com a configurao atual. O
   * usurio deve fazer o desvio manualmente ou refazer os desvios para que no
   * gerem ambiguidades.
   *
   * @param nodesMap mapa com os ns (no desviados) do fluxo indexados por seus
   *        identificadores.
   * @param pendingLinks ligaes que devem ser refeitas.
   * @param bypassedLinks ligaes que devem ser ignoradas pelo fluxo.
   * @return verdadeiro se for possvel refazer as ligaes sem ambiguidade ou
   *         falso caso contrrio.
   */
  protected boolean addPendingLinks(Map<Integer, Node> nodesMap,
    List<FlowLink> pendingLinks, List<FlowLink> bypassedLinks) {
    /*
     * Ligaes agrupadas por output. Somente uma ligao de cada output precisa
     * ser analisada, j que todas devem ser conectadas ao mesmo n.
     */
    Map<LinkParameter, List<FlowLink>> pendingLinksByOutput =
      groupPendingLinksByOutput(pendingLinks);
    /*
     * Mapeia as ligaes que podem ser refeitas a suas ligaes desviadas
     * correspondentes.
     */
    Map<FlowLink, FlowLink> linkMatches = new HashMap<FlowLink, FlowLink>();
    for (List<FlowLink> pendingInputLinks : pendingLinksByOutput.values()) {
      FlowLink pendingInputLink = pendingInputLinks.get(0);
      FlowLink matchingLink = getMatchingLink(bypassedLinks, pendingInputLink);
      if (matchingLink != null) {
        for (FlowLink link : pendingInputLinks) {
          linkMatches.put(link, matchingLink);
        }
      }
    }
    if (areAllPendingLinksMatched(pendingLinks, linkMatches)) {
      if (areAllBypassedLinksMatched(nodesMap, bypassedLinks, linkMatches)) {
        for (FlowLink pendingLink : linkMatches.keySet()) {
          addPendingLink(nodesMap, pendingLink, linkMatches.get(pendingLink));
        }
        return true;
      }
    }
    return false;
  }

  /**
   * Verifica se foi possvel achar as ligaes correspondentes de todas as
   * ligaes pendentes.
   *
   * @param pendingLinks Lista de ligaes pendentes.
   * @param linkMatches Mapa com as ligaes pendentes e as ligaes desviadas
   *        correspondentes.
   * @return verdadeiro se todas as ligaes pendentes tm suas correspondentes
   *         mapeadas ou falso caso contrrio.
   */
  private boolean areAllPendingLinksMatched(List<FlowLink> pendingLinks,
    Map<FlowLink, FlowLink> linkMatches) {
    return linkMatches.keySet().size() == pendingLinks.size();
  }

  /**
   * Verifica se todas as ligaes desviadas que deveriam ser consideradas foram
   * tratadas. @see {@link #countBypassedLinks(List, Map)}
   *
   * @param nodesMap mapa com os ns (no desviados) do fluxo indexados por seus
   *        identificadores.
   * @param bypassedLinks Lista de ligaes que foram desviadas.
   * @param linkMatches Mapa com as ligaes pendentes e as ligaes desviadas
   *        correspondentes.
   * @return verdadeiro se todas as ligaes desviadas que deveriam ser
   *         consideradas foram tratadas ou falso caso contrrio.
   */
  protected boolean areAllBypassedLinksMatched(Map<Integer, Node> nodesMap,
    List<FlowLink> bypassedLinks, Map<FlowLink, FlowLink> linkMatches) {
    /*
     * Adiciona no set para verificar o nmero de ligaes correspondentes
     * nicas
     */
    Set<FlowLink> matchingLinksSet = new HashSet<FlowLink>();
    matchingLinksSet.addAll(linkMatches.values());
    return countBypassedLinks(bypassedLinks, nodesMap) == matchingLinksSet
      .size();
  }

  /**
   * Conta o nmero de ligaes desviadas que deveriam ser tratadas. As ligaes
   * que conectam dois ns desviados so descartas e ignoradas durante a
   * execuo. J as ligaes que s a entrada ou a sada esto conectadas a um
   * n desviado so contadas e devem ser tratadas. Cada ligao dessa deve
   * corresponder a pelo menos uma ligao pendente.
   *
   * @param bypassedLinks Lista de ligaes que foram desviadas.
   * @param nodesMap mapa com os ns (no desviados) do fluxo indexados por seus
   *        identificadores.
   * @return nmero de ligaes desviadas que devem ser tratadas.
   */
  protected int countBypassedLinks(List<FlowLink> bypassedLinks,
    Map<Integer, Node> nodesMap) {
    List<FlowLink> links = new ArrayList<FlowLink>();
    for (FlowLink flowLink : bypassedLinks) {
      Node inputNode = nodesMap.get(flowLink.getInput().getNodeId());
      Node outputNode = nodesMap.get(flowLink.getOutput().getNodeId());
      if (inputNode != null || outputNode != null) {
        links.add(flowLink);
      }
    }
    return links.size();
  }

  /**
   * Adiciona a ligao pendente, substituindo a ligao desviada
   * correspondente.
   *
   * @param nodesMap mapa com os ns (no desviados) do fluxo indexados por seus
   *        identificadores.
   * @param pendingLink Ligao pendente.
   * @param matchingLink Ligao desviada correspondente.
   */
  protected void addPendingLink(Map<Integer, Node> nodesMap,
    FlowLink pendingLink, FlowLink matchingLink) {
    LinkParameter inputParam = pendingLink.getInput();
    LinkParameter outputParam = matchingLink.getOutput();
    int newOutputNodeId = outputParam.getNodeId();
    Node outputNode = nodesMap.get(newOutputNodeId);
    int newInputNodeId = inputParam.getNodeId();
    Node inputNode = nodesMap.get(newInputNodeId);
    addLink(outputNode, outputParam, inputNode, inputParam);
  }

  /**
   * Agrupa as ligaes pendentes pelo parmetro de sada do n ao qual elas
   * esto conectadas.
   *
   * @param pendingLinks Lista de ligaes pendentes.
   * @return mapa com as ligaes pendentes agrupadas pelo parmetro de sada do
   *         n ao qual esto conectadas.
   */
  protected Map<LinkParameter, List<FlowLink>> groupPendingLinksByOutput(
    List<FlowLink> pendingLinks) {
    Map<LinkParameter, List<FlowLink>> pendingLinksByOutput =
      new HashMap<LinkParameter, List<FlowLink>>();
    for (FlowLink link : pendingLinks) {
      List<FlowLink> linkList = pendingLinksByOutput.get(link.getOutput());
      if (linkList == null) {
        linkList = new ArrayList<FlowLink>();
        pendingLinksByOutput.put(link.getOutput(), linkList);
      }
      linkList.add(link);
    }
    return pendingLinksByOutput;
  }

  /**
   * Procura a ligao desviada que corresponde  ligao pendente. Retorna nulo
   * se no encontrar nenhuma ligao correspondente ou encontrar mais de uma
   * ligao correspondente (gerando ambiguidade).
   *
   * @param bypassedLinks Lista de ligaes desviadas.
   * @param pendingInputLink Ligao pendente.
   * @return a ligao desviada que corresponde  ligao pendente ou nulo se
   *         no for possvel encontrar uma nica correspondente
   */
  protected FlowLink getMatchingLink(List<FlowLink> bypassedLinks,
    FlowLink pendingInputLink) {
    List<FlowLink> matchingLinks =
      getMatchingLinksByType(pendingInputLink, bypassedLinks);
    if (matchingLinks != null) {
      if (matchingLinks.size() == 1) {
        return matchingLinks.get(0);
      }
      else if (matchingLinks.size() > 1) {
        matchingLinks = getMatchingLinksByName(pendingInputLink, matchingLinks);
        if (matchingLinks != null && matchingLinks.size() == 1) {
          return matchingLinks.get(0);
        }
      }
    }
    return null;
  }

  /**
   * Retorna uma lista com todas as ligaes desviadas correspondentes que tm o
   * mesmo tipo da ligao pendente especificada (@see
   * {@link #getMatchingLinksByTypeRecursively}). Retorna nulo se no encontrar
   * nenhuma ligao correspondente com o tipo em questo.
   *
   * @param pendingLink Ligao pendente
   * @param candidateLinks Lista de ligaes desviadas que tm o mesmo tipo da
   *        ligao pendente
   * @return lista com as ligaes correspondentes do tipo especificado ou nulo
   *         se no for possvel encontrar nenhuma ligao desviada
   *         correspondente do tipo especificado.
   */
  protected List<FlowLink> getMatchingLinksByType(FlowLink pendingLink,
    List<FlowLink> candidateLinks) {
    List<FlowLink> matchingLinks = new ArrayList<FlowLink>();
    for (FlowLink candidateLink : candidateLinks) {
      if (pendingLink.getOutput().getNodeId() == candidateLink.getInput()
        .getNodeId()) {
        matchingLinks.addAll(getMatchingLinksByTypeRecursively(pendingLink,
          candidateLink, candidateLinks));
      }
    }
    if (matchingLinks.size() > 0) {
      return matchingLinks;
    }
    else {
      return null;
    }
  }

  /**
   * Procura recursivamente as ligaes desviadas correspondentes que tm o
   * mesmo tipo da ligao pendente especificada a partir de uma ligao
   * candidata a ser a correspondente. Caso a ligao candidata esteja conectada
   * a sada de um n desviado, o mtodo  chamado recursivamente para as
   * ligaes de entrada deste n. Quando a ligao candidata est ligada a um
   * n que no foi desviado e tem o tipo correto, a ligao  considerada
   * correspondente.
   *
   * @param pendingLink Ligao pendente
   * @param candidateLink Ligao candidata
   * @param candidateLinks Lista de ligaes desviadas candidatas
   * @return a lista de ligaes desviadas consideradas correspondentes
   */
  protected List<FlowLink> getMatchingLinksByTypeRecursively(
    FlowLink pendingLink, FlowLink candidateLink, List<FlowLink> candidateLinks) {
    List<FlowLink> matchingLinks = new ArrayList<FlowLink>();

    FlowNode parentNode = flow.getNode(candidateLink.getOutput().getNodeId());
    if (!parentNode.isBypassed()) {
      if (getInputParamType(pendingLink).equals(
        getInputParamType(candidateLink))) {
        matchingLinks.add(candidateLink);
      }
    }
    else {
      for (FlowLink link : candidateLinks) {
        if (link.getInput().getNodeId() == parentNode.getId()) {
          matchingLinks.addAll(getMatchingLinksByTypeRecursively(pendingLink,
            link, candidateLinks));
        }
      }
    }
    return matchingLinks;
  }

  /**
   * Retorna uma lista com todas as ligaes desviadas correspondentes que tm o
   * mesmo nome do parmetro de entrada da ligao pendente especificada.
   * Retorna nulo se no encontrar nenhuma ligao correspondente com o nome em
   * questo.
   *
   * @param pendingLink Ligao pendente
   * @param candidateLinks Lista de ligaes desviadas candidatas
   * @return lista com as ligaes desviadas correspondentes que tm o mesmo
   *         nome do parmetro de entrada da ligao pendente especificada.
   */
  protected List<FlowLink> getMatchingLinksByName(FlowLink pendingLink,
    List<FlowLink> candidateLinks) {
    List<FlowLink> matchingLinks = new ArrayList<FlowLink>();
    for (FlowLink candidateLink : candidateLinks) {
      if (pendingLink.getOutput().getNodeId() == candidateLink.getInput()
        .getNodeId()) {
        String linkName = pendingLink.getInput().getName();
        String candidateLinkName = candidateLink.getInput().getName();
        if (linkName.equals(candidateLinkName)) {
          matchingLinks.add(candidateLink);
        }
      }
    }
    if (matchingLinks.size() > 0) {
      return matchingLinks;
    }
    else {
      return null;
    }
  }

  /**
   * Retorna o tipo do parmetro de entrada da ligao especificada.
   *
   * @param link A ligao especificada.
   * @return o tipo do parmetro de entrada da ligao especificada.
   */
  protected String getInputParamType(FlowLink link) {
    LinkParameter input = link.getInput();
    FlowNode node = flow.getNode(input.getNodeId());
    String inputName = input.getName();
    NodeParameter inputParameter = node.getParameter(inputName);
    return inputParameter.getType();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    String name = getName();
    if (name != null) {
      return name;
    }
    return super.toString();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Validation validate(ValidationContext context) throws RemoteException {
    for (Node node : nodesById.values()) {
      AlgorithmConfigurator configurator = node.getConfigurator();
      Validation result = configurator.validate(context);
      if (!result.isWellSucceded()) {
        AlgorithmInfo info = configurator.getAlgorithmVersion().getInfo();
        return new FlowNodeValidationResult(result, info);
      }
    }
    return new ValidationSuccess();
  }
}
