package csbase.logic.algorithms.parameters;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

/**
 * Parmetro que representa um arquivo e pode ter representao de pipe
 *
 *
 * @author Tecgraf/PUC-Rio
 */
public abstract class AbstractFileParameter extends
SimpleParameter<FileURLValue> {

  /**
   * Chave utilizada para informar o <b>caminho</b> do arquivo no mapa de
   * parmetros dos mtodos de importao ({@link #importValue(Map)}) e
   * exportao ({@link #exportValue()}).
   */
  public static final String PATH = ".path";

  /**
   * Chave utilizada para informar o <b>tipo</b> do arquivo no mapa de
   * parmetros dos mtodos de importao ({@link #importValue(Map)}) e
   * exportao ({@link #exportValue()}).
   */

  public static final String TYPE = ".type";

  /**
   * Indica se este parmetro pode aceitar pipe
   * {@value FileParameterPipeAcceptance#TRUE}, no aceita pipe
   * {@value FileParameterPipeAcceptance#FALSE} ou *s* aceita pipe
   * {@value FileParameterPipeAcceptance#ALWAYS}.
   */
  protected FileParameterPipeAcceptance usesPipe;

  /**
   * O tipo de arquivo que este parmetro filtra. Se ele se importa com o tipo
   * do arquivo, ele valer {@code null}.
   */
  protected String fileType;

  /**
   * Indica se este parmetro est ({@code true}) ou no ({@code false}) est
   * participando de uma conexo entre ns de um fluxo de algoritmos.
   */
  protected boolean hasLink;

  /**
   * Associao com os observador do parmetro do tipo arquivo.
   */
  protected transient List<FileParameterListener> listeners;

  /**
   * O modo de funcionamento atual (apenas arquivos, apenas diretrios ou
   * ambos).
   */
  protected FileParameterMode mode;

  /**
   * Indica se o arquivo deve existir.
   */
  protected final boolean mustExist;

  /**
   * Cria um parmetro do tipo arquivo.
   *
   * @param name O nome deste parmetro (No aceita {@code null}).
   * @param label O rtulo deste parmetro (No aceita {@code null}).
   * @param description A descrio deste parmetro (No aceita {@code null}).
   * @param defaultValue O valor-padro (Aceita {@code null}).
   * @param isOptional Indica se o valor do parmetro  opcional.
   * @param isVisible Indica se o parmetro deve ficar visvel.
   * @param commandLinePattern O padro para construo da linha de comando. O
   *        padro ser utilizado para escrever o trecho da linha do comando
   *        referente ao parmetro. Esta string ser passada para o mtodo
   *        MessageFormat.format(String,Object...). O primeiro formato ({0}) 
   *        referente ao nome e o segundo formato ({1})  referente ao valor. Se
   *        {@code null} o parmetro no produzir sada na linha de comando.
   * @param fileType O tipo dos arquivos aceitos neste parmetro (Aceita
   *        {@code null}).
   * @param mode O modo de funcionamento deste parmetro (No aceita
   *        {@code null}).
   * @param usesPipe Indica se este parmetro pode aceitar pipe
   *        {@value FileParameterPipeAcceptance#TRUE}, no aceita pipe
   *        {@value FileParameterPipeAcceptance#FALSE} ou *s* aceita pipe
   *        {@value FileParameterPipeAcceptance#ALWAYS}.
   * @param mustExist Indica se o arquivo deve existir.
   */
  protected AbstractFileParameter(String name, String label,
    String description, FileURLValue defaultValue, boolean isOptional,
    boolean isVisible, String commandLinePattern, String fileType,
    FileParameterMode mode, FileParameterPipeAcceptance usesPipe,
    boolean mustExist) {
    super(name, label, description, defaultValue, isOptional, isVisible,
      commandLinePattern);
    this.listeners = new LinkedList<FileParameterListener>();
    this.fileType = fileType;
    this.hasLink = false;
    this.mustExist = mustExist;
    setUsesPipe(usesPipe);
    setMode(mode);
  }

  /**
   * Indica se este arquivo pode utilizar <i>pipes</i>.
   *
   * @return {@value FileParameterPipeAcceptance#TRUE} se este parmetro pode
   *         aceitar pipe, {@value FileParameterPipeAcceptance#FALSE} se no
   *         aceita pipe, ou {@value FileParameterPipeAcceptance#ALWAYS} se *s*
   *         aceita pipe.
   */
  public final FileParameterPipeAcceptance usesPipe() {
    return this.usesPipe;
  }

  /**
   * Adiciona um observador de parmetros do tipo arquivo a este parmetro.
   *
   * @param listener O observador (No aceita {@code null}).
   */
  public final void addParameterListener(FileParameterListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    this.listeners.add(listener);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final Object getExpressionValue() {
    return null;
  }

  /**
   * <p>
   * Obtm a lista de observadores deste parmetro.
   * </p>
   *
   * <p>
   * A lista retornada  imutvel (veja
   * {@link Collections#unmodifiableList(List)}).
   * </p>
   *
   * <p>
   * Se no houver observadores a lista retornada estar vazia.
   * </p>
   *
   * @return A lista de observadores.
   */
  public final List<FileParameterListener> getFileParameterListeners() {
    return Collections.unmodifiableList(this.listeners);
  }

  /**
   * Obtm o tipo de arquivo que este parmetro aceita.
   *
   * @return O tipo de arquivo ou {@code null} se ele no se importar com um
   *         tipo especfico.
   */
  public final String getFileType() {
    return this.fileType;
  }

  /**
   * Obtm o modo de funcionamento (apenas arquivos, apenas diretrios ou ambos)
   * deste parmetro.
   *
   * @return O modo de funcionamento.
   */
  public final FileParameterMode getMode() {
    return this.mode;
  }

  /**
   * Indica se este parmetro participa de uma conexo entre ns de um fluxo de
   * algoritmos.
   *
   * @return {@code true} se participar ou {@code false} caso contrrio.
   */
  public final boolean hasLink() {
    return this.hasLink;
  }

  /**
   * Remove um observador deste parmetro.
   *
   * @param listener O observador (No aceita {@code null}).
   *
   * @return {@code true} se ele removeu o observador ou {@code false} se o
   *         observador no estiver neste parmetro.
   */
  public final boolean removeFileParameterListener(
    FileParameterListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    return this.listeners.add(listener);
  }

  /**
   * Atribui um tipo de arquivo especfico a este parmetro.
   *
   * @param fileType O tipo de arquivo (Aceita {@code null}).
   */
  public final void setFileType(String fileType) {
    this.fileType = fileType;
    for (FileParameterListener listener : this.listeners) {
      listener.typeWasChanged(this);
    }
    fireVisiblityWasChangedEvent();
  }

  /**
   * Marca se este arquivo est ou no participando de uma conexo entre ns de
   * algoritmos em um fluxo de algoritmos.
   *
   * @param hasLink {@code true} se estiver ou {@code false} caso contrrio.
   */
  public final void setHasLink(boolean hasLink) {
    this.hasLink = hasLink;
    for (FileParameterListener listener : this.listeners) {
      listener.hasLinkWasChanged(this);
    }
    fireVisiblityWasChangedEvent();
  }

  /**
   * Atribui o modo de funcionamento (apenas arquivo, apenas diretrio ou ambos)
   * deste parmetro.
   *
   * @param mode O modo de funcionamento (No aceita {@code null}).
   */
  protected void setMode(FileParameterMode mode) {
    if (mode == null) {
      throw new IllegalArgumentException("O parmetro mode est nulo.");
    }
    this.mode = mode;
  }

  /**
   * Determina se este arquivo pode utilizar <i>pipes</i>.
   *
   * @param usesPipe {@value FileParameterPipeAcceptance#TRUE} se este parmetro
   *        pode aceitar pipe, {@value FileParameterPipeAcceptance#FALSE} se no
   *        aceita pipe, ou {@value FileParameterPipeAcceptance#ALWAYS} se *s*
   *        aceita pipe.
   */
  protected void setUsesPipe(FileParameterPipeAcceptance usesPipe) {
    if (usesPipe == null) {
      throw new IllegalArgumentException("O parmetro usesPipe est nulo.");
    }
    this.usesPipe = usesPipe;
  }

  /**
   * Indica se o arquivo selecionado precisa existir.
   *
   * @return mustExist verdadeiro se o arquivo precisa existir, ou falso, caso
   *         contrrio.
   */
  public boolean mustExist() {
    return mustExist;
  }

  /**
   *
   * @return Indica se este parmetro  de output
   */
  public abstract boolean isOuput();

}
