package csbase.logic;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * Informaes compostas sobre a finalizao de um comando (utilizada para
 * guardar informaes de um fluxo e seus ns).
 */
public class ExtendedCommandFinalizationInfo extends
  AbstractCommandFinalizationInfo {

  /**
   * Identificador do n responsvel por erro na execuo do fluxo (se houver).
   */
  protected Integer guiltyNodeId;

  /**
   * Mapeia os ns do fluxo para suas respectivas informaes de finalizao
   * individuais.
   */
  protected Map<Integer, CommandFinalizationInfo> infoById;

  /**
   * Construtor.
   * 
   * @param finalizationType tipo de finalizao do comando.
   */
  public ExtendedCommandFinalizationInfo(
    CommandFinalizationType finalizationType) {
    this(finalizationType, (Integer) null);
  }

  /**
   * Construtor.
   * 
   * @param finalizationType tipo de finalizao do comando.
   * @param exitCode cdigo de sada do comando.
   */
  public ExtendedCommandFinalizationInfo(
    CommandFinalizationType finalizationType, Integer exitCode) {
    this(finalizationType, null, exitCode);
  }

  /**
   * Construtor.
   * 
   * @param finalizationType tipo de finalizao do comando.
   * @param failureCause o tipo da falha que causou a finalizao do comando ou
   *        null caso o comando no tenha falhado.
   * @param exitCode cdigo de sada do comando.
   */
  private ExtendedCommandFinalizationInfo(
    CommandFinalizationType finalizationType,
    FailureFinalizationType failureCause, Integer exitCode) {
    super(FinalizationInfoType.EXTENDED, finalizationType, failureCause,
      exitCode);
    this.infoById = new HashMap<Integer, CommandFinalizationInfo>();
  }

  /**
   * Construtor.
   * 
   * @param info informao de finalizao de comando.
   */
  public ExtendedCommandFinalizationInfo(CommandFinalizationInfo info) {
    this(info.getFinalizationType(), info.getFailureCause(), info.getExitCode());
  }

  /**
   * Retorna a informao de finalizao de um n especfico de um fluxo.
   * 
   * @param id O identificador do n.
   * @return a informao de finalizao.
   */
  public CommandFinalizationInfo getFinalizationInfoForNode(int id) {
    return infoById.get(id);
  }

  /**
   * Retorna uma cpia das informaes de finalizao individuais mapeadas para
   * os identificadores dos ns.
   * 
   * @return O mapa de informaes de finalizao para cada n.
   */
  public Map<Integer, CommandFinalizationInfo> getFinalizationInfos() {
    return Collections.unmodifiableMap(infoById);
  }

  /**
   * Atribui a informao de finalizao de um n especificado.
   * 
   * @param info A informao de finalizao do n.
   * @param id O identificador do n.
   */
  public void setFinalizationInfoForNode(CommandFinalizationInfo info, int id) {
    this.infoById.put(id, info);
  }

  /**
   * Retorna o identificador do n responsvel por erro na execuo do fluxo ou
   * nulo caso no haja erro.
   * 
   * @return o identificador do n.
   */
  public Integer getGuiltyNodeId() {
    return guiltyNodeId;
  }

  /**
   * Atribui o identificador do n responsvel por erro na execuo do fluxo.
   * 
   * @param errorId o identificador do n.
   */
  public void setGuiltyNodeId(Integer errorId) {
    this.guiltyNodeId = errorId;
  }

  /**
   * Atribui causa da falha.
   * 
   * @param failureCause o tipo da falha que causou a finalizao do comando ou
   *        null caso o comando no tenha falhado.
   */
  public void setFailureCause(FailureFinalizationType failureCause) {
    this.failureCause = failureCause;
  }

}
