/*
 * $Id: DiskOccupation.java 152996 2014-05-30 13:26:14Z mjulia $
 */

package csbase.logic.diskusageservice;

import java.io.Serializable;

/**
 * Estrutura que armazena o ocupao (livre/total) do disco.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class DiskOccupation implements Serializable {
  /**
   * Espao livre
   */
  final private long free;

  /**
   * Espao total
   */
  final private long total;

  /**
   * 
   */
  final public long alertLimit;

  /**
   * 
   */
  final public long warningLimit;

  /**
   * Converte um valor de bytes para mega-bytes
   * 
   * @param bytes o valor em bytes.
   * @return o valor em mega bytes.
   */
  final private double bytesToMb(final long bytes) {
    return bytes / (1024 * 1024); // Mb
  }

  /**
   * @return o valor
   */
  final public double getFreeSpaceMb() {
    return bytesToMb(free);
  }

  /**
   * @return o valor
   */
  final public double getTotalSpaceMb() {
    return bytesToMb(total);
  }

  /**
   * @return o valor
   */
  final public double getUsedSpaceMb() {
    final long used = total - free;
    return bytesToMb(used);
  }

  /**
   * Obtm o percentual livre do disco.
   * 
   * @return o valor percentual de rea livre no disco
   */
  final public double getFreeSpacePerc() {
    final double totalSpaceMb = getTotalSpaceMb();
    if (totalSpaceMb <= 0.0) {
      return 0.0;
    }
    final double freeSpaceMb = getFreeSpaceMb();
    return freeSpaceMb / totalSpaceMb * 100.;
  }

  /**
   * Obtm o percential ocupado no disco.
   * 
   * @return o valor percentual de rea ocupada no disco
   */
  final public double getUsedSpacePerc() {
    final double totalSpaceMb = getTotalSpaceMb();
    if (totalSpaceMb <= 0.0) {
      return 100.0;
    }
    final double usedSpaceMb = getUsedSpaceMb();
    return usedSpaceMb / totalSpaceMb * 100.;
  }

  /**
   * Construtor para objeto vlido
   * 
   * @param free espao livre em bytes
   * @param total espao total em bytes
   * @param alertLimit limite para alerta
   * @param warningLimit limite para aviso
   */
  public DiskOccupation(final long free, final long total,
    final long alertLimit, final long warningLimit) {
    this.free = free;
    this.total = total;
    this.alertLimit = alertLimit;
    this.warningLimit = warningLimit;
  }

  /**
   * Construtor para objeto invlido
   * 
   */
  public DiskOccupation() {
    this.free = -1;
    this.total = -1;
    this.alertLimit = -1;
    this.warningLimit = -1;
  }

  /**
   * Indica que dado  vlido.
   * 
   * @return indicativo
   */
  public boolean isValid() {
    return (free >= 0 && total >= 0);
  }
}
