/**
 * $Id$
 */
package csbase.logic.algorithms.parameters;

import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import csbase.logic.ProjectFileType;
import csbase.logic.algorithms.CommandLineBuilder;
import csbase.logic.algorithms.CommandLineContext;
import csbase.logic.algorithms.EnvironmentVariable;
import csbase.logic.algorithms.parameters.validators.FileParameterValidator;

/**
 * <p>
 * Parmetro do tipo arquivo.
 * </p>
 */
public abstract class FileParameter extends AbstractFileParameter {

  /**
   * Indica se um painel para filtro deve ser exibido.
   */
  private boolean usesFilter;

  /**
   * Cria um parmetro do tipo arquivo.
   *
   * @param name O nome deste parmetro (No aceita {@code null}).
   * @param label O rtulo deste parmetro (No aceita {@code null}).
   * @param description A descrio deste parmetro (No aceita {@code null}).
   * @param defaultValue O valor-padro (Aceita {@code null}).
   * @param isOptional Indica se o valor do parmetro  opcional.
   * @param isVisible Indica se o parmetro deve ficar visvel.
   * @param commandLinePattern O padro para construo da linha de comando. O
   *        padro ser utilizado para escrever o trecho da linha do comando
   *        referente ao parmetro. Esta string ser passada para o mtodo
   *        MessageFormat.format(String,Object...). O primeiro formato ({0}) 
   *        referente ao nome e o segundo formato ({1})  referente ao valor. Se
   *        {@code null} o parmetro no produzir sada na linha de comando.
   * @param fileTypes O tipo dos arquivos aceitos neste parmetro (Aceita
   *        {@code null}).
   * @param mode O modo de funcionamento deste parmetro (No aceita
   *        {@code null}).
   * @param usesPipe Indica se este parmetro pode aceitar pipe
   *        {@link FileParameterPipeAcceptance#TRUE}, no aceita pipe
   *        {@link FileParameterPipeAcceptance#FALSE} ou *s* aceita pipe
   *        {@link FileParameterPipeAcceptance#ALWAYS}.
   * @param usesFilter Indica se o painel de filtro deve ser exibido.
   * @param mustExist Indica se o arquivo deve existir.
   */
  protected FileParameter(String name, String label, String description,
    FileURLValue defaultValue, boolean isOptional, boolean isVisible,
    String commandLinePattern, String[] fileTypes, FileParameterMode mode,
    FileParameterPipeAcceptance usesPipe, boolean usesFilter, boolean mustExist) {
    super(name, label, description, defaultValue, isOptional, isVisible,
      commandLinePattern, fileTypes, mode, usesPipe, mustExist);
    this.usesFilter = usesFilter;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String, Object> exportValue() {
    Map<String, Object> valuesByName = new HashMap<String, Object>();
    Collection<Map<String, String>> exportableValues =
      new LinkedList<Map<String, String>>();
    FileURLValue file = getValue();
    if (file != null) {
      Map<String, String> exportableValue = new HashMap<String, String>();
      exportableValue.put(PATH, file.getPath());
      exportableValue.put(TYPE, file.getType());
      exportableValues.add(Collections.unmodifiableMap(exportableValue));
      valuesByName.put(getName(), Collections
        .unmodifiableCollection(exportableValues));
    }
    return Collections.unmodifiableMap(valuesByName);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final String getCommandValue(CommandLineContext context) {

    FileURLValue file = getValue();
    if (file == null) {
      return null;
    }
    StringBuilder commandValue = new StringBuilder();
    char fileSeparator = context.getFileSeparator();
    if (file.getPath().startsWith(CommandLineBuilder.REFERENCE_VAR_CHAR)) {
      commandValue.append(file.getPath());
    }
    else {
      String path = file.getPath(fileSeparator);
      commandValue.append(CommandLineBuilder.makePathWithEnvironmentVariable(
        EnvironmentVariable.PROJECT_DIR, path, fileSeparator));
    }

    return commandValue.toString();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final String getValueAsText() {
    FileURLValue file = getValue();
    if (file == null) {
      return null;
    }
    String type = file.getType();
    if (type == null || type.trim().isEmpty()) {
      type = ProjectFileType.UNKNOWN;
    }

    String valueAsText = file.getPath() + ":" + type;

    return valueAsText;
  }

  /**
   * Indica se um painel de filtro deve ser exibido. ({@code true}), se sim, (
   * {@code false}), caso contrrio.
   *
   * @return {@code true} se exibir ou {@code false} caso contrrio.
   */
  public boolean usesFilter() {
    return this.usesFilter;
  }

  /**
   * {@inheritDoc}
   */
  @SuppressWarnings("unchecked")
  @Override
  public void importValue(Map<String, Object> parameterValues) {
    Collection<Map<String, String>> exportableValues =
      (Collection<Map<String, String>>) parameterValues.get(getName());
    if (exportableValues != null && !exportableValues.isEmpty()) {
      Map<String, String> exportableValue = exportableValues.iterator().next();
      String path = exportableValue.get(PATH);
      String type = exportableValue.get(TYPE);
      FileURLValue file = new FileURLValue(path, type);
      setValue(file);
    }
    else {
      setValue(null);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final void setValueAsText(String parameterValue) {

    if (parameterValue == null || parameterValue.length() == 0) {
      setValue(null);
      return;
    }
    Pattern withTypePattern = Pattern.compile("^(.+):(.*)$");
    Matcher withTypeMatcher = withTypePattern.matcher(parameterValue);
    String path, type;
    if (withTypeMatcher.matches()) {
      path = withTypeMatcher.group(1);
      type = withTypeMatcher.group(2);
    }
    else {
      path = parameterValue;
      type = null;
    }
    if (type == null || type.trim().isEmpty()) {
      if (mode == FileParameterMode.DIRECTORY) {
        type = ProjectFileType.DIRECTORY_TYPE;
      }
      else {
        type = ProjectFileType.UNKNOWN;
      }
    }
    FileURLValue file = new FileURLValue(path, type);
    setValue(file);
  }

  /**
   * Determina se um painel de filtro pode ser usado.
   *
   * @param usesFilter True, se o parmetro pode exibir um painel de filtro ou
   *        false, caso contrrio.
   */
  public void setUsesFilter(boolean usesFilter) {
    this.usesFilter = usesFilter;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public FileParameterValidator createParameterValidator() {
    return new FileParameterValidator(isOptional());
  }

  /**
   * Cria os atributos transientes.
   *
   * @param in stream com o objeto serializado.
   *
   * @throws IOException em caso de erro de I/O.
   * @throws ClassNotFoundException se a classe do objeto serializado no for
   *         encontrada.
   */
  private void readObject(java.io.ObjectInputStream in) throws IOException,
    ClassNotFoundException {
    in.defaultReadObject();
    listeners = new LinkedList<FileParameterListener>();
  }

}
