package csbase.logic.algorithms.parameters;

import java.util.List;

import tecgraf.javautils.core.io.FileUtils;
import tecgraf.openbus.algorithmservice.v1_0.parameters.OutputFileListParameterHelper;
import csbase.logic.ProjectFileType;

/**
 * <p>
 * Parmetro do tipo lista de arquivo de sada.
 * </p>
 * <p>
 * Ele possui 3 modos de funcionamento: aceitar apenas arquivos regulares,
 * aceitar apenas diretrios ou aceitar arquivos regulares e diretrios (veja
 * {@link FileParameterMode}).
 * </p>
 */
public class OutputFileListParameter extends FileListParameter {

  /** Texto que representa o tipo desse parmetro */
  public static final String TYPE_VALUE = "OUTPUT_FILE";

  /** Indica se o arquivo deve ter uma das extenses pr-definidas. */
  private boolean mustForceExtension;

  /**
   * Cria um parmetro do tipo arquivo.
   *
   * @param name O nome deste parmetro (No aceita {@code null}).
   * @param label O rtulo deste parmetro (No aceita {@code null}).
   * @param description A descrio deste parmetro (No aceita {@code null}).
   * @param defaultValue O valor-padro (Aceita {@code null}).
   * @param isOptional Indica se o valor do parmetro  opcional.
   * @param isVisible Indica se o parmetro deve ficar visvel.
   * @param commandLinePattern O padro para construo da linha de comando. O
   *        padro ser utilizado para escrever o trecho da linha do comando
   *        referente ao parmetro. Esta string ser passada para o mtodo
   *        MessageFormat.format(String,Object...). O primeiro formato ({0}) 
   *        referente ao nome e o segundo formato ({1})  referente ao valor. Se
   *        {@code null} o parmetro no produzir sada na linha de comando.
   * @param fileTypes Os tipos dos arquivos aceitos neste parmetro (Aceita
   *        {@code null}).
   * @param mode O modo de funcionamento deste parmetro (No aceita
   *        {@code null}).
   * @param mustSort Indica se os caminhos para os arquivos precisam ser
   *        ordenados.
   * @param mustForceExtension Indica se o parmetro deve forar o uso da
   *        extenso no arquivo (Se ele no tiver, ele colocar a extenso).
   * @param usesFilter Indica se um painel de filtro pode ser usado.
   */
  public OutputFileListParameter(String name, String label, String description,
    List<FileURLValue> defaultValue, boolean isOptional, boolean isVisible,
    String commandLinePattern, String[] fileTypes, FileParameterMode mode,
    boolean mustSort, boolean usesFilter, boolean mustForceExtension) {
    super(name, label, description, defaultValue, isOptional, isVisible,
      commandLinePattern, fileTypes, mode, mustSort, usesFilter, false);
    if (fileTypes != null && fileTypes.length > 1) {
      String errorMessage =
        "Parmetro no aceita mltiplos tipos de arquivo associados.";
      throw new IllegalArgumentException(errorMessage);
    }
    this.mustForceExtension = mustForceExtension;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getType() {
    return TYPE_VALUE;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getIDLType() {
    return OutputFileListParameterHelper.id();
  }

  /**
   * Indica se deve forar a extenso do arquivo.
   *
   * @return .
   */
  public boolean mustForceExtension() {
    return this.mustForceExtension;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean setValue(List<FileURLValue> files) {
    checkExtension(files);
    return super.setValue(files);
  }

  /**
   * Verifica se deve forar a extenso do arquivo e a coloca caso necessrio.
   *
   * @param files lista de arquivos de sada
   */
  private void checkExtension(List<FileURLValue> files) {
    if ((files != null) && mustForceExtension()) {
      for (FileURLValue file : files) {
        String filePath = file.getPath();
        String fileExtension = FileUtils.getFileExtension(filePath);
        /* Obtm a lista de possveis extenses para o arquivo de sada */
        String[] types= this.getFileTypes();
        if (types == null || types.length != 1) {
          return;
        }

        String type = types[0];
        ProjectFileType projectFileType = ProjectFileType.getFileType(type);
        if (projectFileType == null) {
          return;
        }
        List<String> extensions = projectFileType.getExtensions();
        if (extensions.isEmpty()) {
          return;
        }
        /*
         * Se arquivo no tem extenso ou a extenso no confere com o tipo do
         * arquivo, fora a extenso
         */
        if ((fileExtension == null) || !extensions.contains(fileExtension)) {
          filePath += '.' + extensions.get(0);
          /* Renomeia o arquivo. */
          file.setPath(filePath);
        }
      }
    }
  }

}
