package csbase.logic.algorithms.parsers;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.IntegerColumn;
import csbase.logic.algorithms.parameters.IntegerListParameter;
import csbase.logic.algorithms.parameters.IntegerParameter;
import csbase.logic.algorithms.parameters.ParameterGroup;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.logic.algorithms.parameters.TableParameter;

/**
 * Classe abstrata que serve de base para analisadores de parmetros <T> com
 * valores numricos inteiros como {@link IntegerParameter} e
 * {@link IntegerListParameter}.
 * 
 * @param <T> Tipo do parmetro do qual essa classe faz parsing
 */
public abstract class AbstractIntegerParameterParser<T extends SimpleParameter<?>>
  extends SimpleParameterParser<T> {

  /**
   * <p>
   * O atributo {@value #INTEGER_ELEMENT_MAXIMUM_ATTRIBUTE} dos elementos:
   * <ul>
   * <li>{@link IntegerColumn} de um {@link TableParameter}</li>
   * <li>{@link IntegerListParameter}</li>
   * <li>{@link IntegerParameter}</li>
   * </ul>
   * </p>
   * <p>
   * Indica o valor mximo,  opcional e  do tipo inteiro.
   * </p>
   */
  public static final String INTEGER_ELEMENT_MAXIMUM_ATTRIBUTE = "maximo";

  /**
   * <p>
   * O atributo {@value #INTEGER_ELEMENT_MINIMUM_ATTRIBUTE} dos elementos:
   * <ul>
   * <li>{@link IntegerColumn} de um {@link TableParameter}</li>
   * <li>{@link IntegerListParameter}</li>
   * <li>{@link IntegerParameter}</li>
   * </ul>
   * </p>
   * <p>
   * Indica o valor mnimo,  opcional e  do tipo inteiro.
   * </p>
   */
  public static final String INTEGER_ELEMENT_MINIMUM_ATTRIBUTE = "minimo";

  /**
   * {@inheritDoc}
   */
  @Override
  public T createSimpleParameter(XmlParser parser, String name, String label,
    String description, boolean isOptional, boolean isVisible,
    String commandLinePattern, ParameterGroup group,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    Integer maximum = parser.extractAttributeValueAsInteger(
      INTEGER_ELEMENT_MAXIMUM_ATTRIBUTE, null, null, null);
    Integer minimum = parser.extractAttributeValueAsInteger(
      INTEGER_ELEMENT_MINIMUM_ATTRIBUTE, null, maximum, null);
    return createIntegerParameter(parser, name, label, description, isOptional,
      isVisible, commandLinePattern, maximum, minimum);
  }

  /**
   * Cria uma instncia do parmetro de tipo <T>, a partir dos atributos bsicos
   * de um parmetro numrico do tipo inteiro. As subclasses devem implementar
   * esse mtodo, fazendo a extrao dos demais atributos do parmetro.
   * 
   * @param parser Parser xml do configurador.
   * @param name Nome do parmetro.
   * @param label Rtulo do parmetro.
   * @param description Dica do parmetro.
   * @param isOptional Indica se o parmetro  opcional.
   * @param isVisible Indica se o parmetro deve ficar visvel.
   * @param commandLinePattern Padro da linha de comando do parmetro.
   * @param maximum Valor mximo aceito pelo parmetro.
   * @param minimum Valor mnimo aceito pelo parmetro.
   * @return Uma instncia do parmetro com os atributos bsicos e especficos
   *         preenchidos.
   * @throws ParseException Caso no seja possvel criar a instncia do
   *         parmetro com os atributos especificados.
   */
  protected abstract T createIntegerParameter(XmlParser parser, String name,
    String label, String description, boolean isOptional, boolean isVisible,
    String commandLinePattern, Integer maximum, Integer minimum)
      throws ParseException;

}
