package csbase.logic.algorithms.parsers;

import java.util.ArrayList;
import java.util.List;

import csbase.exception.ParseException;
import csbase.logic.ProjectFileType;
import csbase.logic.algorithms.parameters.FileParameterMode;
import csbase.logic.algorithms.parameters.FileParameterPipeAcceptance;
import csbase.logic.algorithms.parameters.OutputFileListParameter;
import csbase.logic.algorithms.parameters.OutputFileParameter;
import csbase.logic.algorithms.parameters.FileURLValue;

/**
 * <p>
 * Analisador de {@link OutputFileListParameter}.
 * </p>
 *
 * <p>
 * Este parser l os atributos de parmetros do tipo "arquivo de sada"
 * mltiplo. O elemento corrente do {@link XmlParser analisador de XML} precisa
 * ser um elemento {@link OutputFileListParameter}.
 * </p>
 *
 */
class OutputFileListParameterParser extends
AbstractFileParameterParser<OutputFileListParameter> {

  /**
   * <p>
   * O atributo
   * {@value #OUTPUT_FILE_PARAMETER_ELEMENT_MUST_FORCE_EXTENSION_ATTRIBUTE} do
   * elemento {@link OutputFileParameter}.
   * </p>
   *
   * Indica se o {@link OutputFileParameter arquivo de sada} deve criar uma
   * extenso se ela no for informada,  opcional, o seu valor-padro 
   * {@link #OUTPUT_FILE_PARAMETER_ELEMENT_MUST_FORCE_EXTENSION_DEFAULT_VALUE} e
   *  do tipo booleano. </p>
   */
  protected static final String OUTPUT_FILE_PARAMETER_ELEMENT_MUST_FORCE_EXTENSION_ATTRIBUTE =
    "forcar_extensao";

  /**
   * <p>
   * O valor-padro para o atributo
   * {@link #OUTPUT_FILE_PARAMETER_ELEMENT_MUST_FORCE_EXTENSION_ATTRIBUTE} do
   * elemento {@link OutputFileParameter}
   * </p>
   * <p>
   * O seu valor 
   * {@value #OUTPUT_FILE_PARAMETER_ELEMENT_MUST_FORCE_EXTENSION_DEFAULT_VALUE}.
   * </p>
   */
  protected static final boolean OUTPUT_FILE_PARAMETER_ELEMENT_MUST_FORCE_EXTENSION_DEFAULT_VALUE =
    false;

  /**
   * {@inheritDoc}
   */
  @Override
  public OutputFileListParameter createFileParameter(XmlParser parser,
    String name, String label, String description, boolean isOptional,
    boolean isVisible, String commandLinePattern, String[] types, FileParameterMode mode)
      throws ParseException {

    final boolean pipeDefined =
      parser.hasAttributeValue(FILE_PARAMETER_ELEMENT_CAN_USE_PIPE_ATTRIBUTE);
    if (pipeDefined) {
      final FileParameterPipeAcceptance usesPipe =
        parser.extractAttributeValueAsEnumeration(
          FILE_PARAMETER_ELEMENT_CAN_USE_PIPE_ATTRIBUTE,
          new FileParameterPipeAcceptanceConverter());
      if (usesPipe == FileParameterPipeAcceptance.TRUE
        || usesPipe == FileParameterPipeAcceptance.ALWAYS) {
        throw new ParseException(
          "Arquivo de sada (seleo mltipla) no admite pipes");
      }
    }

    if (types != null && types.length > 1) {
      throw new ParseException(
        "Arquivo de sada (seleo mltipla) no admite mltiplos tipos de" +
          " arquivo");

    }

    boolean useRootDirectoryAsDefault =
      parser.extractAttributeValueAsBoolean(
        FILE_PARAMETER_ELEMENT_USE_ROOT_AS_DEFAULT_DIRECTORY_ATTRIBUTE,
        FILE_PARAMETER_ELEMENT_USE_ROOT_AS_DEFAULT_DIRECTORY_DEFAULT_VALUE);
    List<FileURLValue> defaultValue = null;
    if (useRootDirectoryAsDefault) {
      if (mode == FileParameterMode.REGULAR_FILE) {
        throw new ParseException(
          "O atributo {0}  invlido quando o parmetro de sada  um arquivo.",
          FILE_PARAMETER_ELEMENT_USE_ROOT_AS_DEFAULT_DIRECTORY_ATTRIBUTE);
      }
      defaultValue = new ArrayList<FileURLValue>();
      defaultValue.add(new FileURLValue(".", ProjectFileType.DIRECTORY_TYPE));
    }

    boolean mustForceExtension =
      parser.extractAttributeValueAsBoolean(
        OUTPUT_FILE_PARAMETER_ELEMENT_MUST_FORCE_EXTENSION_ATTRIBUTE,
        OUTPUT_FILE_PARAMETER_ELEMENT_MUST_FORCE_EXTENSION_DEFAULT_VALUE);

    boolean mustSort =
      parser.extractAttributeValueAsBoolean(
        FILE_PARAMETER_ELEMENT_MUST_SORT_ATTRIBUTE,
        FILE_PARAMETER_ELEMENT_MUST_SORT_DEFAULT_VALUE);

    boolean useFilter =
      parser.extractAttributeValueAsBoolean(
        FILE_PARAMETER_ELEMENT_USE_FILTER_ATTRIBUTE,
        FILE_PARAMETER_ELEMENT_USE_FILTER_DEFAULT_VALUE);

    OutputFileListParameter parameter =
      new OutputFileListParameter(name, label, description, defaultValue,
        isOptional, isVisible, commandLinePattern, types, mode, mustSort,
        useFilter, mustForceExtension);
    return parameter;
  }
  
  @Override
  protected boolean acceptBothCategory() {
    return false;
  }
}
