package csbase.logic.algorithms.flows;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * <p>A conexo entre os ns de um fluxo de algoritmos.</p>
 *
 * @author lmoreira
 */
public final class FlowLink implements Serializable, Cloneable  {

  /**
   * O identificador.
   */
  private int id;
  
  /**
   * O parmetro de entrada.
   */
  private LinkParameter input;
  
  /**
   * O parmetro de sada.
   */
  private LinkParameter output;
  
  /**
   * A lista de pontos.
   */
  private List<Point> points;

  /**
   * Cria uma conexo.
   * 
   * @param id O identificador.
   * @param output O parmetro de sada (Aceita {@code null}).
   * @param input O parmetro de entrada (Aceita {@code null}).
   * @param points A lista de pontos (No aceita {@code null}).
   */
  public FlowLink(int id, LinkParameter output, LinkParameter input, List<Point> points) {
    setId(id);
    setOutput(output);
    setInput(input);
    setPoints(points);
  }

  @Override
  public FlowLink clone() {
    try {
      FlowLink clone = (FlowLink) super.clone();
      if (this.input != null) {
        clone.input = this.input.clone();
      }
      if (this.output != null) {
        clone.output = this.output.clone();
      }
      clone.points = new ArrayList<Point>(this.points.size());
      for (Point point : this.points) {
        clone.points.add(point.clone());
      }
      return clone;
    } catch (CloneNotSupportedException e) {
      throw new IllegalStateException(e);
    }
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }
    if (obj == null) {
      return false;
    }
    if (getClass() != obj.getClass()) {
      return false;
    }
    FlowLink other = (FlowLink) obj;
    return getId() == other.getId();
  }

  /**
   * Obtm o identificador desta conexo.
   *  
   * @return O identificador.
   */
  public int getId() {
    return this.id;
  }

  /**
   * Obtm o parmetro de entrada deste n.
   * 
   * @return O parmetro de entrada ou {@code null} se no houver parmetro de
   *         entrada.
   */
  public LinkParameter getInput() {
    return this.input;
  }

  /**
   * Obtm o parmetro de sada deste n.
   * 
   * @return O parmetro de sada ou {@code null} se no houver parmetro de
   *         sada.
   */
  public LinkParameter getOutput() {
    return this.output;
  }

  /**
   * <p>Obtm a lista de pontos de conexo.</p>
   * 
   * <p>A lista retornada  imutvel 
   * (veja {@link Collections#unmodifiableList(List)}).</p>
   * 
   * @return A lista de pontos ou uma lista vazia se no houver pontos. 
   */
  public List<Point> getPoints() {
    return Collections.unmodifiableList(this.points);
  }

  @Override
  public int hashCode() {
    return getId();
  }

  /**
   * Atribui um identificador a esta conexo.
   * 
   * @param id O identificador.
   */
  private void setId(int id) {
    this.id = id;
  }

  /**
   * Atribui um parmetro de entrada a esta conexo.
   * 
   * @param input O parmetro de entrada (Aceita {@code null}).
   */
  private void setInput(LinkParameter input) {
    this.input = input;
  }

  /**
   * Atribui um parmetro de sada a esta conexo.
   * 
   * @param output O parmetro de sada (Aceita {@code null}).
   */
  private void setOutput(LinkParameter output) {
    this.output = output;
  }

  /**
   * <p>Atribui pontos a esta conexo.</p>
   * 
   * <p>Armazena uma lista que  uma <i>shallow copy</i> da lista original.</p>
   * 
   * @param points A lista de pontos (No aceita {@code null}).
   */
  private void setPoints(List<Point> points) {
    if (points == null) {
      throw new IllegalArgumentException("O parmetro points est nulo.");
    }
    this.points = new ArrayList<Point>(points);
  }
}
