package csbase.logic.algorithms.parsers;

import java.io.IOException;
import java.io.Reader;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import csbase.exception.OperationFailureException;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.Category;

/**
 * <p>
 * Analisador de {@link Category Categorias}.
 * </p>
 * 
 * <p>
 * Este parser l e escreve um arquivo que representa o repositrio de
 * categorias.
 * </p>
 */
public abstract class CategoryParser {
  /**
   * O nome do atributo que informa o identificador do algoritmo filho de uma
   * categoria.
   */
  private static final String ALGORITHM_ELEMENT_ID_ATTRIBUTE_NAME = "id";

  /**
   * O nome do elemento que descreve o algoritmo filho de uma categoria.
   */
  private static final String ALGORITHM_ELEMENT_NAME = "algorithm";

  /**
   * O nome do atributo que informa o identificador da categoria.
   */
  private static final String CATEGORY_ELEMENT_ID_ATTRIBUTE_NAME = "id";
  /**
   * O nome do elemento que descreve uma categoria.
   */
  private static final String CATEGORY_ELEMENT_NAME = "category";
  /**
   * O nome do atributo que informa o nome da categoria.
   */
  private static final String CATEGORY_ELEMENT_NAME_ATTRIBUTE_NAME = "name";

  /**
   * A fbrica de documentos.
   */
  private DocumentBuilderFactory factory;

  // /**
  // * O repositrio de algoritmos.
  // */
  // private AlgorithmRepository repository;

  /**
   * Cria o analisador.
   */
  protected CategoryParser() {
    factory = DocumentBuilderFactory.newInstance();
    factory.setCoalescing(true);
    factory.setIgnoringComments(true);
    factory.setIgnoringElementContentWhitespace(true);
    factory.setExpandEntityReferences(false);
  }

  /**
   * L as categorias que so razes da floresta de categorias.
   * 
   * @param reader O leitor (No aceita {@code null}).
   * 
   * @return As categorias. Caso no hava categorias o conjunto retornado estar
   *         vazio.
   * 
   * @throws OperationFailureException Erro de sintaxe no XML.
   */
  public final Set<Category> read(Reader reader)
    throws OperationFailureException {
    if (reader == null) {
      throw new IllegalArgumentException("O parmetro reader est nulo.");
    }
    try {
      DocumentBuilder builder = factory.newDocumentBuilder();
      Document document = builder.parse(new InputSource(reader));
      return readCategories(document.getDocumentElement());
    }
    catch (ParserConfigurationException e) {
      throw new OperationFailureException("Erro nas configuraes do parser.",
        e);
    }
    catch (SAXException e) {
      Exception ex = e.getException();
      if (ex == null) {
        ex = e;
      }
      throw new OperationFailureException("Erro de sintaxe no XML.", ex);
    }
    catch (IOException e) {
      throw new OperationFailureException("Erro de IO.", e);
    }
  }

  /**
   * <p>
   * Obtm um algoritmo dado o seu identificador.
   * </p>
   * 
   * @param algorithmId O identificador do algoritmo (No aceita {@code null}).
   * 
   * @return O algoritmo ou {@code null} se o algoritmo no existir.
   * @throws OperationFailureException Se ocorrer um erro ao tentar obter
   *         informaes do algoritmo.
   */
  protected abstract AlgorithmInfo getAlgorithmById(String algorithmId)
    throws OperationFailureException;

  /**
   * Obtm o valor de uma atributo.
   * 
   * @param element O elemento (No aceita {@code null}).
   * @param attributeName O nome do atributo (No aceita {@code null}).
   * 
   * @return O valor do atributo.
   * 
   * @throws OperationFailureException Se o atributo no existir.
   */
  private String getAttributeValue(Element element, String attributeName)
    throws OperationFailureException {
    Attr attribute = element.getAttributeNode(attributeName);
    if (attribute == null) {
      throw new OperationFailureException(
        "O atributo {1} do elemento {0} no foi encontrado.\n", element
          .getNodeName(), attributeName);
    }
    return attribute.getNodeValue();
  }

  /**
   * Obtm os elementos filhos de um elemento.
   * 
   * @param element O elemento (No aceita {@code null}).
   * @param elementName O nome dos elementos (No aceita {@code null}).
   * 
   * @return Os filhos de um elemento (A lista estar vazia, caso no haja
   *         elementos filhos).
   */
  private List<Element> getChildElements(Element element, String elementName) {
    List<Element> elements = new LinkedList<Element>();
    NodeList children = element.getChildNodes();
    for (int i = 0; i < children.getLength(); i++) {
      Node childNode = children.item(i);
      if (childNode.getNodeType() == Node.ELEMENT_NODE) {
        Element childElement = (Element) childNode;
        if (childElement.getNodeName().equals(elementName)) {
          elements.add(childElement);
        }
      }
    }
    return elements;
  }

  /**
   * L todas as categorias filhas que so filhas do elemento informado.
   * 
   * @param element O elemento (No aceita {@code null}).
   * 
   * @return O conjunto com as categorias. Caso no haja categorias, o conjunto
   *         estar vazio.
   * 
   * @throws OperationFailureException Caso haja algum erro no XML.
   */
  private Set<Category> readCategories(Element element)
    throws OperationFailureException {
    Set<Category> categories = new HashSet<Category>();
    List<Element> elements = getChildElements(element, CATEGORY_ELEMENT_NAME);
    for (Element childElement : elements) {
      Category category = readCategory(childElement);
      // TODO REMOVER DEPOIS QUE MUDAR O LEITOR XML
      // System.out.println("CategoryParser: categoria : " + category.getId()
      // + " - " + category.getName());
      categories.add(category);
    }
    return categories;
  }

  /**
   * L a categoria descrita pelo elemento informado.
   * 
   * @param element O elemento que descreve a categoria (No aceita {@code null}
   *        ).
   * 
   * @return A categoria.
   * 
   * @throws OperationFailureException Caso haja algum erro no XML.
   */
  private Category readCategory(Element element)
    throws OperationFailureException {
    String name = getAttributeValue(element,
      CATEGORY_ELEMENT_ID_ATTRIBUTE_NAME);
    String id = getAttributeValue(element,
      CATEGORY_ELEMENT_NAME_ATTRIBUTE_NAME);
    Category newCategory = new Category(null, name, id);
    // TODO REMOVER DEPOIS QUE MUDAR O LEITOR XML
    // System.out.println("CategoryParser: categoria : " + id + " - " + name);
    Set<Category> categories = readCategories(element);
    for (Category childCategory : categories) {
      newCategory.addCategory(childCategory);
    }
    List<Element> elements = getChildElements(element, ALGORITHM_ELEMENT_NAME);
    for (Element algorithmElement : elements) {
      String algorithmId = getAttributeValue(algorithmElement,
        ALGORITHM_ELEMENT_ID_ATTRIBUTE_NAME);
      AlgorithmInfo algorithm = getAlgorithmById(algorithmId);
      if (algorithm != null) {
        newCategory.addAlgorithm(algorithm);
      }
    }
    return newCategory;
  }
}
