package csbase.logic.algorithms.parsers.elements;

import java.io.Serializable;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parsers.elements.attributes.IElementAttribute;

/**
 * Elemento lido pelo parser.
 */
public class ParsedElement implements Serializable {

  /**
   * Mapa de atributos e seus valores lidos pelo parser.
   */
  private final Map<IElementAttribute<?>, Object> attributeValues;
  /**
   * Mapa de elementos filhos lidos pelo parser.
   */
  private final Map<IElementStructure<?>, List<ParsedElement>> children;
  /**
   * Valor lido como contedo do elemento.
   */
  private final String contentValue;

  /**
   * Construtor para elementos sem filhos.
   *
   * @param attributeValues mapa de valores dos atributos.
   * @param contentValue valor do contedo (aceita {@code null}).
   * @throws ParseException em caso de erro.
   */
  public ParsedElement(Map<IElementAttribute<?>, Object> attributeValues,
    String contentValue) throws ParseException {
    this.attributeValues = attributeValues;
    this.children = new HashMap<>();
    this.contentValue = contentValue;
    checkValues();
  }

  /**
   * Construtor para elementos com filhos.
   *
   * @param attributeValues mapa de valores dos atributos.
   * @param children lista de elementos-filhos.
   * @throws ParseException em caso de erro.
   */
  public ParsedElement(Map<IElementAttribute<?>, Object> attributeValues,
    Map<IElementStructure<?>, List<ParsedElement>> children) throws
    ParseException {
    this.attributeValues = attributeValues;
    this.contentValue = null;
    if (children != null) {
      this.children = children;
    }
    else {
      this.children = new HashMap<>();
    }
    checkValues();
  }

  /**
   * Obtm o valor lido do contedo do elemento.
   *
   * @return o contedo do elemento.
   */
  public String getContentValue() {
    return contentValue;
  }


  /**
   * Verifica os valores dos atributos.
   *
   * @throws ParseException em caso de erro nos valores.
   */
  public void checkValues() throws ParseException {
    //TODO BELLA: Verificar filhos e contedo aqui tambm
    for (IElementAttribute<?> elementAttribute : getAttributes()) {
      elementAttribute.validate(this);
    }
  }

  /**
   * Verifica se existe algum elemento filho com um determinado nome.
   *
   * @param name o nome do elemento filho.
   * @return verdadeiro se existem filhos com o nome especificado ou falso, caso
   * contrrio.
   */
  public boolean hasChildren(String name) {
    IElementStructure<?> childElement = getChildElement(name);
    return childElement != null;
  }

  /**
   * Obtm a lista de elementos-filhos com um determinado nome.
   *
   * @param name o nome dos filhos.
   * @return a lista dos elementos encontrandos.
   */
  public List<ParsedElement> getChildren(String name) {
    IElementStructure<?> childElement = getChildElement(name);
    return children.get(childElement);
  }

  //TODO: Remover cast forando a barra aqui
  /**
   * Obtm o valor do atributo com o nome especificado.
   *
   * @param name o nome do atributo.
   * @param <E> o tipo do valor do atributo.
   * @return o valor do atributo.
   */
  public <E> E getAttributeValue(String name) {
    IElementAttribute<?> elementAttribute = getAttribute(name);
    if (elementAttribute == null) {
      throw new IllegalArgumentException("Atributo " + name + " no definido ");
    }
    return (E) attributeValues.get(elementAttribute);
  }

  /**
   * Obtm o atributo com o nome especificado.
   *
   * @param name o nome do atributo.
   * @return o atributo.
   */
  public IElementAttribute<?> getAttribute(String name) {
    for (IElementAttribute<?> elementAttribute : getAttributes()) {
      if (elementAttribute.getName().equals(name)) {
        return elementAttribute;
      }
    }
    return null;
  }

  /**
   * Obtm o elemento filho com o nome especificado.
   *
   * @param name o nome do elemento filho.
   * @return o atributo.
   */
  private IElementStructure<?> getChildElement(String name) {
    for (IElementStructure<?> child : getChildren()) {
      if (child.getName().equals(name)) {
        return child;
      }
    }
    return null;
  }

  /**
   * Obtm o conjunto de elementos filhos.
   *
   * @return o conjunto.
   */
  public Set<IElementStructure<?>> getChildren() {
    return children.keySet();
  }

  /**
   * Obtm o conjunto de atributos.
   *
   * @return o conjunto.
   */
  public Set<IElementAttribute<?>> getAttributes() {
    return attributeValues.keySet();
  }

  /**
   * Indica se o atributo foi definido.
   *
   * @param name o nome do atributo.
   * @return verdadeiro se o atributo foi definido ou falso, caso contrrio.
   */
  public boolean hasAttributeValue(String name) {
    return getAttribute(name) != null;
  }

}
