/**
 * $Id: ClientProjectFileList.java 175310 2016-08-03 14:38:03Z fpina $
 */
package csbase.logic;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

/**
 * Lista de filhos de um {@link ClientProjectFile}. Gerencia aspectos como
 * ordenao dos filhos e referncias para os ns-pai.
 * 
 * @author Tecgraf
 */
class ClientProjectFileList implements Serializable,
  Iterable<ClientProjectFile> {
  /**
   * Lista efetiva dos filhos.
   */
  private List<ClientProjectFile> children;
  /**
   * N-pai.
   */
  private ClientProjectFile parent;

  /**
   * Cria uma lista de filhos, inicializando-a com os elementos de um array.
   * 
   * @param parent n-pai
   */
  ClientProjectFileList(ClientProjectFile parent) {
    this.parent = parent;
    this.children = new ArrayList<ClientProjectFile>();
  }

  /**
   * Acrescenta um ou mais novos filhos  lista, caso estes ainda no existam.
   * 
   * @param clientSide <code>true</code> se o arquivo est sendo inserido no
   *        cliente. Se este parmetro for <code>false</code> o parmetro
   *        <code>fixParentsRecursively</code> ser ignorado.
   * @param fixParentsRecursively <code>true</code> se as referncias para os
   *        pais devem ser corrigidas recursivamente. Neste caso, os novos
   *        filhos apontaro para seu novo pai, e este processo se repetir
   *        recursivamente.<br>
   *        ATENO: as correes afetam apenas os novos filhos, os filhos j
   *        existentes no so alterados. Se este parmetro for
   *        <code>false</code>, apenas as referncias dos novos filhos diretos
   *        sero corrigidas.<br>
   * @param newChildren novos filhos
   */
  void addIfDoesntExist(boolean clientSide, boolean fixParentsRecursively,
    ClientProjectFile... newChildren) {
    if (!prepareNewChildren(clientSide, fixParentsRecursively, newChildren)) {
      return;
    }
    for (ClientProjectFile child : newChildren) {
      if (children.contains(child)) {
        continue;
      }
      _add(child, false);
    }
    sort();
  }

  /**
   * Acrescenta um ou mais filhos  lista.
   * <p>
   * ATENO: este mtodo permite repetio (i.e. o mesmo n pode ser inserido
   * mais de uma vez). Use-o apenas se tiver certeza de que os ns ainda no
   * pertencem  lista.
   * 
   * @param clientSide <code>true</code> se o arquivo est sendo inserido no
   *        cliente. Se este parmetro for <code>false</code> o parmetro
   *        <code>fixParentsRecursively</code> ser ignorado.
   * @param fixParentsRecursively <code>true</code> se as referncias para os
   *        pais devem ser corrigidas recursivamente. Neste caso, os novos
   *        filhos apontaro para seu novo pai, e este processo se repetir
   *        recursivamente.<br>
   *        ATENO: as correes afetam apenas os novos filhos, os filhos j
   *        existentes no so alterados. Se este parmetro for
   *        <code>false</code>, apenas as referncias dos novos filhos diretos
   *        sero corrigidas.<br>
   * @param newChildren novos filhos
   * 
   * @see #addIfDoesntExist(boolean, boolean, ClientProjectFile...)
   */
  void add(boolean clientSide, boolean fixParentsRecursively,
    ClientProjectFile... newChildren) {
    if (!prepareNewChildren(clientSide, fixParentsRecursively, newChildren)) {
      return;
    }
    for (ClientProjectFile child : newChildren) {
      _add(child, false);
    }
    sort();
  }

  /**
   * Ordena a lista de filhos.
   */
  private void sort() {
    Collections.sort(children);
  }

  /**
   * Prepara novos filhos para serem inseridos.
   * 
   * @param clientSide <code>true</code> se o arquivo est sendo inserido no
   *        cliente. Se este parmetro for <code>false</code> o parmetro
   *        <code>fixParentsRecursively</code> ser ignorado.
   * @param fixParentsRecursively <code>true</code> se as referncias para os
   *        pais devem ser corrigidas recursivamente. Neste caso, os novos
   *        filhos apontaro para seu novo pai, e este processo se repetir
   *        recursivamente.<br>
   *        ATENO: as correes afetam apenas os novos filhos, os filhos j
   *        existentes no so alterados. Se este parmetro for
   *        <code>false</code>, apenas as referncias dos novos filhos diretos
   *        sero corrigidas.<br>
   * @param newChildren novos filhos
   * @return <code>true</code> se o a preparao transcorreu com sucesso
   */
  private boolean prepareNewChildren(boolean clientSide,
    boolean fixParentsRecursively, ClientProjectFile... newChildren) {
    if (newChildren == null) {
      return false;
    }
    if (clientSide) {
      setParents(fixParentsRecursively, parent, newChildren);
    }
    return true;
  }

  /**
   * Acrescenta um novo filho  lista. No corrige as referncias para os pais.
   * 
   * @param child novo filho
   * @param sort <code>true</code> se a lista deve ser ordenada aps o acrscimo
   */
  private void _add(ClientProjectFile child, boolean sort) {
    children.add(child);
    if (sort) {
      sort();
    }
  }

  /**
   * Remove um filho da lista.
   * 
   * @param child filho a ser removido
   * @return ndice do filho, ou -1 se o filho no pertencia  lista
   */
  int remove(ClientProjectFile child) {
    int i = children.indexOf(child);
    if (i != -1) {
      children.remove(i);
    }
    return i;
  }

  /**
   * Cria um array contendo todos os filhos
   * 
   * @return array contendo todos os filhos
   */
  ClientProjectFile[] toArray() {
    return children.toArray(new ClientProjectFile[children.size()]);
  }

  /**
   * Corrige recursivamente as referncias para os pais.
   * 
   * @param recursive <code>true</code> se a correo deve ser aplicada
   *        recursivamente
   * @param parent n-pai
   * @param children lista com os filhos diretos do n-pai (pode ser
   *        <code>null</code>)
   */
  private static void setParents(boolean recursive, ClientProjectFile parent,
    ClientProjectFileList children) {
    if (children == null) {
      return;
    }
    for (ClientProjectFile child : children) {
      setParent(recursive, parent, child);
    }
  }

  /**
   * Corrige recursivamente as referncias para os pais.
   * 
   * @param recursive <code>true</code> se a correo deve ser aplicada
   *        recursivamente
   * @param parent n-pai
   * @param children array com os filhos diretos do n-pai
   */
  private static void setParents(boolean recursive, ClientProjectFile parent,
    ClientProjectFile... children) {
    for (ClientProjectFile child : children) {
      setParent(recursive, parent, child);
    }
  }

  /**
   * Corrige recursivamente a referncia para o n-pai.
   * 
   * @param recursive <code>true</code> se a correo deve ser aplicada
   *        recursivamente
   * @param parent n-pai
   * @param child filho
   */
  private static void setParent(boolean recursive, ClientProjectFile parent,
    ClientProjectFile child) {
    child.setParent(parent);
    if (recursive && child.isDirectory()) {
      setParents(recursive, child, child.getChildrenList());
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Iterator<ClientProjectFile> iterator() {
    return children.iterator();
  }

  /**
   * Redefine o n-pai.
   * 
   * @param newParent novo n-pai
   */
  void setParent(ClientProjectFile newParent) {
    for (ClientProjectFile child : children) {
      child.setParent(newParent);
    }
    this.parent = newParent;
  }

  /**
   * Indica se a lista est vazia.
   * 
   * @return {@code true} se a lista est vazia ou {@code false} caso contrrio.
   */
  boolean isEmpty() {
    return children.isEmpty();
  }
}
