/**
 * $Id: ProjectPermissions.java 176168 2016-09-22 21:12:51Z fpina $
 */
package csbase.logic;

import java.util.HashSet;
import java.util.Set;

import tecgraf.javautils.core.lng.LNG;

/**
 * 
 * 
 * @author Tecgraf
 */
public class ProjectPermissions {

  /**
   * Tipo do compartilhamento. Para evitar problemas de serializao, este enum
   * no  serializado diretamente para {@link CommonProjectInfo}; ao invs
   * disto, os inteiros associados a cada constante so armazenados.
   * 
   * ATENO: A ORDEM DAS CONSTANTES NO DEVE SER ALTERADA! CASO CONTRRIO, AS
   * CONFIGURAES J EXISTENTES FICARO INCORRETAS.
   * 
   * @author Tecgraf
   */
  public static enum SharingType {
    /**
     * Acesso privado.
     */
    PRIVATE,
    /**
     * Acesso pblico, apenas leitura.
     */
    ALL_RO,
    /**
     * Acesso pblico, leitura e escrita.
     */
    ALL_RW,
    /**
     * Acesso seletivo.
     * 
     * @see ProjectPermissions#USERS_RO
     * @see ProjectPermissions#USERS_RW
     */
    PARTIAL,
  }

  /**
   * Nome do parmetro que indica o tipo de compartilhamento.
   */
  private static final String SHARING_TYPE = "SHARING_TYPE";
  /**
   * Nome do parmetro que contm a lista de usurios (
   * <code>List&lt;Object&gt;</code>) com acesso apenas de leitura.
   * 
   * @see #USERS_RW
   */
  private static final String USERS_RO = "USERS_RO";
  /**
   * Nome do parmetro que contm a lista de usurios (
   * <code>List&lt;Object&gt;</code>) com acesso de leitura e escrita.
   * 
   * @see #USERS_RO
   */
  private static final String USERS_RW = "USERS_RW";

  /**
   * Constante interna usada para se obter as constantes de {@link SharingType}
   * a partir dos seus respectivos ordinais.
   */
  private static final SharingType[] SHARING_TYPE_CONSTANTS =
    ProjectPermissions.SharingType.values();

  /**
   * Obtm o tipo de compartilhamento armazenado nas informaes do projeto.
   * 
   * @param prjInfo - informaes do projeto
   * @return tipo de compartilhamento do projeto
   * @throws IllegalStateException se o tipo de compartilhamento ainda  o
   *         modelo antigo de permisses (apenas uma lista de usurios)
   */
  public static SharingType getSharingType(CommonProjectInfo prjInfo) {
    Integer typeOrdinal =
      (Integer) prjInfo.getAttribute(ProjectPermissions.SHARING_TYPE);
    if (typeOrdinal == null) {
      throw new IllegalStateException(LNG.get("csbase.logic.OldSharingType"));
    }
    return SHARING_TYPE_CONSTANTS[typeOrdinal];
  }

  /**
   * Define o tipo de compartilhamento do projeto.
   * 
   * @param prjInfo - informaes do projeto
   * @param type - tipo de compartilhamento
   */
  public static void setSharingType(CommonProjectInfo prjInfo, SharingType type) {
    prjInfo.setAttribute(SHARING_TYPE, type.ordinal());
  }

  /**
   * Obtm o conjunto de usurios que tm acesso apenas de leitura a um projeto.
   * <p>
   * <b>ATENO:</b>
   * <ul>
   * <li>retorna o prprio conjunto. Se for necessrio alter-lo, deve-se tomar
   * as devidas precaues
   * <li>o conjunto (vazio ou no) ser retornado mesmo que o tipo de
   * compartilhamento seja privado ou pblico
   * </ul>
   * 
   * @param prjInfo - informaes do projeto
   * @return conjunto de usurios que tm acesso apenas de leitura ao projeto
   */
  public static Set<Object> getUsersRO(CommonProjectInfo prjInfo) {
    return getUsers(prjInfo, true);
  }

  /**
   * Obtm o conjunto de usurios que tm acesso de leitura e escrita a um
   * projeto.
   * <p>
   * <b>ATENO:</b>
   * <ul>
   * <li>retorna o prprio conjunto. Se for necessrio alter-lo, deve-se tomar
   * as devidas precaues
   * <li>o conjunto (vazio ou no) ser retornado mesmo que o tipo de
   * compartilhamento seja privado ou pblico
   * </ul>
   * 
   * @param prjInfo - informaes do projeto
   * @return conjunto de usurios que tm acesso de leitura e escrita ao projeto
   */
  public static Set<Object> getUsersRW(CommonProjectInfo prjInfo) {
    return getUsers(prjInfo, false);
  }

  /**
   * Retorna um conjunto com a unio dos conjuntos dos usurios que tm acesso
   * RO e dos usurios que tm acesso RW a um projeto compartilhado
   * seletivamente.
   * <p>
   * <b>ATENO:</b>
   * <ul>
   * <li>retorna um novo conjunto a cada chamada, contendo todos os usurios de
   * ambos os conjuntos (RO e RW). Portanto, esta chamada no  necessariamente
   * "barata".
   * <li>o conjunto (vazio ou no) ser retornado mesmo que o tipo de
   * compartilhamento seja privado ou pblico
   * </ul>
   *
   * @param prjInfo - informaes do projeto
   * @return <b>novo</b> conjunto que  a unio do conjunto de usurios com
   *         acesso RO + usurios com acesso RW
   */
  public static Set<Object> getAllUsers(CommonProjectInfo prjInfo) {
    HashSet<Object> users = new HashSet<Object>(getUsersRO(prjInfo));
    users.addAll(getUsersRW(prjInfo));
    return users;
  }

  /**
   * Mtodo utilitrio para se obter os conjuntos de usurios que tm acesso a
   * um projeto.
   * 
   * @param prjInfo - informaes do projeto
   * @param readOnly - flag que indica se deseja-se os usurios que tm acesso
   *        RO (<code>true</code>) ou RW (<code>false</code>)
   * @return conjunto de usurios que tm o acesso especificado ao projeto
   */
  private static Set<Object> getUsers(CommonProjectInfo prjInfo,
    boolean readOnly) {
    Set<Object> users =
      (Set<Object>) prjInfo.getAttribute(readOnly ? USERS_RO : USERS_RW);
    if (users == null) {
      /*
       * no existia o conjunto de usurios com o acesso especificado, criamos
       * um novo conjunto vazio e o associamos ao usurio
       */
      users = new HashSet<Object>();
      setUsers(prjInfo, users, readOnly);
    }
    return users;
  }

  /**
   * Define o conjunto de usurios que tm um determinado tipo de acesso a um
   * projeto.
   * 
   * @param prjInfo - informaes do projeto
   * @param users - conjunto de usurios
   * @param readOnly - tipo de acesso: RO (<code>true</code>) ou RW (
   *        <code>false</code>)
   */
  public static void setUsers(CommonProjectInfo prjInfo, Set<Object> users,
    boolean readOnly) {
    prjInfo.setAttribute(readOnly ? USERS_RO : USERS_RW, users);
  }

  /**
   * Define o conjunto de usurios que tm acesso RO a um projeto.
   * 
   * @param prjInfo - informaes do projeto
   * @param usersRO - usurios que tm acesso RO ao projeto
   */
  public static void setUsersRO(CommonProjectInfo prjInfo, Set<Object> usersRO) {
    setUsers(prjInfo, usersRO, true);
  }

  /**
   * Define o conjunto de usurios que tm acesso RW a um projeto.
   * 
   * @param prjInfo - informaes do projeto
   * @param usersRW - usurios que tm acesso RW ao projeto
   */
  public static void setUsersRW(CommonProjectInfo prjInfo, Set<Object> usersRW) {
    setUsers(prjInfo, usersRW, false);
  }

  /**
   * Verifica se um projeto  pblico (compartilhamento == ALL_RO || ALL_RW).
   * 
   * @param prjInfo - informaes do projeto
   * @return true se o compartilhamento  ALL_RO ou ALL_RW
   */
  public static boolean isPublic(CommonProjectInfo prjInfo) {
    SharingType type = getSharingType(prjInfo);
    return type == SharingType.ALL_RO || type == SharingType.ALL_RW;
  }

  /**
   * Verifica se um projeto  privado.
   * 
   * @param prjInfo - informaes do projeto
   * @return true se o projeto  privado
   */
  public static boolean isPrivate(CommonProjectInfo prjInfo) {
    return getSharingType(prjInfo) == SharingType.PRIVATE;
  }

  /**
   * Verifica se um projeto  compartilhado.
   * 
   * @param prjInfo - informaes do projeto
   * @return true se o projeto  compartilhado
   */
  public static boolean isShared(CommonProjectInfo prjInfo) {
    return getSharingType(prjInfo) == SharingType.PARTIAL;
  }

  /**
   * Verifica se o usurio tem acesso de qualquer tipo ao projeto. Isto acontece
   * se qualquer uma das opes seguintes for satisfeita:
   * <ul>
   * <li>projeto  pblico (RO ou RW)
   * <li>usurio  o dono do projeto ou o admin
   * <li>compartilhamento  PARTIAL e usurio tem acesso RO ou RW ao projeto
   * </ul>
   * 
   * @param prjInfo - informaes do projeto
   * @param userID - identificador do usurio
   * @return <code>true</code> se o usurio tem acesso ao projeto (RO ou RW)
   */
  public static boolean userHasAccess(CommonProjectInfo prjInfo, Object userID) {
    if (isPublic(prjInfo) || prjInfo.userId.equals(userID)
      || User.isAdmin(userID)) {
      return true;
    }
    SharingType sharingType = getSharingType(prjInfo);
    if (sharingType == SharingType.PARTIAL) {
      return getUsersRO(prjInfo).contains(userID)
        || getUsersRW(prjInfo).contains(userID);
    }
    return false;
  }

  /**
   * Verifica se o compartilhamento  seletivo e um determinado usurio tem
   * acesso RO ao projeto.
   * 
   * @param prjInfo - informaes do projeto
   * @param userID - identificador do usurio
   * @return <code>true</code> se o compartilhamento  seletivo e o usurio tem
   *         acesso RO ao projeto
   */
  public static boolean userHasSelectiveAccessRO(CommonProjectInfo prjInfo,
    Object userID) {
    return getSharingType(prjInfo) == SharingType.PARTIAL
      && getUsersRO(prjInfo).contains(userID);
  }

  /**
   * Verifica se o usurio tem qualquer tipo de acesso RO ao projeto, ou seja,
   * se uma das seguintes condies  satisfeita:
   * <ul>
   * <li>o projeto  pblico apenas para leitura
   * <li>o compartilhamento  seletivo e o usurio possui acesso RO
   * </ul>
   * 
   * @param prjInfo - informaes do projeto
   * @param userID - identificador do usurio
   * @return <code>true</code> se o usurio tem qualquer tipo de acesso RO ao
   *         projeto
   */
  public static boolean userHasAccessRO(CommonProjectInfo prjInfo, Object userID) {
    return getSharingType(prjInfo) == SharingType.ALL_RO
      || userHasSelectiveAccessRO(prjInfo, userID);
  }

  /**
   * Verifica se o compartilhamento  seletivo e um determinado usurio tem
   * acesso RW ao projeto.
   * 
   * @param prjInfo - informaes do projeto
   * @param userID - identificador do usurio
   * @return <code>true</code> se o compartilhamento  seletivo e o usurio tem
   *         acesso RW ao projeto
   */
  public static boolean userHasSelectiveAccessRW(CommonProjectInfo prjInfo,
    Object userID) {
    return getSharingType(prjInfo) == SharingType.PARTIAL
      && getUsersRW(prjInfo).contains(userID);
  }

  /**
   * Verifica se o compartilhamento  seletivo e um determinado usurio tem
   * acesso RO ou RW ao projeto.
   * 
   * @param prjInfo - informaes do projeto
   * @param userID - identificador do usurio
   * @return <code>true</code> se o compartilhamento  seletivo e o usurio tem
   *         acesso RO ou RW ao projeto
   */
  public static boolean userHasSelectiveAccess(CommonProjectInfo prjInfo,
    Object userID) {
    return getSharingType(prjInfo) == SharingType.PARTIAL
      && (getUsersRO(prjInfo).contains(userID) || getUsersRW(prjInfo).contains(
        userID));
  }

  /**
   * Verifica se o usurio tem qualquer tipo de acesso RW ao projeto, ou seja,
   * se uma das seguintes condies  satisfeita:
   * <ul>
   * <li>usurio  o admin ou o dono do projeto
   * <li>o projeto  pblico para leitura e escrita
   * <li>o compartilhamento  seletivo e o usurio possui acesso RW
   * </ul>
   * 
   * @param prjInfo - informaes do projeto
   * @param userID - identificador do usurio
   * @return <code>true</code> se o usurio tem acesso RW ao projeto
   */
  public static boolean userHasAccessRW(CommonProjectInfo prjInfo, Object userID) {
    SharingType sharingType = getSharingType(prjInfo);
    if (sharingType == SharingType.ALL_RW || userOwnsProject(prjInfo, userID)) {
      return true;
    }
    return userHasSelectiveAccessRW(prjInfo, userID);
  }

  /**
   * Verifica se um usurio  dono de um projeto.
   * 
   * @param prjInfo - informaes do projeto
   * @param userID - identificador do usurio
   * @return <code>true</code> se o usurio  o dono do projeto ou  o admin
   */
  public static boolean userOwnsProject(CommonProjectInfo prjInfo, Object userID) {
    return User.isAdmin(userID) || prjInfo.userId.equals(userID);
  }
}
