/*
 * SGASet.java
 *
 * $Author: ericaflr $ $Revision: 167255 $ - $Date: 2007-12-19 11:39:33 -0200
 * (qua, 19 dez 2007) $
 */
package csbase.logic;

import java.util.Arrays;
import java.util.Collection;
import java.util.Enumeration;
import java.util.Map;
import java.util.Set;
import java.util.Vector;

/**
 * A classe <code>SGASet</code> representa para o cliente o gerenciador de uma
 * mquina hospedeira (ou servidor), capaz de atender  solicitao de execuo
 * remota de algoritmos. Alm de comandar a execuo remota de um algoritmo, um
 * objeto SGASet permite a consulta  configurao da mquina hospedeira
 * (<i>hostname</i>, plataforma de execuo, nmero de processadores, ...) e a
 * seu estado corrente (carga no processador, memria livre disponvel). Quando
 * o SGA  um cluster, a classe SGASet representa o cluster como um todo. Os
 * mtodos int getNumProcessors() int getRAMMemoryInfoMb() int
 * getSwapMemoryInfoMb() double getCPULoad1() double getCPULoad5() double
 * getCPULoad15() double getRAMFreeMemory() double getSwapFreeMemory() obtm a
 * informao do n principal, mas poderiam ser reescritos para refletir o
 * cluster (mdia ou sumatrio?). Quando o SGA no  um cluster, a classe SGASet
 * se confunde com a SGAInfo, que representa cada mquina (do cluster)
 * individualmente...
 */
public class SGASet implements MonitoringSet, Comparable<SGASet> {
  /** Estado do servidor */
  public enum Status {
    /** Disponvel */
    AVAILABLE,
    /** Sem acesso ao disco */
    DISKLESS,
    /** Reservado */
    RESERVED,
    /** Inacessvel */
    INACCESSIBLE,
    /** Ainda dentro do periodo de backoff */
    BACKOFF;

    /**
     * Retorna o ndice da constante no array 'values', ou -1 se no encontrada.
     *
     * @param constantToFind - a constante a ser procurada
     * @return .
     */
    public static int getIndex(Enum<Status> constantToFind) {
      for (int i = 0; i < values().length; i++) {
        if (values()[i] == constantToFind) {
          return i;
        }
      }
      return -1;
    }
  }

  /** Informaes do SGA que podem ser serializadas para o cliente. */
  private SGAInfo[] info = null;

  /** Nome que representa o SGA */
  private String name = null;

  /** Plataforma do SGA */
  private String platform = null;

  /** Indica se o SGA  um cluster. */
  private boolean isCluster;

  /** Indica se o SGA est apto a executar comandos. */
  private boolean enabled;

  /** Indica se o SGA est acessvel. */
  private boolean alive;

  /** Indica se o SGA tem acesso ao disco. */
  private boolean hasDiskAccess;

  /** Indica se o tempo de backoff do SGA expirou */
  private boolean backoffExpired;

  /** Infomaes dos jobs em execuo no SGA. */
  private String jobsInfo;

  /** Infomaes se o SGA est usando o CSFS. */
  private boolean CSFSEnabled;

  /** Mapa com todas as propriedades do SGA */
  // TODO mjulia Decidir se os campos acima podems ficar apenas como propriedades gerais.
  private Map<String, String> properties;
  
  /** Timestamp relativo  ltima atualizao de dados do SGA */
  private long updateTimestamp;

  /**
   * Calcula a plataforma do SGA.
   *
   * @return a plataforma do SGA.
   */
  private String updatePlatformId() {
    StringBuilder allPlats = new StringBuilder();
    if (info != null) {
      Vector<String> plats = new Vector<String>();
      for (int i = 0; i < info.length; i++) {
        String platformId = info[i].getPlatformId();
        if (!plats.contains(platformId)) {
          plats.add(platformId);
        }
      }
      for (Enumeration<String> e = plats.elements(); e.hasMoreElements();) {
        allPlats.append(e.nextElement());
        if (e.hasMoreElements()) {
          allPlats.append('/');
        }
      }
    }
    return allPlats.toString();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int compareTo(SGASet object) {
    return toString().compareToIgnoreCase(object.toString());
  }

  /**
   * Retorna as informaes de um n do SGA.
   *
   * @param hostName o nome do n desejado (se for passado null,  retornado o
   *        dado do n principal).
   *
   * @return um objeto do tipo <code>SGAInfo</code> com as informaes.
   */
  public SGAInfo getHost(String hostName) {
    int i = info.length - 1;
    while ((i > 0) && !(info[i].getHostName()).equals(hostName)) {
      i--;
    }
    return info[i];
  }

  /**
   * Retorna as informaes de um n do SGA.
   *
   * @param index o ndice do n desejado (se for passado zero,  retornado o
   *        dado do n principal).
   *
   * @return um objeto do tipo <code>SGAInfo</code> com as informaes.
   */
  public SGAInfo getInfo(int index) {
    return info[index];
  }

  /**
   * Retorna todas as informaes do SGA.
   *
   * @return as informaes do SGA.
   */
  public SGAInfo[] getAllInfo() {
    return info;
  }

  /**
   * Retorna todos os nomes dos ns do SGA.
   *
   * @return os nomes dos ns do SGA.
   */
  public String[] getAllNames() {
    String[] names = new String[info.length];
    for (int i = 0; i < info.length; i++) {
      names[i] = info[i].getHostName();
    }
    Arrays.sort(names);
    return names;
  }

  /**
   * Obtm as capacidades dos sgas medidas atravs dos benchmarks.
   *
   * @param capacityType Tipo de capacidade a ser obtida.
   * @return O valor da capacidade.
   */
  public long getCapacity(CapacityType capacityType) {
    return info[0].getCapacity(capacityType);
  }

  /**
   * Retorna o nmero de mquinas do SGA.
   *
   * @return o nmero de mquinas do SGA.
   */
  public int getNumNodes() {
    return info.length;
  }

  /**
   * Retorna o nmero de processadores.
   *
   * @return o nmero de processadores.
   */
  public int getNumProcessors() {
    return info[0].getNumProcessors(); //XXX
  }

  /**
   * Obtm o tamanho da memria RAM do SGA
   *
   * @return tamanho da memria RAM (em Mb)
   */
  public int getRAMMemoryInfoMb() {
    return info[0].getRAMMemoryInfoMb(); //XXX
  }

  /**
   * Obtm o tamanho da memria Swap do SGA
   *
   * @return tamanho da memria Swap (em Mb)
   */
  public int getSwapMemoryInfoMb() {
    return info[0].getSwapMemoryInfoMb(); //XXX
  }

  /**
   * Verifica se o SGA est apto a executar comandos.
   *
   * @return true se o SGA pode executar comandos, false caso contrrio
   */
  public boolean getEnabled() {
    return enabled;
  }

  /**
   * Atualiza disponibilidade do SGA para a execuo de comandos.
   *
   * @param enabled true se o SGA pode executar comandos, false caso contrrio
   */
  public void setEnabled(boolean enabled) {
    this.enabled = enabled;
  }

  /**
   * Obtm o nome do SGA.
   *
   * @return nome do SGA.
   */
  public String getName() {
    return this.name;
  }

  /**
   * Verifica se uma plataforma  suportada pelo SGA.
   *
   * @param platformId o identificador da plataforma.
   *
   * @return true se a plataforma for suportada, false caso contrrio.
   */
  public boolean isPlatformSupported(Collection<String> platformId) {
    for (int i = 0; i < info.length; i++) {
      String nodePlatform = info[i].getPlatformId();
      if (platformId.contains(nodePlatform)) {
        return true;
      }
    }
    return false;
  }

  /**
   * Obtm a plataforma do SGA.
   *
   * @return o identificador da plataforma
   */
  public String getPlatformId() {
    return this.platform;
  }

  /**
   * Obtm o nome de um n de uma das plataforma especificadas.
   *
   * @param plats a lista de identificadores das plataformas.
   *
   * @return o nome do n.
   */
  public String getHostOfPlatform(Collection<String> plats) {
    for (int i = 0; i < info.length; i++) {
      String platformId = info[i].getPlatformId();
      if (plats.contains(platformId)) {
        return info[i].getHostName();
      }
    }
    return null;
  }

  /**
   * Obtm a carga na mquina hospedeira no ltimo minuto.
   *
   * @return carga (percentual) no ltimo minuto.
   */
  public double getCPULoad1() {
    return info[0].getCPULoad1(); // XXX
  }

  /**
   * Obtm a carga na mquina hospedeira nos ltimos 5 minutos.
   *
   * @return carga (percentual) nos ltimos 5 minutos.
   */
  public double getCPULoad5() {
    return info[0].getCPULoad5(); // XXX
  }

  /**
   * Obtm a carga na mquina hospedeira nos ltimos 15 minutos.
   *
   * @return carga (percentual) nos ltimos 15 minutos.
   */
  public double getCPULoad15() {
    return info[0].getCPULoad15(); // XXX
  }

  /**
   * Obtm o percentual de memria RAM livre.
   *
   * @return percentual de memria RAM livre
   */
  public double getRAMFreeMemory() {
    return info[0].getRAMFreeMemory(); // XXX
  }

  /**
   * Obtm o percentual de memria RAM livre em MB
   *
   * @return percentual de memria RAM livre em MB
   */
  public double getRAMFreeMemoryMb() {
    double maxMem = info[0].getRAMFreeMemoryMb();
    int nnode, maxnode = info.length;
    for (nnode = 1; nnode < maxnode; nnode++) {
      if (info[nnode].getRAMFreeMemoryMb() > maxMem) {
        maxMem = info[nnode].getRAMFreeMemoryMb();
      }
    }
    return maxMem; // XXX
  }

  /**
   * Obtm o percentual de memria Swap livre.
   *
   * @return percentual de memria Swap livre
   */
  public double getSwapFreeMemory() {
    return info[0].getSwapFreeMemory(); // XXX
  }

  /**
   * Verifica se todos os ns atendem um determinado requisito.
   *
   * @param requirement o requisito que se deseja verificar.
   *
   * @return true se todos os ns atendem ao requisito
   */
  public boolean hasRequirement(String requirement) {
    for (SGAInfo node : info) {
      if (!node.hasRequirement(requirement)) {
        return false;
      }
    }
    return true;
  }

  /**
   * Verifica se todos os ns atendem a TODOS os determinados requisitos.
   *
   * @param requirements os requisitos que se deseja verificar.
   *
   * @return true se todos os ns atendem aos requisitos
   */
  public boolean hasRequirements(Set<String> requirements) {
    for (String requirement : requirements) {
      for (SGAInfo node : info) {
        if (!node.hasRequirement(requirement)) {
          return false;
        }
      }
    }
    return true;
  }

  /**
   * Retorna qual o estado do SGA: acessvel, inacessvel, reservado ou sem
   * acesso ao disco.
   *
   * @return o estado do SGA.
   */
  public Status getStatus() {
    if (getAlive()) {
      if (getEnabled()) {
        if (hasDiskAccess()) {
          if (!this.backoffExpired) {
            return Status.BACKOFF;
          }
          return Status.AVAILABLE;
        }
        return Status.DISKLESS;
      }
      return Status.RESERVED;
    }
    return Status.INACCESSIBLE;
  }

  /**
   * Indica se o SGA est apto a executar algum comando: est vivo, habilitado,
   * tem acesso ao disco e est fora do perodo de backoff.
   *
   * @return true se o SGA est apto, false caso contrrio
   */
  public boolean mayExecuteCommand() {
    return (getAlive() && getEnabled() && hasDiskAccess() && isBackoffExpired());
  }

  /**
   * Verifica se o SGA est acessvel.
   *
   * @return true se o SGA est acessvel, false caso contrrio
   */
  public boolean getAlive() {
    return alive;
  }

  /**
   * Atualiza acessibilidade do SGA. Invalida (zera) as informaes dinmicas se
   * o SGA est inacessvel.
   *
   * @param alive true se o SGA est acessvel, false caso contrrio
   */
  public void setAlive(boolean alive) {
    this.alive = alive;
    if (alive) {
      return;
    }
    for (int i = 0; i < info.length; i++) {
      info[i].setAlive(false);
    }
  }

  /**
   * Verifica se o SGA  um cluster.
   *
   * @return true se o SGA  um cluster, false se o SGA no  um cluster
   */
  public boolean isCluster() {
    return isCluster;
  }

  /**
   * Verifica se o SGA tem acesso ao disco.
   *
   * @return true se o SGA tem acesso, false caso contrrio
   */
  public boolean hasDiskAccess() {
    return hasDiskAccess;
  }

  /**
   * Informao sobre os jobs em execuo no SGA.
   *
   * @return informao sobre os jobs em execuo no SGA.
   */
  public String getJobsInfo() {
    return jobsInfo;
  }

  /**
   * Verifica se o SGA est usando o CSFS.
   *
   * @return true se o SGA est usnado o CSFS, false caso contrrio
   */
  public boolean isCSFSEnabled() {
    return CSFSEnabled;
  }

  /**
   * Valor de uma propriedade qualquer.
   *
   * @param key chave da propriedade
   * @return valor da propriedade
   */
  public String getProperty(String key) {

    return properties.get(key);
  }

  /**
   * Obtm lista de chaves de propriedades
   *
   * @return lista de chaves de propriedades
   *
   */
  public String[] getPropertiesKeys() {
    String[] keys = new String[properties.size()];
    return properties.keySet().toArray(keys);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(MonitoringSet o) {
    if (!(o instanceof SGASet)) {
      return false;
    }
    SGASet other = (SGASet) o;
    if (!getName().equals(other.getName())) {
      return false;
    }
    if ((getAlive() != other.getAlive())
      || (getEnabled() != other.getEnabled())) {
      return false;
    }
    if (hasDiskAccess() != other.hasDiskAccess()) {
      return false;
    }
    if (getCPULoad1() != other.getCPULoad1()) {
      return false;
    }
    if (getRAMMemoryInfoMb() != other.getRAMMemoryInfoMb()) {
      return false;
    }
    if (getSwapMemoryInfoMb() != other.getSwapMemoryInfoMb()) {
      return false;
    }
    if (!getPlatformId().equals(other.getPlatformId())) {
      return false;
    }

    // TODO mjulia Verificar se as propriedades deveriam entrar no critrio de comparao.
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    StringBuffer buffer = new StringBuffer();
    buffer.append(getName());
    buffer.append(Boolean.toString(getAlive()));
    buffer.append(Boolean.toString(getEnabled()));
    buffer.append(Double.toString(getCPULoad1()));
    buffer.append(Integer.toString(getRAMMemoryInfoMb()));
    buffer.append(Integer.toString(getSwapMemoryInfoMb()));
    buffer.append(getPlatformId());
    return buffer.toString().hashCode();
  }

  /**
   * Obtm o n principal.
   *
   * @return O n principal.
   */
  public SGAInfo getMainInfo() {
    return this.info[0];
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getKey() {
    return getName();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return getName();
  }

  /**
   * Verifica se o perodo de backoff do SGA j expirou.
   *
   * @return {@code true} se o perodo de backoff expirou e {@code false} caso
   *         contrrio
   */
  public boolean isBackoffExpired() {
    return this.backoffExpired;
  }
  
  /**
   * Obtm timestamp de update das informaes.
   * 
   * @return timestamp de update das informaes
   */
  public long getUpdateTimestamp() {
	  return this.updateTimestamp;
  }

  /**
   * Constri a representao de um SGA.
   *
   * @param info nome da mquina hospedeira
   * @param name referncia remota para o SGA
   * @param enabled .
   * @param alive .
   * @param hasDiskAccess indica se o SGA est conseguindo escrever em seu
   *        disco.
   * @param backoffExpired indica se o perodo de backoff do SGA j expirou
   * @param CSFSEnabled indica se o SGA est usando o CSFS
   * @param jobsInfo informao dos jobs em execuo no SGA.
   * @param properties
   */

  public SGASet(SGAInfo[] info, String name, boolean enabled, boolean alive,
    boolean hasDiskAccess, boolean backoffExpired, String jobsInfo,
    boolean CSFSEnabled, Map<String, String> properties, long updateTimestamp) {
    this.info = info;
    this.name = name;
    this.isCluster = (info.length > 1) ? true : false;
    this.enabled = enabled;
    setAlive(alive);
    this.hasDiskAccess = hasDiskAccess;
    this.backoffExpired = backoffExpired;
    this.platform = updatePlatformId();
    this.jobsInfo = jobsInfo;
    this.CSFSEnabled = CSFSEnabled;
    this.properties = properties;
    this.updateTimestamp = updateTimestamp;
  }
}
