package csbase.logic.algorithms.parameters;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import csbase.exception.algorithms.ExpressionFunctionExecutionException;
import tecgraf.javautils.core.lng.LNG;

/**
 * Assinatura de um mtodo.
 * 
 * @author lmoreira
 */
public abstract class ExpressionFunction {

  /** Nome do mtodo. */
  private String name;

  /** Lista dos tipos dos parmetros. */
  private List<Class<?>> parameterTypes;

  /**
   * Cria uma assinatura.
   * 
   * @param name O nome do mtodo.
   * @param parameterTypes O array com os tipos dos parmetros.
   */
  public ExpressionFunction(String name, Class<?>... parameterTypes) {
    setName(name);
    setParameterTypes(parameterTypes);
  }

  /**
   * Valida os parmetros do mtodo.
   * 
   * @param parameters os parmetros.
   * @return verdadeiro, se parmetros so aceitos ou falso, caso contrrio.
   */
  public final boolean acceptParameters(Object... parameters) {
    if (parameters == null) {
      throw new IllegalArgumentException(MessageFormat.format(LNG.get(
    		  "csbase.logic.algorithms.nullParameter"),
    		  "parameters"));
    }
    if (parameters.length != this.parameterTypes.size()) {
      return false;
    }
    for (int i = 0; i < this.parameterTypes.size(); i++) {
      Object parameter = parameters[i];
      Class<?> expectedParameterType = this.parameterTypes.get(i);
      if (parameter == null) {
        if (expectedParameterType.isPrimitive()) {
          return false;
        }
      }
      else {
        Class<? extends Object> currentParameterType = parameter.getClass();
        if (currentParameterType.isPrimitive()) {
          currentParameterType = getWrapperType(currentParameterType);
        }
        if (!expectedParameterType.isAssignableFrom(currentParameterType)) {
          return false;
        }
      }
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final boolean equals(Object other) {
    if (other == null) {
      return false;
    }
    if (!(other instanceof ExpressionFunction)) {
      return false;
    }
    ExpressionFunction function = (ExpressionFunction) other;
    if (!function.name.equals(this.name)) {
      return false;
    }
    if (function.parameterTypes.size() != this.parameterTypes.size()) {
      return false;
    }
    for (int i = 0; i < this.parameterTypes.size(); i++) {
      if (!this.parameterTypes.get(i).isAssignableFrom(
        function.parameterTypes.get(i))) {
        return false;
      }
    }
    return true;
  }

  /**
   * Executa o mtodo.
   * 
   * @param configurator o configurador.
   * @param parameters os parmetros do mtodo.
   * @return object o retorno do mtodo.
   * 
   * @throws ExpressionFunctionExecutionException em caso de falha na execuo
   *         do mtodo.
   */
  public final Object execute(SimpleAlgorithmConfigurator configurator,
    Object... parameters) throws ExpressionFunctionExecutionException {
    if (configurator == null) {
      throw new IllegalArgumentException(MessageFormat.format(LNG.get(
    		  "csbase.logic.algorithms.nullParameter"),
    		  "configurator"));
    }
    if (parameters == null) {
      throw new IllegalArgumentException(MessageFormat.format(LNG.get(
    		  "csbase.logic.algorithms.nullParameter"),
    		  "parameters"));
    }
    if (!acceptParameters(parameters)) {
      throw new ExpressionFunctionExecutionException(LNG.get(
		  "csbase.logic.algorithms.parameters.InvalidFunctionParameter"),
		  toString(), Arrays.asList(parameters));
    }
    return doOperation(configurator, parameters);
  }

  /**
   * Obtm o nome do mtodo.
   * 
   * @return O nome do mtodo.
   */
  public final String getName() {
    return this.name;
  }

  /**
   * Obtm uma lista imutvel de tipos dos parmetros do mtodo.
   * 
   * @return a lista de tipos.
   */
  public final List<Class<?>> getParameterTypes() {
    return Collections.unmodifiableList(this.parameterTypes);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final int hashCode() {
    return this.name.hashCode();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final String toString() {
    StringBuffer buffer = new StringBuffer();
    String separator = "";
    buffer.append(this.name);
    buffer.append("(");
    for (Class<?> parameterType : this.parameterTypes) {
      buffer.append(separator);
      buffer.append(getName(parameterType));
      separator = ", ";
    }
    buffer.append(")");
    return buffer.toString();
  }

  /**
   * Mtodo que executa a operao.
   * 
   * @param configurator o configurador.
   * @param parameters os parmetros.
   * @return o resultado da operao
   * 
   * @throws ExpressionFunctionExecutionException em caso de falha durante a
   *         execuo.
   */
  protected abstract Object doOperation(
    SimpleAlgorithmConfigurator configurator, Object... parameters)
    throws ExpressionFunctionExecutionException;

  /**
   * Obtm o nome de uma classe.<br>
   * Quando temos array uni ou multidimensionais, este mtodo colocar os "[]"
   * adequados.
   * 
   * @param typeClass A classe.
   * 
   * @return O nome da classe.
   */
  private static String getName(Class<?> typeClass) {
    if (typeClass.isArray()) {
      return getName(typeClass.getComponentType()) + "[]";
    }
    return typeClass.getName();
  }

  /**
   * Obtm o tipo que "empacota" o tipo primitivo definido.
   * 
   * @param primitiveType o tipo primitivo
   * @return wrapper type o tipo que "empacota" o primitivo.
   */
  private static Class<?> getWrapperType(Class<?> primitiveType) {
    if (boolean.class.equals(primitiveType)) {
      return Boolean.class;
    }
    if (byte.class.equals(primitiveType)) {
      return Byte.class;
    }
    if (char.class.equals(primitiveType)) {
      return Character.class;
    }
    if (double.class.equals(primitiveType)) {
      return Double.class;
    }
    if (float.class.equals(primitiveType)) {
      return Float.class;
    }
    if (int.class.equals(primitiveType)) {
      return Integer.class;
    }
    if (long.class.equals(primitiveType)) {
      return Long.class;
    }
    if (short.class.equals(primitiveType)) {
      return Short.class;
    }
    throw new IllegalArgumentException(LNG.get(
    		"csbase.logic.algorithms.parameters.NotPrimitiveType") +
    		primitiveType);
  }

  /**
   * Atribui o nome do mtodo.
   * 
   * @param name o nome do mtodo.
   */
  private void setName(String name) {
    if (name == null) {
      throw new IllegalArgumentException(MessageFormat.format(LNG.get(
    		  "csbase.logic.algorithms.nullParameter"),
    		  "name"));
    }
    this.name = name;
  }

  /**
   * Ajusta a lista dos tipos dos parmetros do mtodo.
   * 
   * @param parameterTypes a lista de tipos.
   */
  private void setParameterTypes(Class<?>... parameterTypes) {
    if (parameterTypes == null) {
      throw new IllegalArgumentException(MessageFormat.format(LNG.get(
    		  "csbase.logic.algorithms.nullParameter"),
    		  "parameterTypes"));
    }
    this.parameterTypes = new ArrayList<Class<?>>(parameterTypes.length);
    for (Class<?> parameterType : parameterTypes) {
      if (parameterType == null) {
        throw new IllegalArgumentException(LNG.get(
            "csbase.logic.algorithms.parameters.UnacceptedNullParameters") +
    		Arrays.asList(parameterTypes));
      }
      if (parameterType.isPrimitive()) {
        parameterType = getWrapperType(parameterType);
      }
      this.parameterTypes.add(parameterType);
    }
  }
}
