package csbase.logic.eventlogservice;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;

/**
 * Agregador das informaes obtidas dos logs para o perdo especificado. No
 * contm obrigatoriamente os dados de todas as tabelas. Estes dados sero
 * preenchidos conforme demanda, atravs do parser de logs de eventos.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class LogsInfo implements Serializable {

  /**
   * Tabela com os dados de logins regitrados num determinado perodo, as
   * colunas so:
   * <p>
   * 0 - Data (yyyy/MM/dd)<br>
   * 1 - Usurio
   * </p>
   */
  private List<String[]> loginTable = new ArrayList<String[]>();

  /**
   * Tabela com os dados de aplicaes registrados nos logs num determinado
   * perodo, as colunas so:
   * <p>
   * 0 - Data (yyyy/MM/dd)<br>
   * 1 - Id da Aplicao<br>
   * 2 - Descrio da Aplicao <br>
   * </p>
   */
  private List<String[]> applicationsTable = new ArrayList<String[]>();

  /**
   * Tabela com os dados de execues regitrados num determinado perodo, as
   * colunas so:
   * <p>
   * 0 - Data (yyyy/MM/dd)<br>
   * 1 - Algoritmo <br>
   * 2 - Usurio <br>
   * </p>
   */
  private List<String[]> executionsTable = new ArrayList<String[]>();

  /**
   * Parmetro que indica a data de incio do perodo pesquisado.
   */
  private final Date startDate;

  /**
   * Parmetro que indica a data de trmino do perodo pesquisado.
   */
  private final Date endDate;

  /**
   * Construtor.
   */
  public LogsInfo() {
    this(null, null);
  }

  /**
   * COnstrutor.
   * 
   * @param startDate data de incio
   * @param endDate data de trmino
   */
  public LogsInfo(Date startDate, Date endDate) {
    this.startDate = startDate;
    this.endDate = endDate;
  }

  /**
   * @return data de incio
   */
  public Date getStartDate() {
    return this.startDate;
  }

  /**
   * @return data de trmino
   */
  public Date getEndDate() {
    return this.endDate;
  }

  /**
   * @return informaes dos logins realizados no perodo associado.
   */
  public List<String[]> getLoginTable() {
    return loginTable;
  }

  /**
   * @return informaes das aplicaes utilizadas no perodo associado.
   */
  public List<String[]> getApplicationsTable() {
    return applicationsTable;
  }

  /**
   * 
   * @return informaes das execues realizadas no perodo associado.
   */
  public List<String[]> getExecutionsTable() {
    return executionsTable;
  }

  /**
   * Adiciona uma linha de informao na tabela de logins.
   * 
   * @param strDate string que representa a data no formato yyyy/MM/dd
   * @param user usurio associado a operao
   */
  public void addLineLogin(String strDate, String user) {
    loginTable.add(new String[] { strDate, user });
  }

  /**
   * Adiciona uma linha de informao na tabela de aplicaes.
   * 
   * @param strDate string que representa a data no formato yyyy/MM/dd
   * @param appId identificador da aplicao
   * @param appDesc descrio da aplicao
   */
  public void addLineApplication(String strDate, String appId, String appDesc) {
    applicationsTable.add(new String[] { strDate, appId, appDesc });
  }

  /**
   * Adiciona uma linha de informao na tabela de execues.
   * 
   * @param strDate string que representa a data no formato yyyy/MM/dd
   * @param algorithm algoritmo associado a operao
   * @param user usurio associado a operao
   */
  public void addLineExecution(String strDate, String algorithm, String user) {
    executionsTable.add(new String[] { strDate, algorithm, user });
  }

  /**
   * @return todas as aplicaes associadas aos dados da busca. Pode retornar um
   *         conjunto vazio, caso no existam dados de aplicaes.
   */
  public Set<String> getApplications() {
    Set<String> applications = new TreeSet<String>();
    for (String[] line : applicationsTable) {
      String app = line[2]; // coluna 2 descrio do algoritmo
      applications.add(app);
    }
    return applications;
  }

  /**
   * @return todos os algoritmos associados aos dados da busca. Pode retornar um
   *         conjunto vazio, caso no existam dados de execues.
   */
  public Set<String> getAlgorithms() {
    Set<String> algorithms = new TreeSet<String>();
    for (String[] line : executionsTable) {
      String algorithm = line[1]; // coluna 1 algoritmo
      algorithms.add(algorithm);
    }
    return algorithms;
  }

  /**
   * Consulta um mapa com o nmero total de execues por usurios (chave).
   * Considera na pesquisa apenas as execues dos algoritmos fornecidos como
   * parmetro.
   * 
   * @param filterAlgorithms lista de algoritmos da qual se deseja obter os
   *        dados de execuo
   * 
   * @return mapa com nmero de execues por USURIO
   */
  public Map<String, Integer> getExecutionsByUser(List<String> filterAlgorithms) {
    Map<String, Integer> map = new TreeMap<String, Integer>();
    for (String[] line : executionsTable) {

      String algorithm = line[1]; // coluna 1 algoritmo
      if (!filterAlgorithms.contains(algorithm)) {
        continue;
      }

      String user = line[2]; // coluna 2 usurio
      Integer value = map.get(user);
      if (value == null) {
        value = 0;
      }
      map.put(user, value + 1);
    }
    return map;
  }

  /**
   * Consulta um mapa com o nmero total de execues por algoritmos (chave).
   * Considera na pesquisa apenas as execues dos algoritmos fornecidos como
   * parmetro.
   * 
   * @param filterAlgorithms lista de algoritmos da qual se deseja obter os
   *        dados de execuo
   * 
   * @return mapa com nmero de execues por ALGORITMO
   */
  public Map<String, Integer> getExecutionsByAlgorithm(
    List<String> filterAlgorithms) {
    Map<String, Integer> map = new TreeMap<String, Integer>();
    for (String[] line : executionsTable) {

      String algorithm = line[1]; // coluna 1 algoritmo
      if (!filterAlgorithms.contains(algorithm)) {
        continue;
      }

      Integer value = map.get(algorithm);
      if (value == null) {
        value = 0;
      }
      map.put(algorithm, value + 1);
    }
    return map;
  }

  /**
   * Consulta um mapa com o nmero total de execues por usurio (chave).
   * 
   * @return mapa com nmero de logins por USURIO
   */
  public Map<String, Integer> getLoginsByUser() {
    Map<String, Integer> map = new TreeMap<String, Integer>();
    for (String[] line : loginTable) {
      String user = line[1];
      Integer currentValue = map.get(user);
      if (currentValue == null) {
        currentValue = 0;
      }
      map.put(user, currentValue + 1);
    }
    return map;
  }

}
