/*
 * $Id: FileTransferRequest.java 149541 2014-02-11 12:51:59Z pietroguedes $
 */
package csbase.logic.filetransferservice;

import java.io.Serializable;

/**
 * Transferncia do servio de transferncia de arquivos. Ver
 * FileTransferService.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class FileTransferRequest implements Serializable {

  /**
   * Identificador da requisio (interno)
   */
  final private String id;

  /**
   * tipo
   */
  final private FileTransferRequestType type;

  /**
   * conexo.
   */
  final private FileTransferConnection connection;

  /**
   * Arquivo remoto associado
   */
  final private String remoteFilePath;

  /**
   * Arquivo local associado
   */
  final private String[] localFilePath;

  /**
   * Tempo do enfileiramento.
   */
  private long queueTime = -1;

  /**
   * Tempo atual
   */
  private long currentTime = -1;

  /**
   * Tempo de incio.
   */
  private long startTime = -1;

  /**
   * Estado da requisio.
   */
  private FileTransferRequestStatus status = FileTransferRequestStatus.NONE;

  /**
   * Tempo do trmino
   */
  private long endTime = -1;

  /**
   * Mensagem de erro (se fizer sentido).
   */
  private String errorMessage = null;

  /**
   * Tamanho tarnsmitido.
   */
  private long totalBytesTransferred = 0;

  /**
   * Tamanho total da transferncia
   */
  final private long totalBytesSize;

  /**
   * Acumula tamanho atual da transferncia.
   * 
   * @param bytesTransferred tamnaho a ser adicionado.
   */
  final public void addTransferredSize(final long bytesTransferred) {
    this.totalBytesTransferred += bytesTransferred;
    currentTime = System.currentTimeMillis();
  }

  /**
   * Consulta o valor de connection
   * 
   * @return o valor
   */
  public final FileTransferConnection getConnection() {
    return connection;
  }

  /**
   * @return o valor de endTime.
   */
  public final long getEndTime() {
    return endTime;
  }

  /**
   * Consulta o valor de errorMessage
   * 
   * @return o valor
   */
  public final String getErrorMessage() {
    return errorMessage;
  }

  /**
   * Consulta o valor de id
   * 
   * @return o valor
   */
  public final String getId() {
    return id;
  }

  /**
   * Consulta o valor de localFilePath
   * 
   * @return o valor
   */
  public final String[] getLocalFilePath() {
    return localFilePath;
  }

  /**
   * @return o percentual
   */
  public double getPercentage() {
    if (totalBytesSize <= 0) {
      return -1.0;
    }
    final double ratio = ((double) totalBytesTransferred) / totalBytesSize;
    return ratio * 100.;
  }

  /**
   * @return o valor de queueTime.
   */
  public final long getQueueTime() {
    return queueTime;
  }

  /**
   * Consulta o valor de remoteFilePath
   * 
   * @return o valor
   */
  public final String getRemoteFilePath() {
    return remoteFilePath;
  }

  /**
   * @return o valor de startTime.
   */
  public final long getStartTime() {
    return startTime;
  }

  /**
   * Consulta o valor de status
   * 
   * @return o valor
   */
  synchronized public final FileTransferRequestStatus getStatus() {
    return status;
  }

  /**
   * Consulta o tamanho atual da transferncia.
   * 
   * @return tamanho transferido.
   */
  final public long getTransferredSize() {
    return totalBytesTransferred;
  }

  /**
   * Consulta a taxa global de transferncia.
   * 
   * @return a taxa
   */
  final public double getGlobalTransferRateKBytesSec() {
    if (totalBytesTransferred < 0) {
      return -1;
    }

    if (status != FileTransferRequestStatus.RUNNING) {
      return -2;
    }
    final long dt = (currentTime - getStartTime()) / 1000;
    return totalBytesTransferred / 1024. / dt;
  }

  /**
   * Consulta o tamanho total da transferncia
   * 
   * @return o tamanho
   */
  public final long getTotalBytesSize() {
    return totalBytesSize;
  }

  /**
   * Consulta o valor de type
   * 
   * @return o valor
   */
  public final FileTransferRequestType getType() {
    return type;
  }

  /**
   * Marca a requisio como iniciada.
   */
  synchronized final public void markCancelled() {
    status = FileTransferRequestStatus.INTERRUPTED;
    endTime = System.currentTimeMillis();
  }

  /**
   * Marcaco do estado como nenhum apenas para efeito de debbuging, caso seja
   * necessrio. Esse cdigo no tem efeito prtico.
   */
  public void markDeleted() {
    status = FileTransferRequestStatus.NONE;

  }

  /**
   * Marca requisio com terminada.
   */
  synchronized final public void markEnd() {
    if (status == FileTransferRequestStatus.ERROR) {
      return;
    }
    if (status == FileTransferRequestStatus.INTERRUPTED) {
      return;
    }
    status = FileTransferRequestStatus.FINISHED;
    endTime = System.currentTimeMillis();
    totalBytesTransferred = totalBytesSize;
  }

  /**
   * Marca requisio com terminada.
   * 
   * @param msg mensgem de erro.
   */
  synchronized final public void markError(final String msg) {
    status = FileTransferRequestStatus.ERROR;
    endTime = System.currentTimeMillis();
    this.errorMessage = msg;
  }

  /**
   * Marca a requisio como enfileirada.
   */
  synchronized final public void markQueued() {
    status = FileTransferRequestStatus.QUEUED;
    queueTime = System.currentTimeMillis();
  }

  /**
   * Marca a requisio como iniciada.
   */
  synchronized final public void markRunning() {
    status = FileTransferRequestStatus.RUNNING;
    startTime = System.currentTimeMillis();
  }

  /**
   * Construtor
   * 
   * @param requestType tipo da requisio
   * @param connection conexo
   * @param remoteFilePath path remoto
   * @param localFilePath path local
   * @param size tamanho da transferncia.
   */
  public FileTransferRequest(final FileTransferRequestType requestType,
    final FileTransferConnection connection, final String remoteFilePath,
    final String[] localFilePath, final long size) {
    this.type = requestType;
    this.connection = connection;
    this.remoteFilePath = remoteFilePath;
    this.localFilePath = localFilePath;
    this.status = FileTransferRequestStatus.NONE;
    this.totalBytesSize = size;

    final long milis = System.currentTimeMillis();
    final String userName = connection.getUserName();
    final Object projectId = connection.getProjectId();
    final String prefix = userName + "::" + projectId + "::" + remoteFilePath;
    this.id = "[[" + prefix + "::" + requestType + "::" + milis + "]]";
  }
}
