package csbase.logic.algorithms.serializer;

import java.util.HashMap;
import java.util.Map;

import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.flows.configurator.FlowAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import tecgraf.javautils.core.lng.LNG;

/**
 * Fbrica de serializadores de configuraco de AlgorithmConfigurator.
 * 
 * @author Tecgraf / PUC-Rio
 */
public class AlgorithmConfigurationSerializerFactory {

  private static final IAlgorithmConfigurationSerializer DEFAULT_SERIALIZER;

  private static final Map<Class<?>, IAlgorithmConfigurationSerializer> SERIALIZERS;

  static {
    DEFAULT_SERIALIZER = new DefaultAlgorithmConfigurationSerializer();

    SERIALIZERS = new HashMap<Class<?>, IAlgorithmConfigurationSerializer>();
    SERIALIZERS.put(SimpleAlgorithmConfigurator.class, DEFAULT_SERIALIZER);
    SERIALIZERS.put(FlowAlgorithmConfigurator.class,
      new FlowAlgorithmConfigurationSerializer());
  }

  /**
   * 
   * @return O serializador padro utilizado para (des)serializar as
   *         configuraces de um AlgorithmConfigurator.
   */
  public static final IAlgorithmConfigurationSerializer getDefaultSerializer() {
    return DEFAULT_SERIALIZER;
  }

  /**
   * @param <T> tipo do configurador de algortmos que deseja ter suas
   *        configuraces serializadas.
   * @param algConfClass classe do configurador de algortmos que deseja ter
   *        suas configuraces serializadas (No aceita {@code null}).
   * 
   * @return um serializador utilizado para (des)serializar as configuraces de
   *         um AlgorithmConfigurator do tipo T. Caso nenhum tenha sido
   *         encontrado, o serializador padro ser retornado.
   */
  public static final <T extends AlgorithmConfigurator> IAlgorithmConfigurationSerializer getSerializer(
    Class<T> algConfClass) {

    if (null == algConfClass) {
      throw new IllegalArgumentException(LNG.get(
        "csbase.logic.algorithms.serializer.NullAlgConfClass"));
    }

    IAlgorithmConfigurationSerializer serializer =
      SERIALIZERS.get(algConfClass);
    if (null != serializer) {
      return serializer;
    }

    return getDefaultSerializer();
  }

  /**
   * @param serializerClassName nome da classe de um serializador (No aceita
   *        {@code null}).
   * 
   * @return o serializador com o nome da classe que foi passado como parmetro,
   *         ou o serializador padro caso o primeiro no tenha sido encontrado.
   */
  public static IAlgorithmConfigurationSerializer getSerializer(
    String serializerClassName) {

    if (null == serializerClassName) {
      throw new IllegalArgumentException(LNG.get(
        "csbase.logic.algorithms.serializer.NullSerializerClassName"));
    }

    for (IAlgorithmConfigurationSerializer serializer : SERIALIZERS.values()) {
      if (serializer.getClass().getName().equals(serializerClassName)) {
        return serializer;
      }
    }

    return getDefaultSerializer();
  }
}
