/*
 * $Id$
 */
package csbase.logic.algorithms.xml.algorithmspack;

import java.io.IOException;
import java.io.Writer;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;

import csbase.logic.algorithms.AlgorithmVersionId;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import tecgraf.javautils.xml.ImprovedXMLListIterator;
import tecgraf.javautils.xml.XMLBasicElement;
import tecgraf.javautils.xml.XMLElementInterface;

/**
 * Esta classe implementa o elemento XML que representa um conjunto de verses
 * do algoritmo.
 *
 */
public class XMLVersionSetElement extends XMLBasicElement {
  /**
   * Mapa de verses do algoritmo
   */
  private Hashtable<AlgorithmVersionId, AlgorithmVersionInfo> versions;

  /**
   * Retorna as verses do algoritmo.
   *
   * @return retorna as verses do algoritmo
   */
  public Hashtable<AlgorithmVersionId, AlgorithmVersionInfo> getVersions() {
    return versions;
  }

  /**
   * Constri o elemento xml que descreve um conjunto de verses do algoritmo.
   */
  public XMLVersionSetElement() {
    super();
    setTag(XmlAlgorithmsPackUtils.VERSIONS_TAG);
    versions = new Hashtable<AlgorithmVersionId, AlgorithmVersionInfo>();
  }

  /**
   * Executado quando a tag de fim do elemento XML  encontrada. Deve-se criar o
   * objeto correspondente na camada da aplicao.
   *
   * @see tecgraf.javautils.xml.XMLElementInterface#endTag(java.util.List)
   */
  @Override
  public void endTag(List<XMLElementInterface> childrenList) {
    int major = 0, minor = 0, patch = 0;
    AlgorithmVersionId versionId = null;
    AlgorithmVersionInfo versionInfo = null;

    ImprovedXMLListIterator iterator = new ImprovedXMLListIterator(
      childrenList);

    while (iterator.hasNext()) {
      XMLElementInterface nextElem = iterator.next();

      //L as verses do algoritmo
      if (isVersionElement(nextElem)) {
        major = getPropertyMajor(nextElem);
        minor = getPropertyMinor(nextElem);
        patch = getPropertyPatch(nextElem);
        if (major != -1 && minor != -1 && patch != -1) {
          versionId = new AlgorithmVersionId(major, minor, patch);
          String versionDesc = ((XMLVersionElement) nextElem).getDescription();
          Map<String, String> versionProps = ((XMLVersionElement) nextElem)
            .getProperties();
          versionInfo = createVersionInfo(versionId, versionDesc, versionProps);
          versions.put(versionId, versionInfo);
        }
      }
    }
  }

  /**
   * Cria as informaes da verso do algoritmo a partir dos dados
   * especificados.
   *
   * @param id identificador do algoritmo
   * @param description descrio da verso do algoritmo
   * @param propertyValues valores das propriedades do algoritmo
   *
   * @return retorna a verso do algoritmo com as informaes bsicas
   */
  private AlgorithmVersionInfo createVersionInfo(AlgorithmVersionId id,
    String description, Map<String, String> propertyValues) {
    return new AlgorithmVersionInfo(id, description, propertyValues, null);
  }

  /**
   * Verifica se  o elemento que representa uma verso do algoritmo.
   *
   * @param xmlElem elemento xml
   * @return retorna true se o elemento representa uma verso do algoritmo, caso
   *         contrrio, retorna false
   */
  private boolean isVersionElement(XMLElementInterface xmlElem) {
    return xmlElem.getTag().equals(XmlAlgorithmsPackUtils.VERSION_TAG);
  }

  /**
   * Obtm o valor equivalente ao patch da verso.
   *
   * @param xmlElem elemento xml
   *
   * @return o valor equivalente ao patch da verso
   */
  private int getPropertyPatch(XMLElementInterface xmlElem) {
    String patchStrValue = xmlElem.getAttributeStrValue(
      XmlAlgorithmsPackUtils.VERSION_PATCH_ATTR);
    return Integer.parseInt(patchStrValue);
  }

  /**
   * Obtm o valor equivalente ao minor da verso.
   *
   * @param xmlElem elemento xml
   *
   * @return o valor equivalente ao minor da verso
   */
  private int getPropertyMinor(XMLElementInterface xmlElem) {
    String minorStrValue = xmlElem.getAttributeStrValue(
      XmlAlgorithmsPackUtils.VERSION_MINOR_ATTR);
    return Integer.parseInt(minorStrValue);
  }

  /**
   * Obtm o valor equivalente ao major da verso.
   *
   * @param xmlElem elemento xml
   *
   * @return o valor equivalente ao major da verso
   */
  private int getPropertyMajor(XMLElementInterface xmlElem) {
    String majorStrValue = xmlElem.getAttributeStrValue(
      XmlAlgorithmsPackUtils.VERSION_MAJOR_ATTR);
    return Integer.parseInt(majorStrValue);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void write(Writer arg0, String arg1) throws IOException {
    // TODO Auto-generated method stub

  }

  /**
   * @return Mapa das verses do algoritmo.
   */
  public Hashtable<AlgorithmVersionId, AlgorithmVersionInfo> getVersionsValues() {
    return versions;
  }

}
