/*
 * $Id: UserGroup.java 170738 2015-12-16 17:39:22Z fpina $
 */
package csbase.logic;

import java.io.Serializable;
import java.rmi.RemoteException;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.List;
import java.util.Observable;
import java.util.Observer;
import java.util.Vector;

import csbase.exception.administration.AdministrationDeleteException;
import csbase.remote.ClientRemoteLocator;
import csbase.util.restart.RestartListener;
import csbase.util.restart.RestartManager;
import tecgraf.javautils.core.lng.LNG;

/**
 * A classe <code>UserGroup</code> representa um tipo de Grupo de Usurios. Todo
 * usurio no CSBase est associado (pertence) a um grupo. Mantm uma cache dos
 * UserGroups instanciados localmente.
 * 
 * @author $Author: fpina $
 * @version $Revision: 170738 $
 */
public class UserGroup implements Serializable, IdInterface {
  /**
   * Representa a UserGroup na Hashtable de atributos do usurio na UserInfo
   */
  public static final String USERGROUP_ID = "unId";

  /**
   * Cache local de UserGroups
   */
  private static Hashtable<Object, UserGroup> userGroups =
    new Hashtable<Object, UserGroup>();

  /**
   * Indica se a cache est completa
   */
  private static boolean hasAllUserGroups = false;

  /**
   * Identificao de uma UserGroup
   */
  private final Object id;

  /**
   * Informaes de uma UserGroup
   */
  private final UserGroupInfo info;

  /**
   * Objeto que permite a observao da classe UserGroup
   */
  private static Observable observable = new Observable() {
    @Override
    public void notifyObservers(Object arg) {
      setChanged();
      super.notifyObservers(arg);
    }
  };

  /**
   * Cria uma UserGroup. S ser executado no servidor. Objetos so serializados
   * para o cliente.
   * 
   * @param id o identificador da UserGroup
   * @param info os dados da UserGroup
   */
  public UserGroup(Object id, UserGroupInfo info) {
    this.id = id;
    this.info = info;
  }

  /**
   * Verifica se dois UserGroups so iguais. Um UserGroup  igual ao outro se
   * ambos possuem o mesmo identificador.
   * 
   * @param obj o outro objeto com o qual esse UserGroup est sendo comparado
   * 
   * @return .
   */
  @Override
  public boolean equals(Object obj) {
    if (!(obj instanceof UserGroup)) {
      return false;
    }
    UserGroup userGroup = (UserGroup) obj;
    return userGroup.getId().equals(id);
  }

  /**
   * Calcula o cdigo hash do objeto.
   * 
   * @return Inteiro que representa o cdigo hash do objeto.
   */
  @Override
  public int hashCode() {
    return id.hashCode();
  }

  /**
   * Indica se contedo  igual.
   * 
   * @param obj objeto
   * @return indicativo
   */
  public boolean equalContents(Object obj) {
    if (!(obj instanceof UserGroup)) {
      return false;
    }
    UserGroup userGroup = (UserGroup) obj;
    return userGroup.getId().equals(id)
      && userGroup.getUserGroupInfo().equals(info);
  }

  /**
   * Obtem a UserGroup que possui uma determinada identificao. A UserGroup 
   * guardada no cache, caso ela no esteja l.
   * 
   * @param id a identificao de uma UserGroup
   * 
   * @return a UserGroup que possui a identificao requisitada ou null caso
   *         essa UserGroup no exista.
   * 
   * @throws RemoteException falha de rmi
   */
  public static UserGroup getUserGroup(Object id) throws RemoteException {
    if (id == null) {
      return null;
    }
    UserGroup userGroup = userGroups.get(id);
    if ((userGroup == null) && !hasAllUserGroups) {
      userGroup = ClientRemoteLocator.administrationService.getUserGroup(id);
      if (userGroup != null) {
        userGroups.put(id, userGroup);
      }
    }
    return userGroup;
  }

  /**
   * Solicita uma UserGroup que possui um determinado nome.
   * 
   * @param name o nome da UserGroup procurada
   * 
   * @return a UserGroup que possui o nome procurado ou null caso essa UserGroup
   *         no exista.
   * 
   * @throws RemoteException falha de rmi
   */
  public static UserGroup getUserGroupByName(String name)
    throws RemoteException {
    List<UserGroup> allUserGroups = UserGroup.getAllUserGroups();
    for (UserGroup userGroup : allUserGroups) {
      if (userGroup.getName().equalsIgnoreCase(name)) {
        return userGroup;
      }
    }
    return null;
  }

  /**
   * Solicita uma UserGroup a partir de um usurio.
   * 
   * @param user usurio que se deseja obter a UserGroup.
   * 
   * @return UserGroup
   * 
   * @throws RemoteException falha de rmi
   */
  public static UserGroup getUserGroupFromUser(User user)
    throws RemoteException {
    Object userGroupId = user.getAttribute(USERGROUP_ID);
    if (userGroupId == null) {
      return null;
    }
    UserGroup userGroup = getUserGroup(userGroupId);
    return userGroup;
  }

  /**
   * Solicita a lista de todos os grupos de usurios cadastrados no sistema. Os
   * UserGroups so colocados no cache.
   * 
   * @return um vetor com todos os UserGroups
   * 
   * @throws RemoteException falha de rmi
   */
  public static List<UserGroup> getAllUserGroups() throws RemoteException {
    if (hasAllUserGroups) {
      return userGroupsToVector();
    }
    List<UserGroup> allUserGroups =
      ClientRemoteLocator.administrationService.getAllUserGroups();
    for (UserGroup userGroup : allUserGroups) {
      if (!userGroups.containsKey(userGroup.getId())) {
        userGroups.put(userGroup.getId(), userGroup);
      }
    }
    if (allUserGroups.size() > 0) {
      hasAllUserGroups = true;
    }
    return allUserGroups;
  }

  /**
   * Cria uma nova UserGroup no sistema. No se pode criar uma nova UserGroup
   * com o mesmo nome de outra j existente. A UserGroup criada  colocada na
   * cache.
   * 
   * @param info os dados da nova UserGroup
   * 
   * @return a UserGroup criada ou null caso j exista uma UserGroup com o mesmo
   *         nome
   * 
   * @throws Exception em aso de falha.
   */
  public static UserGroup createUserGroup(UserGroupInfo info) throws Exception {
    if ((info.name == null) || info.name.trim().equals("")) {
      throw new Exception(LNG.get("csbase.logic.EmptyName"));
    }
    if (getUserGroupByName(info.name) != null) {
      return null;
    }
    UserGroup userGroup =
      ClientRemoteLocator.administrationService.createUserGroup(info);
    if (userGroup != null) {
      userGroups.put(userGroup.getId(), userGroup);
    }
    return userGroup;
  }

  /**
   * Modifica um UserGroup no sistema. No se pode modificar o nome de um
   * UserGroup se j existir uma outro com esse nome. O UserGroup modificado 
   * colocado no cache.
   * 
   * @param id o identificador do grupo a ser modificado
   * @param info os dados do grupo a ser modificado
   * 
   * @return o UserGroup modificado ou null caso j exista um UserGroup com o
   *         mesmo nome ou no exista UserGroup com o id especificado
   * 
   * @throws RemoteException falha de rmi
   */
  public static UserGroup modifyUserGroup(Object id, UserGroupInfo info)
    throws RemoteException {
    UserGroup userGroup = getUserGroup(id);
    if (userGroup == null) {
      return null;
    }
    if (!userGroup.getName().equals(info.name)
      && existsAnotherUserGroup(id, info.name)) {
      return null;
    }
    userGroup =
      ClientRemoteLocator.administrationService.modifyUserGroup(id, info);
    userGroups.put(id, userGroup);
    return userGroup;
  }

  /**
   * Verifica se j existe um outro UserGroup com o mesmo nome.
   * 
   * @param id o identificador de um UserGroup
   * @param name o nome sendo procurado
   * 
   * @return true se existir uma outro UserGroup com o nome procurado ou false
   *         no caso de no existir
   * 
   * @throws RemoteException falha de rmi
   */
  public static boolean existsAnotherUserGroup(Object id, String name)
    throws RemoteException {
    UserGroup anotherUserGroup = getUserGroupByName(name);
    if ((anotherUserGroup != null) && !anotherUserGroup.getId().equals(id)) {
      return true;
    }
    return false;
  }

  /**
   * Remove um UserGroup do sistema. O UserGroup removido  retirado do cache.
   * 
   * @param id a identificao do UserGroup a ser removido do sistema.
   * 
   * @throws RemoteException falha de rmi
   * @throws AdministrationDeleteException caso haja usurio cadastrado no grupo
   */
  public static void deleteUserGroup(Object id) throws RemoteException,
    AdministrationDeleteException {
    ClientRemoteLocator.administrationService.deleteUserGroup(id);
    userGroups.remove(id);
  }

  /**
   * Devolve um vetor com todos os UserGroups da cache.
   * 
   * @return um vetor com todos os UserGroups da cache
   */
  private static Vector<UserGroup> userGroupsToVector() {
    Vector<UserGroup> allUserGroups = new Vector<UserGroup>();
    for (Enumeration<UserGroup> e = userGroups.elements(); e.hasMoreElements();) {
      allUserGroups.add(e.nextElement());
    }
    return allUserGroups;
  }

  /**
   * Obtm um comparator de <code>UserGroup</code> pelo critrio de ordem
   * alfabtica do nome do grupo de usurios.
   * 
   * @return um comparador por nome
   */
  public static Comparator<UserGroup> getNameComparator() {
    Comparator<UserGroup> userGroupComparator = new Comparator<UserGroup>() {
      @Override
      public int compare(UserGroup ug1, UserGroup ug2) {
        return ug1.getName().compareTo(ug2.getName());
      }
    };
    return userGroupComparator;
  }

  /**
   * Obtm um comparator de <code>UserGroup</code> pelo critrio de ordem
   * alfabtica da descrio do grupo de usurios.
   * 
   * @return um comparador por descrio
   */
  public static Comparator<UserGroup> getDescComparator() {
    Comparator<UserGroup> userGroupComparator = new Comparator<UserGroup>() {
      @Override
      public int compare(UserGroup ug1, UserGroup ug2) {
        return ug1.getDesc().compareTo(ug2.getDesc());
      }
    };
    return userGroupComparator;
  }

  /**
   * Obtm um getter para o nome do <code>UserGroup</code>.
   * 
   * @return um getter para o nome
   */
  public static Getter getNameGetter() {
    Getter nameGetter = new Getter() {
      @Override
      public Object get(Object o) {
        UserGroup userGroup = (UserGroup) o;
        return userGroup.getName();
      }
    };
    return nameGetter;
  }

  /**
   * Obtm um getter para o identificador do <code>UserGroup</code>.
   * 
   * @return um getter para o identificador
   */
  public static Getter getIdGetter() {
    Getter idGetter = new Getter() {
      @Override
      public Object get(Object o) {
        UserGroup userGroup = (UserGroup) o;
        return userGroup.getId();
      }
    };
    return idGetter;
  }

  /**
   * Obtm um getter para a descrio de <code>UserGroup</code>.
   * 
   * @return um getter para a descrio
   */
  public static Getter getDescGetter() {
    Getter infoGetter = new Getter() {
      @Override
      public Object get(Object o) {
        UserGroup userGroup = (UserGroup) o;
        return userGroup.getDesc();
      }
    };
    return infoGetter;
  }

  /**
   * Adiciona um observador local da classe. Sempre que a classe  avisada pelo
   * servio de administrao de que alguma mudana ocorreu, os observadores
   * locais tambm so notificados.
   * 
   * @param o um observador local
   */
  public static void addObserver(Observer o) {
    observable.addObserver(o);
  }

  /**
   * Remove um observador local da classe.
   * 
   * @param o o observador a ser removido
   */
  public static void deleteObserver(Observer o) {
    observable.deleteObserver(o);
  }

  /**
   * Esse mtodo  chamado quando um servio de administrao sofre alguma
   * alterao relativa aos grupos de usurios. Atualiza o cache de grupo de
   * usurios e notifica seus observadores locais.
   * 
   * @param event o evento que ocorreu no servio de administrao
   */
  public static void update(AdministrationEvent event) {
    UserGroup userGroup = (UserGroup) event.item;
    Object id = userGroup.getId();
    switch (event.type) {
      case AdministrationEvent.CREATE:
      case AdministrationEvent.MODIFY:
        userGroups.put(id, userGroup);
        break;
      case AdministrationEvent.DELETE:
        userGroups.remove(id);
        break;
    }
    observable.notifyObservers(event);
  }

  /**
   * Obtm o identificador do grupo de usurios.
   * 
   * @return o identificador do grupo de usurios
   */
  @Override
  public Object getId() {
    return id;
  }

  /**
   * Obtm o nome do grupo.
   * 
   * @return o nome do grupo
   */
  public String getName() {
    return info.name;
  }

  /**
   * Obtm a descrio do grupo de usurios.
   * 
   * @return a descrio do grupo de usurios
   */
  public String getDesc() {
    return info.desc;
  }

  /**
   * Obtm as informaes do grupo de usurios
   * 
   * @return UserGroupInfo
   */
  public UserGroupInfo getInfo() {
    return info;
  }

  /**
   * Obtm uma cpia do UserGroupInfo deste grupo de usurios.
   * 
   * @return a cpia do UserGroupInfo
   */
  public UserGroupInfo getUserGroupInfo() {
    return (UserGroupInfo) info.clone();
  }

  /**
   * Muda o nome do grupo de usurios.
   * 
   * @param name o novo nome
   */
  public void setName(String name) {
    this.info.name = name;
  }

  /**
   * Muda a descrio do grupo de usurios.
   * 
   * @param desc a nova informao
   */
  public void setDesc(String desc) {
    this.info.desc = desc;
  }

  /**
   * Obtm uma descrio de um grupo de usurios.
   * 
   * @return um texto que descreve as informaes desse UserGroup.
   */
  @Override
  public String toString() {
    String userGroupString = "";
    userGroupString += info.name;
    userGroupString += "\t";
    userGroupString += info.desc;
    return userGroupString;
  }

  static {
    RestartManager.getInstance().addListener(new RestartListener() {
      @Override
      public void restart() {
        UserGroup.userGroups = new Hashtable<Object, UserGroup>();
        UserGroup.hasAllUserGroups = false;
        UserGroup.observable.deleteObservers();
      }
    });
  }
}
