/**
 * $Id: AbstractConsoleApp.java 154918 2014-08-01 00:49:36Z fpina $
 */
package csbase.console;

import java.io.Console;
import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.rmi.RemoteException;
import java.util.Collection;
import java.util.Map;

import org.kohsuke.args4j.CmdLineException;
import org.kohsuke.args4j.CmdLineParser;

import tecgraf.javautils.core.lng.FormatUtils;
import csbase.console.remote.ClientRemoteMonitor;
import csbase.exception.CSBaseException;
import csbase.logic.User;
import csbase.logic.Version;
import csbase.logic.diagnosticservice.DeploymentInfo;
import csbase.logic.diagnosticservice.ServerBasicInfo;
import csbase.remote.ClientRemoteLocator;

/**
 * Superclasse para os clientes console.
 * 
 * @author Tecgraf
 */
public abstract class AbstractConsoleApp {

  /**
   * Identao para a sada.
   */
  protected static final String IDENT = "    ";
  /**
   * URI do servidor.
   */
  private String serverURI;
  /**
   * Monitor da conexo com o servidor.
   */
  protected ClientRemoteMonitor clientMonitor;
  /**
   * Porta para conexo com o servidor.
   */
  private int port;
  /**
   * Senha.
   */
  private String password;
  /**
   * Parmetros da linha de comando.
   */
  private BasicParams params;
  /**
   * Stream de sada (pode apontar para um arauivo).
   */
  private PrintStream out = System.out;
  /**
   * Indica se a sada est sendo redirecionada para um arquivo.
   */
  private boolean outputToFile = false;
  /**
   * Diretrio corrente, que pode ter sido definido via
   * <code>-DcurrDir=xxx</code>.
   */
  private String currentDir;

  /**
   * Construtor.
   * 
   * @param _serverURI URI do servidor
   * @param _port porta do servidor (-1 para obter da linha de comando)
   * @param _password senha
   */
  protected AbstractConsoleApp(String _serverURI, int _port, String _password) {
    setPassword(_password);
    setServerURI(_serverURI);
    setPort(_port);
    createClientMonitor();
  }

  /**
   * Construtor.
   * 
   * @param args parmetros recebidos pela linha de comando
   */
  protected AbstractConsoleApp(String[] args) {
    params = createParams();
    CmdLineParser parser = new CmdLineParser(params);
    try {
      parser.parseArgument(args);
    }
    catch (CmdLineException e) {
      if (!params.showHelp) {
        System.err.println(e.getMessage());
        showUsage(parser, System.err);
        System.exit(1);
      }
    }
    /*
     * se usurio quis ver o help, mostramos e samos com cdigo 0
     */
    if (params.showHelp) {
      showUsage(parser, System.out);
      System.exit(0);
    }
    /*
     * processa os parmetros bsicos
     */
    processBasicParams();
    /*
     * processa os parmetros adicionais
     */
    processExtraParams();
    createClientMonitor();
  }

  /**
   * Exibe informaes complementares ao help dos parmetros.
   * 
   * @param parser parser dos parmetros
   * @param stream stream de sada
   */
  private static void showUsage(CmdLineParser parser, PrintStream stream) {
    parser.printUsage(stream);
    stream.println();
    stream.println("OBSERVAES:\n");
    stream
      .println("- projetos de outros usurios podem ser referenciados atravs");
    stream.println("  da sintaxe user:prj/path");
  }

  /**
   * Processa os parmetros bsicos (URI, porta, senha).
   */
  protected void processBasicParams() {
    setServerURI(params.serverURI);
    setPort(params.port);
    currentDir = System.getProperty("currDir");
    if (currentDir == null) {
      currentDir = System.getProperty("user.dir");
      printInfo("usando " + currentDir + " como diretrio corrente");
    }
    /*
     * para permitir execuo no Eclipse onde o console no  interativo,
     * verificamos se a propriedade "password" foi definida na linha de comando
     * via -Dpassword=...
     * 
     * Esta funcionalidade s deve ser usada em desenvolvimento e
     * propositadamente no foi documentada.
     */
    setPassword(System.getProperty("password"));
    if (getPassword() == null) {
      setPassword(readPassword());
    }
    if (params.outputFile != null) {
      File file = new File(params.outputFile);
      try {
        if (!file.exists()) {
          file.createNewFile();
        }
        out = new PrintStream(file);
        outputToFile = true;
      }
      catch (IOException e) {
        e.printStackTrace();
        System.exit(1);
      }
    }
  }

  /**
   * Obtm a senha interativamente.
   * 
   * @return senha digitada pelo usurio
   */
  protected String readPassword() {
    Console cons = getConsole();
    return new String(cons.readPassword("senha [%s]: ", getLogin()));
  }

  /**
   * Exibe um prompt e aguarda entrada do usurio.
   * 
   * @param prompt prompt
   * @param params parmetros para o prompt
   * @return entrada do usurio, sem espaos antes ou depois
   */
  protected String ask(String prompt, Object... params) {
    Console cons = getConsole();
    String answer = new String(cons.readLine(prompt + ' ', params)).trim();
    return answer;
  }

  /**
   * Exibe um prompt para obter uma confirmao. Acrescenta
   * <code>"[s\n]?"</code> ao prompt.
   * 
   * @param prompt prompt
   * @param params parmetros para o prompt
   * @return <code>true</code> apenas se o usurio confirmou a operao (i.e.
   *         digitou "s" ou "S")
   */
  protected boolean confirm(String prompt, Object... params) {
    String option = ask(prompt + " [s/n]", params);
    return option.equalsIgnoreCase("s");
  }

  /**
   * Obtm um console para interao com o usurio. Lana {@link AssertionError}
   * se o console no for interativo.
   * 
   * @return console
   */
  protected Console getConsole() {
    Console cons = System.console();
    if (cons == null) {
      throw new AssertionError(
        "console no  interativo, no foi possvel obter a senha");
    }
    return cons;
  }

  /**
   * Cria o objeto que ser usado para processar a linha de comando.
   * 
   * @return objeto que ser usado para processar a linha de comando
   */
  protected abstract BasicParams createParams();

  /**
   * Processa os parmetros adicionais.
   */
  protected void processExtraParams() {
    // por default no fazemos nada
  }

  /**
   * Cria o monitor da comunicao com o servidor.
   */
  protected void createClientMonitor() {
    clientMonitor =
      new ClientRemoteMonitor(getServerURI(), getPort(), getLogin(),
        getPassword(), false);
  }

  /**
   * Autentica o usurio no servidor.
   * 
   * @return <code>true</code> se a autenticao foi bem-sucedida.
   * @throws RemoteException erro na comunicao RMI
   */
  protected boolean login() throws RemoteException {
    try {
      if (!clientMonitor.lookup()) {
        printError("Autenticao de " + getLogin() + " falhou");
        return false;
      }
      postLoginInit();
      return true;
    }
    catch (CSBaseException e) {
      printError(e.getMessage());
      return false;
    }
  }

  /**
   * Inicializaes feitas aps um login bem-sucedido.
   * 
   * @throws RemoteException erro na comunicao RMI
   * 
   */
  protected void postLoginInit() throws RemoteException {
    // por default no fazemos nada
  }

  /**
   * Encerra a conexo com o servidor.
   */
  protected void logout() {
    if (preLogout()) {
      clientMonitor.shutdown();
    }
    out.close();
  }

  /**
   * Aes realizadas imediatamente antes do logout. Indica se o logout pode ser
   * efetuado.
   * 
   * @return <code>true</code> se o logout deve ser efetuado
   */
  protected boolean preLogout() {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return getServerURI() + '\n' + getLogin();
  }

  /**
   * Verifica se o usurio  o administrador.
   * 
   * @param verbose se igual a <code>true</code>, exibe mensagem de erro se o
   *        usurio no  o admin
   * 
   * @return <code>true</code> se o usurio  o administrador
   */
  protected final boolean isAdmin(boolean verbose) {
    if (!User.isAdmin(getLogin())) {
      if (verbose) {
        printError("esta operao s pode ser executada pelo administrador");
      }
      return false;
    }
    return true;
  }

  /**
   * Obtm os parmetros j processados.
   * 
   * @return parmetros j processados
   */
  protected BasicParams getParams() {
    return params;
  }

  /**
   * Obtm um mapa com as propriedades runtime do servidor.
   * 
   * @return mapa com as propriedades runtime do servidor
   * @throws RemoteException erro na comunicao RMI
   */
  public Map<String, String> getRuntimeProperties() throws RemoteException {
    return ClientRemoteLocator.serverService.getRuntimeProperties();
  }

  /**
   * Retorna o identificador da verso do sistema.
   * 
   * @return identificador da verso do sistema
   * 
   * @throws RemoteException erro na comunicao RMI
   */
  public String getSystemVersion() throws RemoteException {
    String versionName = ClientRemoteLocator.server.getVersionName();
    return versionName.isEmpty() ? Version.BAD_VERSION_STR : versionName;
  }

  /**
   * Obtm informaes sobre o ambiente de instalao do servidor.
   * 
   * @return informaes sobre o ambiente de instalao do servidor
   * @throws RemoteException erro na comunicao RMI
   */
  public DeploymentInfo getDeploymentInfo() throws RemoteException {
    return ClientRemoteLocator.diagnosticService.getDeploymentInfo();
  }

  /**
   * Define a URI do sistema.
   * 
   * @param serverURI URI
   * @return o prprio objeto, para encadeamento
   */
  public AbstractConsoleApp setServerURI(String serverURI) {
    this.serverURI = serverURI;
    return this;
  }

  /**
   * Obtm a URI do servidor.
   * 
   * @return URI do servidor
   */
  public String getServerURI() {
    return serverURI;
  }

  /**
   * Define a porta para comunicao com o servidor.
   * 
   * @param port porta
   * @return o prprio objeto, para encadeamento
   */
  public AbstractConsoleApp setPort(int port) {
    this.port = port;
    return this;
  }

  /**
   * Obtm a porta para comunicao com o servidor.
   * 
   * @return porta para comunicao com o servidor
   */
  public int getPort() {
    return port;
  }

  /**
   * Obtm o login do usurio.
   * 
   * @return login do usurio
   */
  public abstract String getLogin();

  /**
   * Define a senha.
   * 
   * @param password senha
   * @return o prprio objeto, para encadeamento
   */
  protected AbstractConsoleApp setPassword(String password) {
    this.password = password;
    return this;
  }

  /**
   * Obtm a senha fornecida pelo usurio.
   * 
   * @return senha fornecida pelo usurio
   */
  public String getPassword() {
    return password;
  }

  /**
   * Exibe uma mensagem de erro. Erros sempre so exibidos (tambm) em
   * {@link System#err}, mesmo que a sada tenha sido redirecionada.
   * 
   * @param msg mensagem
   * @param params parmetros para a mensagem
   */
  protected final void printError(String msg, Object... params) {
    String errorMsg = String.format("ERRO: " + msg, params);
    System.err.println(errorMsg);
    if (outputToFile) {
      println(errorMsg);
    }
  }

  /**
   * Exibe uma mensagem de informao.
   * 
   * @param msg mensagem
   * @param params parmetros para a mensagem
   */
  protected final void printInfo(String msg, Object... params) {
    println("INFO: " + msg, params);
  }

  /**
   * Exibe uma mensagem de alerta.
   * 
   * @param msg mensagem
   * @param params parmetros para a mensagem
   */
  protected final void printWarning(String msg, Object... params) {
    println("ALERTA: " + msg, params);
  }

  /**
   * Exibe uma linha em branco.
   */
  protected final void println() {
    println("");
  }

  /**
   * Exibe uma mensagem. Se no foi fornecido nenhum parmetro, exibe a mensagem
   * como uma string "simples", sem interpretar parmetros de formatao ('%s',
   * '%d' etc. so exibidos literalmente).
   * 
   * @param msg
   * @param params parmetros para a mensagem
   */
  protected final void println(String msg, Object... params) {
    if (params.length == 0) {
      printf("%s\n", msg);
    }
    else {
      printf(msg + '\n', params);
    }
  }

  /**
   * Exibe uma mensagem formatada. No acrescenta o terminador automaticamente.
   * 
   * @param format formato da mensagem
   * @param args parmetros para a mensagem
   * 
   * @see PrintStream#printf(String, Object...)
   */
  protected final void printf(String format, Object... args) {
    out.printf(format, args);
  }

  /**
   * Exive uma mensagem, sem terminador de linha.
   * 
   * @param msg
   */
  protected void print(String msg) {
    out.print(msg);
  }

  /**
   * L uma linha interativamente.
   * 
   * @param format formato da mensagem
   * @param args parmetros para a mensagem
   * 
   * @return linha fornecida pelo usurio
   * 
   * @see PrintStream#printf(String, Object...)
   */
  protected String readLine(String format, Object... args) {
    return getConsole().readLine(format, args);
  }

  /**
   * Exibe o stacktrace de uma exceo. O stacktrace sempre  exibido (tambm)
   * em {@link System#err} , mesmo que a sada tenha sido redirecionada.
   * 
   * @param e exceo
   */
  protected void printStackTrace(Exception e) {
    e.printStackTrace(System.err);
    if (outputToFile) {
      e.printStackTrace(out);
    }
  }

  /**
   * Exibe informaes bsicas sobre o sistema (verso, nome, hora de incio).
   * 
   * @throws RemoteException erro na comunicao RMI
   * 
   */
  protected void showBasicInfo() throws RemoteException {
    ServerBasicInfo info =
      ClientRemoteLocator.diagnosticService.getServerBasicInfo();
    println("\nSistema:");
    println("%snome: %s", IDENT, info.systemName);
    println("%sverso: %s", IDENT, info.systemVersion);
    long startupTime = info.startUpTime;
    long delta = System.currentTimeMillis() - startupTime;
    long mins = delta / (1000 * 60);
    long hours = mins / 60;
    long days = hours / 24;
    println("%siniciado em: %s (%dd %dh %dmin)", IDENT,
      FormatUtils.format(startupTime), days, hours % 24, mins % 60);
  }

  /**
   * Calcula a largura mnima para exibir uma coleo de objetos como strings.
   * 
   * @param <T> tipo dos objetos
   * @param strs coleo dos objetos
   * @return largura mnima para mostrar todos os objetos
   */
  static <T> int getMaxStrLen(Collection<T> strs) {
    int maxLen = 0;
    for (T t : strs) {
      maxLen = Math.max(maxLen, t.toString().length());
    }
    return maxLen;
  }

  /**
   * Retorna o diretrio corrente.
   * 
   * @return diretrio corrente
   */
  protected String getCurrentDir() {
    return currentDir;
  }
}
