#!/bin/ksh
#
# $Id: functions.ksh 159265 2014-12-01 12:38:04Z pietroguedes $

# habilitamos colorizao de mensagens para sistemas que tm suporte a
# isto (p.ex. Linux)
# Se a varivel de ambiente NO_COLOR for != "", no habilita as cores
if [ -z "$NO_COLOR" ]; then
    case $TEC_UNAME in
        Linux*)
            COLOR_ERR="\033[1;37;41m"
            COLOR_WRN="\033[33;1m"
            COLOR_NORMAL="\033[0m"
        ;;
    esac
fi

# Exibe uma mensagem de erro para o usurio e aborta a execuo.
#
# $1 = mensagem
# $2 = cdigo de retorno (opcional, default=1)
function abort
{
    error "$1"    
    exit ${2:-1}
}

# Exibe uma mensagem para o usurio.
#
# $1 = mensagem
function info
{
    print "[INFO ] $1"
}

# Exibe uma mensagem de aviso para o usurio.
#
# $1 = mensagem
function warn
{
    print "$COLOR_WRN[WARN ] $1$COLOR_NORMAL"
}

# Exibe uma mensagem de erro para o usurio.
#
# $1 = mensagem
function error
{
    print "$COLOR_ERR[ERROR] $1$COLOR_NORMAL"
}


# Faz uma pergunta ao usurio, e atribui o resultado  varivel
# especificada. Opcionalmente, usa um valor default.
# Se a varivel de ambiente _ASSUME_DEFAULTS foi definida (i.e. possui
# qualquer valor) e foi fornecido um valor default, usa o mesmo e no
# consulta o usurio. Se no foi fornecido default, sempre consulta o
# usurio.
#
# $1 = mensagem
# $2 = nome da var que vai conter o resultado (opcional)
# $3 = valor default (opcional)
function ask
{
    unset l_opt
    print "$1 \c"
    test -n "$3" && print "[$3] \c"
    # s precisamos do dado se _ASSUME_DEFAULTS no foi definida ou se
    # foi definida mas no existe default
    if [ -z "$_ASSUME_DEFAULTS" -o -z "$3" ]; then
        read l_opt
    else
        # j que no houve interao, simulamos o ENTER
        print
    fi
    if [ -z "$l_opt" ]; then
        l_opt=$3
    fi
    test -n "$2" && eval "$2='$l_opt'"
}

# Tenta mudar a permisso do arquivo para u+w. Se no conseguir, aborta
# a execuo
#
# $1 = arquivo
function makeWritable
{
    chmod u+w $1 2> /dev/null || {
        abort "voce nao tem permissao de escrita no arquivo $1"
    }
}

# Substitui todas as ocorrncias de um determinado padro por um texto
#
# $1 = path para o arquivo
# $2 = texto a ser trocado
# $3 = novo texto
function replaceTxt
{   
    makeWritable $1
    ex $1 > /dev/null << EOF
%s/$2/$3/g
wq!
EOF
}

# Substitui todas as ocorrncias de um determinado caminho por outro
#
# $1 = path para o arquivo
# $2 = caminho a ser trocado
# $3 = novo caminho
function replacePath
{
  makeWritable $1
  ex $1 > /dev/null << EOF
%s*$2*$3*g
wq!
EOF
}

# Substitui um padro apenas em uma linha (a 1a encontrada).
#
# $1 = path para o arquivo
# $2 = padro para localizar a linha desejada
# $3 = texto a ser trocado
# $4 = novo texto
function replaceSingleLine
{
    makeWritable $1
    ex $1 > /dev/null << EOF
/$2/s/$3/$4/g
wq!
EOF
}

# Substitui um padro apenas em uma linha encontrada a partir de um
# padro + deslocamento
#
# $1 = path para o arquivo
# $2 = padro para localizar a linha desejada
# $3 = deslocamento (num. de linhas. p.ex. +1 ou -2)
# $4 = texto a ser trocado
# $5 = novo texto
function replaceSingleLineWithOffset
{
    makeWritable $1
    ex $1 > /dev/null << EOF
/$2/${3}s/$4/$5/g
wq!
EOF
}

# Encontra todas as linhas com um determinado padro e substitui em cada
# uma delas um 2o padro por um texto. Esta funo difere de replaceTxt
# porque aqui os padres de busca das linhas e de substituio do texto
# so distintos.
#
# $1 = path para o arquivo
# $2 = padro para localizar as linhas desejadas
# $3 = texto a ser trocado
# $4 = novo texto
function replaceMultipleLines
{
    makeWritable $1
    ex $1 > /dev/null << EOF
g/$2/s/$3/$4/g
wq!
EOF
}

# Encontra todas as linhas com um determinado padro, aplica um
# deslocamento e substitui em cada uma delas um 2o padro por um texto.
# Esta funo difere de replaceTxt porque aqui os padres de busca das
# linhas e de substituio do texto so distintos.
#
# $1 = path para o arquivo
# $2 = padro para localizar as linhas desejadas
# $3 = deslocamento (num. de linhas. p.ex. +1 ou -2)
# $4 = texto a ser trocado
# $5 = novo texto
function replaceMultipleLinesWithOffset
{
    makeWritable $1
    ex $1 > /dev/null << EOF
g/$2/${3}s/$3/$4/g
wq!
EOF
}

# Remove espaos no incio e no fim de uma determinada string
#
# $1 = string a ser processada
function trim {
    print "$1" | sed 's#^ *\(.*\) *$#\1#'
}

# Converte um path relativo para absoluto.
#
function makeAbsolutePath {
    if [ "/" = `echo $1 | cut -c1` ]; then
        # j  um path absoluto
        print "$1"
    elif [ -d $1 ]; then
        #  um dir com path relativo
        (cd $1; pwd)
    else
        #  um arq com path relativo
        (cd $(dirname $1); print "$PWD/$(basename $1)")
    fi
}

# Copia JARs de um dir para outro, filtrando por um conjunto de nomes.
# As referncias para os JARs podem estar separadas por ":" (da forma
# como  feito no script [CSBASE]/conf/csbase.sh).
# 
# $1 = dir de origem (apenas um)
# $2 = dir de destino
# $3-n = JARs
function copyLibs {
    orig=$1
    dest=$2
    shift 2
    scratch=$(mktemp /tmp/copylib.XXXXXX)
    libs=$(print "$@" | tr : \ )
    for lib in $libs; do
        print "$(basename $lib)" >> $scratch
    done
    cp $(find $orig -type f -name \*.jar | grep -F -f $scratch) $dest
    rm -f $scratch
}

# L o valor de uma propriedade de um arquivo de propriedade e o associa
# a uma varivel de ambiente.
# 
# $1 = nome da propriedade
# $2 = arquivo de propriedades (com path)
# $3 = nome da varivel de ambiente que vai armazenar o valor da
#      propriedade
# 
# Retorna 0 em caso de sucesso ou 1 se a propriedade no existe ou no
# possui valor. Aborta a execuo se o arquivo no existe.
function readProperty {
    test -z "$CSBASE_DIR" && abort "CSBASE_DIR nao foi definida"
    propFile=$(makeAbsolutePath $2)
    propVal=$(cd $CSBASE_DIR && java -cp .:csbase/libs/javautils/javautils-core-2.0.0.jar csbase.tools.PropertiesDumper $propFile $1)
    eval "$3=\"$propVal\""
    # o retorno da funo ser o resultado do teste abaixo
    test "$propVal" != "null"
}

# L uma propriedade de um arquivo de propriedades qualquer. Retorna o valor
# associado  propriedade ou a string vazia.
# 
# IMPORTANTE: l apenas propriedades definidas em uma nica linha;
# propriedades espalhadas em mais de uma linha com '\' sero truncadas
# 
# $1 = nome da propriedade (completo, case-sensitive)
# $2 = path para o arquivo de propriedades
function getProperty {
    prop=$1
    arq=$2
    test -z "$arq" && abort "arquivo de propriedades nao foi informado"
    test -f $arq || abort "arquivo de propriedades invalido"
    grep "^ *$prop *= *" $arq | sed "s/^ *$prop *= *\(.*\)$/\1/"
}

# Mtodo utilizado para logar o resultado de uma ao de incio/trmino 
# de um processo.
# Recebe:
#   mensagem a ser logada 
set -A results
function logResult {
  print $1
  results[${#results[*]}]=$1
}
# Mtodo que reimprime os resultados logados.
function printResults {

  print " \n---------------\n- RESULTADOS: -\n---------------"
  typeset inx=0
  while [ $inx -lt ${#results[*]} ]; do
    print ${results[$inx]}
    (( inx += 1 ))
  done
}

# Verifica se um determinado processo esta vivo.
# Recebe:
#   o identificador do processo
# Retorna:
#   0 se o processo esta vivo
#   1 se o processo esta morto
function isProcessAlive {
  # Identificador do processo.
  typeset PID=$1

  if ps -p $PID | grep $PID > /dev/null ; then
    return 0
  else
    return 1
  fi
}

# Verifica se um determinado processo esta vivo.
# Recebe:
#   o caminho para o arquivo contendo o PID do processo
# Retorna:
#   0 se o processo esta vivo
#   1 se o processo esta morto ou se o arquivo pid no foi encontrado
function isPIDFileProcessAlive {
  typeset PID_FILE=$1

  if [ ! -e ${PID_FILE} ]
  then
    return 1
  else
    typeset PID=`cat ${PID_FILE}`
    isProcessAlive $PID
    return $?
  fi
}

# Requisita a morte de um processo e de seus processos filhos.
# Recebe como parmetro:
#   o identificador do processo
function killProcess {
  typeset PPID=$1

  typeset CHILD_PID=0
# A opo de pedir explicitamente os campos desejados (pid e ppid) no
# funcionou no Windows (cygwin).
# for CHILD_PID in `ps -eo "pid,ppid" | awk '$2 == '${PPID}' { print $1 }'`; do
  for CHILD_PID in `ps -ef| awk '$3 == '${PPID}' { print $2 }'`; do
    killProcess $CHILD_PID
  done

  kill $PPID 2> /dev/null
}

# Requisita a morte de um processo e espera at que ele morra ou um
# determinado 
# tempo passe.
# Recebe um parmetro obrigatrio e um opcional:
#   o identificador do processo
#   o tempo, em segundos, que ele deve esperar pela morte do processo
#   (padro=15)
# Retorna:
#   0 se o processo foi morto
#   1 se o processo nao esta executando
#   2 se o processo ainda existia quando o tempo de espera terminou   
function killProcessAndWait {
  # Identificador do processo.
  typeset PID=$1
  # Nmero de vezes que deve se verificar se o processo ainda existe. 
  # Como existe um sleep de 1 segundo entre cada verificao, esse 
  # valor equivale ao tempo, em segundos, que se deve esperar pela 
  # morte do comando.
  typeset VERIFICATIONS=${2:-15}

  if isProcessAlive $PID ; then
    # Mata o processo caso ele exista.
    killProcess $PID
  else
    # Indica que o processo no existia.
    return 1
  fi

  # Espera o processo morrer ou o nmero de verificaes terminar.
  while [[ $VERIFICATIONS -gt 0 ]];do
    if isProcessAlive $PID ; then
      print "Esperando o processo $PID ser finalizado... $VERIFICATIONS"
    else
      return 0
      break
    fi
    # Dorme 1 segundo antes da prxima verificao.  
    sleep 1
    (( VERIFICATIONS -= 1 ))
  done
  return 2
}

# Requisita a morte de um processo e imprime o resultado na tela, dado
# um 
# arquivo de pid. Esse arquivo ser removido caso o processo termine com
# sucesso.
# 
# Recebe um parmetro obrigatrio e um opcional:
#   o nome do processo que ser utilizado na impresso do resultado
#   o caminho para o arquivo contendo o PID do processo      
function terminatePIDFileProcess {
  typeset PROCESS_NAME=$1
  typeset PID_FILE=$2

  if [ ! -e ${PID_FILE} ]
  then
    logResult "${PROCESS_NAME} no pode ser finalizado, pois o arquivo '${PID_FILE}' no foi encontrado."
  else
    typeset PID=`cat ${PID_FILE}`

    # Requisita a morte do filerChecker e verifica o resultado
    killProcessAndWait $PID 15
    case $? in
      0)
        rm ${PID_FILE}
        logResult "$PROCESS_NAME finalizado"
        ;;
      1)
        logResult "$PROCESS_NAME no est executando"
        ;;
      2)
        logResult "ERRO! O $PROCESS_NAME no pde ser finalizado (processo $PID)"
        TERMINATE_ERROR=true
        ;;
    esac
  fi
}
