package csbase.login;

import java.util.Properties;
import java.util.logging.Logger;

import tecgraf.javautils.core.properties.PropertiesUtils;
import tecgraf.javautils.core.properties.PropertyException;

import csbase.logic.User;
import csbase.server.plugin.service.loginservice.ILoginService;

import br.com.petrobras.cenpes.te.marlim.ValidaAcesso;
import br.com.petrobras.security.ISecurityContext;
import br.com.petrobras.security.configuration.SecuritySettings;
import br.com.petrobras.security.configuration.StandaloneSecurityConfigurer;
import br.com.petrobras.security.exception.ExpiredPasswordException;
import br.com.petrobras.security.exception.InactivatedApplicationEnvironmentException;
import br.com.petrobras.security.exception.InactivatedUserException;
import br.com.petrobras.security.exception.InvalidArgumentException;
import br.com.petrobras.security.exception.InvalidCredentialsException;
import br.com.petrobras.security.exception.InvalidOperationException;

/**
 * Protocolo CA para login do CSBase
 *
 * @author Tecgraf
 */
public class CALoginProtocol implements ILoginService {

  /** No */
  private static final String CAVERSION_PROP_NAME = "CAVersion";

  /** Contexto de Segurana */
  private ISecurityContext securityContext = ISecurityContext.getContext();

  /** Logger */
  private final Logger logger = Logger.getLogger(this.getClass()
    .getCanonicalName());

  /** Configurador */
  private StandaloneSecurityConfigurer configurer;

  /** Verso da biblioteca CA */
  private int version;

  /** Propriedades do plugin */
  private Properties pluginProperties;

  /**
   * {@inheritDoc}
   */
  @Override
  public LoginProtocol getProtocolType() {
    return LoginProtocol.PROTOCOL_CA;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setProperties(Properties pluginProperties) {
    try {
      version = PropertiesUtils.getIntValue(pluginProperties,
        CAVERSION_PROP_NAME);
      if (version == 4) {
        checkCAv4Properties();
      }

      this.pluginProperties = pluginProperties;
    }
    catch (PropertyException e) {
      logger.severe(String.format(
        "Propriedade '%s' no est configurada corretamente.",
        CAVERSION_PROP_NAME));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public User authenticate(User user, String password) {
    if (version == 4)
      return authenticateCAv4(user, password);

    if (version == 3)
      return authenticateCAv3(user, password);

    return null;
  }

  /**
   * Autentica o usurio utilizando a biblioteca CAv3
   *
   * @param user - Login do usurio
   * @param password - Senha do usurio
   * @return o usurio autenticado ou nulo em caso de falha
   */
  private User authenticateCAv3(User user, String password) {
    if (ValidaAcesso.Autenticacao(user.getLogin(), password))
      return user;
    return null;
  }

  /**
   * Autentica o usurio utilizando a biblioteca CAv4
   *
   * @param user - Login do usurio
   * @param password - Senha do usurio
   * @return o usurio autenticado ou nulo em caso de falha
   */
  private User authenticateCAv4(User user, String password) {
    if (this.configurer == null) {
      this.configurer = new StandaloneSecurityConfigurer();
      SecuritySettings.load(configurer, this.pluginProperties, false);
    }

    try {
      /*
       *  necessrio realizar o logoff aps um login bem sucedido, pois o
       * autenticador aceita apenas um login por vez
       */
      securityContext.getUserAuthenticator().logon(user.getLogin(), password);
      securityContext.getUserAuthenticator().logoff();
      return user;
    }
    catch (InvalidArgumentException e) {
      logger.warning(String.format("Argumento invlido: %s", e.getMessage()));
    }
    catch (InvalidCredentialsException e) {
      logger.warning(String.format("Credenciais invlidas: usurio '%s'", user
        .getLogin()));
    }
    catch (NullPointerException e) {
      /*
       * Por alguma razo percebemos agora que a api da Petrobras tambm d NPE
       * quando senha est errada, por isso o throw abaixo
       */
      logger.warning(String.format("Credenciais incorretas: usurio '%s'.%n%s",
        user.getLogin(), e.getMessage()));
    }
    catch (InactivatedApplicationEnvironmentException e) {
      logger.warning(String.format("Ambiente de aplicao inativo: %s", e
        .getMessage()));
    }
    catch (InactivatedUserException e) {
      logger.warning(String.format("Usurio '%s' inativo.", user.getLogin()));
    }
    catch (InvalidOperationException e) {
      logger.warning(String.format("Operao invlida: %s", e.getMessage()));
    }
    catch (ExpiredPasswordException e) {
      logger.warning(String.format("Usurio '%s' com senha expirada.", user
        .getLogin()));
    }
    catch (Exception e) {
      /*
       * outras excees sistmicas (erros do CA server ou da api do cav4), mas
       * que no tem a ver com login/senha invlidos
       */
      logger.warning(String.format(
        "Erro na autenticao com o servidor CAv4.%n%s", e.getMessage()));
    }

    return null;
  }

  /** Verifica se as propriedades obrigatrias do CAv4 esto preenchidas. */
  private void checkCAv4Properties() {
    String[] cav4ConfigKeys = new String[] { "regional_id",
        "application_catalog_id", "application_password", "environment_id",
        "endpoint_wsdl" };
    for (String prop : cav4ConfigKeys) {
      try {
        PropertiesUtils.getValue(pluginProperties, prop);
      }
      catch (Exception e) {
        logger.warning(String.format("Propriedade '%s' no encontrada.", prop));
      }
    }
  }

}
