package csbase.rest.adapter.project.v1;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.function.Predicate;

import csbase.exception.InfoException;
import csbase.exception.PermissionException;
import csbase.exception.ServiceFailureException;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.CommonProjectInfo;
import csbase.logic.ProjectAdminInfo;
import csbase.logic.ProjectPermissions;
import csbase.logic.User;
import csbase.logic.UserProjectInfo;
import csbase.remote.AdministrationServiceInterface;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.ProjectServiceInterface;
import ibase.common.NotFoundException;
import ibase.exception.InternalServiceException;
import ibase.exception.InvalidParameterException;
import ibase.rest.api.project.v1.adapter.Project;
import ibase.rest.api.project.v1.adapter.ProjectFile;
import ibase.rest.api.project.v1.adapter.ProjectInfo;
import ibase.rest.api.project.v1.adapter.ProjectService;
import ibase.rest.api.project.v1.adapter.SharingType;
import tecgraf.javautils.core.io.FileUtils;

/**
 * Implementação do adaptador CSBase para o serviço de projetos.
 *
 * @author Tecgraf/PUC-Rio
 */
public class CSBaseProjectServiceImpl implements ProjectService {

  /**
   * {@inheritDoc}
   */
  @Override
  public Project createProject(String name, String userId, String description, String type, SharingType sharingType) {
    ProjectServiceInterface service = ClientRemoteLocator.projectService;
    CommonProjectInfo projectInfo = new CommonProjectInfo();
    projectInfo.userId = userId;
    projectInfo.description = description;
    if (type != null) {
      projectInfo.setAttribute(CSBaseProject.PROJECT_TYPE, type);
    }
    projectInfo.name = name;
    try {
      CommonClientProject project = service.createProject(projectInfo);
      if (sharingType != null) {
        setCSBaseSharingType(project.getId(), sharingType, project.getUsersRO(), project.getUsersRW());
      }
      return new CSBaseProject(project);
    }
    catch (PermissionException e) {
      throw new ibase.exception.PermissionException(e.getMessage());
    }
    catch (InfoException e) {
      throw new InvalidParameterException(e.getMessage());
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Project updateProject(String projectId, String description, String type, SharingType sharingType) {
    try {
      ProjectServiceInterface service = ClientRemoteLocator.projectService;
      CommonClientProject project = service.openProject(projectId, false);
      if (description != null) {
        project.setDescription(description);
        project.modify();
      }
      if (type != null) {
        project.setAttribute(CSBaseProject.PROJECT_TYPE, type);
        project.modify();
      }
      if (sharingType != null) {
        Set<Object> usersRO = project.getUsersRO();
        Set<Object> usersRW = project.getUsersRW();
        setCSBaseSharingType(project.getId(), sharingType, usersRO, usersRW);
      }
      return new CSBaseProject(project);
    }
    catch (PermissionException e) {
      throw new ibase.exception.PermissionException(e.getMessage());
    }
    catch (InfoException e) {
      throw new InvalidParameterException(e.getMessage());
    }
    catch (ServiceFailureException e) {
      throw new NotFoundException(e.getMessage());
    }
    catch (Throwable e) {
      e.printStackTrace();
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Project getProject(String projectId) {
    ProjectServiceInterface service = ClientRemoteLocator.projectService;
    try {
      CommonClientProject project = service.openProject(projectId, false);
      return new CSBaseProject(project);
    }
    catch (PermissionException e) {
      throw new ibase.exception.PermissionException(e.getMessage());
    }
    catch (InfoException e) {
      throw new InvalidParameterException(e.getMessage());
    }
    catch (ServiceFailureException e) {
      throw new NotFoundException(e.getMessage());
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setLocale(Locale locale) {
    ClientRemoteLocator.administrationService.setLocale(locale);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ProjectFile getProjectFile(String projectId, String fileId) {
    try {
      ClientProjectFile file = getFileFromId(projectId, fileId);
      return new CSBaseProjectFile(file);
    }
    catch (PermissionException e) {
      throw new ibase.exception.PermissionException(e.getMessage());
    }
    catch (ServiceFailureException | InfoException e) {
      throw new NotFoundException(e.getMessage());
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * Obtm o objeto que representa um arquivo, a partir do seu identificador.
   * 
   * @param projectId identificador do projeto que contm o arquivo.
   * @param fileId identificador do arquivo.
   * 
   * @return o objeto que representa um arquivo.
   */
  private static ClientProjectFile getFileFromId(String projectId, String fileId) throws RemoteException {
    ProjectServiceInterface service = ClientRemoteLocator.projectService;

    String[] path = FileUtils.splitPath(fileId, "/"); // Ver CSBaseProjectFile.getId.
    return service.getChild(projectId, path);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<ProjectInfo> getProjectsSharedWith(String userId) {
    ProjectServiceInterface service = ClientRemoteLocator.projectService;
    try {
      List<UserProjectInfo> infos = service.getProjectsSharedWithUser(userId);
      return filter(infos);
    }
    catch (InfoException e) {
      throw new InvalidParameterException(e.getMessage());
    }
    catch (ServiceFailureException e) {
      throw new NotFoundException(e.getMessage());
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<ProjectInfo> getAllProjects(String userId) {
    ProjectServiceInterface projectService = ClientRemoteLocator.projectService;
    AdministrationServiceInterface adminService = ClientRemoteLocator.administrationService;
    try {
      List<User> users = adminService.getAllUsers();
      List<UserProjectInfo> allProjects = new ArrayList<UserProjectInfo>();
      for (User user : users) {
        allProjects.addAll(projectService.getProjectsFromUser(user.getId()));
      }
      return filter(allProjects);
    }
    catch (InfoException e) {
      throw new InvalidParameterException(e.getMessage());
    }
    catch (ServiceFailureException e) {
      throw new NotFoundException(e.getMessage());
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<ProjectInfo> getProjectsCreatedBy(String userId) {
    ProjectServiceInterface service = ClientRemoteLocator.projectService;
    try {
      List<UserProjectInfo> infos = service.getProjectsFromUser(userId);
      return filter(infos);
    }
    catch (InfoException e) {
      throw new InvalidParameterException(e.getMessage());
    }
    catch (ServiceFailureException e) {
      throw new NotFoundException(e.getMessage());
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void removeProject(String projectId) {
    ProjectServiceInterface service = ClientRemoteLocator.projectService;
    try {
      service.removeProject(projectId);
    }
    catch (PermissionException e) {
      throw new ibase.exception.PermissionException(e.getMessage());
    }
    catch (InfoException e) {
      throw new InvalidParameterException(e.getMessage());
    }
    catch (ServiceFailureException e) {
      throw new NotFoundException(e.getMessage());
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean existsProjectFile(String projectId, String fileId) {
    ProjectServiceInterface service = ClientRemoteLocator.projectService;
    try {
      String[] path = getFileFromId(projectId, fileId).getPath();
      return service.existsFile(projectId, path);
    }
    catch (PermissionException e) {
      throw new ibase.exception.PermissionException(e.getMessage());
    }
    catch (InfoException e) {
      throw new InvalidParameterException(e.getMessage());
    }
    catch (ServiceFailureException e) {
      throw new NotFoundException(e.getMessage());
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void copyFile(String sourceProjectId, String sourceFileId, String targetProjectId, String targetFileId) {
    ProjectServiceInterface service = ClientRemoteLocator.projectService;
    try {
      String[] sourceFilePath = getFileFromId(sourceProjectId, sourceFileId).getPath();
      String[] targetDirPath = getFileFromId(targetProjectId, targetFileId).getPath();
      service.copyFile(sourceProjectId, sourceFilePath, targetProjectId, targetDirPath);
    }
    catch (PermissionException e) {
      throw new ibase.exception.PermissionException(e.getMessage());
    }
    catch (InfoException e) {
      throw new InvalidParameterException(e.getMessage());
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);

    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void moveFile(String sourceProjectId, String sourceFileId, String targetProjectId, String targetFileId) {
    ProjectServiceInterface service = ClientRemoteLocator.projectService;
    try {
      String[] sourceFilePath = getFileFromId(sourceProjectId, sourceFileId).getPath();
      String[] targetDirPath = getFileFromId(targetProjectId, targetFileId).getPath();
      service.moveFile(sourceProjectId, sourceFilePath, targetProjectId, targetDirPath);
    }
    catch (PermissionException e) {
      throw new ibase.exception.PermissionException(e.getMessage());
    }
    catch (InfoException e) {
      throw new InvalidParameterException(e.getMessage());
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ProjectFile renameFile(String projectId, String fileId, String name) {
    ProjectServiceInterface service = ClientRemoteLocator.projectService;
    try {
      ClientProjectFile file = getFileFromId(projectId, fileId);
      String[] path = file.getPath();
      service.renameFile(projectId, path, name);
      path[path.length - 1] = name;
      ClientProjectFile newFile = service.getChild(projectId, path);
      return new CSBaseProjectFile(newFile);
    }
    catch (PermissionException e) {
      throw new ibase.exception.PermissionException(e.getMessage());
    }
    catch (ServiceFailureException e) {
      throw new NotFoundException(e.getMessage());
    }
    catch (InfoException e) {
      throw new InvalidParameterException(e.getMessage());
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ProjectFile createFolder(String projectId, String fileId, String folderName) {
    ProjectServiceInterface service = ClientRemoteLocator.projectService;
    try {
      ClientProjectFile targetFolder = getFileFromId(projectId, fileId);
      String[] targetFolderPath = targetFolder.getPath();
      String[] targetPath = Arrays.copyOf(targetFolderPath, targetFolderPath.length + 1); // Add an extra space for fileName after folder.
      targetPath[targetFolderPath.length] = folderName;
      service.createDirectory(projectId, targetPath);
      ClientProjectFile child = service.getChild(projectId, targetPath);
      return new CSBaseProjectFile(child);
    }
    catch (PermissionException e) {
      throw new ibase.exception.PermissionException(e.getMessage());
    }
    catch (InfoException e) {
      throw new InvalidParameterException(e.getMessage());
    }
    catch (Throwable e) {
      throw new InternalServiceException(e);
    }
  }

  /////////////////////
  // Auxiliar Methods
  /////////////////////

  @SuppressWarnings("javadoc")
  private static List<ProjectInfo> filter(List<UserProjectInfo> infos) {
    List<ProjectInfo> projectInfos = new ArrayList<ProjectInfo>();
    infos.stream().filter(isAvailable()).forEach(i -> projectInfos.add(new CSBaseProjectInfo(i)));
    return projectInfos;
  }

  @SuppressWarnings("javadoc")
  private static Predicate<UserProjectInfo> isAvailable() {
    return new Predicate<UserProjectInfo>() {
      @Override
      public boolean test(UserProjectInfo info) {
        ProjectServiceInterface service = ClientRemoteLocator.projectService;
        ProjectAdminInfo projectAdminInfo;
        try {
          projectAdminInfo = service.getProjectAdminInfo(info.getProjectId());
          return !(projectAdminInfo != null && (projectAdminInfo.isLocked() || projectAdminInfo.isWaitingAreaFree()));
        }
        catch (RemoteException e) {
          return false;
        }
      }
    };
  }

  @SuppressWarnings("javadoc")
  private void setCSBaseSharingType(Object projectId, SharingType type, Set<Object> usersRO, Set<Object> usersRW) {
    ProjectServiceInterface service = ClientRemoteLocator.projectService;
    try {
      switch (type) {
        case PRIVATE:
          service.updateUsers(projectId, ProjectPermissions.SharingType.PRIVATE, null, null);
          break;
        case PUBLIC_RO:
          service.updateUsers(projectId, ProjectPermissions.SharingType.ALL_RO, null, null);
          break;
        case PUBLIC_RW:
          service.updateUsers(projectId, ProjectPermissions.SharingType.ALL_RW, null, null);
          break;
        case SELECTIVE:
          service.updateUsers(projectId, ProjectPermissions.SharingType.PARTIAL, usersRO, usersRW);
          break;
      }
    }
    catch (RemoteException e) {
    }
  }

}
