package csbase.rest.adapter.project.v1;

import csbase.remote.ServerEntryPoint;

import java.io.BufferedInputStream;
import java.nio.ByteBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;

/**
 * Classe auxiliar para detectar o charset de um arquivo de projeto.
 */
public class CharsetDetector {

  public static Charset detectCharset(CSBaseProjectFile f) {
    String[] charsetNames = new String[] {"UTF-8", "ISO-8859-1",
      "US-ASCII", "windows-1252", "MacRoman", "IBM500" };
    return detectCharset(f, charsetNames);
  }

  /**
   * Detecta o charset a partir de um vetor de strings de charsets para um determinado arquivo de projeto.
   *
   * @param f Arquivo de projeto.
   * @param charsets vetor de strings com os charsets para serem verificados.
   * @return o objeto Charset que foi detectado ou null caso no seja possvel detectar o charset.
   */
  public static Charset detectCharset(CSBaseProjectFile f, String[] charsets) {

    Charset charset = null;

    for (String charsetName : charsets) {
      charset = detectCharset(f, Charset.forName(charsetName));
      if (charset != null) {
        break;
      }
    }

    return charset;
  }

  /**
   * Detecta o charset a partir de um arquivo de projeto de um determinado objeto Charset.
   *
   * @param f Arquivo de projeto.
   * @param charset o Charset que ser verificado.
   * @return o objeto Charset que foi detectado ou null caso no seja possvel detectar o charset.
   */
  private static Charset detectCharset(CSBaseProjectFile f, Charset charset) {
    try {
      BufferedInputStream input = new BufferedInputStream(f.getInputStream());

      CharsetDecoder decoder = charset.newDecoder();
      decoder.reset();

      byte[] buffer = new byte[512];
      boolean identified = false;
      while ((input.read(buffer) != -1) && (!identified)) {
        identified = identify(buffer, decoder);
      }

      input.close();

      if (identified) {
        return charset;
      } else {
        return null;
      }

    } catch (Exception e) {
      return null;
    }
  }

  /**
   * Identifica se um vetor de bytes pode ser decodificado a partir de um objeto CharsetDecoder.
   *
   * @param bytes vetor de bytes.
   * @param decoder o decoder que ser utilizado para tentar decodificar o vetor de bytes.
   * @return True caso consiga decodificar o vetor de bytes ou false caso contrrio.
   */
  private static boolean identify(byte[] bytes, CharsetDecoder decoder) {
    try {
      decoder.decode(ByteBuffer.wrap(bytes));
    } catch (CharacterCodingException e) {
      return false;
    }
    return true;
  }
}
