package csbase.server.plugin.service.sgaservice;

import sgaidl.CompletedCommandInfo;
import sgaidl.InvalidCommandException;
import sgaidl.InvalidParameterException;
import sgaidl.InvalidSGAException;
import sgaidl.NoPermissionException;
import sgaidl.RetrievedInfo;
import sgaidl.SGAAlreadyRegisteredException;
import sgaidl.SGACommandOperations;
import sgaidl.SGADaemonOperations;
import sgaidl.SGANotRegisteredException;
import sgaidl.SGAProperties;

/**
 * Interface pblica do servio de gerncias dos SGAs.
 *
 * @author Tecgraf/PUC-Rio
 */
public interface ISGAService {
  /**
   * Registro de um SGA.
   *
   * @param sgaReference referncia para o sga
   * @param sgaName nome do SGA
   * @param sgaProperties propriedades do SGA
   * @param updateInterval informa de quanto em quanto tempo o SGA deve dizer
   *        que est vivo.
   *
   * @return true se o o registro do SGA for aceito, false caso o registro no
   *         possa ser aceito
   */
  public abstract boolean registerSGA(SGADaemonOperations sgaReference,
    String sgaName, SGAProperties sgaProperties,
    org.omg.CORBA.IntHolder updateInterval) throws InvalidParameterException,
    NoPermissionException, SGAAlreadyRegisteredException;

  /**
   * Solicitao de de-registro de um SGA.
   *
   * @param sgaReference referncia para o sga
   * @param sgaName nome do SGA
   */
  public abstract void unregisterSGA(SGADaemonOperations sgaReference,
    String sgaName) throws NoPermissionException, SGANotRegisteredException;

  /**
   * SGA verifica se o SSI est acessvel e se seu registro est vlido.
   *
   * @param sgaReference referncia para o SGA
   * @param sgaName nome da mquina hospedeira do SGA
   *
   * @return true se o o registro do SGA estiver vlido, false caso contrrio.
   *         Um retorno "false"  uma indicao de que o SSI esteve fora do ar,
   *         o que exige um novo registro do SGA junto ao mesmo.
   */
  public abstract boolean isRegistered(SGADaemonOperations sgaReference,
    String sgaName) throws InvalidSGAException, NoPermissionException;

  /**
   * SGA atualiza seus dados dinmicos.
   *
   * @param sgaReference referncia para o SGA
   * @param sgaName nome da mquina hospedeira do SGA
   * @param sgaProperties informaes dinmicas do SGA
   *
   * @return true se conseguiu atualizar seus dados ou false caso contrrio.
   */
  public abstract boolean updateSGAInfo(SGADaemonOperations sgaReference,
    String sgaName, SGAProperties sgaProperties)
    throws InvalidParameterException, NoPermissionException,
    SGANotRegisteredException;

  /**
   * Lista de propriedades obrigatrias do SGA.
   *
   * @return o array com o nome das propriedades obrigatrias de qualquer SGA.
   */
  public abstract String[] mandatorySGAProperties();

  /**
   * Lista de propriedades obrigatrias dos ns do SGA.
   *
   * @return o array com o nome das propriedades obrigatrias de qualquer ns de
   *         SGA.
   */
  public abstract String[] mandatorySGANodeProperties();

  /**
   * Notificao de trmino de um comando.
   *
   * @param sgaName nome do SGA.
   * @param command referncia para o comando concludo.
   * @param commandId identificador do comando no SGA.
   * @param info informaes sobre o comando.
   *
   * @return true se notificao aceita, false se o SSI desconhece o SGA
   *
   * @throws InvalidSGAException
   * @throws NoPermissionException
   * @throws InvalidCommandException
   */
  public abstract boolean commandCompleted(String sgaName,
    SGACommandOperations command, String commandId, CompletedCommandInfo info)
    throws InvalidSGAException, NoPermissionException, InvalidCommandException;

  /**
   * Notificao de comando no recuperado.
   *
   * @param sgaName nome do SGA.
   * @param commandId identificador do comando no SGA.
   *
   * @return true se notificao aceita, false se o SSI desconhece o SGA
   *
   * @throws InvalidSGAException
   * @throws NoPermissionException
   * @throws InvalidCommandException
   */
  public abstract boolean commandLost(String sgaName, String commandId)
    throws InvalidSGAException, NoPermissionException, InvalidCommandException;

  /**
   * Registra comandos recuperados em um SGA. Os demais comandos em execuo no
   * SGA (no-recuperados) so marcados como perdidos.
   *
   * @param sgaName nome que identifica o sga.
   * @param infos informaes sobre os comandos recuperados
   *
   * @return <code>true</code> se existe um SGA com o nome especificado,
   *         <code>false</code> caso contrrio
   * 
   * @throws InvalidSGAException
   * @throws NoPermissionException
   * @throws InvalidCommandException
   */
  public abstract boolean commandRetrieved(String sgaName, RetrievedInfo[] infos)
    throws InvalidSGAException, NoPermissionException, InvalidCommandException;

}
