/**
 * $Id: ProjectService.java 170919 2015-12-29 17:02:16Z isabella $
 */

package csbase.server.services.projectservice;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.lang.reflect.Constructor;
import java.rmi.RemoteException;
import java.text.MessageFormat;
import java.util.*;
import java.util.regex.Pattern;

import csbase.logic.*;
import csbase.logic.projectservice.DefaultProjectTemplate;
import csbase.logic.projectservice.ProjectTemplate;
import org.apache.commons.io.FilenameUtils;

import tecgraf.ftc.common.logic.RemoteFileChannelInfo;
import tecgraf.javautils.core.io.FileFinder;
import csbase.exception.InfoException;
import csbase.exception.InvalidRequestException;
import csbase.exception.PermissionException;
import csbase.exception.ServiceFailureException;
import csbase.logic.ProjectPermissions.SharingType;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.ProjectServiceInterface;
import csbase.remote.RemoteEvent;
import csbase.remote.RemoteObserver;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;
import csbase.server.services.administrationservice.AdministrationService;
import csbase.server.services.loginservice.LoginService;
import csbase.server.services.mailservice.MailService;
import csbase.server.services.messageservice.MessageService;
import csbase.server.services.projectservice.FileLockInterface.LockStatus;
import csbase.util.messages.Message;
import tecgraf.javautils.core.io.FileUtils;

/**
 * Implementa a interface {@link ProjectServiceInterface} para prover o servio
 * de projetos do {@link Server}. Essa classe cria os objetos do tipo
 * {@link CommonClientProject} que sero enviados para o usurio. Tambm  ela
 * que gera identificadores para cada CommonClientProject. Dado um identificador
 * de um CommonClientProject, esta classe deve saber achar o ServerProject
 * correspondente. Implementa o servio de acesso a arquivos, baseado em
 * conexes socket.
 *
 * @author Tecgraf/PUC-Rio
 */
public class ProjectService extends Service implements ProjectServiceInterface {

  /**
   * Prefixo utilizado nos nomes das chaves de traduo.
   */
  private static final String LNG_PREFIX = "ProjectService.";

  /**
   * Modo de otimizao de acesso a arquivos de projeto. O padro  sem
   * otimizao.
   */
  private ClientOptimizationMode clientOptimizationMode =
    ClientOptimizationMode.NONE;

  /**
   * Caminho local para acesso aos arquivos de projeto. Utilizado apenas quando
   * o modo de otimizao  <code>ClientOptimizationMode.GLOBAL</code>.
   */
  private String clientLocalProjectPath = null;

  /**
   * O identificador da fonte de dados do servio DataService publicado no
   * barramento.
   *
   * Esse identificador  utilizado pelo DataService para criar a chave de
   * dados, ento no pode ser nulo.
   */
  private String sourceId = null;
  /**
   * O nome da propriedade que representa o identificador de uma fonte de dados
   * em um servio de dados.
   */
  public static String DATA_SOURCE_ID_PROPERTY_NAME = "openbus.data_source_id";

  /**
   * Mensagem para parmetros invlidos/inadequados passsados pelo cliente.
   */
  private static final String INVALID_REQUEST_MSG =
    LNG_PREFIX + "error.invalid.request";

  /**
   * Instncia do administrador de projetos.
   */
  private ProjectAdministrator prjAdmin = null;

  /**
   * Informaes dos projetos compartilhados entre usurios. No inclui projetos
   * pblicos nem privados.
   */
  private Hashtable<Object, Vector<UserProjectInfo>> sharedProjects;

  /**
   * Conjunto de projetos pblicos, ou seja, projetos em que todos os usurios
   * tm acesso.
   */
  private Set<UserProjectInfo> publicProjects;

  /**
   * O tamanho mnimo do identificador nico dos arquivos.
   */
  public static final int MINIMUM_FILE_ID_SIZE = 1;
  /**
   * O separador do identificador nico dos arquivos.
   */
  public static final String FILE_ID_SEPARATOR = "@/@";

  /**
   * Indica se o servio de projetos utiliza reserva de rea.
   */
  private boolean areaReserved;
  /**
   * Caminho para o repositrio de projeto;
   */
  private final String projectRepositoryPath;

  /**
   * Responsvel por agendar a atualizao de arquivos de projetos.
   */
  private ProjectFileUpdateScheduler updateScheduler;

  /**
   * Diretrio de projetos.
   */
  private File prjDir;

  /**
   * Repositrio de tipos.
   */
  private ProjectFileTypeRepository typeRepository;

  /**
   * Lista de templates de projeto.
   */
  private List<ProjectTemplate> projectTemplates;

  /**
   * Obtm uma instncia do servio.
   *
   * @return ProjectService
   */
  public static ProjectService getInstance() {
    return (ProjectService) getInstance(SERVICE_NAME);
  }

  //
  // Implementao dos mtodos pblicos da interface ProjectServiceInterface.
  //
  // ATENO: Esses devem ser os nicos mtodos pblicos dessa classe, para
  // garantir que os demais servios utilizem apenas os recursos da interface
  // pblica.
  //

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean userHasHisOwnProjects(Object userId) {
    return ServerProject.userHasHisOwnProjects(userId);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<UserProjectInfo> getProjectsFromUser(Object userId) {
    User user = Service.getUser();
    List<UserProjectInfo> result = new ArrayList<UserProjectInfo>();
    List<Object> projectIds = ServerProject.getAllProjectIds(userId);
    for (Object projectId : projectIds) {
      ServerProject project = ServerProject.openProject(projectId, false);
      if (project.userHasAccess(user.getId())) {
        String projectName = ServerProject.getProjectName(projectId);
        result.add(new UserProjectInfo(projectId, projectName, userId));
      }
    }
    return result;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<UserProjectInfo> getProjectsSharedWithUser(Object userId) {
    List<UserProjectInfo> projects = new ArrayList<UserProjectInfo>();
    if (publicProjects != null) {
      synchronized (publicProjects) {
        for (UserProjectInfo info : publicProjects) {
          if (!info.getOwnerId().equals(userId)) {
            projects.add(info);
          }
        }
      }
    }
    if (sharedProjects != null) {
      List<UserProjectInfo> prjsSharedWithUser = sharedProjects.get(userId);
      if (prjsSharedWithUser != null) {
        projects.addAll(prjsSharedWithUser);
      }
    }
    return projects;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean userParticipatesOnSharedProjects(Object userId) {
    if (publicProjects != null && !publicProjects.isEmpty()) {
      /*
       * se existe algum projeto pblico, j basta
       */
      return true;
    }
    if (sharedProjects == null || sharedProjects.isEmpty()) {
      return false;
    }
    return sharedProjects.containsKey(userId);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void updateUsers(Object projectId, SharingType sharingType,
    Set<Object> usersRO, Set<Object> usersRW) {
    checkWritePermission(projectId);
    ServerProject sp = ServerProject.openProject(projectId, false);
    SharingType oldSharingType = sp.getSharingType();
    Set<Object> oldUsersRO = sp.getUsersRO();
    Set<Object> oldUsersRW = sp.getUsersRW();
    sp.setSharingType(sharingType);
    sp.setUsersRO(usersRO);
    sp.setUsersRW(usersRW);
    sp.modify();
    switch (sharingType) {
      case ALL_RO:
      case ALL_RW:
        prjAccessBecamePublic(sp, oldSharingType, oldUsersRO, oldUsersRW);
        break;
      case PARTIAL:
        /*
         * Acesso passou a ser seletivo. Temos que comparar as listas de acesso
         * antigas com as atuais e notificar aqueles usurios que mudaram de
         * status.
         */
        prjAccessBecamePartial(sp, oldSharingType, oldUsersRO, oldUsersRW);
        break;
      case PRIVATE:
        /*
         * Acesso passou a ser privado. Temos que avisar a todos que
         * eventualmente tinham acesso que agora no tm mais.
         */
        prjAccessBecamePrivate(sp, oldSharingType, oldUsersRO, oldUsersRW);
        break;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean removeUser(Object projectId, Object userId) {
    checkWritePermission(projectId);
    ServerProject sp = ServerProject.openProject(projectId, false);
    if (sp.userHasSelectiveAccessRO(userId)) {
      Set<Object> newUsers = removeUserFrom(userId, sp.getUsersRO());
      updateUsersRO(sp, newUsers);
      return true;
    }
    if (sp.userHasSelectiveAccessRW(userId)) {
      Set<Object> newUsers = removeUserFrom(userId, sp.getUsersRW());
      updateUsersRW(sp, newUsers);
      return true;
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void removeUsers(Object projectId, Set<Object> usersId) {
    checkWritePermission(projectId);
    ServerProject sp = ServerProject.openProject(projectId, false);
    Set<Object> usersRO = new HashSet<Object>(sp.getUsersRO());
    if (usersRO.removeAll(usersId)) {
      updateUsersRO(sp, usersRO);
    }
    Set<Object> usersRW = new HashSet<Object>(sp.getUsersRW());
    if (usersRW.removeAll(usersId)) {
      updateUsersRW(sp, usersRW);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean removeUserProjects(Object userId) {
    if (!Service.getUser().isAdmin()) {
      throw new PermissionException();
    }
    List<UserProjectInfo> infoList = getProjectsFromUser(userId);
    for (Iterator<UserProjectInfo> iterator = infoList.iterator(); iterator
      .hasNext();) {
      UserProjectInfo userProjectInfo = iterator.next();
      removeProject(userProjectInfo.getProjectId());
    }
    return ServerProject.removeBasePrjDirForUser(userId);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public CommonClientProject openProject(Object projectId, boolean notify) {
    this.checkReadPermission(projectId);
    if (prjAdmin.isLocked(projectId)) {
      String InfoMsg = getFormattedString("ProjectService.info.project.locked",
        new Object[] { getProjectName(projectId) });
      throw new InfoException(InfoMsg);
    }
    ServerProject sp = ServerProject.openProject(projectId, notify);
    return buildClientProject(sp);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public CommonClientProject createProject(CommonProjectInfo info) {
    if (!Service.getUser().getId().equals(info.userId)) {
      checkAdminPermission();
    }
    ServerProject sp = ServerProject.createProject(info);
    CommonClientProject cp = buildClientProject(sp);
    Server.logFineMessage("Projeto " + info.name + " do usurio " + info.userId
      + " criado com sucesso");
    AdministrationEvent action =
      new AdministrationEvent(AdministrationEvent.CREATE, info);
    notifyProject(action, sp.getId());
    /* Notificando ProjectAdminObserver da criao do projeto */
    info.projectId = sp.getId();
    ProjectUserEvent event =
      new ProjectUserEvent(ProjectUserEvent.CREATE, info);
    notifyProject(event, sp.getId());
    return cp;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void createProjectWithAllocation(CommonProjectInfo info, long size) {
    if (!Service.getUser().getId().equals(info.userId)) {
      checkAdminPermission();
    }
    ServerProject sp = ServerProject.createProject(info);
    ProjectAdminInfo adminInfo = prjAdmin.lockProject(sp.getId(), size);

    Server.logInfoMessage("Projeto " + info.name + " do usurio " + info.userId
      + " criado com sucesso e aguardando liberao do administrador.");

    AdministrationEvent action =
      new AdministrationEvent(AdministrationEvent.CREATE, info);
    notifyProject(action, sp.getId());

    String message = String.format(
      getString(LNG_PREFIX + "msg.project.created.with.allocation"), info.name,
      info.userId);
    notifyAdmin(adminInfo, message);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void unlockProjectsWithAllocation(ProjectAdminInfo[] infos) {
    checkAdminPermission();
    for (int i = 0; i < infos.length; i++) {
      ServerProject sp =
        ServerProject.openProject(infos[i].getProjectId(), false);
      Hashtable<String, Object> attributes = sp.getAttributes();
      attributes.put(ProjectAttribute.LOCKING_AREA_SIZE.getAttributeKey(),
        Long.valueOf(infos[i].getAreaLockedSize()));
      sp.setAttributes(attributes);
      sp.modify();
      sp.close(false);
      ProjectAdminInfo info = prjAdmin.unlockProject(infos[i].getProjectId(),
        infos[i].getAreaLockedSize());
      infos[i] = info;
      /*
       * envia a notificao
       */
      final Notification notification =
        new ProjectNotification(getSenderName(), info);
      String[] ids = new String[] { (String) infos[i].getOwnerId() };
      try {
        MessageService.getInstance().send(new Message(notification), ids);
      }
      catch (RemoteException e) {
        Server.logSevereMessage("Erro ao enviar evento de concesso de rea.",
          e);
      }

      AdministrationEvent event =
        new AdministrationEvent(AdministrationEvent.MODIFY, infos);
      sendAdmistrationEvent(event, sp);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void freeAreaForProjects(ProjectAdminInfo[] infos) {
    checkAdminPermission();
    for (int i = 0; i < infos.length; i++) {
      ServerProject sp =
        ServerProject.openProject(infos[i].getProjectId(), false);
      infos[i] = prjAdmin.freeProjectArea(infos[i].getProjectId());

      AdministrationEvent event =
        new AdministrationEvent(AdministrationEvent.DELETE, infos[i]);
      sendAdmistrationEvent(event, sp);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public CommonProjectInfo modifyProject(Object projectId,
    CommonProjectInfo info) {
    this.checkWritePermission(projectId);
    ServerProject sp = ServerProject.openProject(projectId, false);
    sp.setName(info.name);
    sp.setDescription(info.description);
    sp.setAttributes(info.getAttributes());
    sp.modify();
    info.name = sp.getName();
    info.description = sp.getDescription();
    info.setAttributes(sp.getAttributes());
    return info;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void removeProject(Object projectId) {
    ServerProject sp = ServerProject.openProject(projectId, false);
    this.checkWritePermission(projectId);

    /* Remove os usurios do projeto e notifica */
    Set<Object> removedUsers = removeUsers(sp);
    if (removedUsers != null) {
      notifyUsers(UsersNotification.REMOVE_PROJECT, removedUsers, sp);
    }

    /* Guarda os usurio que devem receber eventos administrativos */
    Set<Object> adminUsers = new HashSet<Object>(0);
    try {
      adminUsers.addAll(User.getAdminIds());
    }
    catch (Exception e) {
      Server.logSevereMessage(
        "Erro ao obter a lista de usurios adminstradores.", e);
    }

    adminUsers.add(sp.getUserId());

    /* Guarda os usurio que devem receber eventos do projeto */
    Set<Object> allUsers = new HashSet<Object>(0);
    allUsers.addAll(adminUsers);
    if (removedUsers != null) {
      allUsers.addAll(removedUsers);
    }

    sp.remove();

    if (prjAdmin.isUnlockedWithAreaAllocated(projectId)) {
      ProjectAdminInfo info = prjAdmin.setAllocatedProjectRemoved(projectId);

      AdministrationEvent event =
        new AdministrationEvent(AdministrationEvent.MODIFY, info);
      notifyUsers(event, adminUsers.toArray(new String[0]));

      final String message = String.format(
        getString(LNG_PREFIX + "msg.project.removed.with.allocation"),
        info.getProjectName(), info.getOwnerId());
      notifyAdmin(info, message);
    }
    else if (prjAdmin.isLocked(projectId)) {
      final ProjectAdminInfo info = prjAdmin.getProjectAdminInfo(projectId);
      prjAdmin.removeProjectAdminInfo(projectId);
      final String message = String.format(
        getString(LNG_PREFIX + "msg.project.removed.without.allocation"),
        info.getProjectName(), info.getOwnerId());
      notifyAdmin(info, message);
    }

    /* Notificando ProjectAdminObserver da remoo do projeto */
    ProjectUserEvent event =
      new ProjectUserEvent(ProjectUserEvent.DELETE, sp.getInfo());
    notifyUsers(event, allUsers.toArray(new String[0]));

    Server.logInfoMessage("ProjectService:removeProject: projeto" + projectId
      + " removido com sucesso");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void closeProject(Object projectId, boolean notify) {
    this.checkReadPermission(projectId);
    ServerProject sp = ServerProject.getProject(projectId);
    if (sp == null) {
      String InfoMsg =
        getFormattedString("ProjectService.info.project.not.found",
          new Object[] { getProjectName(projectId) });
      throw new InfoException(InfoMsg);
    }
    sp.close(notify);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void rebuildTree(Object projectId) {
    this.checkReadPermission(projectId);
    ServerProject sp = ServerProject.openProject(projectId, false);
    sp.rebuildTree();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void refreshTree(Object projectId) {
    this.checkReadPermission(projectId);
    ServerProject sp = ServerProject.openProject(projectId, false);
    sp.refreshTree();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void rebuildDir(Object projectId, String[] path) {
    if (projectId == null) {
      throw new InvalidRequestException(getString(INVALID_REQUEST_MSG),
        "projectId == null");
    }
    if (path == null) {
      throw new InvalidRequestException(getString(INVALID_REQUEST_MSG),
        "path == null");
    }
    this.checkReadPermission(projectId);
    ServerProjectFile spf = ServerProjectFile.findFile(projectId, path);
    spf.rebuildDir();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void refreshDir(Object projectId, String[] path) {
    if (projectId == null) {
      throw new InvalidRequestException(getString(INVALID_REQUEST_MSG),
        "projectId == null");
    }
    if (path == null) {
      throw new InvalidRequestException(getString(INVALID_REQUEST_MSG),
        "path == null");
    }
    this.checkReadPermission(projectId);
    ServerProjectFile spf = ServerProjectFile.findFile(projectId, path);
    spf.refreshDir();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void createDirectory(Object projectId, String[] path) {
    this.checkWritePermission(projectId);
    String[] root = new String[] {};
    ServerProjectFile parentDir = ServerProjectFile.findFile(projectId, root);
    for (String dirName : path) {
      ServerProjectFile currentDir = parentDir.getChild(dirName);
      if (currentDir != null) {
        //  preciso verificar se  mesmo um diretrio
        if (!currentDir.isDirectory()) {
          String InfoMsg =
            getFormattedString("ProjectService.info.not.directory",
              new Object[] { currentDir.getName(), parentDir.getName() });
          throw new InfoException(InfoMsg);
        }
      }
      else {
        Object userId = Service.getUser().getId();
        parentDir.createFile(dirName, ProjectFileType.DIRECTORY_TYPE, userId);
        currentDir = parentDir.getChild(dirName);
      }
      parentDir = currentDir;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ClientProjectFile[] getChildren(Object projectId, String[] path) {
    return getChildren(projectId, path, false);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ClientProjectFile[] getChildren(Object projectId, String[] path,
    boolean recursive) {
    this.checkReadPermission(projectId);
    final ServerProjectFile dir = ServerProjectFile.findFile(projectId, path);
    if (!dir.isDirectory()) {
      return null;
    }
    final ServerProjectFile[] children = dir.getChildren();
    if (children == null) {
      return null;
    }
    final int sz = children.length;
    final ClientProjectFile[] clientFiles = new ClientProjectFile[sz];
    for (int i = 0; i < sz; i++) {
      ServerProjectFile srvChild = children[i];
      if (recursive && srvChild.isDirectory()) {
        clientFiles[i] = buildClientProjectSubtree(srvChild);
      }
      else {
        clientFiles[i] = buildSingleClientProjectFile(srvChild);
      }
    }
    return clientFiles;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ClientProjectFile getChild(Object projectId, String[] path,
    String name) {
    this.checkReadPermission(projectId);
    final ServerProjectFile dir = ServerProjectFile.findFile(projectId, path);
    if (!dir.isDirectory()) {
      return null;
    }
    ServerProjectFile srvChild = dir.getChild(name);
    if (null == srvChild) {
      return null;
    }
    return buildSingleClientProjectFile(srvChild);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ClientProjectFile getChild(Object projectId, String[] path) {
    this.checkReadPermission(projectId);
    final ServerProjectFile file = ServerProjectFile.findFile(projectId, path);
    return buildSingleClientProjectFile(file);
  }

  /**
   * Obtm o pai de um arquivo/diretrio. Lana {@link ServiceFailureException}
   * se o arquivo/diretrio no existe.
   *
   * @param projectId identificador do projeto
   * @param path path para o arquivo/diretrio
   * @return pai do arquivo/diretrio ou <code>null</code> caso este seja o
   *         diretrio-raiz
   */
  public ClientProjectFile getParent(Object projectId, String[] path) {
    this.checkReadPermission(projectId);
    ServerProjectFile file = ServerProjectFile.findFile(projectId, path);
    ServerProjectFile parent = file.getParent();
    if (parent == null) {
      return null;
    }
    return buildSingleClientProjectFile(parent);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void createFile(Object projectId, String[] path, String name,
    String type) {
    this.checkWritePermission(projectId);
    ServerProjectFile dir = ServerProjectFile.findFile(projectId, path);
    Object userId = Service.getUser().getId();
    dir.createFile(name, type, userId);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void createFiles(Object projectId, String[] parentPath,
    List<ProjectFileInfo> fileInfoList) {
    this.checkWritePermission(projectId);
    if (projectId == null) {
      throw new InvalidRequestException(
        "ProjectService: createFile: projectId == null");
    }
    if (parentPath == null) {
      throw new InvalidRequestException(
        "ProjectService: createFile: parentPath == null");
    }
    ServerProjectFile dir = ServerProjectFile.findFile(projectId, parentPath);
    Object userId = Service.getUser().getId();
    dir.createFiles(fileInfoList, userId);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void renameFile(Object projectId, String[] path, String name) {
    this.checkWritePermission(projectId);
    ServerProjectFile spf = ServerProjectFile.findFile(projectId, path);
    spf.rename(name);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void changeFileType(Object projectId, String[] path, String type) {
    this.checkWritePermission(projectId);
    ServerProjectFile spf = ServerProjectFile.findFile(projectId, path);
    spf.changeType(type);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void copyFile(Object projectId, String[] filePath, String[] dirPath) {
    this.checkWritePermission(projectId);
    ServerProjectFile file = ServerProjectFile.findFile(projectId, filePath);
    ServerProjectFile dir = ServerProjectFile.findFile(projectId, dirPath);
    file.copy(dir);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void copyFile(Object sourceProjectId, String[] filePathSource,
    Object targetProjectId, String[] dirPathTarget) {
    this.checkWritePermission(targetProjectId);
    ServerProjectFile file =
      ServerProjectFile.findFile(sourceProjectId, filePathSource);
    ServerProjectFile dir =
      ServerProjectFile.findFile(targetProjectId, dirPathTarget);
    file.copy(dir);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void moveFile(Object projectId, String[] filePath, String[] dirPath) {
    moveFile(projectId, filePath, projectId, dirPath);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void moveFile(Object sourceProjectId, String[] filePathSource,
    Object targetProjectId, String[] dirPathTarget) {
    this.checkWritePermission(sourceProjectId);
    this.checkWritePermission(targetProjectId);
    ServerProjectFile file =
      ServerProjectFile.findFile(sourceProjectId, filePathSource);
    ServerProjectFile dir =
      ServerProjectFile.findFile(targetProjectId, dirPathTarget);
    file.move(dir);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void removeFile(Object projectId, String[] path) {
    this.checkWritePermission(projectId);
    ServerProjectFile spf = ServerProjectFile.findFile(projectId, path);
    spf.remove();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ClientProjectFileInfo getUpdatedFileInfo(Object projectId,
    String[] path) {
    this.checkReadPermission(projectId);
    ServerProjectFile spf = ServerProjectFile.findFile(projectId, path);
    return spf.getUpdatedInfo();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void removeFiles(Object projectId, String[][] paths) {
    this.checkWritePermission(projectId);
    paths = filterPathsToRemove(paths);
    HashMap<ServerProjectFile, List<String[]>> parentMap =
      new HashMap<ServerProjectFile, List<String[]>>();
    for (String[] path : paths) {
      ServerProjectFile spf = ServerProjectFile.findFile(projectId, path);
      if (!spf.removeFile()) {
        String errMsg = "Falha ao remover " + spf.getAbsolutePath();
        throw (new ServiceFailureException(errMsg));
      }
      ServerProjectFile parent = spf.getParent();
      List<String[]> removedPathsList = parentMap.get(parent);
      if (removedPathsList == null) {
        removedPathsList = new ArrayList<String[]>();
      }
      removedPathsList.add(spf.getPath());
      parentMap.put(parent, removedPathsList);
    }
    Iterator<List<String[]>> iter = parentMap.values().iterator();
    while (iter.hasNext()) {
      List<String[]> removedPathsList = iter.next();
      String[][] removedFilePaths = new String[removedPathsList.size()][];
      for (int i = 0; i < removedPathsList.size(); i++) {
        removedFilePaths[i] = removedPathsList.get(i);
      }
      fireFilesDeletedEvent(projectId, removedFilePaths);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getFileDescription(Object projectId, String[] path) {
    this.checkReadPermission(projectId);
    ServerProjectFile spf = ServerProjectFile.findFile(projectId, path);
    return spf.getDescription();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setFileDescription(Object projectId, String[] path, String text) {
    this.checkWritePermission(projectId);
    ServerProjectFile spf = ServerProjectFile.findFile(projectId, path);
    spf.setDescription(text);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void appendFileDescription(Object projectId, String[] path,
    String text) {
    this.checkWritePermission(projectId);
    ServerProjectFile spf = ServerProjectFile.findFile(projectId, path);
    spf.appendDescription(text);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean existsFile(Object projectId, String[] path) {
    this.checkReadPermission(projectId);
    return ServerProjectFile.testFile(projectId, path) != null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public RemoteFileChannelInfo openFileChannel(Object projectId, String[] path,
    boolean readOnly) {
    if (readOnly) {
      this.checkReadPermission(projectId);
    }
    else {
      this.checkWritePermission(projectId);
    }
    ServerProjectFile spf = null;
    spf = ServerProjectFile.findFile(projectId, path);
    return spf.openChannel(readOnly);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getProjectLocationInServer(Object projectId) {
    this.checkReadPermission(projectId);
    ServerProject sp = ServerProject.openProject(projectId, false);
    File file = new File(sp.getAbsolutePath());
    try {
      return file.getCanonicalPath();
    }
    catch (IOException e) {
      throw new ServiceFailureException(e.getMessage(), e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object acquireExclusiveLock(Object projectId, String[] path) {
    this.checkWritePermission(projectId);
    ServerProjectFile spf = ServerProjectFile.findFile(projectId, path);
    return spf.acquireExclusiveLock();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object acquireExclusiveLock(Object projectId, String[] path,
    FileLockListenerInterface listener, long timeout) {
    this.checkWritePermission(projectId);
    ServerProjectFile spf = ServerProjectFile.findFile(projectId, path);
    return spf.acquireExclusiveLock(listener, timeout);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object acquireSharedLock(Object projectId, String[] path) {
    this.checkReadPermission(projectId);
    ServerProjectFile spf = ServerProjectFile.findFile(projectId, path);
    return spf.acquireSharedLock();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object acquireSharedLock(Object projectId, String[] path,
    FileLockListenerInterface listener, long timeout) {
    this.checkReadPermission(projectId);
    ServerProjectFile spf = ServerProjectFile.findFile(projectId, path);
    return spf.acquireSharedLock(listener, timeout);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int releaseLock(Object projectId, String[] path, Object lockId) {
    this.checkReadPermission(projectId);
    ServerProjectFile spf = ServerProjectFile.findFile(projectId, path);
    return spf.releaseLock(lockId);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int forceReleaseLock(Object projectId, String[] path) {
    checkLockPermission(projectId, path);
    ServerProjectFile spf = ServerProjectFile.findFile(projectId, path);
    return spf.forceReleaseLock();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean userOwnsLock(Object projectId, String[] path) {
    ServerProjectFile spf = ServerProjectFile.findFile(projectId, path);
    LockStatus status =
      FileLockManager.getInstance().isLocked(spf, Service.getKey().toString());
    return status == LockStatus.LOCKED_BY_USER;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long fileSize(Object projectId, String[] path) {
    this.checkReadPermission(projectId);
    ServerProjectFile spf = ServerProjectFile.findFile(projectId, path);
    return spf.size();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public long getModificationDate(Object projectId, String[] path) {
    this.checkReadPermission(projectId);
    ServerProjectFile spf = ServerProjectFile.findFile(projectId, path);
    return spf.getModificationDate();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void addObserver(RemoteObserver observer, Object arg) {
    //Para observar deve-se usar o MessageService
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized boolean deleteObserver(RemoteObserver observer,
    Object arg) {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String, ProjectFileTypeInfo> getAllFileTypes(Locale loc) {
    return typeRepository.getInfos(loc);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ProjectFileTypeInfo getFileType(String type) {
    return typeRepository.getInfo(getDefaultLocale(), type);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getMimeType(String type) {
    ProjectFileTypeInfo info = getFileType(type);
    return info.getMimeType();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ProjectAdminInfo[] getLockedProjectAdminInfo() {
    //
    // Ateno: esse mtodo personifica o administrador para poder abrir os
    // projetos e verificar se o usurio corrente possui acesso a cada um,
    // descartando os que ele no possuir acesso. Esse cdigo deve ser feito
    // com cuidado para evitar quebra da segurana.
    Object userId = Service.getUser().getId();
    try {
      Service.setUserId(User.getAdminId());
      ProjectAdminInfo[] infos = prjAdmin.getAllProjectAdminInfo();
      List<ProjectAdminInfo> filteredInfos = new LinkedList<ProjectAdminInfo>();
      for (int i = 0; i < infos.length; i++) {
        ServerProject project = null;
        boolean waitingAreaFree =
          infos[i].getState() == ProjectAllocationState.WAITING_AREA_FREE;
        if (!waitingAreaFree) {
          Object projectId = ServerProject.getId(infos[i].getOwnerId(),
            infos[i].getProjectName());
          project = ServerProject.openProject(projectId, false);
        }
        if (waitingAreaFree
          || (project != null && project.userHasAccess(userId))) {
          filteredInfos.add(infos[i]);
        }
      }
      return filteredInfos.toArray(new ProjectAdminInfo[0]);
    }
    finally {
      Service.setUserId(userId);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isUnlockedWithAreaAllocated(Object projectId) {
    this.checkReadPermission(projectId);
    return prjAdmin.isUnlockedWithAreaAllocated(projectId);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setFileModificationDate(Object projectId, String[] path,
    long date) {
    this.checkWritePermission(projectId);
    ServerProjectFile spf = ServerProjectFile.findFile(projectId, path);
    spf.setModificationDate(date);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setUnderConstruction(Object projectId, String[] path,
    boolean underConstruction) {
    this.checkWritePermission(projectId);
    ServerProjectFile spf = ServerProjectFile.findFile(projectId, path);
    spf.setUnderConstruction(underConstruction);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void startUpdate(Object projectId, String[] path, long interval,
    boolean notification, Serializable extraInfo) {
    this.checkWritePermission(projectId);
    ServerProjectFile spf = ServerProjectFile.findFile(projectId, path);
    UpdatableFileInfo updatableFileInfo = spf.getUpdatableFileInfo();
    if (updatableFileInfo == null) {
      throw new ServiceFailureException(
        MessageFormat.format(this.getString(LNG_PREFIX + "error.not.updatable"),
          new Object[] { spf.getName() }));
    }
    updatableFileInfo.setExtraUserData(extraInfo);
    this.updateScheduler.startUpdate(spf, interval, notification);
    ServerProjectFile parent = spf.getParent();
    parent.refreshDir();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void startUpdate(Object projectId, String[] path, long interval,
    boolean notification) {
    startUpdate(projectId, path, interval, notification, null);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void stopUpdate(Object projectId, String[] path) {
    this.checkWritePermission(projectId);
    ServerProjectFile spf = ServerProjectFile.findFile(projectId, path);
    this.updateScheduler.stopUpdate(spf);
    ServerProjectFile parent = spf.getParent();
    parent.refreshDir();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<FileInfoSearchResult> getAllFileInfoSearchResult(Object projectId,
    String text, boolean isCaseInsensitive, boolean isRegex) {
    this.checkReadPermission(projectId);
    String projectPath = ServerProject.getAbsolutePath(projectId);
    if (projectPath == null) {
      return null;
    }
    FileFinder finder = new FileFinder().matchAll(FileFinder.NO_DOTFILE_FILTER,
      createFileFinderFilter(text, isCaseInsensitive, isRegex));
    File startDir = new File(projectPath);
    List<File> fileList;
    try {
      fileList = finder.findIn(startDir);
    }
    catch (IOException e) {
      throw new ServiceFailureException(
        String.format(getString(LNG_PREFIX + "error.invalidDir"), projectPath));
    }
    int startDirPrefixLength = startDir.getPath().length();
    final List<FileInfoSearchResult> result =
      new ArrayList<FileInfoSearchResult>(fileList.size());
    String projectName = ServerProject.getProjectName(projectId);
    String ownerName = ServerProject.getOwnerName(projectId);
    User user = Service.getUser();
    Object userId = user.getId();
    for (File file : fileList) {
      boolean isWritable = user.isAdmin() || userHasAccessRW(projectId, userId);
      /*
       * no queremos exibir o path antes
       */
      String relativePath = file.getParent().substring(startDirPrefixLength);
      result.add(new FileInfoSearchResult(file.getName(),
        relativePath.isEmpty() ? "/" : relativePath, projectName, projectId,
        ownerName, isWritable));
    }
    return result;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<Object> getManageableProjectsId() {
    Set<Object> manageables = new HashSet<Object>();
    User user = Service.getUser();
    if (user.isAdmin()) {
      Set<Object> allUsers =
        AdministrationService.getInstance().getAllUserIds();
      for (Object anUserId : allUsers) {
        loadAllProjectsId(anUserId, manageables);
      }
    }
    else {
      loadAllProjectsId(user.getId(), manageables);
    }
    return manageables;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setUnallocatedProjectAsLocked(Object projectId, int areaSize)
    throws RemoteException {
    this.checkIsAdmin();
    if (projectId == null) {
      throw new IllegalArgumentException(
        getString(LNG_PREFIX + "illegal.argument.exception.msg"));
    }
    if (areaSize <= 0) {
      throw new IllegalArgumentException(
        getString(LNG_PREFIX + "illegal.argument.value.exception.msg"));
    }
    CommonClientProject ccp1 = openProject(projectId, false);
    if (ccp1 == null) {
      String infoMsg =
        getFormattedString("ProjectService.info.project.not.found",
          new Object[] { getProjectName(projectId) });
      throw new InfoException(infoMsg);
    }
    ccp1.close(false);
    if (isUnlockedWithAreaAllocated(projectId)) {
      throw new ServiceFailureException(
        getString(LNG_PREFIX + "project.allocated.exception.msg"));
    }
    prjAdmin.lockProject(projectId, areaSize);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setAllocatedProjectAsUnallocated(Object projectId)
    throws RemoteException {
    this.checkIsAdmin();
    if (projectId == null) {
      throw new IllegalArgumentException(
        getString(LNG_PREFIX + "illegal.argument.exception.msg"));
    }
    CommonClientProject ccp = openProject(projectId, false);
    if (ccp == null) {
      String infoMsg =
        getFormattedString("ProjectService.info.project.not.found",
          new Object[] { getProjectName(projectId) });
      throw new InfoException(infoMsg);
    }
    ccp.close(false);
    if (!isUnlockedWithAreaAllocated(projectId)) {
      throw new ServiceFailureException(
        getString(LNG_PREFIX + "project.unallocated.exception.msg"));
    }
    prjAdmin.removeProjectAdminInfo(projectId);
    ccp = openProject(projectId, false);
    if (ccp == null) {
      throw new ServiceFailureException(
        getString(LNG_PREFIX + "modified.project.access.error.exception.msg"));
    }
    ccp.setLockingAreaSize(0);
    ccp.close(false);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setUnallocatedProjectAsAllocated(Object projectId, int areaSize)
    throws RemoteException {
    this.checkIsAdmin();
    if (projectId == null) {
      throw new IllegalArgumentException(
        getString(LNG_PREFIX + "illegal.argument.exception.msg"));
    }
    if (areaSize <= 0) {
      throw new IllegalArgumentException(
        getString(LNG_PREFIX + "illegal.argument.value.exception.msg"));
    }
    CommonClientProject ccp = openProject(projectId, false);
    if (ccp == null) {
      String infoMsg =
        getFormattedString("ProjectService.info.project.not.found",
          new Object[] { getProjectName(projectId) });
      throw new InfoException(infoMsg);
    }
    ccp.close(false);
    if (isUnlockedWithAreaAllocated(projectId)) {
      throw new ServiceFailureException(
        getString(LNG_PREFIX + "project.allocated.exception.msg"));
    }
    prjAdmin.lockProject(projectId, areaSize);
    prjAdmin.unlockProject(projectId, areaSize);
    ccp = openProject(projectId, false);
    if (ccp == null) {
      throw new ServiceFailureException(
        getString(LNG_PREFIX + "modified.project.access.error.exception.msg"));
    }
    ccp.setLockingAreaSize(areaSize);
    ccp.close(false);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ProjectAdminInfo getProjectAdminInfo(Object projectId) {
    this.checkReadPermission(projectId);
    return prjAdmin.getProjectAdminInfo(projectId);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isAreaReserved() {
    return areaReserved;
  }

  //
  // Implementao dos mtodos de Service.
  //

  /**
   * Constri a instncia do servio.
   *
   * @throws ServerException se ocorrer erro.
   */
  public static void createService() throws ServerException {
    new ProjectService();
  }

  /**
   * Cria o ServerSocketChannel e faz o accept. Essa comunicao, que serve
   * apenas para leitura e escrita de arquivos, deve ser no-bloqueante. Deve
   * haver, tambm, um pool de threads que recebam/enviem os dados disponveis
   * de/para os clientes. Nesta primeira implementao, criamos um novo thread
   * para cada conexo recebida.
   */
  @Override
  public void initService() throws ServerException {
    this.prjDir = new File(projectRepositoryPath);
    checkRepository();
    this.prjAdmin = new ProjectAdministrator(this);
    areaReserved = getBooleanProperty("area.reserved");
    loadProjects();
    this.updateScheduler = new ProjectFileUpdateScheduler(
      this.getStringListProperty("updater.class"));
    FileLockManager.getInstance().startThread();
    this.updateScheduler.init();
  }

  /**
   * {@inheritDoc}
   *
   * Encerra o servio pelo trmino de <code>prjAdmin</code>.
   */
  @Override
  public void shutdownService() throws csbase.server.ServerException {
    FileLockManager.getInstance().stopThread();
    if (updateScheduler != null) {
      this.updateScheduler.shutdown();
    }
    if (prjAdmin != null) {
      prjAdmin.finish();
    }
    ServerProjectFile.clearAllCache();
  }

  //
  // Implementao dos demais mtodos pblicos do servio. Esses mtodos so
  // exportados para uso pelos demais servios. Como no pertencem  definio
  // da interface remota, no esto disponveis para os clientes RMI.
  //

  /**
   * Obtm o identificador de um projeto a partir do identificador do dono e do
   * nome do projeto.  importante observar que mesmo que o identificador seja
   * retornado, o projeto pode no existir.
   *
   * @param userId Identificador do usurio dono do projeto.
   * @param projectName Nome do projeto.
   * @return O identificador do projeto.
   */
  public Object getProjectId(Object userId, String projectName) {
    return ServerProject.getId(userId, projectName);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean existsProject(Object projectId) {
    return ServerProject.existsProject(projectId);
  }

  /**
   * Informa o nome de um projeto.
   *
   * @param projectId Identificador do projeto.
   * @return O nome do projeto.
   */
  public String getProjectName(Object projectId) {
    return ServerProject.getProjectName(projectId);
  }

  /**
   * Informa o caminho para um projeto, a partir da raiz da rea de projetos.
   *
   * @param projectId Identificador do projeto.
   * @return O caminho para um projeto, a partir da raiz da rea de projetos.
   */
  public String[] getProjectPath(Object projectId) {
    this.checkReadPermission(projectId);
    ServerProject sp = ServerProject.openProject(projectId, false);
    return sp.getPath();
  }

  /**
   * Obtm a caminho para o repositrio de projetos.
   *
   * @return O caminho.
   */
  public String getProjectRepositoryPath() {
    return projectRepositoryPath;
  }

  /**
   * Recupera a propriedade do caminho dos arquivos de tipos de arquivos.
   *
   * @return caminho dos arquivos de tipos de arquivos.
   */
  public String getFileTypesProperty() {
    String prop = getStringProperty("FileTypesPath");
    return getOSPropertyPath(prop);
  }

  /**
   * Retona o identificador do usurio dono do projeto.
   *
   * @param projectId Identificador do projeto.
   * @return O identificador do usurio dono do projeto.
   */
  public Object getOwnerId(Object projectId) {
    this.checkReadPermission(projectId);
    ServerProject sp = ServerProject.openProject(projectId, false);
    return sp.getUserId();
  }

  /**
   * Retona o conjunto de todos os usurios que tem acesso ao projeto.
   *
   * @param projectId Identificador do projeto.
   * @return O conjunto com todos os usurios que tem acesso ao projeto.
   */
  public Collection<Object> getAllUsers(Object projectId) {
    this.checkReadPermission(projectId);
    ServerProject sp = ServerProject.openProject(projectId, false);
    return ProjectPermissions.getAllUsers(sp.getInfo());
  }

  /**
   * Informa se um usurio possui acesso a um projeto. Isto acontece se qualquer
   * uma das opes seguintes for satisfeita:
   * <ul>
   * <li>projeto  pblico (RO ou RW)
   * <li>usurio  o dono do projeto ou o admin
   * <li>compartilhamento  PARTIAL e usurio tem acesso RO ou RW ao projeto
   * </ul>
   *
   * @param projectId Identificador do projeto.
   * @param userId Identificador do usurio.
   * @return Verdadeiro se o usurio tem acesso ao projeto, falso caso
   *         contrrio.
   */
  public boolean userHasAccess(Object projectId, Object userId) {
    try {
      ServerProject sp = ServerProject.openProject(projectId, false);
      return ProjectPermissions.userHasAccess(sp.getInfo(), userId);
    }
    catch (PermissionException pe) {
      return false;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean userCanWrite(Object projectId, Object userId)
    throws RemoteException {
    return userHasAccessRW(projectId, userId);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<Boolean> userCanWrite(List<Object> projects, Object userId)
    throws RemoteException {
    List<Boolean> result = new ArrayList<Boolean>(projects.size());
    for (Object prjId : projects) {
      result.add(userHasAccessRW(prjId, userId));
    }
    return result;
  }

  /**
   * Verifica se um usurio tem permisso de escrita em um projeto.
   *
   * @param projectId identificador do projeto
   * @param userId identificador do usurio
   * @return <code>true</code> se o usurio tem permisso de escrita no projeto
   */
  public boolean userHasAccessRW(Object projectId, Object userId) {
    ServerProject sp = ServerProject.openProject(projectId, false);
    return ProjectPermissions.userHasAccessRW(sp.getInfo(), userId);
  }

  /**
   * Informa o tipo de compartilhamento de um projeto.
   *
   * @param projectId Identificador do projeto.
   * @return O tipo de compartilhamento do projeto.
   */
  public SharingType getSharingType(Object projectId) {
    if (isInternalServerRequest()) {
      ServerProject sp = ServerProject.openServerProject(projectId);
      return ProjectPermissions.getSharingType(sp.getInfo());
    }
    else {
      this.checkReadPermission(projectId);
      ServerProject sp = ServerProject.openProject(projectId, false);
      return ProjectPermissions.getSharingType(sp.getInfo());
    }
  }

  /**
   * Obtm o caminho absoluto para um arquivo de um projeto.
   *
   * @param projectId Identificador do projeto.
   * @param path Caminho do arquivo, a partir da raiz.
   * @return O caminho absoluto para o arquivo.
   */
  public String getAbsolutePath(Object projectId, String[] path) {
    this.checkReadPermission(projectId);
    ServerProjectFile spf = ServerProjectFile.findFile(projectId, path);
    return spf.getAbsolutePath();
  }

  /**
   * Obtm um InputStream para ler o contedo de um arquivo de um projeto.
   *
   * @param projectId Identificador do projeto.
   * @param path Caminho do arquivo, a partir da raiz.
   * @return Um InputStream associado ao arquivo.
   */
  public InputStream getInputStream(Object projectId, String[] path) {
    this.checkReadPermission(projectId);
    ServerProjectFile spf = ServerProjectFile.findFile(projectId, path);
    return spf.openInputStream();
  }

  /**
   * Obtm um OutputStream para escrever o contedo de um arquivo de um projeto.
   * Sobrescreve dados j existentes.
   *
   * @param projectId Identificador do projeto.
   * @param path Caminho do arquivo, a partir da raiz.
   * @return Um OutputStream associado ao arquivo.
   *
   * @see #getOutputStream(Object, String[], boolean)
   */
  public OutputStream getOutputStream(Object projectId, String[] path) {
    return getOutputStream(projectId, path, false);
  }

  /**
   * Obtm um OutputStream para escrever o contedo de um arquivo de um projeto,
   * com a possibilidade de escrever a partir do final do arquivo.
   *
   * @param projectId Identificador do projeto.
   * @param path Caminho do arquivo, a partir da raiz.
   * @param append <code>true</code> para escrever a partir do final do arquivo
   * @return Um OutputStream associado ao arquivo.
   */
  public OutputStream getOutputStream(Object projectId, String[] path,
    boolean append) {
    this.checkWritePermission(projectId);
    ServerProjectFile spf = ServerProjectFile.findFile(projectId, path);
    return spf.openOutputStream(append);
  }

  /**
   * Obtm um File para acesso ao contedo de um arquivo de um projeto. Esse
   * mtodo s existe por causa do WIOService e ser removido quando o WIO for
   * removido. <b>No use!</b>
   *
   * @param projectId Identificador do projeto.
   * @param path Caminho do arquivo, a partir da raiz.
   * @return Um InputStream associado ao arquivo.
   * @deprecated Esse mtodo s existe por causa do WIOService e ser removido
   *             quando o WIO for removido. <b>No use!</b>
   */
  @Deprecated
  public File getFile(Object projectId, String[] path) {
    this.checkReadPermission(projectId);
    ServerProjectFile spf = ServerProjectFile.findFile(projectId, path);
    return spf.getFile();
  }

  /**
   * Agenda a atualizao de um arquivo.
   *
   * @param projectId Identificador do projeto.
   * @param path Caminho do arquivo, a partir da raiz.
   */
  public void singleUpdate(Object projectId, String[] path) {
    this.checkReadPermission(projectId);
    ServerProjectFile spf = ServerProjectFile.findFile(projectId, path);
    this.updateScheduler.singleUpdate(spf);
  }

  /**
   * Obtm os usurios de um projeto que devem ser notificados.
   *
   * @param projectId identificador do projeto
   *
   * @return os usurios que deber ser notificados
   */
  public String[] getUserToNotify(Object projectId) {
    Set<Object> userIds = new HashSet<Object>();

    // O administrador deve receber a notificao de todos os projetos
    try {
      userIds.addAll(User.getAdminIds());
    }
    catch (Exception e) {
      Server.logSevereMessage(
        "Erro ao obter a lista de usurios adminstradores.", e);
    }

    // O dono do projeto deve sempre receber a notificao
    userIds.add(ServerProject.getOwnerId(projectId));

    //Coloca na tread null para indicar que essa  uma chamada interna
    Object user = Service.getKey();
    Service.setUserId(null);
    SharingType sharingType = getSharingType(projectId);

    switch (sharingType) {
      case PRIVATE:
        /*
         * Se o projeto  privado, o dono do projeto e o admin devem receber a
         * notificao. Os dois j foram includos.
         */
        break;
      case PARTIAL:
        /*
         * Se o projeto est sendo compartilhado, notificamos os usurios que
         * tm acesso ao mesmo.
         */
        userIds.addAll(getAllUsers(projectId));
        break;
      case ALL_RO:
      case ALL_RW:
        /*
         * Se o projeto  visivel pblicamente, todos os usurios logados no
         * momento devem receber a notificao.
         */
        final LoginService loginService = LoginService.getInstance();
        UserOutline[] loggedUserOutlines = loginService.getLoggedUsers();
        for (UserOutline aLoggedUser : loggedUserOutlines) {
          userIds.add(aLoggedUser.getId());
        }
        break;
      default:
        Server.logSevereMessage("Tipo de compartilhamento do projeto "
          + projectId + " desconhecido: " + getSharingType(projectId) + ".\n");
        Service.setUserId(user);
        return null;
    }
    Service.setUserId(user);
    return userIds.toArray(new String[0]);
  }

  //
  // Implementao dos demais mtodos no pblicos da classe, acessveis apenas
  //  prpria classe, s suas subclasses ou ao pacote.
  //
  // ATENO: no deve haver nenhum mtodo pblico abaixo, para garantir que
  // os demais servios utilizem apenas os recursos da interface pblica.
  //

  /**
   * Construtor
   *
   * @throws ServerException se houver falha de servio
   */
  protected ProjectService() throws ServerException {
    super(SERVICE_NAME);
    // XXX - Temporrio. Aguardando a soluo do Bug #5847.
    ClientRemoteLocator.projectService = this;
    this.projectRepositoryPath = this.getStringProperty("base.project.dir");
    this.sourceId = this.getStringProperty("sourceid");
    this.typeRepository = new ProjectFileTypeRepository(getFileTypesProperty());
    this.projectTemplates = createProjectTemplates();
    System.out.println("Diretrio base de projetos: " + projectRepositoryPath);

    readOptimizationProperties();
  }

  /**
   * L as configuraes de templates dos projetos.
   *
   * @return lista de templates.
   */
  private List<ProjectTemplate> createProjectTemplates() {
    List<ProjectTemplate> fileInfos = new ArrayList<>();
    List<String> templateDirs = this.getStringListProperty("template.directory");
    for (int i = 1; i <= templateDirs.size(); i++) {
      String templateDir = templateDirs.get(i - 1);
      String[] path = FileUtils.splitPath(templateDir);
      checkTemplatePath(path);

      /* Para cada diretrio de template configurado, buscamos as configuraes opcionais:
        tipo do diretrio e classe que implementa o template. */
      String typeProp = "template.type." + i;
      String type;
      if (hasProperty(getName() + "." + typeProp)) {
        type = getStringProperty(typeProp);
      }
      else {
        type = ProjectFileType.DIRECTORY_TYPE;
      }

      ProjectFileInfo fileInfo = new ProjectFileInfo(path, type);
      String classProp = "template.class." + i;
      String templateClassName;
      if (hasProperty(getName() + "." + classProp)) {
        templateClassName = getStringProperty(classProp);
      }
      else {
        templateClassName = DefaultProjectTemplate.class.getName();
      }

      String paramsProp = classProp + ".params";
      String[] templateParams;
      if (hasProperty(getName() + "." + paramsProp)) {
        String paramList = getStringProperty(paramsProp);
        // Remove os espaos e obtm cada parmetro da lista separada por vrgula
        templateParams = paramList.replaceAll("\\s","").split(",");
      }
      else {
        templateParams = new String[0];
      }

      ProjectTemplate template = createTemplate(templateClassName, fileInfo, templateParams);
      fileInfos.add(template);
    }
    return fileInfos;
  }

  /**
   * Verifica se o caminho definido para o template  vlido.
   *
   * @param path o caminho.
   */
  private void checkTemplatePath(String[] path) {
    for (String component : path) {
      String fixedFileName = FileUtils.fixFileName(component);
      if (!fixedFileName.equals(component)) {
        String error = getFormattedString(LNG_PREFIX + "error.template.path", new Object[] { component });
        throw new ServiceFailureException(error);
      }
    }
  }

  /**
   * Cria um template via reflexo a partir da classe que o implementa e do diretrio-base especificado.
   *
   * @param templateClassName nome da classe que implementa o template.
   * @param fileInfo diretrio-base do template.
   * @param templateParams parmetros para configurao do template.
   * @return o template.
   */
  private ProjectTemplate createTemplate(String templateClassName, ProjectFileInfo fileInfo, String[] templateParams) {
    ProjectTemplate template;
    try {
      Class<?> clazz = Class.forName(templateClassName);
      if (ProjectTemplate.class.isAssignableFrom(clazz)) {
        Constructor<?> constructor = clazz.getConstructor(ProjectFileInfo.class, String[].class);
        template = (ProjectTemplate) constructor.newInstance(fileInfo, (Object) templateParams);
      }
      else {
        String error = getFormattedString(LNG_PREFIX + "error.template.class", new Object[] { templateClassName });
        throw new ServiceFailureException(error);
      }
    }
    catch (Exception e) {
      String error = getFormattedString(LNG_PREFIX + "error.template.class", new Object[] { templateClassName });
      throw new ServiceFailureException(error, e);
    }
    return template;
  }

  /**
   * L as propriedades de otimizao de acesso aos arquivos de projeto
   *
   * @throws ServerException
   */
  private void readOptimizationProperties() throws ServerException {
    String modeStr = this.getStringProperty("client.optimization.mode");
    clientOptimizationMode =
      ClientOptimizationMode.getClientOptimizationMode(modeStr);
    if (clientOptimizationMode == null) {
      String err =
        "A propriedade optimization.mode contm um valor invlido. Vlidos: [NONE] ou [GLOBAL]";
      Server.logSevereMessage(err);
      throw new ServerException(err);
    }
    clientLocalProjectPath =
      this.getStringProperty("client.optimization.project.path");
    String msg = "Modo de Otimizao: " + modeStr;
    if (clientOptimizationMode == ClientOptimizationMode.GLOBAL) {
      msg += "\nPath local de Projeto: " + clientLocalProjectPath;
    }
    Server.logInfoMessage(msg);
    System.out.println(msg);
  }

  /**
   * Retorna o diretrio raiz de um projeto.
   *
   * @param projectId Identificador do projeto.
   * @return File representando o diretrio raiz do projeto.
   */
  File getRoot(Object projectId) {
    this.checkReadPermission(projectId);
    String path = ServerProject.getAbsolutePath(projectId);
    return new File(path);
  }

  /**
   * O observador s  notificado o identificador do usurio no seu argumento
   * for igual ao identificador do usurio no evento gerado.
   *
   * @param arg .
   * @param event .
   * @return .
   */
  @Override
  protected boolean has2Update(Object arg, Object event) {
    return false;
  }

  /**
   * Verifica se o usurio informado  o administrador ou, caso contrrio, se
   * ele tem permisso de administrao sobre o servio de projeto do servidor
   * corrente.
   *
   * @param user usurio.
   * @return indicativo.
   */
  private boolean checkAdminPermission(User user) {
    if (user.isAdmin()) {
      return true;
    }
    Server server = Server.getInstance();
    String serverName = AdminPermission.LOCAL + server.getSystemName();
    Permission p = null;
    try {
      p = user.getMatchAttributesPermission(ProjectAdminPermission.class,
        serverName);
    }
    catch (Exception e) {
      String errorMsg = String.format(
        "Ocorreu um erro ao buscar a permisso para gerenciar projetos para o usurio %s.",
        user.getLogin());
      Server.logSevereMessage(errorMsg, e);
    }
    if (p != null) {
      return true;
    }
    return false;
  }

  /**
   * Verifica se o usurio que efetuou a chamada  o administrador ou, caso
   * contrrio, se ele tem permisso de administrao sobre o servio de projeto
   * do servidor corrente. Em caso negativo, lana PermissionException.
   *
   * TODO: VERIFICAR OS CASOS DE ADMINISTRACAO PARA PROJETOS READ ONLY
   *
   * @throws PermissionException em caso de falha de permisso.
   */
  private void checkAdminPermission() throws PermissionException {
    if (!checkAdminPermission(Service.getUser())) {
      throw new PermissionException();
    }
  }

  /**
   * Verifica se o usurio  administrador.
   */
  private void checkIsAdmin() {
    User user = Service.getUser();
    if (user == null || user.isAdmin()) {
      return;
    }
    throw new PermissionException();
  }

  /**
   * Verifica se o usurio  o dono do lock (o administrador  o dono de
   * qualquer lock).
   *
   * @param projectId identificador do projeto
   * @param path para para o arquivo (relativo ao projeto)
   */
  private void checkLockPermission(Object projectId, String[] path) {
    User user = Service.getUser();
    ServerProject sp = ServerProject.openProject(projectId, false);
    Object userId = user.getId();
    if (user.isAdmin() || sp.getUserId().equals(userId)
      || userOwnsLock(projectId, path)) {
      return;
    }
    throw new PermissionException();
  }

  /**
   * Verifica se o usurio corrente tem permisso de leitura no projeto, seja RO
   * ou RW. O administrador possui permisso de leitura em qualquer projeto.
   *
   * @param projectId identificador do projeto
   */
  public void checkReadPermission(Object projectId) {
    if (isInternalServerRequest()) {
      return;
    }
    Object userId = Service.getUser().getId();
    ServerProject sp = ServerProject.openProject(projectId, false);
    if (sp.userHasAccess(userId)) {
      return;
    }
    throw new PermissionException();
  }

  /**
   * Verifica se o usurio corrente tem permisso de escrita no projeto. O
   * administrador possui permisso de escrita em qualquer projeto.
   *
   * @param projectId O identificador do projeto.
   * @throws PermissionException se o usurio no possui permisso
   */
  public void checkWritePermission(Object projectId)
    throws PermissionException {
    User user = Service.getUser();
    if (user == null) {
      String msg = "Usurio no existe.";
      throw new PermissionException(msg);
    }
    Object userId = user.getId();
    ServerProject sp = ServerProject.openProject(projectId, false);
    if ((isProjectRO(sp) && !hasROProjectWritePermission(sp))
      || !sp.userHasAccessRW(userId)) {
      String msg = MessageFormat.format(
        "O usurio {0} no possui permisso de escrita no projeto {1}.",
        getUser().getLogin(), projectId);
      throw new PermissionException(msg);
    }
  }

  /**
   * Verifica se projeto foi criado pela instancia atual do servidor.
   *
   * @param sp
   * @return true caso uma das opes sejam satisfeitas: 1) O nome da instncia
   *         do servidor  igual ao nome do servidor salvo no arquivo das
   *         propriedades do projeto ou se o projeto no possuir ainda essa
   *         varivel. 2) Se o nome do servidor que criou o projeto no for
   *         fazio, ou seja, o projeto possui um servidor dono associado. Caso
   *         contrrio, significa que o projeto foi criado anteriormente a
   *         incluso da propriedade SEVER_NAME em ProjectProperties. 3)
   *         Administrador sempre consegue manipular o projeto independente do
   *         projeto ser somente leitura ou no.
   */
  private boolean isProjectRO(ServerProject sp) {
    return !Server.getInstance().getSystemName().equals(sp.getOwnerServerName())
      && !isEmptyServerProjectOwner(sp)
      && !checkAdminPermission(Service.getUser());
  }

  /**
   * Verifica se projeto foi criado pela instancia atual do servidor.
   *
   * @param sp
   * @return true se o usuario possui a permisso para escrever no projeto
   *         read-only, false caso contrrio.
   */
  private boolean hasROProjectWritePermission(ServerProject sp) {
    boolean hasPermission = false;
    String systemId = null; // Sistema corrente
    try {
      if (ReadOnlyProjectPermission.checkSystemAndOwnerPermission(
        Service.getUser(), systemId, sp.getOwnerServerName())) {
        hasPermission = true;
      }
    }
    catch (Exception e) {
      hasPermission = false;
    }
    return hasPermission;
  }

  /**
   * Verifica se o nome do servidor que criou o projeto ainda est fazio ou no.
   * Isso acontee para os projetos criados anteriormente a incluso da
   * propriedade SEVER_NAME em ProjectProperties.
   *
   * @param sp
   * @return true se o nome da instncia do servidor ainda for vazia.
   */
  private boolean isEmptyServerProjectOwner(ServerProject sp) {
    return sp.getOwnerServerName().equals("");
  }

  /**
   * Verifica se o servidor corrente pode escrever na rea de projetos. Um
   * servidor pode escrever na rea de projetos quando: 1) O Projeto no for
   * "read-only"; ou 2) Quando o projeto  "read-only" mas o usurio possui
   * mesmo assim permisso cadastrada para escrever no projeto.
   *
   * @param sp
   * @return true se o determinado servidor pode escrever no projeto. false caso
   *         contrrio.
   */
  private boolean isProjectWritableFromServer(ServerProject sp) {
    if (isProjectRO(sp) && !hasROProjectWritePermission(sp)) {
      return false;
    }
    return true;
  }

  /**
   * Filtra dentre os paths passados, aqueles que devem realmente ser removidos.
   * Esse filtro existe para o caso de tentar-se remover um diretrio e um ou
   * mais arquivos que este contm; no  necessrio remover os arquivos, basta
   * remover o diretrio. O filtro tambm remove eventuais caminhos nulos.
   *
   * @param inputPaths lista de caminhos de arquivos candidatos  remoo.
   * @return lista de caminhos de arquivos que realmente precisam ser removidos.
   */
  private String[][] filterPathsToRemove(String[][] inputPaths) {
    if (inputPaths == null) {
      return null;
    }
    LinkedHashSet<String[]> candidates = new LinkedHashSet<String[]>();
    Iterator<String[]> iter = null;
    inputPathsLoop: for (int inputIndex =
      0; inputIndex < inputPaths.length; inputIndex++) {
      if (inputPaths[inputIndex] == null) {
        continue inputPathsLoop;
      }
      if (candidates.size() == 0) {
        candidates.add(inputPaths[inputIndex]);
        continue inputPathsLoop;
      }
      LinkedHashSet<String[]> candidatesClone =
        (LinkedHashSet<String[]>) candidates.clone();
      iter = candidatesClone.iterator();
      while (iter.hasNext()) {
        String[] candidatePath = iter.next();
        if (ServerProjectFile.isAncestor(candidatePath,
          inputPaths[inputIndex])) {
          continue inputPathsLoop;
        }
        if (ServerProjectFile.isAncestor(inputPaths[inputIndex],
          candidatePath)) {
          candidates.remove(candidatePath);
        }
      }
      candidates.add(inputPaths[inputIndex]);
    }
    String[][] outputPaths = new String[candidates.size()][];
    iter = candidates.iterator();
    int i = 0;
    while (iter.hasNext()) {
      String[] candidatePath = iter.next();
      outputPaths[i++] = candidatePath;
    }
    return outputPaths;
  }

  /**
   * Notifica aos observadores que um arquivo foi removido.
   *
   * @param projectId .
   * @param paths .
   */
  private void fireFilesDeletedEvent(Object projectId, String[][] paths) {
    Object[] arg = { paths };
    ServerProject sp = ServerProject.openProject(projectId, false);
    notifyProject(
      ProjectEvent.makeEvent(projectId, ProjectEvent.FILES_DELETED, arg),
      projectId);
  }

  /**
   * Atualiza a lista de usurios com acesso RO a um projeto.
   *
   * @param prj - projeto
   * @param usersRO - nova lista de usurios com acesso RO ao projeto
   *
   * @see #updateUsers(Object, SharingType, Set, Set)
   */
  private void updateUsersRO(ServerProject prj, Set<Object> usersRO) {
    updateUsers(prj.getId(), prj.getSharingType(), usersRO, null);
  }

  /**
   * Atualiza a lista de usurios com acesso RW a um projeto.
   *
   * @param prj - projeto
   * @param usersRW - nova lista de usurios com acesso RW ao projeto
   *
   * @see #updateUsers(Object, SharingType, Set, Set)
   */
  private void updateUsersRW(ServerProject prj, Set<Object> usersRW) {
    updateUsers(prj.getId(), prj.getSharingType(), null, usersRW);
  }

  /**
   * Compartilhamento do projeto passou a ser seletivo.
   *
   * @param sp - projeto
   * @param oldSharingType - modo de compartilhamento anterior
   * @param oldUsersRO - usurios que possuam acesso RO
   * @param oldUsersRW - usurios que possuam acesso RW
   */
  private void prjAccessBecamePartial(ServerProject sp,
    SharingType oldSharingType, Set<Object> oldUsersRO,
    Set<Object> oldUsersRW) {
    Set<Object> newUsersRO = sp.getUsersRO();
    Set<Object> newUsersRW = sp.getUsersRW();
    Set<Object> usersToNotify;
    switch (oldSharingType) {
      case PRIVATE:
        /*
         * Projeto era privado. Precisamos registrar e avisar a todos os
         * usurios que ganharam acesso.
         */
        usersGainedAccessToProject(sp, newUsersRO, UsersNotification.INSERT);
        usersGainedAccessToProject(sp, newUsersRW, UsersNotification.INSERT);
        break;
      case ALL_RO:
      case ALL_RW:
        /*
         * Projeto era pblico. Precisamos remov-lo da lista de projetos
         * pblicos, registrar e avisar aos usurios para os quais o tipo de
         * acesso foi alterado e avisar aos usurio que perderam acesso.
         */
        synchronized (publicProjects) {
          publicProjects.remove(
            new UserProjectInfo(sp.getId(), sp.getName(), sp.getUserId()));
        }
        if (oldSharingType == SharingType.ALL_RO) {
          usersGainedAccessToProject(sp, newUsersRO, -1);
          usersGainedAccessToProject(sp, newUsersRW, UsersNotification.CHANGE);
        }
        else {
          usersGainedAccessToProject(sp, newUsersRO, UsersNotification.CHANGE);
          usersGainedAccessToProject(sp, newUsersRW, -1);
        }
        usersToNotify = new HashSet<Object>(
          AdministrationService.getInstance().getAllUserIds());
        usersToNotify.remove(sp.getUserId());
        usersToNotify.removeAll(newUsersRO);
        usersToNotify.removeAll(newUsersRW);
        /*
         * No chamamos usersLostAccessToProject porque os projetos pblicos no
         * usam o usersGainedAccessToProject: eles no inserem o projeto na
         * lista de projetos aos quais o usurio tem acesso (projetos pblicos
         * tm tratamento especial para isto).
         */
        notifyUsers(UsersNotification.REMOVE, usersToNotify, sp);

        notifyUsers(
          new SharedProjectEvent(usersToNotify, sp.getInfo(), newUsersRO,
            newUsersRW, sp.getSharingType()),
          usersToNotify.toArray(new String[0]));
        break;
      case PARTIAL:
        /*
         * Determina e trata os usurios que perderam todo tipo de acesso.
         */
        Set<Object> oldUsers = new HashSet<Object>();
        oldUsers.addAll(oldUsersRO);
        oldUsers.addAll(oldUsersRW);
        oldUsers.removeAll(newUsersRO);
        oldUsers.removeAll(newUsersRW);
        usersLostAccessToProject(sp, oldUsers, true);
        /*
         * Determina e trata os usurios que no tinham acesso e agora tem.
         */
        Set<Object> newUsers = new HashSet<Object>();
        newUsers.addAll(newUsersRO);
        newUsers.addAll(newUsersRW);
        newUsers.removeAll(oldUsersRO);
        newUsers.removeAll(oldUsersRW);
        usersGainedAccessToProject(sp, newUsers, UsersNotification.INSERT);
        /*
         * Determina e trata os usurios cujo tipo de acesso foi modificado.
         */
        Set<Object> changedUsers = new HashSet<Object>();
        changedUsers.addAll(oldUsersRO);
        changedUsers.retainAll(newUsersRW);
        notifyUsers(UsersNotification.CHANGE, changedUsers, sp);
        changedUsers.clear();
        changedUsers.addAll(oldUsersRW);
        changedUsers.retainAll(newUsersRO);
        notifyUsers(UsersNotification.CHANGE, changedUsers, sp);
        break;
    }
    /*
     * Precisamos enviar o evento de alterao no compartilhamento de projeto
     * para a unio de todos os conjuntos de usurios: os que tinham acesso (RO
     * + RW) e os que ganharam acesso (RO + RW). As listas que so enviadas no
     * evento so as dos usurios que ainda tm acesso.
     */
    usersToNotify = new HashSet<Object>();
    usersToNotify.addAll(oldUsersRO);
    usersToNotify.addAll(newUsersRO);
    usersToNotify.addAll(oldUsersRW);
    usersToNotify.addAll(newUsersRW);
    usersToNotify.add(sp.getUserId());
    try {
      usersToNotify.addAll(User.getAdminIds());
    }
    catch (Exception e) {
      Server.logSevereMessage(
        "Erro ao obter a lista de usurios adminstradores.", e);
    }

    SharedProjectEvent event = new SharedProjectEvent(usersToNotify,
      sp.getInfo(), newUsersRO, newUsersRW, sp.getSharingType());
    notifyUsers(event, usersToNotify.toArray(new String[0]));
  }

  /**
   * Acesso ao projeto tornou-se privado. Precisamos remover o acesso a quem j
   * tinha, notificando-os.
   *
   * @param sp - projeto
   * @param oldSharingType - modo de compartilhamento anterior
   * @param oldUsersRO - usurios que possuam acesso RO anteriormente
   * @param oldUsersRW - usurios que possuam acesso RW anteriormente
   */
  private void prjAccessBecamePrivate(ServerProject sp,
    SharingType oldSharingType, Set<Object> oldUsersRO,
    Set<Object> oldUsersRW) {
    Set<Object> usersToNotify = null;
    switch (oldSharingType) {
      case PARTIAL:
        /*
         * O compartilhamento anterior era seletivo, precisamos registrar e
         * notificar todos os usurios que tinham acesso (RO ou RW).
         */
        usersLostAccessToProject(sp, oldUsersRO, true);
        usersLostAccessToProject(sp, oldUsersRW, true);
        break;
      case ALL_RO:
      case ALL_RW:
        /*
         * Projeto era pblico (RO ou RW), precisamos remov-lo da lista de
         * projetos pblicos e avisar a todos os usurios que eles perderam a
         * permisso de acesso
         */
        synchronized (publicProjects) {
          publicProjects.remove(
            new UserProjectInfo(sp.getId(), sp.getName(), sp.getUserId()));
        }
        usersToNotify = AdministrationService.getInstance().getAllUserIds();
        usersToNotify.remove(sp.getUserId());
        usersLostAccessToProject(sp, usersToNotify, true);
        break;
      case PRIVATE:
        /*
         * Nada a fazer.
         */
        break;
    }

    SharedProjectEvent event =
      new SharedProjectEvent(sp.getInfo(), sp.getSharingType());
    notifyProject(event, sp.getId());
  }

  /**
   * Notifica os usurios quando um projeto passou a ser pblico (ALL_RO ou
   * ALL_RW). Caso o projeto j estivesse compartilhado seletivamente, os
   * usurios que j possuam o novo tipo de acesso no so notificados. P.ex.
   * se o projeto tornou-se ALL_RO, os usurios que j tinham acesso RO no so
   * notificados.
   *
   * @param sp - projeto
   * @param oldSharingType - modo de compartilhamento anterior
   * @param oldUsersRO - usurios que possuam acesso RO
   * @param oldUsersRW - usurios que possuam acesso RW
   */
  private void prjAccessBecamePublic(ServerProject sp,
    SharingType oldSharingType, Set<Object> oldUsersRO,
    Set<Object> oldUsersRW) {
    synchronized (publicProjects) {
      publicProjects
        .add(new UserProjectInfo(sp.getId(), sp.getName(), sp.getUserId()));
    }
    SharingType sharingType = sp.getSharingType();
    Set<Object> users;
    switch (oldSharingType) {
      case PARTIAL:
        /*
         * Se o projeto tornou-se pblico, devemos remover os usurios das
         * listas e avisar aos usurios cujo tipo de acesso tenha sido alterado.
         */
        Set<Object> oldUsers = new HashSet<Object>();
        oldUsers.addAll(oldUsersRO);
        oldUsers.addAll(oldUsersRW);
        usersLostAccessToProject(sp, oldUsers, false);
        if (sharingType == SharingType.ALL_RO) {
          notifyUsers(UsersNotification.CHANGE, oldUsersRW, sp);
        }
        else {
          notifyUsers(UsersNotification.CHANGE, oldUsersRO, sp);
        }
        break;
      case PRIVATE:
        users = new HashSet<Object>(
          AdministrationService.getInstance().getAllUserIds());
        users.remove(sp.getUserId());
        /*
         * No chamamos usersGainedAccessToProject() porque no queremos inserir
         * este projeto na lista de projetos aos quais o usurio tem acesso
         * (projetos pblicos tm tratamento especial para isto).
         */
        notifyUsers(UsersNotification.INSERT, users, sp);
        break;
      case ALL_RO:
      case ALL_RW:
        users = new HashSet<Object>(
          AdministrationService.getInstance().getAllUserIds());
        users.remove(sp.getUserId());
        notifyUsers(UsersNotification.CHANGE, users, sp);
        break;
    }
    SharedProjectEvent event =
      new SharedProjectEvent(sp.getInfo(), sharingType);
    notifyProject(event, sp.getId());
  }

  /**
   * Cria um <b>novo</b> conjunto de usurios igual ao original, e remove deste
   * o usurio especificado.
   *
   * @param userID - identificador do usurio
   * @param users - conjunto de usurios de onde o usurio ser removido
   * @return <b>novo</b> conjunto de usurios igual ao original, de onde o
   *         usurio foi removido
   */
  private Set<Object> removeUserFrom(Object userID, Set<Object> users) {
    HashSet<Object> newUsers = new HashSet<Object>(users);
    newUsers.remove(userID);
    return newUsers;
  }

  /**
   * Cria um {@link CommonClientProject} a partir de um {@link ServerProject}.
   * Os filhos <b>no</b> so recuperados.
   *
   * @param sp representao do projeto no servidor
   * @return representao do projeto no cliente
   */
  private CommonClientProject buildClientProject(ServerProject sp) {
    CommonProjectInfo info = new CommonProjectInfo();
    info.userId = sp.getUserId();
    info.name = sp.getName();
    info.description = sp.getDescription();
    info.setAttributes(sp.getAttributes());
    ServerProjectFile serverTree = sp.getTree();
    ClientProjectFile clientTree = buildSingleClientProjectFile(serverTree);
    CommonClientProject comProj =
      new CommonClientProject(sp.getId(), info, clientTree, sp.getPath());
    comProj.setServerCanWriteProject(isProjectWritableFromServer(sp));
    return comProj;
  }

  /**
   * Cria um {@link ClientProjectFile} com seus respectivos filhos a partir de
   * um {@link ServerProjectFile}.
   * <p>
   * <b>IMPORTANTE: Todos os filhos so obtidos recursivamente.</b>
   *
   * @param spf <code>ServerProjectFile</code>
   * @return <code>ClientProjectFile</code>. Se  um diretrio, toda a sua
   *         subrvore foi preenchida.
   *
   * @see #buildSingleClientProjectFile(ServerProjectFile)
   */
  ClientProjectFile buildClientProjectSubtree(ServerProjectFile spf) {
    ClientProjectFile clientFile = buildSingleClientProjectFile(spf);
    if (!spf.isDirectory()) {
      return clientFile;
    }
    ServerProjectFile[] serverChildren = spf.getChildren();
    if (serverChildren != null) {
      ClientProjectFile[] clientChildren = null;
      clientChildren = new ClientProjectFile[serverChildren.length];
      for (int j = 0; j < serverChildren.length; j++) {
        clientChildren[j] = buildClientProjectSubtree(serverChildren[j]);
      }
      clientFile.setChildren(false, false, clientChildren);
    }
    return clientFile;
  }

  /**
   * Criao de client project file. Apenas o n  criado, seus filhos
   * <b>no</b> so recuperados.
   *
   * @param spf arquivo servidor
   * @return o arquivo
   */
  ClientProjectFile buildSingleClientProjectFile(ServerProjectFile spf) {
    String updateUserLogin = null;
    long updateInterval = 0;
    Object key = Service.getKey();
    if (key != null) {
      ProjectService prjSrv = ProjectService.getInstance();
      if (prjSrv != null) {
        UpdatableFileLocation fileLocation =
          prjSrv.updateScheduler.getFileLocation(spf);
        FileUpdateInfo info =
          prjSrv.updateScheduler.getFileUpdateInfo(fileLocation);
        if (info != null) {
          updateUserLogin = info.getUserLogin();
          updateInterval = info.getInterval();
        }
      }
    }
    ClientProjectFile clientFile =
      new ClientProjectFile(spf.getProjectId(), spf.getName(), spf.getPath(),
        null, spf.getType(), spf.isDirectory(), spf.hasChildren(),
        spf.isUnderConstruction(), spf.whoCreated(), spf.getCreationDate(),
        spf.size(), spf.getModificationDate(), spf.isLocked(),
        (spf.getUpdatableFileInfo() != null), updateUserLogin, updateInterval);
    return clientFile;
  }

  /**
   * L o arquivo de configurao de um projeto e registra as informaes de
   * compartlhamento, se for o caso. Os projetos pblicos so registrados em
   * {@link #publicProjects}, e os compartilhados em {@link #sharedProjects}.
   *
   * @param prjConfigFile arquivo de configurao do projeto
   * @param ownerId Dono do projeto.
   */
  private void registerIfSharedProject(final File prjConfigFile,
    final Object ownerId) {
    if (!prjConfigFile.exists()) {
      return;
    }
    if (!prjConfigFile.isFile()) {
      String path = prjConfigFile.getAbsolutePath();
      String fmt = "Dado de controle de projeto no  arquivo: %s";
      String err = String.format(fmt, path);
      Server.logSevereMessage(err);
      return;
    }
    /*
     * l a configurao do projeto
     */
    CommonProjectInfo info = null;

    info = ServerProject.readProjectInfoFromConfigFile(prjConfigFile, ownerId);
    if (info == null) {
      return;
    }
    Object id = ServerProject.getId(ownerId, info.name);
    UserProjectInfo userProj = new UserProjectInfo(id, info.name, info.userId);
    if (ProjectPermissions.isPublic(info)) {
      /*
       * projeto  pblico
       */
      synchronized (publicProjects) {
        publicProjects.add(userProj);
      }
    }
    else if (ProjectPermissions.getSharingType(info) == SharingType.PARTIAL) {
      /*
       * projeto  compartilhado, obtm todos os usurios que tm acesso ao
       * projeto
       */
      Set<Object> u = ProjectPermissions.getAllUsers(info);
      for (Object user : u) {
        associateUserToSharedProject(user, userProj);
      }
    }
  }

  /**
   * Adiciona um projeto  lista de projetos compartilhados com um determinado
   * usurio.
   *
   * @param user usurio
   * @param prjInfo informaes do projeto
   */
  private void associateUserToSharedProject(Object user,
    UserProjectInfo prjInfo) {
    synchronized (sharedProjects) {
      Vector<UserProjectInfo> prjsSharedWithUser = sharedProjects.get(user);
      if (prjsSharedWithUser == null) {
        /*
         * usurio ainda no est associado a nenhum projeto
         */
        prjsSharedWithUser = new Vector<UserProjectInfo>();
        sharedProjects.put(user, prjsSharedWithUser);
      }
      /*
       * a rigor, permitimos duplicidade por usarmos uma lista ao invs de um
       * conjunto... mas, por construo, no deve acontecer (caso contrrio,
       * deveramos verificar se j no existe antes de adicionar)
       */
      prjsSharedWithUser.add(prjInfo);
    }
  }

  /**
   * Verifica se um login pertence a um usurio
   *
   * @param login o login do usurio.
   * @return id do usurio.
   */
  private Object getUserIdByLogin(String login) {
    try {
      User user = User.getUserByLogin(login);
      if (user == null) {
        return null;
      }
      return user.getId();
    }
    catch (Exception e) {
      return null;
    }
  }

  /**
   * Concatena texto de erro.
   *
   * @param sourceString texto original
   * @param err mensagem de erro
   * @return nova string./
   */
  private String addErrorStringItem(String sourceString, String err) {
    String destString = sourceString + "  # " + err + "\n\n";
    return destString;
  }

  /**
   * Faz check do repositrio de projetos enviando email para o administrador
   * com o estado geral do repositrio e suas correes.
   *
   * @throws ServerException em caso de falha na checagem do repositrio.
   */
  private void checkRepository() throws ServerException {
    if (!prjDir.exists()) {
      String fmt = "Repositrio de projetos no encontrado: %s!";
      String err = String.format(fmt, prjDir.getAbsolutePath());
      try {
        System.out.println(MessageFormat.format(
          "\nVerifique a propriedade ProjectService.base.project.dir que define o diretrio do repositrio de projetos, pois o diretrio {0} no existe.\n",
          prjDir.getCanonicalPath()));
      }
      catch (IOException e) {
      }
      Server.logSevereMessage(err);
      throw new ServerException(err);
    }
    if (!prjDir.isDirectory()) {
      String fmt = "Repositrio de projetos no  diretrio: %s!";
      String err = String.format(fmt, prjDir.getAbsolutePath());
      Server.logSevereMessage(err);
      throw new ServerException(err);
    }

    // Buscando servio de email.
    final MailService mailService = MailService.getInstance();

    // Texto com contedo de e-mail para admin quando do aparecimento de
    // situaes no ortodoxas no start-up dos projetos.
    String adminMailContent = "";

    // Buscando arquivos soltos na rea de diretrios de usurios
    // no repositrio de projetos. Apenas alertando o administrador...
    // O filtro j exclui o arquivo de controle de alocao
    File[] dummyFiles =
      prjDir.listFiles(getUnknownFileInsideRepositoryFilter());
    for (File dummyFile : dummyFiles) {
      String fmt = "Ignorando arquivo invlido %s na rea de projetos (%s).";
      String dummyName = dummyFile.getName();
      String dummyPath = dummyFile.getAbsolutePath();
      String err = String.format(fmt, dummyName, dummyPath);
      Server.logSevereMessage(err);
      adminMailContent = addErrorStringItem(adminMailContent, err);
    }

    // Todos os diretrios da raiz so logins de usurios.
    // Arquivos presentes nessa rea sero descartados.
    File[] usersDirs = prjDir.listFiles(getDirectoryFilter());
    for (File userDir : usersDirs) {
      String dirPath = userDir.getAbsolutePath();
      String dirName = userDir.getName();
      String ownerLogin = dirName;
      Object ownerId = getUserIdByLogin(ownerLogin);
      if (ownerId == null) {
        String fmt = "Ignorando usurio inexistente (diretrio) %s ";
        fmt = fmt + "na rea de projetos (%s).";
        String err = String.format(fmt, ownerLogin, dirPath);
        Server.logWarningMessage(err);
        adminMailContent = addErrorStringItem(adminMailContent, err);
        continue;
      }

      // Buscando arquivos soltos na rea de diretrios de projetos de usurio
      // dentro do repositrio de projetos. Apenas alertando o administrador...
      File[] trashFiles =
        userDir.listFiles(getUnknownFilesInsideUserAreaFilter());
      if (trashFiles == null) {
        String fmt =
          "Erro ao recuperar os arquivos na rea de projetos para usurio %s (%s).";
        String err = String.format(fmt, ownerLogin, userDir);
        Server.logSevereMessage(err);
        adminMailContent = addErrorStringItem(adminMailContent, err);
      }
      else {
        for (File trashFile : trashFiles) {
          String fmt = "Ignorando arquivo invlido %s na rea de projetos ";
          fmt = fmt + "do usurio %s (%s)";
          String trashName = trashFile.getName();
          String trashPath = trashFile.getAbsolutePath();
          String err = String.format(fmt, trashName, ownerLogin, trashPath);
          Server.logWarningMessage(err);
          adminMailContent = addErrorStringItem(adminMailContent, err);
        }
      }
      // Texto com contedo de e-mail para admin, se ocorrer problemas
      // no start-up dos projetos.
      String ownerMailContent = "";

      // No diretorio dos usurios esto os projetos e suas configuraes.
      File[] projectsDirs = userDir.listFiles(getDirectoryFilter());
      if ((projectsDirs == null) || (projectsDirs.length == 0)) {
        continue;
      }

      for (File projectDir : projectsDirs) {
        String projectName = projectDir.getName();
        if (!ServerProject.isValidBaseDirectory(projectDir)) {
          String prjDirPath = projectDir.getAbsolutePath();
          String fmt = "Ignorando diretrio oculto %s do usurio %s (%s).";
          String err = String.format(fmt, projectName, ownerLogin, prjDirPath);
          Server.logSevereMessage(err);
          adminMailContent = addErrorStringItem(adminMailContent, err);
          continue;
        }
        File configFile = ServerProject.getConfigFile(projectDir);
        CommonProjectInfo cpi = ServerProject.createProjectInfo(configFile);
        if (cpi == null) {
          boolean ok =
            ServerProject.generateConfigFile(configFile, ownerId, projectName);
          String status = "recuperado (acesso reconfigurado para \"privado\")";
          if (!ok) {
            status = "*NO* recuperado aps tentativa de restaurao";
          }
          cpi = ServerProject.createProjectInfo(configFile);
          if (cpi == null) {
            status = " *NO recuperado aps sucesso de restaurao";
          }
          String fmt = "Projeto %s do usurio %s %s.";
          String err = String.format(fmt, projectName, ownerLogin, status);
          Server.logSevereMessage(err);
          adminMailContent = addErrorStringItem(adminMailContent, err);
          ownerMailContent = addErrorStringItem(ownerMailContent, err);
        }
      }

      // Envio de email com resumo geral dos projetos do usurio (owner)
      if (!ownerMailContent.trim().isEmpty()) {
        Object[] addresse = new Object[] { ownerId };
        mailService.mailSomeUsersFromService(this, addresse, ownerMailContent);
      }
    }

    // Envio de email com resumo geral de todos projetos ao admin.
    if (!adminMailContent.trim().isEmpty()) {
      Vector<Object> addresse = new Vector<Object>();
      try {
        addresse.addAll(User.getAdminIds());
      }
      catch (Exception e) {
        Server.logSevereMessage(
          "Erro ao obter a lista de usurios adminstradores.", e);
      }
      mailService.mailSomeUsersFromService(this,
        addresse.toArray(new Object[0]), adminMailContent);
    }
  }

  /**
   * Carrega os projetos do sistema, preenchendo o conjunto de projetos
   * pblicos. Alm disso, carrega o mapa de usurios e projetos.
   */
  private void loadProjects() {
    publicProjects =
      Collections.synchronizedSet(new HashSet<UserProjectInfo>());
    sharedProjects = new Hashtable<Object, Vector<UserProjectInfo>>();

    // Testando a existncia do diretrio de projetos
    File rootDir = new File(projectRepositoryPath);
    if (!rootDir.exists() || !rootDir.isDirectory()) {
      return;
    }

    // Todos os diretrios da raiz so logins de usurios.
    // Arquivos presentes nessa rea sero descartados.
    File[] usersDirs = rootDir.listFiles(getDirectoryFilter());
    for (File userDir : usersDirs) {
      String dirName = userDir.getName();
      String ownerLogin = dirName;
      Object ownerId = getUserIdByLogin(ownerLogin);
      if (ownerId == null) {
        continue;
      }

      /* No diretorio dos usurios esto os projetos e suas configuraes */
      File[] projectsDirs = userDir.listFiles(getDirectoryFilter());
      if ((projectsDirs == null) || (projectsDirs.length == 0)) {
        continue;
      }

      for (int j = 0; j < projectsDirs.length; j++) {
        File projectDir = projectsDirs[j];
        String projectName = projectDir.getName();
        if (!ServerProject.isValidBaseDirectory(projectDir)) {
          continue;
        }

        // Para registrar o projeto como compartilhado, basta termos
        // o configFile. Mas, como instrumentao, estamos criando
        // um objeto info para o caso de haver corrupo no arquivo nesse
        // meio tempo. Se isso ocorrer e no tivermos como reconstruir,
        // o projeto no ser registrado.
        File configFile = ServerProject.getConfigFile(projectDir);
        CommonProjectInfo cpi =
          ServerProject.readProjectInfoFromConfigFile(configFile, ownerId);
        if (cpi != null) {
          registerIfSharedProject(configFile, ownerId);
        }
        else {
          String fmt = "Falha na carga do projeto %s do usurio %s";
          String err = String.format(fmt, projectName, ownerLogin);
          Server.logSevereMessage(err);
        }
      }
    }
  }

  /**
   * Constroi um filtro de diretrios
   *
   * @return o filtro
   */
  private FileFilter getDirectoryFilter() {
    return new FileFilter() {
      @Override
      public boolean accept(File file) {
        return file.isDirectory();
      }
    };
  }

  /**
   * Constroi um filtro de arquivos presentes na rea de usurios no repositrio
   * de projetos, desconsiderando o arquivo de controle de alocao
   * {@link ProjectAdministrator#LOCKED_PROJECTS_FILE_NAME}.
   *
   * @return o filtro
   */
  private FileFilter getUnknownFileInsideRepositoryFilter() {
    return new FileFilter() {
      @Override
      public boolean accept(File file) {
        String lockingFileName = ProjectAdministrator.LOCKED_PROJECTS_FILE_NAME;
        String fileName = file.getName();
        return file.isFile() && !fileName.equals(lockingFileName);
      }
    };
  }

  /**
   * Constroi um filtro de arquivos simples.
   *
   * @return o filtro
   */
  private FileFilter getUnknownFilesInsideUserAreaFilter() {
    return new FileFilter() {
      @Override
      public boolean accept(File file) {
        String fileName = file.getName();
        boolean isCtrl = fileName.endsWith(ServerProject.INFO_EXTENSION);
        boolean isDesc = fileName.endsWith(FileConfig.DESCRIPTION_EXTENSION);
        return file.isFile() && !isCtrl && !isDesc;
      }
    };
  }

  /**
   * Notifica o administrador sobre um evento do servio de projetos.
   *
   * @param info A informao sobre o evento ocorrido.
   * @param mailMessage .
   */
  private void notifyAdmin(ProjectAdminInfo info, String mailMessage) {
    try {
      List<Object> adminUsers = new ArrayList<Object>();
      List<User> allUsers = User.getAllUsers();
      for (int i = 0; i < allUsers.size(); i++) {
        User user = allUsers.get(i);
        if (checkAdminPermission(user)) {
          Object uid = user.getId();
          adminUsers.add(uid);
          if (mailMessage != null) {
            final MailService mSrv = MailService.getInstance();
            mSrv.mailUserFromService(this, uid, mailMessage);
          }
        }
      }
      final Notification notification =
        new ProjectNotification(getSenderName(), info);
      MessageService.getInstance().send(new Message(notification),
        adminUsers.toArray(new String[0]));
    }
    catch (RemoteException e) {
      Server.logSevereMessage("ProjectService:notifyAdmin", e);
    }
  }

  /**
   * Envia notificaes para os usurios quando estes ganham ou perdem acesso a
   * um projeto. Isto acontece nas seguintes situaes:
   * <ul>
   * <li>projeto do qual o usurio participava foi removido
   * <li>projeto tornou-se pblico
   * <li>usurio ganhou acesso ao projeto
   * <li>usurio perdeu acesso ao projeto
   * </ul>
   *
   * @param eventType - tipo do evento (ver constantes de
   *        {@link UsersNotification})
   * @param usersToNotify - usurios a serem notificados
   * @param project - projeto para o qual se ganhou ou perdeu acesso
   */
  private void notifyUsers(int eventType, Collection<Object> usersToNotify,
    ServerProject project) {
    if (usersToNotify.isEmpty()) {
      return;
    }

    /*
     * O administrador nunca perde/ganha acesso a projetos, portanto no h
     * porque notific-lo.
     */
    //TODO Remover todos os usurios admins
    usersToNotify.remove(User.getAdminId());

    final Notification notification = new UsersNotification(getSenderName(),
      eventType, usersToNotify, project.getUserId(), project.getName());

    try {
      MessageService.getInstance().send(new Message(notification),
        usersToNotify.toArray(new String[0]));
    }
    catch (RemoteException e) {
      Server
        .logSevereMessage("Erro ao enviar notificao para usurios do projeto "
          + project.getName() + ".", e);
    }
  }

  /**
   * Envia notificao aos usurios com acesso ao projeto.
   *
   * @param event evento que deve ser notificado
   * @param projectId identificador do projeto
   */
  private void notifyProject(RemoteEvent event, Object projectId) {
    try {
      String[] users = getUserToNotify(projectId);
      MessageService.getInstance().send(new Message(event), users);
    }
    catch (RemoteException e) {
      Server
        .logSevereMessage("Erro ao enviar notificao para usurios do projeto "
          + projectId + ".", e);
    }
  }

  /**
   * Envia notificao a usurios.
   *
   * @param event evento que deve ser notificado
   * @param users os usurios
   */
  private void notifyUsers(RemoteEvent event, String[] users) {
    try {
      MessageService.getInstance().send(new Message(event), users);
    }
    catch (RemoteException e) {
      Server.logSevereMessage("Erro ao enviar notificao para usurios.", e);
    }
  }

  /**
   * Envia um evento administrativo.
   *
   * @param event evento que deve ser notificado
   * @param project identificador do projeto
   */
  private void sendAdmistrationEvent(AdministrationEvent event,
    ServerProject project) {
    Set<Object> destinations = new HashSet<Object>();

    try {
      destinations.addAll(User.getAdminIds());
    }
    catch (Exception e) {
      Server.logSevereMessage(
        "Erro ao obter a lista de usurios adminstradores.", e);
    }
    destinations.add(project.getUserId());
    try {
      MessageService.getInstance().send(new Message(event),
        destinations.toArray(new String[0]));
    }
    catch (RemoteException e) {
      Server.logSevereMessage("Erro ao enviar notificao para usurios.", e);
    }
  }

  /**
   * Atualiza a estrutura em memria dos projetos de que cada usurio participa,
   * removendo alguns usurios de um projeto especfico. Envia notificaes para
   * cada usurio que foi removido.
   *
   * @param sp Projeto do qual os usurios foram removidos.
   * @param usersRemoved Identificadores dos usurios removidos do projeto.
   * @param notify Indica se os usurios devem ser notificados.
   */
  private void usersLostAccessToProject(ServerProject sp,
    Set<Object> usersRemoved, boolean notify) {
    if (usersRemoved == null || usersRemoved.isEmpty()) {
      return;
    }
    UserProjectInfo prjInfo =
      new UserProjectInfo(sp.getId(), sp.getName(), sp.getUserId());
    synchronized (sharedProjects) {
      /*
       * Para cada usurio, verificamos se ele estava associado ao projeto em
       * questo; se estava, removemos a associao, e se esta era sua nica
       * associao, removemos o usurio da tabela de associaes.
       */
      for (Object uid : usersRemoved) {
        Vector<UserProjectInfo> prjsSharedWithUser = sharedProjects.get(uid);
        if (prjsSharedWithUser != null) {
          if (prjsSharedWithUser.remove(prjInfo)) {
            if (prjsSharedWithUser.isEmpty()) {
              sharedProjects.remove(uid);
            }
          }
        }
      }
    }
    if (notify) {
      notifyUsers(UsersNotification.REMOVE, usersRemoved, sp);
    }

    SharedProjectEvent event =
      new SharedProjectEvent(sp.getInfo(), sp.getSharingType());
    notifyUsers(event, usersRemoved.toArray(new String[0]));
  }

  /**
   * Atualiza a estrutura em memria dos projetos de que cada usurio participa,
   * adicionando alguns usurios a um projeto especfico. Envia notificaes
   * para cada usurio que foi adicionado.
   *
   * @param sp Projeto do qual os usurios foram removidos.
   * @param usersAdded Identificadores dos usurios adicionados ao projeto.
   * @param event O tipo de evento a ser enviado na notificao aos usurios. Se
   *        o valor fornecido for -1, nenhuma notificao  enviada.
   */
  private void usersGainedAccessToProject(ServerProject sp,
    Set<Object> usersAdded, int event) {
    if (usersAdded == null || usersAdded.isEmpty()) {
      return;
    }
    UserProjectInfo prjInfo =
      new UserProjectInfo(sp.getId(), sp.getName(), sp.getUserId());
    /*
     * Para cada usurio, adicionamos o projeto em questo  sua lista de
     * acessos a projetos.
     */
    for (Object uid : usersAdded) {
      associateUserToSharedProject(uid, prjInfo);
    }
    if (event != -1) {
      notifyUsers(event, usersAdded, sp);
    }
  }

  /**
   * Remove os usurios de um projeto.
   *
   * @param sp Identificador do dono do projeto.
   * @return Vetor de identificadores dos usurios que foram removidos do
   *         projeto.
   */
  private Set<Object> removeUsers(ServerProject sp) {
    if (sp.isPublic()) {
      Set<Object> allUsers =
        AdministrationService.getInstance().getAllUserIds();
      allUsers.remove(sp.getUserId());
      synchronized (publicProjects) {
        publicProjects.remove(
          new UserProjectInfo(sp.getId(), sp.getName(), sp.getUserId()));
      }
      return allUsers;
    }
    Set<Object> users = new HashSet<Object>();
    synchronized (sharedProjects) {
      for (Enumeration<Object> e = sharedProjects.keys(); e
        .hasMoreElements();) {
        Object user = e.nextElement();
        Vector<UserProjectInfo> prjsSharedWithUser = sharedProjects.get(user);
        UserProjectInfo info =
          new UserProjectInfo(sp.getId(), sp.getName(), sp.getUserId());
        boolean removed = prjsSharedWithUser.remove(info);
        if (prjsSharedWithUser.isEmpty()) {
          sharedProjects.remove(user);
        }
        if (removed) {
          users.add(user);
        }
      }
    }
    if (users.size() <= 0) {
      return null;
    }
    return users;
  }

  /**
   * Carrega na lista {@code projectsId} o identificador dos projetos
   * pertencentes ao usurio {@code ownerId}.
   *
   * @param ownerId identificador nico do usurio.
   * @param projectsId lista que receber a carga de identificadores nicos dos
   *        projetos pertencentes ao usurio {@code ownerId}.
   */
  private void loadAllProjectsId(Object ownerId,
    Collection<Object> projectsId) {
    List<UserProjectInfo> projects = getProjectsFromUser(ownerId);
    if (projects != null) {
      for (UserProjectInfo project : projects) {
        projectsId.add(project.getProjectId());
      }
    }
  }

  /**
   * Define as informaes sobre a atualizao de um arquivo.
   *
   * @param file O arquivo.
   * @param updatableFileInfo As informaes sobre a atualizao do arquivo.
   *
   * @return <code>true</code>, caso o arquivo tenha sido atualizado, ou
   *         <code>false</code>, caso contrrio.
   */
  public boolean setUpdatableFileInfo(ClientProjectFile file,
    UpdatableFileInfo updatableFileInfo) {
    Object projectId = file.getProjectId();
    this.checkWritePermission(projectId);
    ServerProjectFile spf =
      ServerProjectFile.findFile(projectId, file.getPath());
    if (spf == null) {
      return false;
    }
    spf.setUpdatableFileInfo(updatableFileInfo);
    return true;
  }

  /**
   * Obtm o UpdatableFileInfo de um arquivo.
   *
   * @param projectId o identificador do projeto
   * @param path o caminho para o arquivo
   *
   * @return o UpdatableFileInfo
   */
  public UpdatableFileInfo getUpdatableFileInfo(Object projectId,
    String[] path) {
    this.checkReadPermission(projectId);
    ServerProjectFile spf = ServerProjectFile.findFile(projectId, path);
    return spf.getUpdatableFileInfo();
  }

  /**
   * Cria a informao de um canal de leitura e escrita (a ser determinado no
   * momento da sua abertura) para o arquivo indicado.
   *
   * @param file O arquivo para o qual a informao do canal ser criada.
   * @return A informao de canal para o arquivo fornecido.
   * @throws Exception
   */
  public RemoteFileChannelInfo createFileChannelInfo(ClientProjectFile file)
    throws Exception {
    Object projectId = file.getProjectId();
    this.checkReadPermission(projectId);
    ServerProjectFile spf =
      ServerProjectFile.findFile(projectId, file.getPath());
    Object userId = Service.getUser().getId();
    ServerProject sp = ServerProject.openProject(projectId, false);
    boolean isReadOnly = (isProjectRO(sp) && !hasROProjectWritePermission(sp))
      || !sp.userHasAccessRW(userId);
    return spf.openChannel(isReadOnly);
  }

  /**
   * Cria um filtro para buscas por arquivos nos projetos.
   *
   * @param text texto a ser usado na comparao
   * @param caseInsensitive <code>true</code> se no devemos diferenciar
   *        maisculas de minsculas
   * @param isRegex <code>true</code> se o texto  uma regex
   * @return filtro que combina os parmetros acima
   */
  private FileFilter createFileFinderFilter(String text,
    boolean caseInsensitive, boolean isRegex) {
    if (text.equals("*")) {
      // retornar todos os arquivos
      return null;
    }
    if (!caseInsensitive) {
      if (text.lastIndexOf('*') == 0) {
        // *text -> qualquer arquivo que termina com o texto informado
        return FileFinder.suffixFilter(text.substring(1));
      }
      else if (text.indexOf('*') == text.length()) {
        // text* -> qualquer arquivo que comea com o texto informado
        return FileFinder.prefixFilter(text.substring(0, text.length() - 1));
      }
      else if (Pattern.matches("\\*.+?\\*", text)) {
        // busca por "*xxx*"
        return FileFinder.namePartFilter(text.substring(1, text.length() - 1));
      }
      else if (text.contains("*")) {
        /*
         * busca possui '*', transformamos em regex (escape do '.' e converso
         * de '*' para ".*")
         */
        String regex = text.replace(".", "\\.").replace("*", ".*");
        return FileFinder.regexFilter(regex);
      }
      else if (isRegex) {
        // text  uma expresso regular
        return FileFinder.regexFilter(text);
      }
      else {
        // nenhum '*', nome do arquivo deve ser idntico ao texto informado
        return FileFinder.nameFilter(text);
      }
    }
    else {
      if (text.lastIndexOf('*') == 0) {
        // *text -> qualquer arquivo que termina com o texto informado
        return FileFinder.suffixIgnoreCaseFilter(text.substring(1));
      }
      else if (text.indexOf('*') == text.length()) {
        // text* -> qualquer arquivo que comea com o texto informado
        return FileFinder
          .prefixIgnoreCaseFilter(text.substring(0, text.length() - 1));
      }
      else if (Pattern.matches("\\*.+?\\*", text)) {
        // busca por "*xxx*"
        return FileFinder
          .namePartIgnoreCaseFilter(text.substring(1, text.length() - 1));
      }
      else if (text.contains("*")) {
        /*
         * busca possui '*', transformamos em regex (escape do '.' e converso
         * de '*' para ".*")
         */
        String regex = text.replace(".", "\\.").replace("*", ".*");
        Pattern pattern = Pattern.compile(regex, Pattern.CASE_INSENSITIVE);
        return FileFinder.regexFilter(pattern);
      }
      else if (isRegex) {
        return FileFinder
          .regexFilter(Pattern.compile(text, Pattern.CASE_INSENSITIVE));
      }
      else {
        // nenhum '*', nome do arquivo deve ser idntico ao texto informado
        return FileFinder.nameIgnoreCaseFilter(text);
      }
    }
  }

  /**
   * Obtm o path-pai de um determinado path (i.e. o caminho at o
   * arquivo/diretrio, sem inclui-lo).
   *
   * @param path caminho at o arquivo/diretrio
   * @return caminho at o arquivo/diretrio, sem inclui-lo, ou
   *         <code>null</code> caso o path seja vazio
   */
  public static String[] getParentPath(String[] path) {
    if (path.length == 0) {
      return null;
    }
    return Arrays.copyOf(path, path.length - 1);
  }

  /**
   * Obtm o nome de um arquivo/diretrio a partir do seu path.
   *
   * @param path path para o arquivo
   * @return o nome do arquivo (ltimo componente do path), ou <code>null</code>
   *         se o path  vazio
   */
  public static String getFileName(String[] path) {
    if (path.length == 0) {
      return null;
    }
    return path[path.length - 1];
  }

  /**
   * Obtm a propriedade SourceId
   *
   * @return a propriedade SourceId
   */
  public String getSourceId() {
    return sourceId;
  }

  /**
   * Obtem o tipo a partir da extensao do nome do arquivo.
   *
   * @param name Nome do arquivo
   * @param isDirectory Indica se o tipo deve aceitar diretrios ({@code true})
   *        ou arquivos {@code false}).
   *
   * @return Tipo do arquivo que possui a extensao ou "UNKNOWN" caso nenhum
   *         arquivo tenha este tipo.
   */
  public ProjectFileTypeInfo getTypeFromExtension(String name,
    boolean isDirectory) {
    String extension = FilenameUtils.getExtension(name);
    ProjectFileTypeInfo info =
      typeRepository.findInfoByExtension(getDefaultLocale(), extension);
    if (info.getCode().equals(ProjectFileType.UNKNOWN) && isDirectory) {
      info = typeRepository.getInfo(getDefaultLocale(),
        ProjectFileType.DIRECTORY_TYPE);
    }
    return info;
  }

  /**
   *
   * {@inheritDoc}
   */
  @Override
  public ClientOptimizationMode getOptimizationMode() throws RemoteException {
    return clientOptimizationMode;
  }

  /**
   *
   * {@inheritDoc}
   */
  @Override
  public String getLocalProjectPath() throws RemoteException {
    return clientLocalProjectPath;
  }

  /**
   * Obtm os templates de projeto.
   *
   * @return a lista de templates.
   */
  List<ProjectTemplate> getProjectTemplate() {
    return projectTemplates;
  }
}
