/*
 * $Id: TransactionManager.java 109447 2010-08-24 19:45:16Z clinio $
 */

package csbase.server;

import csbase.remote.TransactionCallbackInterface;
import csbase.remote.TransactionInterface;

/**
 * Esta classe tem por objetivo controlar a sincronizao entre servidores
 * locais e o servidor central, evitando que sejam feitas alteraes ao
 * repositrio enquanto a sincronizao estiver em progresso.
 */
public class TransactionManager implements TransactionInterface {
  /**
   * Tempo entre verificaes da validade do lock.
   */
  private static final long CHECK_LOCK_SLEEP_TIME = 5000; // 5 sec

  /**
   * Tempo de espera entre tantativas de estabelecer um lock.
   */
  private static final long TRY_LOCK_SLEEP_TIME = 100;

  /**
   * Nmero mximo de tantativas de estabelecer um lock.
   */
  private static final long MAX_LOCK_TRIES = 100;

  /**
   * Thread de verificao da validade do lock.
   */
  private LockValidation checkAlive;

  /**
   * Objeto remoto de validao do lock.
   */
  private TransactionCallbackInterface cb = null;

  /**
   * Validao de lck
   * 
   * @author Tecgraf/PUC-Rio
   */
  private class LockValidation extends Thread {
    /**
     * {@inheritDoc}
     */
    @Override
    public void run() {
      while (!interrupted()) {
        try {
          sleep(CHECK_LOCK_SLEEP_TIME);
        }
        catch (InterruptedException e) {
          break;
        }
        synchronized (TransactionManager.this) {
          if (cb == null) {
            break;
          }
          boolean alive = false;
          try {
            alive = cb.isAlive();
          }
          catch (Exception e) {
            alive = false;
          }
          if (!alive) {
            cb = null;
            break;
          }
        }
      }
      synchronized (TransactionManager.this) {
        TransactionManager.this.notifyAll();
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized boolean lock(TransactionCallbackInterface cb) {
    int numTries = 0;
    while (isLocked() && numTries++ < MAX_LOCK_TRIES) {
      try {
        wait(TRY_LOCK_SLEEP_TIME);
      }
      catch (InterruptedException e) {
      }
    }
    if (isLocked())
      return false;
    this.cb = cb;
    checkAlive = new LockValidation();
    checkAlive.start();
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized boolean isLocked() {
    return cb != null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized void unlock(TransactionCallbackInterface cb) {
    if (this.cb == null || !this.cb.equals(cb)) {
      throw new Error("TransactionManager:unlock(" + this.cb + "->" + cb + ")");
    }
    this.cb = null;
    checkAlive.interrupt();
    checkAlive = null;
  }
}
