package csbase.server.services.opendreamsservice.opendreams.v1_7;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.Serializable;
import java.rmi.RemoteException;
import java.text.MessageFormat;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Map;
import java.util.Set;
import java.util.Vector;

import csbase.exception.PermissionException;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommandFinalizationType;
import csbase.logic.CommandInfo;
import csbase.logic.CommandNotification;
import csbase.logic.CommandStatus;
import csbase.logic.CommandSubmission;
import csbase.logic.CommonClientProject;
import csbase.logic.Priority;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.parameters.FileURLValue;
import csbase.server.Server;
import csbase.server.Service;
import csbase.server.services.commandpersistenceservice.CommandPersistenceService;
import csbase.server.services.messageservice.MessageService;
import csbase.server.services.openbusservice.OpenBusService;
import csbase.server.services.projectservice.ProjectService;
import csbase.server.services.schedulerservice.SchedulerService;
import csbase.server.services.sgaservice.SGAService;
import csbase.util.messages.IMessageListener;
import csbase.util.messages.Message;
import csbase.util.messages.filters.BodyTypeFilter;
import tecgraf.javautils.core.io.FileUtils;
import tecgraf.openbus.DRMAA.v1_7.AlreadyActiveSessionException;
import tecgraf.openbus.DRMAA.v1_7.AuthorizationException;
import tecgraf.openbus.DRMAA.v1_7.DeniedByDrmException;
import tecgraf.openbus.DRMAA.v1_7.DrmCommunicationException;
import tecgraf.openbus.DRMAA.v1_7.ExitTimeoutException;
import tecgraf.openbus.DRMAA.v1_7.InternalException;
import tecgraf.openbus.DRMAA.v1_7.InvalidArgumentException;
import tecgraf.openbus.DRMAA.v1_7.InvalidContactStringException;
import tecgraf.openbus.DRMAA.v1_7.InvalidJobException;
import tecgraf.openbus.DRMAA.v1_7.InvalidJobTemplateException;
import tecgraf.openbus.DRMAA.v1_7.JobControlAction;
import tecgraf.openbus.DRMAA.v1_7.JobInfo;
import tecgraf.openbus.DRMAA.v1_7.JobState;
import tecgraf.openbus.DRMAA.v1_7.JobTemplate;
import tecgraf.openbus.DRMAA.v1_7.NoActiveSessionException;
import tecgraf.openbus.DRMAA.v1_7.Version;
import tecgraf.openbus.DRMAA.v1_7.JobTemplatePackage.PARAMETRIC_INDEX;
import tecgraf.openbus.opendreams.v1_7.OpenDreamsJobTemplate;
import tecgraf.openbus.opendreams.v1_7.OpenDreamsJobTemplateImpl;

/**
 * Implementa uma sesso no OpenDreams. Guarda o projeto ao qual essa sesso
 * est associada.
 *
 * @author Tecgraf PUC-Rio
 */
public class OpenDreamsSession extends Session {
  /**
   * Tempo de sleep entre cada tentativa de terminar o job (em ms).
   */
  private static final int TERMINATE_SLEEP_TRY = 50;
  /**
   * O projeto associado a essa sesso.
   */
  private CommonClientProject project;
  /**
   * O identificador do usurio associado a essa sesso.
   */
  private String userId;

  /**
   * Indica se essa sesso j est ativa.
   */
  private boolean isActive = false;
  /**
   * Conjunto de job templates vlidos nessa sesso.
   */
  private Set<JobTemplate> validJobTemplates;
  /**
   * Fila que monitora o trmino da execuo de comandos.
   */
  private JobInfoQueue jobInfoQueue = new JobInfoQueue();
  /**
   * Verso do servidor CSBase.
   */
  private Version drmsVersion;
  /**
   * Nome do servidor CSBase.
   */
  private String drmsInfo;
  /**
   * Nome e verso do servidor CSBase
   */
  private String drmaaImplementation;
  /**
   * Host e porta onde o servidor CSBase est executando
   */
  private String drmsContact;
  /**
   * Mapa com os contextos de cada comando
   */
  private Map<String, OpenDreamsJobTemplate> contexts;
  /**
   * Identificador da sesso no {@link MessageService}
   */
  private Serializable consumerId;

  /**
   * Construtor.
   */
  public OpenDreamsSession() {
    csbase.logic.ManifestVersion version = new csbase.logic.ManifestVersion(
      getClass());
    this.drmsVersion = new Version(version.getMajorVersion(), version
      .getMinorVersion());
    this.drmsContact = OpenBusService.getInstance().getEntityName();
    this.drmsInfo = Server.getInstance().getSystemName();
    this.drmaaImplementation = MessageFormat.format("{0} - {1} - {2}",
      new Object[] { drmsContact, drmsInfo, version.getVersionName() });

    contexts = new Hashtable<String, OpenDreamsJobTemplate>();

    registerListener();
  }

  /**
   * {@inheritDoc}
   *
   * Usa o sistema <b>CSBase</b> como sistema DRMS. O valor do parmetro
   * <code>contactString</code>, quando no for null ou vazio, deve ser igual ao
   * retornado por {@link #contact()}.
   */
  @Override
  public void init(String contactString) throws AlreadyActiveSessionException,
  InternalException, AuthorizationException, InvalidContactStringException {
    String userId = OpenDreams.UNDEFINED;
    try {
      userId = OpenBusService.getInstance().initCSBaseAccess();
      Server.logFineMessage("OpenDreamsSession: init (" + contactString + ")");
      OpenDreams.checkUser(userId); // Verifica se o usurio da credencial est cadastrado
      if (contactString != null && !contactString.equals("")) {
        if (!contactString.equals(drmsContact)) {
          throw new InvalidContactStringException(
            "No existe um DRMS com nome " + contactString
            + ". O correto deve ser: " + drmsContact);
        }
      }
      if (this.isActive) {
        throw new AlreadyActiveSessionException("A sesso j est ativa");
      }
      this.isActive = true;
      this.validJobTemplates = new HashSet<JobTemplate>();
    }
    catch (AuthorizationException e) {
      String msg =
        "Falha na tentativa de iniciar uma sesso OpenDreams do usurio "
          + userId + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (InvalidContactStringException e) {
      String msg =
        "Falha na tentativa de iniciar uma sesso OpenDreams do usurio "
          + userId + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (AlreadyActiveSessionException e) {
      String msg =
        "Falha na tentativa de iniciar uma sesso OpenDreams do usurio "
          + userId + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (Throwable e) {
      String msg = "Erro na inicializao da sesso do usurio " + userId;
      Server.logSevereMessage(msg, e);
      throw new InternalException(OpenDreams.formatMessage(e, msg));
    }
    finally {
      OpenBusService.getInstance().finishCSBaseAccess();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void exit() throws AuthorizationException, NoActiveSessionException,
  InternalException {
    String userId = OpenDreams.UNDEFINED;
    try {
      userId = OpenBusService.getInstance().initCSBaseAccess();
      Server.logFineMessage("OpenDreamsSession: exit()");
      OpenDreams.checkUser(userId); // Verifica se o usurio da credencial est cadastrado
      if (!this.isActive) {
        throw new NoActiveSessionException("A sesso j est fechada");
      }
      this.isActive = false;
      this.validJobTemplates.clear();
      this.validJobTemplates = null;
    }
    catch (AuthorizationException e) {
      String msg =
        "Falha na tentativa de fechar uma sesso OpenDreams do usurio "
          + userId + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (NoActiveSessionException e) {
      String msg =
        "Falha na tentativa de fechar uma sesso OpenDreams do usurio "
          + userId + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (Throwable e) {
      String msg =
        "Erro na finalizao da sesso OpenDreams do usurio " + userId;
      Server.logSevereMessage(msg, e);
      throw new InternalException(OpenDreams.formatMessage(e, msg));
    }
    finally {
      OpenBusService.getInstance().finishCSBaseAccess();
      //      MessageService.getInstance().clearServerMessageListener(consumerId);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public JobTemplate createJobTemplate() throws AuthorizationException,
  NoActiveSessionException, InternalException {
    String userId = OpenDreams.UNDEFINED;
    try {
      userId = OpenBusService.getInstance().initCSBaseAccess();
      Server.logFineMessage("OpenDreamsSession: createJobTemplate()");
      OpenDreams.checkUser(userId); // Verifica se o usurio da credencial est cadastrado
      if (!this.isActive) {
        throw new NoActiveSessionException("A sesso no est ativa");
      }
      JobTemplate jt = new OpenDreamsJobTemplateImpl();
      this.validJobTemplates.add(jt);
      return jt;
    }
    catch (AuthorizationException e) {
      String msg =
        "Falha na tentativa de criar um JobTemplate pelo usurio " + userId
        + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (NoActiveSessionException e) {
      String msg =
        "Falha na tentativa de criar um JobTemplate pelo usurio " + userId
        + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (Throwable e) {
      String msg = "Erro na criao de um JobTemplate pelo usurio " + userId;
      Server.logSevereMessage(msg, e);
      throw new InternalException(OpenDreams.formatMessage(e, msg));
    }
    finally {
      OpenBusService.getInstance().finishCSBaseAccess();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void deleteJobTemplate(JobTemplate jt)
    throws InvalidJobTemplateException, AuthorizationException,
    NoActiveSessionException, InternalException {
    String userId = OpenDreams.UNDEFINED;
    try {
      userId = OpenBusService.getInstance().initCSBaseAccess();
      Server.logFineMessage("OpenDreamsSession: deleteJobTemplate()");
      OpenDreams.checkUser(userId); // Verifica se o usurio da credencial est cadastrado
      if (!this.isActive) {
        throw new NoActiveSessionException("A sesso no est ativa");
      }
      if (!isValidJobTemplate(jt)) {
        throw new InvalidJobTemplateException(
          "No  um job template vlido nessa sesso");
      }
      this.validJobTemplates.remove(jt);
    }
    catch (AuthorizationException e) {
      String msg =
        "Falha na tentativa de remover um JobTemplate pelo usurio " + userId
        + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (NoActiveSessionException e) {
      String msg =
        "Falha na tentativa de remover um JobTemplate pelo usurio " + userId
        + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (InvalidJobTemplateException e) {
      String msg =
        "Falha na tentativa de remover um JobTemplate pelo usurio " + userId
        + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (Throwable e) {
      String msg = "Erro na remoo de um jobtemplate pelo usurio " + userId;
      Server.logSevereMessage(msg, e);
      throw new InternalException(OpenDreams.formatMessage(e, msg));
    }
    finally {
      OpenBusService.getInstance().finishCSBaseAccess();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String[] runBulkJobs(JobTemplate jt, int beginIndex, int endIndex,
    int step) throws InvalidArgumentException, InternalException,
    AuthorizationException, DeniedByDrmException, InvalidJobTemplateException,
    NoActiveSessionException {
    String userId = OpenDreams.UNDEFINED;
    try {
      userId = OpenBusService.getInstance().initCSBaseAccess();
      Server.logFineMessage("OpenDreamsSession: runBulkJobs (" + beginIndex
        + ", " + endIndex + ", " + step + ")");
      OpenDreams.checkUser(userId); // Verifica se o usurio da credencial est cadastrado
      if (!this.isActive) {
        throw new NoActiveSessionException("A sesso no est ativa");
      }
      if (!isValidJobTemplate(jt)) {
        throw new InvalidJobTemplateException("No  um job template vlido");
      }
      if (beginIndex < 1) {
        throw new InvalidArgumentException(
          "O valor do argumento beginIndex deve ser maior que 1");
      }
      if (beginIndex > endIndex) {
        throw new InvalidArgumentException(
          "O valor do argumento beginIndex deve ser menor ou igual ao valor do argumento endIndex");
      }
      if (step <= 0) {
        throw new InvalidArgumentException(
          "O valor do argumento step deve ser positivo");
      }
      if (OpenDreamsJobTemplate.class.isInstance(jt)) {
        Vector<String> jobsIds = new Vector<String>();
        String baseOutputPath = jt.outputPath;
        for (int i = beginIndex; i <= endIndex; i += step) {
          //Duplica o JobTemplate e modifica o diretrio de sada, se existir,
          //colocando o identificador do subJob
          OpenDreamsJobTemplate subJobTemplate =
            OpenDreamsJobTemplate.class.cast(OpenDreamsJobTemplateImpl.class
              .cast(jt).clone());
          if (baseOutputPath != null && baseOutputPath.length() > 0) {
            subJobTemplate.outputPath =
              baseOutputPath.replaceAll(PARAMETRIC_INDEX.value, String
                .valueOf(i));
          }
          //Executa o subJob e armazena seu identificador
          jobsIds.add(executeAlgorithm(userId, subJobTemplate));
        }
        return jobsIds.toArray(new String[0]);
      }
      return null;
    }
    catch (AuthorizationException e) {
      String msg =
        "Falha na submisso do comando pelo usurio  " + userId + ": "
          + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (NoActiveSessionException e) {
      String msg =
        "Falha na submisso do comando pelo usurio " + userId + ": "
          + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (InvalidJobTemplateException e) {
      String msg =
        "Falha na submisso do comando pelo usurio " + userId + ": "
          + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (InvalidArgumentException e) {
      String msg =
        "Falha na submisso do comando pelo usurio " + userId + ": "
          + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (AlgorithmTemplateException e) {
      String msg =
        "Falha na submisso do comando pelo usurio " + userId + ": "
          + e.getMessage();
      Server.logWarningMessage(msg);
      throw new DeniedByDrmException(msg);
    }
    catch (PermissionException e) {
      String msg =
        "Falha na submisso do comando pelo usurio " + userId + ": "
          + e.getMessage();
      Server.logWarningMessage(msg);
      throw new AuthorizationException(msg);
    }
    catch (Throwable e) {
      String msg = "Erro na submisso do comando pelo usurio " + userId;
      Server.logSevereMessage(msg, e);
      throw new InternalException(OpenDreams.formatMessage(e, msg));
    }
    finally {
      OpenBusService.getInstance().finishCSBaseAccess();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String runJob(JobTemplate jt) throws InternalException,
  AuthorizationException, DeniedByDrmException, InvalidJobTemplateException,
  NoActiveSessionException {
    String userId = OpenDreams.UNDEFINED;
    try {
      userId = OpenBusService.getInstance().initCSBaseAccess();
      Server.logFineMessage("OpenDreamsSession: runJob");
      OpenDreams.checkUser(userId); // Verifica se o usurio da credencial est cadastrado
      if (!this.isActive) {
        throw new NoActiveSessionException("A sesso no est ativa");
      }
      if (!isValidJobTemplate(jt)) {
        throw new InvalidJobTemplateException("No  um job template vlido");
      }
      if (OpenDreamsJobTemplate.class.isInstance(jt)) {
        String result =
          executeAlgorithm(userId, OpenDreamsJobTemplate.class.cast(jt));
        return result;
      }
      return null;
    }
    catch (AuthorizationException e) {
      String msg =
        "Falha na submisso do comando pelo usurio  " + userId + ": "
          + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (NoActiveSessionException e) {
      String msg =
        "Falha na submisso do comando pelo usurio " + userId + ": "
          + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (InvalidJobTemplateException e) {
      String msg =
        "Falha na submisso do comando pelo usurio " + userId + ": "
          + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (AlgorithmTemplateException e) {
      String msg =
        "Falha na submisso do comando pelo usurio " + userId + ": "
          + e.getMessage();
      Server.logWarningMessage(msg);
      throw new DeniedByDrmException(msg);
    }
    catch (PermissionException e) {
      String msg =
        "Falha na submisso do comando pelo usurio " + userId + ": "
          + e.getMessage();
      Server.logWarningMessage(msg);
      throw new AuthorizationException(msg);
    }
    catch (Throwable e) {
      String msg = "Erro na submisso do comando pelo usurio " + userId;
      Server.logSevereMessage(msg, e);
      throw new InternalException(OpenDreams.formatMessage(e, msg));
    }
    finally {
      OpenBusService.getInstance().finishCSBaseAccess();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void synchronize(String[] jobList, long timeout, boolean dispose)
    throws AuthorizationException, ExitTimeoutException, InternalException,
    InvalidJobException {
    String userId = OpenDreams.UNDEFINED;
    try {
      userId = OpenBusService.getInstance().initCSBaseAccess();
      StringBuffer jobListString = new StringBuffer("{ ");
      for (String jobName : jobList) {
        jobListString.append(jobName + " ");
      }
      jobListString.append("}");
      Server.logFineMessage("OpenDreamsSession: synchronize ("
        + jobListString.toString() + ", " + timeout + ", " + dispose + ")");
      OpenDreams.checkUser(userId); // Verifica se o usurio da credencial est cadastrado
      if (jobList[0].equals(Session.JOB_IDS_SESSION_ALL)) {
        jobInfoQueue.waitForAllJobs(timeout, dispose);
      }
      else {
        jobInfoQueue.waitFor(jobList, timeout, dispose);
      }
    }
    catch (AuthorizationException e) {
      String msg =
        "Falha na consulta synchronize ao trmino do comando pelo usurio  "
          + userId + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (ExitTimeoutException e) {
      String msg =
        "Timeout esgotado na consulta synchronize do trmino do comando pelo usurio  "
          + userId + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (InvalidJobException e) {
      String msg =
        "Falha na consulta synchronize ao trmino do comando pelo usurio  "
          + userId + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (Throwable e) {
      String msg =
        "Erro na consulta synchronize ao trmino do comando pelo usurio "
          + userId;
      Server.logSevereMessage(msg, e);
      throw new InternalException(OpenDreams.formatMessage(e, msg));
    }
    finally {
      OpenBusService.getInstance().finishCSBaseAccess();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public JobInfo _wait(String jobName, long timeout)
    throws AuthorizationException, ExitTimeoutException, InternalException,
    InvalidJobException {
    String userId = OpenDreams.UNDEFINED;
    try {
      userId = OpenBusService.getInstance().initCSBaseAccess();
      Server.logFineMessage("OpenDreamsSession: _wait (" + jobName + ", "
        + timeout + ")");
      OpenDreams.checkUser(userId); // Verifica se o usurio da credencial est cadastrado
      if (jobName.equals(Session.JOB_IDS_SESSION_ANY)) {
        return jobInfoQueue.waitForAnyJob(timeout);
      }
      return jobInfoQueue.waitFor(jobName, timeout);
    }
    catch (AuthorizationException e) {
      String msg =
        "Falha na consulta wait ao trmino do comando pelo usurio  " + userId
        + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (ExitTimeoutException e) {
      String msg =
        "Falha na consulta wait ao trmino do comando pelo usurio  " + userId
        + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (InvalidJobException e) {
      String msg =
        "Falha na consulta wait ao trmino do comando pelo usurio  " + userId
        + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (Throwable e) {
      String msg =
        "Erro na consulta wait ao trmino do comando pelo usurio " + userId;
      Server.logSevereMessage(msg, e);
      throw new InternalException(OpenDreams.formatMessage(e, msg));
    }
    finally {
      OpenBusService.getInstance().finishCSBaseAccess();
    }
  }

  /**
   * {@inheritDoc}
   *
   * O OpenDreams verifica se o job j finalizou usando um cache prprio que
   * mantm as informaes sobre jobs que finalizaram. Se no encontrar a
   * informao nesse cache, pede ao servio {@link CommandPersistenceService}
   * que possui persistido em arquivo as informaes sobre todos os jobs
   * submetidos.
   *
   *  importante observar que nem sempre o estado de retorno DONE (finalizado
   * normalmente) significa que o comando terminou com cdigo de retorno igual a
   * zero. Se o configurador do algoritmo no tiver definido a opo que captura
   * a sada do comando, o estado retornado pode ser DONE tambm. Se o
   * configurador tiver definido a opo que captura a sada do comando e se o
   * comando terminar com um valor diferente de zero, o estado  retornado como
   * FAILED.
   *
   * Os seguintes estados podem ser retornados:
   * <ul>
   * <li>
   * JobState.QUEUED_ACTIVE: se o comando estiver na fila. Corresponde ao estado
   * SCHEDULED do {@link CommandStatus}</li>
   * <li>
   * JobState.RUNNING: se o comando j tiver sido removido da fila para ser
   * executado. Corresponde aos seguintes estados do {@link CommandStatus}:
   * INIT, UPLOADING, EXECUTING e DOWNLOADING.</li>
   * <li>
   * JobState.DONE: se o comando tiver finalizado e o cdigo de sada do job no
   * tiver sido capturado ou se o cdigo de sada do job tiver sido capturado
   * com valor zero. Corresponde ao estado FINISHED do {@link CommandStatus} e
   * as seguintes condies de trmino do job, no
   * {@link CommandFinalizationType}: UNKNOWN, END, SUCCESS.</li>
   * <li>
   * JobState.FAILED: se o comando tiver finalizado e o cdigo de sada do job
   * tiver sido capturado com valor diferente de zero ou se houve falha na
   * inicializao do comando ou se ele foi cancelado ou se houve falha na
   * obteno do cdigo de sada. Corresponde ao estado FINISHED do
   * {@link CommandStatus} e as seguintes condies de trmino do job, no
   * {@link CommandFinalizationType}: EXECUTION_ERROR, FAILED, KILLED e
   * NO_EXIT_CODE</li>
   * <li>
   * JobState.UNDETERMINED: em qualquer condio que no se enquadra nas
   * descritas acima.</li>
   * </ul>
   */
  @Override
  public JobState jobStatus(String jobName) throws AuthorizationException,
  InternalException, InvalidJobException {
    String userId = OpenDreams.UNDEFINED;
    try {
      userId = OpenBusService.getInstance().initCSBaseAccess();
      Server.logFineMessage("OpenDreamsSession: jobStatus (" + jobName + ")");
      OpenDreams.checkUser(userId); // Verifica se o usurio da credencial est cadastrado
      try {
        synchronized (jobInfoQueue) {
          JobInfo jobInfo = jobInfoQueue.getJobInfo(jobName);
          if (jobInfo.hasExited) {
            return jobInfo.exitStatus == 0 ? JobState.DONE : JobState.FAILED;
          }
        }
      }
      catch (InvalidJobException exception) {
        // Se no for encontrado um jobName, tenta o servio de persistncia.
      }
      CommandInfo commandInfo =
        CommandPersistenceService.getInstance().getCommandInfo(project.getId(),
          jobName);
      if (commandInfo == null) {
        throw new InvalidJobException("No foi encontrado um job com o nome: "
          + jobName);
      }
      CommandStatus status = commandInfo.getStatus();
      if (status == CommandStatus.SCHEDULED) {
        return JobState.QUEUED_ACTIVE;
      }
      if (status == CommandStatus.INIT || status == CommandStatus.UPLOADING
        || status == CommandStatus.EXECUTING
        || status == CommandStatus.DOWNLOADING) {
        return JobState.RUNNING;
      }
      if (status == CommandStatus.FINISHED) {
        CommandFinalizationType finnalizationType =
          commandInfo.getFinalizationType();
        if (finnalizationType == CommandFinalizationType.UNKNOWN
          || finnalizationType == CommandFinalizationType.END
          || finnalizationType == CommandFinalizationType.SUCCESS) {
          return JobState.DONE;
        }
        else if (finnalizationType == CommandFinalizationType.EXECUTION_ERROR
          || finnalizationType == CommandFinalizationType.FAILED
          || finnalizationType == CommandFinalizationType.KILLED
          || finnalizationType == CommandFinalizationType.NO_EXIT_CODE) {
          return JobState.FAILED;
        }
      }
      return JobState.UNDETERMINED;
    }
    catch (AuthorizationException e) {
      String msg =
        "Falha na recuperao do estado do job " + jobName + " pelo usurio "
          + userId + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (InvalidJobException e) {
      String msg =
        "Falha na recuperao do estado do job " + jobName + " pelo usurio "
          + userId + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (Throwable e) {
      Server.logSevereMessage("Erro na recuperao do estado do job " + jobName
        + " pelo usurio " + userId, e);
      throw new InternalException(e.toString());
    }
    finally {
      OpenBusService.getInstance().finishCSBaseAccess();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String contact() {
    return this.drmsContact;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void control(String jobName, JobControlAction operation)
    throws NoActiveSessionException, AuthorizationException,
    InvalidArgumentException, InvalidJobException, DrmCommunicationException,
    InternalException {
    String userId = OpenDreams.UNDEFINED;
    try {
      userId = OpenBusService.getInstance().initCSBaseAccess();
      Server.logFineMessage("OpenDreamsSession: control (" + jobName + ", "
        + operation.toString());
      OpenDreams.checkUser(userId); // Verifica se o usurio da credencial est cadastrado
      if (!this.isActive) {
        throw new NoActiveSessionException("A sesso no est ativa");
      }
      if (operation.equals(JobControlAction.HOLD)
        || operation.equals(JobControlAction.RELEASE)
        || operation.equals(JobControlAction.SUSPEND)
        || operation.equals(JobControlAction.RESUME)) {
        throw new InvalidArgumentException("Operao " + operation.toString()
          + " no implementada");
      }
      if (jobName.equals(Session.JOB_IDS_SESSION_ALL)) {
        String[] jobIds = jobInfoQueue.getAllJobs();
        for (String jobId : jobIds) {
          terminateJob(jobId);
        }
      }
      else {
        terminateJob(jobName);
      }
    }
    catch (NoActiveSessionException e) {
      String msg =
        "Falha na tentativa de alterar o estado de um comando pelo usurio "
          + userId + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (AuthorizationException e) {
      String msg =
        "Falha na tentativa de alterar o estado de um comando pelo usurio "
          + userId + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (InvalidArgumentException e) {
      String msg =
        "Falha na tentativa de alterar o estado de um comando pelo usurio "
          + userId + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (InvalidJobException e) {
      String msg =
        "Falha na tentativa de alterar o estado de um comando pelo usurio "
          + userId + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (DrmCommunicationException e) {
      String msg =
        "Falha na tentativa de alterar o estado de um comando pelo usurio "
          + userId + ": " + e.message;
      Server.logWarningMessage(msg);
      throw e;
    }
    catch (Throwable e) {
      String msg =
        "Erro na tentativa de alterar o estado de um comando pelo usurio "
          + userId;
      Server.logSevereMessage(msg, e);
      throw new InternalException(OpenDreams.formatMessage(e, msg));
    }
    finally {
      OpenBusService.getInstance().finishCSBaseAccess();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Version version() {
    return this.drmsVersion;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String drmaaImplementation() {
    return this.drmaaImplementation;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String drmsInfo() {
    return this.drmsInfo;
  }

  /**
   * Altera o projeto associado a essa sesso. O projeto j deve ter sido
   * aberto.
   *
   * @param project o projeto associado a essa sesso.
   */
  public void setProject(CommonClientProject project) {
    this.project = project;
  }

  /**
   * Altera o usurio associado a essa sesso.
   *
   * @param userId o identificador do usurio associado a essa sesso.
   */
  public void setUser(String userId) {
    this.userId = userId;
  }

  /**
   * Obtm o projeto associado a essa sesso.
   *
   * @return o projeto associado a essa sesso.
   */
  public CommonClientProject getProject() {
    return this.project;
  }

  /**
   * Verifica se um job template est vlido.
   *
   * @param jt o job template a ser validado
   * @return verdadeiro, se o job template  vlido e falso, caso contrrio.
   */
  private boolean isValidJobTemplate(JobTemplate jt) {
    return validJobTemplates.contains(jt);
  }

  /**
   * Executa um algoritmo especificado por um job template do opendreams.
   *
   * @param userId o identificador do usurio que fez a submisso
   * @param jt o job tempate com os argumentos para execuo
   * @return o identificador do comando submetido no opendreams
   * @throws AlgorithmTemplateException se houver um erro na descrio do job
   *         template
   */
  private String executeAlgorithm(final String userId,
    final OpenDreamsJobTemplate jt) throws AlgorithmTemplateException {
    /* Faz a validao dos atributos do job template */
    AlgorithmTemplate validator;
    validator = new AlgorithmTemplate(project, userId, jt);
    /* Se o modo de execuo do algoritmo for mltipla, altera para simples */
    final AlgorithmConfigurator configurator =
      validator.getAlgorithmConfigurator();
    /* Cria o comando de submisso de job */
    CommandSubmission submission =
      new CommandSubmission(configurator, project.getId());
    submission.setDescription(jt.jobDescription);
    submission.setPriority(Priority.values()[jt.jobPriority]);
    submission.setEmailList(jt.email);
    submission.setMailAtEnd(!jt.blockEmail);
    submission.configureSimpleExecution(jt.jobExecutionServer.isEmpty() ? null
      : jt.jobExecutionServer);

    Set<CommandInfo> commandInfos = null;
    try {
      commandInfos = SchedulerService.getInstance().submitCommand(submission);
    }
    catch (RemoteException e) {
    }
    String cmdId = commandInfos.toArray(new CommandInfo[0])[0].getId();
    jobInfoQueue.jobSubmited(cmdId);
    contexts.put(cmdId, jt);
    return cmdId;
  }

  /**
   * Registra o ouvinte de mensagens.
   */
  private void registerListener() {
    MessageService messageService = MessageService.getInstance();
    consumerId =
      messageService.setServerMessageListener(new IMessageListener() {
        @Override
        public void onMessagesReceived(Message... messages) throws Exception {
          for (Message message : messages) {
            CommandNotification notification =
              (CommandNotification) message.getBody();
            String cmdId = (String) notification.getCommandId();

            OpenDreamsJobTemplate jt = contexts.get(cmdId);

            if (jt != null) {
              try {
                Service.setUserId(userId);
                commandFinished(notification, jt, userId);
              }
              catch (Exception e) {
                Server.logSevereMessage("Erro ao finalizar o comando "
                  + jt.jobName, e);
              }
              finally {
                Service.setUserId(null);
              }
            }
          }
        }
      }, new BodyTypeFilter(CommandNotification.class));
  }

  /**
   * Trata o fim da execuo de um comando submetido.
   *
   * @param notification a notificao de trmino de execuo
   * @param template o job template correspondente a submisso
   * @param userId o identificador do usurio que fez a submisso
   */
  private void commandFinished(CommandNotification notification,
    OpenDreamsJobTemplate template, String userId) {
    try {
      if (template.outputPath != null && !template.outputPath.isEmpty()) {
        String jobId = notification.getCommandId().toString();
        CommandInfo info =
          CommandPersistenceService.getInstance().getCommandInfo(
            project.getId(), jobId);
        // Cria o arquivo de sada definido em outputFilePath
        try {
          Set<FileURLValue> files = info.getConfigurator().getStandardOutputFiles();
          createOutputFile(userId, files, template.outputPath);
        }
        catch (RemoteException e) {
          // Essa exceo no deve ser lanada j que no  uma chamada remota
          Server.logSevereMessage(
            "Erro na recuperao do configurador do comando.", e);
        }
      }
    }
    catch (Throwable e) {
      // Se ocorrer algum erro durante o tratamento do arquivo de sada
      Server.logSevereMessage(
        "Erro na recuperao do log de sada do comando.", e);
    }
    jobInfoQueue.jobExit(notification);
  }

  /**
   * Cria os arquivos de sada.
   *
   * @param userId identificador do usurio
   * @param stdOutputFiles lista de arquivos para criar
   * @param outputFile diretrio onde criar os arquivos
   *
   * @return true se os arquivos foram criados e false caso contrrio
   */
  private boolean createOutputFile(String userId, Set<FileURLValue> stdOutputFiles,
    String outputFile) {
    ProjectService projectService = ProjectService.getInstance();
    String[] outputFilePath = FileUtils.splitPath(outputFile);
    // Remove o arquivo definido em outputFilePath se j existir um com o
    // mesmo nome
    if (projectService.existsFile(project.getId(), outputFilePath)) {
      projectService.removeFile(project.getId(), outputFilePath);
    }
    // Caminho para o diretrio do arquivo de sada
    String[] outputDirPath =
      FileUtils.splitPath(FileUtils.getFilePath(outputFile));
    // Nome do arquivo que deve ser criado no diretrio do projeto
    String newFileName = FileUtils.getFileName(outputFile);
    // Cria o arquivo de output de sada
    projectService.createFile(project.getId(), outputDirPath, newFileName,
      "TEXT");
    ClientProjectFile cpf =
      projectService.getChild(project.getId(), outputDirPath, newFileName);
    // Transfere os arquivos de sada do comando para o arquivo outputFile
    OutputStream fileTo = null;
    try {
      fileTo = projectService.getOutputStream(project.getId(), outputFilePath);
      PrintWriter printWriter = new PrintWriter(new OutputStreamWriter(fileTo));
      for (final FileURLValue stdOutputFile : stdOutputFiles) {
        String[] stdOutputFilePath = stdOutputFile.getPathAsArray();
        InputStream fileFrom = null;
        try {
          if (stdOutputFiles.size() > 1) {
            printWriter.printf("\n---- %s ----\n", stdOutputFile.getPath());
          }
          if (!projectService.existsFile(project.getId(), stdOutputFilePath)) {
            continue;
          }
          fileFrom =
            projectService.getInputStream(project.getId(), stdOutputFilePath);
          BufferedReader bufferedReader =
            new BufferedReader(new InputStreamReader(fileFrom));
          String line = bufferedReader.readLine();
          while (line != null) {
            printWriter.println(line);
            line = bufferedReader.readLine();
          }
          printWriter.flush();
        }
        catch (IOException e) {
          Server.logSevereMessage("Erro na gravao do arquivo "
            + stdOutputFile.getPath() + " em " + cpf.getStringPath(), e);
        }
        catch (Exception e) {
          Server.logSevereMessage(
            "Erro na recuperao do inputstream do arquivo "
              + stdOutputFile.getPath(), e);
        }
        finally {
          try {
            if (fileFrom != null) {
              fileFrom.close();
            }
          }
          catch (IOException e) {
            Server.logSevereMessage(
              "Erro no fechamento do inputStream do arquivo "
                + stdOutputFile.getPath(), e);
          }
        }
      }
    }
    catch (Exception e) {
      Server.logSevereMessage(
        "Erro na recuperao do outputsream do arquivo de sada "
          + cpf.getStringPath(), e);
      return false;
    }
    finally {
      if (fileTo != null) {
        try {
          fileTo.close();
        }
        catch (IOException e) {
          Server.logSevereMessage(
            "Erro no fechamento do outputsream do arquivo de sada "
              + cpf.getStringPath(), e);
        }
      }
    }
    return true;
  }

  /**
   * Termina um job submetido para execuo.
   *
   * @param jobName nome do job
   * @throws AuthorizationException se o usurio no tem permisso para cancelar
   *         o job
   * @throws InvalidJobException se o identificador no foi encontrado
   * @throws DrmCommunicationException se o SGA no est disponvel
   */
  public void terminateJob(String jobName) throws AuthorizationException,
  InvalidJobException, DrmCommunicationException {
    CommandPersistenceService commandPersistenceService =
      CommandPersistenceService.getInstance();
    Object projectId = project.getId();
    CommandInfo commandInfo =
      commandPersistenceService.getCommandInfo(projectId, jobName);
    if (commandInfo.getStatus().equals(CommandStatus.SYSTEM_FAILURE)) {
      throw new InvalidJobException(
        "No foi encontrado informao sobre o job com identificador "
          + jobName);
    }
    if (commandInfo.getStatus().equals(CommandStatus.FINISHED)) {
      Server.logInfoMessage("O comando " + jobName
        + " j terminou e no pode mais ser cancelado");
    }
    SchedulerService schedulerService = SchedulerService.getInstance();
    try {
      // Tenta remover o comando da fila
      if (schedulerService.removeCommand(jobName)) {
        return;
      }
      while (true) {
        commandInfo =
          commandPersistenceService.getCommandInfo(projectId, jobName);
        if (commandInfo.getStatus().equals(CommandStatus.FINISHED)) {
          // Comando terminou e no pode mais ser cancelado
          return;
        }
        if (commandInfo.getStatus().equals(CommandStatus.EXECUTING)) {
          SGAService sgaService = SGAService.getInstance();
          if (sgaService.killCommand(jobName)) {
            return;
          }
          if (!sgaService.getSGASet(commandInfo.getSGAName()).getAlive()) {
            throw new DrmCommunicationException(
              "O servidor perdeu a conexo com o SGA "
                + commandInfo.getSGAName());
          }
        }
        try {
          Thread.sleep(TERMINATE_SLEEP_TRY);
        }
        catch (InterruptedException e) {
        }
      }
    }
    catch (PermissionException e) {
      throw new AuthorizationException("O usurio " + Service.getUser().getId()
        + " no possui permisso para cancelar o job " + jobName);
    }
  }
}
