/*
 * CentralAdministrationObserver.java $Author$ $Revision$ -
 * $Date$
 */
package csbase.server.services.administrationservice;

import java.rmi.RemoteException;
import java.rmi.server.UnicastRemoteObject;

import csbase.logic.AdministrationEvent;
import csbase.logic.Permission;
import csbase.logic.Platform;
import csbase.logic.Role;
import csbase.logic.User;
import csbase.logic.UserGroup;
import csbase.remote.AdministrationServiceInterface;
import csbase.remote.RemoteEvent;
import csbase.remote.RemoteObserver;
import csbase.remote.TransactionCallbackInterface;
import csbase.server.LocalServerRemoteMonitor;
import csbase.server.Server;
import csbase.server.Service;

/**
 * A classe <code>CentralAdministrationObserver</code> implementa um observador
 * remoto do servio central de administrao. Os eventos recebidos devem ser
 * refletidos no servio local. Por comodidade, ele tambm implementa a
 * interface TransactionCallbackInterface, para que possa ser utilizado como
 * 'callback' durante transaes com o servio central de administrao.
 */
class CentralAdministrationObserver extends UnicastRemoteObject implements
  RemoteObserver, TransactionCallbackInterface {
  /** Servio de administrao local. */
  LocalAdministrationService service;

  /** Indica se o lock deve ou no ser mantido. */
  boolean lockMode;

  CentralAdministrationObserver(LocalAdministrationService service)
    throws RemoteException {
    super(Server.getInstance().getRMIExportPort());
    this.service = service;
    lockMode = false;
  }

  void setLockMode(boolean lockMode) {
    this.lockMode = lockMode;
  }

  @Override
  public boolean isAlive() {
    return lockMode;
  }

  @Override
  public void install() throws RemoteException {
    LocalServerRemoteMonitor.getInstance().addObserver(
      AdministrationServiceInterface.SERVICE_NAME, this, null);
  }

  @Override
  public void uninstall() throws RemoteException {
    LocalServerRemoteMonitor.getInstance().deleteObserver(
      AdministrationServiceInterface.SERVICE_NAME, this, null);
    unexportObject(this, true);
  }

  /**
   * Esse mtodo  chamado sempre que alguma alterao ocorre no servio de
   * administrao. Essa alterao deve ser informada ao servidor local.
   * 
   * @param remoteEvent Descreve o evento ocorrido no servidor central.
   * 
   * @throws RemoteException .
   */
  @Override
  public void update(RemoteEvent remoteEvent) throws RemoteException {
    try {
      // Registra que o usurio deste thread  o administrador
      Service.setUserId(User.getAdminId());
      AdministrationEvent event = (AdministrationEvent) remoteEvent;
      if (event.item instanceof User) {
        service.updateUser(event);
      }
      else if (event.item instanceof UserGroup) {
        service.updateUserGroup(event);
      }
      else if (event.item instanceof Role) {
        service.updateRole(event);
      }
      else if (event.item instanceof Permission) {
        service.updatePermission(event);
      }
      else if (event.item instanceof Platform) {
        service.updatePlatform(event);
      }
      else {
        Server.logSevereMessage("Evento no reconhecido: " + event + " - item="
          + event.item);
      }
    }
    catch (Exception e) {
      String msg = "Falha na atualizao de usurio em servidor local.";
      Server.logSevereMessage(msg, e);
      throw new RemoteException(msg, e);
    }
    finally {
      Service.setUserId(null);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void update(RemoteEvent[] events) throws RemoteException {
    for (int i = 0; i < events.length; i++) {
      update(events[i]);
    }
  }
}
