/*
 * $Id$
 */

package csbase.server.services.dbmanagerservice;

import java.lang.reflect.Constructor;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.Statement;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.WeakHashMap;

import csbase.remote.DBManagerServiceInterface;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;

/**
 * Classe que implementa o gerente de conexes com o banco de dados, que ser
 * representado por um singleton.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class DBManagerService extends Service implements
  DBManagerServiceInterface {
  /**
   * Lista de pools de conexes com o banco de dados.
   */
  private final Hashtable<String, Pool> pools = new Hashtable<String, Pool>();

  /**
   * Usado para que o release da conexo possa descobrir a qual pool pertence
   */
  private final WeakHashMap<Connection, Pool> connMap =
    new WeakHashMap<Connection, Pool>();

  /**
   * Indica que o mdulo deve ser fechado.
   */
  private boolean moduleClosing = false;

  /**
   * Cria a chave de identificao do pool. A chave  a concatenao do nome
   * simples da classe, driver, nome de usurio e a url de conexo com a da base
   * de dados.
   * 
   * @param clazz A classe que implementa o pool
   * @param driver o driver jdbc de conexo com o banco de dados.
   * @param url a url jdbc de conexo com o banco de dados.
   * @param user a chave do usurio do banco de dados.
   * 
   * @return A chave que identifica uma conexo desse usurio nessa instncia de
   *         banco de dados.
   */
  private String createKey(final Class<?> clazz, final String driver,
    final String url, final String user) {
    return String.format("%s:%s:%s:%s", clazz.getSimpleName(), driver, user,
      url);
  }

  /**
   * Cria um novo Pool de conexes com o banco de dados
   * 
   * @param clazz A classe que implementa o pool
   * @param driver .
   * @param url A url jdbc
   * @param user O usurio do banco para conexo
   * @param password A senha do banco para conexo
   * 
   * @return Um novo pool
   */
  private Pool createPool(final Class<?> clazz, final String driver,
    final String url, final String user, final String password) {
    Pool p = null;
    try {
      final Constructor<?> constructor = clazz.getConstructor(String.class);
      p = (Pool) constructor.newInstance(createKey(clazz, driver, url, user));
      if (driver != null) {
        p.setDriver(driver);
      }
      p.setUrl(url);
      p.setUser(user);
      p.setPassword(password);
      if (!p.init()) {
        Server.logSevereMessage("Erro ao inicializar o pool de conexes : "
          + clazz.getName());
        return null;
      }
    }
    catch (final Exception e) {
      Server.logSevereMessage("Erro instanciando pool.", e);
    }
    return p;
  }

  /**
   * Encerra o gerente de banco de dados fechando todas as conexes.
   */
  private void destroy() {
    final Enumeration<Pool> elem = this.pools.elements();
    while (elem.hasMoreElements()) {
      final Pool pool = elem.nextElement();
      pool.destroy();
    }
  }

  /**
   * Obtm uma conexo com o banco de dados para uma implementao especfica do
   * pool. Driver, url e user  usado como chave para identificar o pool.
   * 
   * @param clazz A classe que implementa o pool de conexes
   * @param driver .
   * @param url A url para conexo com o banco
   * @param user O usurio do banco
   * @param password A senha do usurio do banco
   * 
   * @return uma conexo com o banco de dados
   */
  @Override
  public Connection getConnection(final Class<?> clazz, final String driver,
    final String url, final String user, final String password) {
    if (this.moduleClosing) {
      return null;
    }
    Pool pool = null;
    final String key = createKey(clazz, driver, url, user);
    synchronized (this.pools) {
      pool = this.pools.get(key);
      if (pool == null) {
        Server.logInfoMessage("DBManagerService: criando Pool - "
          + clazz.getName() + " - " + key);
        pool = createPool(clazz, driver, url, user, password);
        if (pool != null) {
          this.pools.put(key, pool);
        }
        else {
          return null;
        }
      }
    }
    pool.checkPassword(password);
    final Connection c = pool.getConnection();
    synchronized (this.connMap) {
      if (c != null) {
        this.connMap.put(c, pool);
      }
    }
    return c;
  }

  /**
   * Obtm uma conexo com o banco de dados. Usa implementao do DBPool.
   * Driver, url e user  usado como chave para identificar o pool.
   * 
   * @param driver o driver do banco de dados utilizado.
   * @param url a url do banco de dados.
   * @param user a chave do usurio do banco de dados.
   * @param password a senha do usurio do banco de dados.
   * 
   * @return uma conexo com o banco de dados.
   */
  @Override
  public Connection getConnection(final String driver, final String url,
    final String user, final String password) {
    return getConnection(DBPool.class, driver, url, user, password);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean has2Update(final Object arg, final Object event) {
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void initService() {
    this.moduleClosing = false;
  }

  /**
   * Devolve uma conexo com banco de dados, tornando-a disponvel novamente.
   * 
   * @param conn a conexo a ser devolvida.
   */
  @Override
  public void releaseConnection(final Connection conn) {
    releaseConnection(conn, null, null);
  }

  /**
   * Devolve uma conexo com banco de dados, tornando-a disponvel novamente.
   * 
   * @param conn a conexo a ser devolvida.
   * @param stmt o Statement a ser fechado (pode ser nulo).
   * @param rset o ResultSet a ser fechado (pode ser nulo).
   */
  public void releaseConnection(final Connection conn, final Statement stmt,
    final ResultSet rset) {
    if (conn == null) {
      Server.logWarningMessage("Conexo nula devolvida!");
      return;
    }

    Pool pool = null;
    synchronized (this.connMap) {
      pool = this.connMap.remove(conn);
    }
    if (pool != null) {
      pool.releaseConnection(conn, stmt, rset);
    }
    else {
      Server.logSevereMessage("Conexo devolvida no pertence a nenhum pool!");
      try {
        conn.close();
      }
      catch (final Exception e) {
        /* nada a fazer */
      }
    }

  }

  /**
   * {@inheritDoc}
   * 
   * Termina o servio de gerenciamento de conexes com o banco de dados
   * fechando todos os pools e suas respectivas conexes.
   */
  @Override
  public void shutdownService() throws ServerException {
    this.moduleClosing = true;
    destroy();
  }

  /**
   * Mtodo que instancia o gerente de conexes.
   * 
   * @throws ServerException em caso de falha no servio.
   */
  public static void createService() throws ServerException {
    new DBManagerService();
  }

  /**
   * Mtodo que retorna a instncia do gerente de conexes(singleton).
   * 
   * @return a instncia de DBManagerService.
   */
  public static DBManagerService getInstance() {
    final String serviceName = DBManagerServiceInterface.SERVICE_NAME;
    return (DBManagerService) Service.getInstance(serviceName);
  }

  /**
   * Construtor da classe.
   * 
   * @throws ServerException em caso de falha no servio.
   */
  protected DBManagerService() throws ServerException {
    super(DBManagerServiceInterface.SERVICE_NAME);
  }
}
