/*
 * $Id$
 */
package csbase.server.services.dbmanagerservice;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.LinkedList;

import csbase.server.Server;
import csbase.server.ServerException;

/**
 * Implementa pools de conexes com bancos de dados.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class DBPool extends Pool {

  /**
   * Estrutura de dados mantida no pool.
   */
  private class ConnectionData {
    /** conexo com o banco */
    Connection conn;
    /** instante da criao da conexo */
    long creationTime;
    /** tempo de uso da conexo */
    long inUseTime;
    /** contador de utilizao da conexo */
    int usedTimes;
    /** marca se a conexo ainda est vlida */
    boolean invalid = false;

    /**
     * Instancia essa estrutura
     * 
     * @param conn A conexo com o banco a ser mantida na estrutura
     */
    ConnectionData(final Connection conn) {
      this.conn = conn;
      this.creationTime = System.currentTimeMillis();
      this.usedTimes = 0;
    }
  }

  /**
   * Implementa o monitor do pool. O monitor garante a poltica de manuteno do
   * tamanho do pool, isto , do nmero de conexes abertas com o banco. A cada
   * perodo de tempo especificado em ms por {@link #decayTime}, metade das
   * conexes abertas e no utilizadas no perodo so fechadas por este monitor.
   * Esse critrio de fechamento das conexes respeita o nmero mnimo de
   * conexes abertas e livres que devem ser mantidas. Este valor  dado pelo
   * campo {@link #freeConnections}.
   */
  private class PoolMonitor extends Thread {
    /**
     * {@inheritDoc}
     */
    @Override
    public void run() {
      Server.logInfoMessage("Thread iniciada: PoolMonitor");
      synchronized (DBPool.this.pool) {
        long lastDecayTime = 0;
        while (!DBPool.this.moduleClosing) {
          long waitTime;
          if (DBPool.this.pool.size() < DBPool.this.freeConnections
            && DBPool.this.numConnections < DBPool.this.maxConnections) {
            Server.logInfoMessage("POOL " + DBPool.this.name
              + ": Criando conexo extra.");
            final ConnectionData data = newConnection();
            if (data != null) {
              DBPool.this.pool.add(data);
            }
            waitTime = DBPool.this.delayBetweenOpen;
          }
          else {
            final long now = System.currentTimeMillis();
            if (now - lastDecayTime > DBPool.this.decayTime) {
              int spareConnections =
                (int) Math
                  .ceil((DBPool.this.numConnections - DBPool.this.usedConnections) / 2F);
              final int eligibleConnections =
                DBPool.this.pool.size() - DBPool.this.freeConnections;
              spareConnections =
                Math.max(0, Math.min(spareConnections, eligibleConnections));
              for (int i = 0; i < spareConnections; i++) {
                final ConnectionData data = DBPool.this.pool.removeFirst();
                disposeConnection(data.conn);
              }
              Server.logInfoMessage("POOL " + DBPool.this.name + ": usadas="
                + DBPool.this.usedConnections + " fechadas=" + spareConnections
                + " restaram=" + DBPool.this.numConnections);
              lastDecayTime = System.currentTimeMillis();
              DBPool.this.usedConnections = 0;
              waitTime = DBPool.this.decayTime;
            }
            else {
              waitTime = DBPool.this.decayTime - (now - lastDecayTime);
            }
          }
          try {
            DBPool.this.pool.wait(waitTime);
          }
          catch (final InterruptedException e) {
            Server.logSevereMessage("POOL " + DBPool.this.name
              + ": Erro no PoolMonitor - run", e);
          }
        }
      }
      Server.logInfoMessage("Thread terminada: PoolMonitor");
    }
  }

  /**
   * O cdigo de erro utilizado para indicar nome de usurio e senha invlidos
   * para obteno de uma conexo.
   */
  private int userPasswordErrorCode;

  /**
   * Nmero mnimo de conexes livres abertas no pool.
   */
  private int freeConnections;

  /**
   * Nmero mximo de conexes abertas no pool.
   */
  private int maxConnections;

  /**
   * Nmero mximo de vezes que uma conexo pode ser utilizada.
   */
  private int maxUseTimes;

  /**
   * Tempo, em ms, de decaimento para fechar metade das conexes livres.
   */
  private long decayTime;

  /**
   * Tempo de espera, em ms, entre tentativas de criar conexo com o banco.
   */
  private long delayBetweenOpen;

  /**
   * Tempo mximo, em ms, de espera na criao de uma conexo com o banco.
   */
  private long connectionTimeout;

  /**
   * Nmero de conexes abertas no pool.
   */
  private int numConnections;

  /**
   * Nmero de conexes utilizadas no perodo.
   */
  private int usedConnections;

  /**
   * Pool de conexes: guarda as conexes abertas.
   */
  private LinkedList<ConnectionData> pool;

  /**
   * Guarda as conexes em uso.
   */
  private Hashtable<Connection, ConnectionData> using;

  /**
   * Monitor: ajusta nmero de conexes abertas em funo da demanda.
   */
  private PoolMonitor poolMonitor;

  /**
   * Indica que o mdulo deve ser fechado.
   */
  private boolean moduleClosing;

  /**
   * {@inheritDoc}
   */
  @Override
  public void checkPassword(final String passwd) {
    if (passwd == null) {
      throw new IllegalArgumentException("password == null");
    }

    if (!password.equals(passwd)) {
      Server.logInfoMessage("Houve mudana de senha.");
      setPassword(passwd);

      /* conexes disponveis so removidas */
      synchronized (this.pool) {
        Server.logInfoMessage("Removendo " + this.pool.size()
          + " conexes invlidas do pool..");
        while (!this.pool.isEmpty()) {
          final ConnectionData connData = this.pool.removeFirst();
          try {
            connData.conn.commit();
            connData.conn.close();
          }
          catch (final Exception e) {
            e.printStackTrace();
          }
          finally {
            this.numConnections--;
          }
        }
      }

      /* conexes em uso so marcadas para posterior remoo */
      synchronized (this.using) {
        Server.logInfoMessage("Invalidando " + this.using.values().size()
          + " conexes em uso para serem descartadas..");
        final Iterator<ConnectionData> it = this.using.values().iterator();
        while (it.hasNext()) {
          final ConnectionData connData = it.next();
          connData.invalid = true;
        }
      }
    }
  }

  /**
   * Trmino do mdulo de acesso a conexes  base de dados do Bandeira Brasil.
   */
  @Override
  public void destroy() {
    Server.logInfoMessage("POOL " + this.name + ": terminando...");
    synchronized (this.pool) {
      this.moduleClosing = true;
      this.maxConnections = 0;
      while (this.numConnections > 0) {
        final ConnectionData data = internalGetConnection();
        if (data != null) {
          disposeConnection(data.conn);
        }
      }
    }
    Server.logInfoMessage("POOL " + this.name + ": terminado.");
  }

  /**
   * Fecha uma conexo com o banco de dados.
   * 
   * @param conn A conexo a ser fechada.
   */
  private void disposeConnection(final Connection conn) {
    if (conn == null) {
      return;
    }
    synchronized (this.pool) {
      try {
        conn.commit();
        conn.close();
      }
      catch (final SQLException e) {
        Server.logSevereMessage("POOL " + this.name + ": Erro em disposeConnection", e);
      }
      finally {
        this.numConnections--;
      }
    }
  }

  /**
   * Obtm uma conexo para acesso a base de dados. Toda conexo retornada pelo
   * pool sempre est em modo auto-commit. O usurio da conexo  livre para
   * mudar a conexo para o modo de commits manuais e no necessita retorn-la
   * para o modo auto-commit antes de devolv-la ao pool. A conexo <b>deve</b>
   * ser posteriormente devolvida atravs do mtodo {@link #releaseConnection}.
   * 
   * @return A conexo a ser utilizada ou <code>null</code> caso o mdulo j
   *         esteja fechado.
   */
  @Override
  public Connection getConnection() {
    synchronized (this.pool) {
      if (this.moduleClosing) {
        return null;
      }
      final ConnectionData data = internalGetConnection();
      if (data == null) {
        return null;
      }
      data.inUseTime = System.currentTimeMillis();
      this.using.put(data.conn, data);
      this.usedConnections =
        Math.max(this.usedConnections, this.numConnections - this.pool.size());
      this.pool.notifyAll();
      return data.conn;
    }
  }

  /**
   * Inicializao do pool de conexes a uma base de dados. O pool deve ser
   * configurado antes de ser inicializado.
   * 
   * @return <code>true</code> em caso de sucesso, <code>false</code> caso
   *         contrrio.
   */
  @Override
  public boolean init() {
    this.numConnections = 0;
    this.usedConnections = 0;
    this.pool = new LinkedList<ConnectionData>();
    this.using = new Hashtable<Connection, ConnectionData>();
    this.moduleClosing = false;
    Connection conn = null;
    try {
      Class.forName(this.driver);
      conn = getConnection();
      if (conn == null) {
        Server.logInfoMessage("POOL " + this.name + ": Erro na conexo com a base.");
        return false;
      }
    }
    catch (final ClassNotFoundException e) {
      Server.logSevereMessage("POOL " + this.name + ": Erro na carga do driver "
        + this.driver, e);
      return false;
    }
    finally {
      if (conn != null) {
        releaseConnection(conn, null, null);
      }
    }
    this.poolMonitor = new PoolMonitor();
    this.poolMonitor.start();
    return true;
  }

  /**
   * Cria uma conexo para acesso a base de dados.
   * 
   * @return Uma estrutura ConnectionData contendo a conexo criada.
   */
  private ConnectionData internalGetConnection() {
    synchronized (this.pool) {

      if (!this.pool.isEmpty()) {
        return this.pool.removeFirst();
      }

      if (this.numConnections < this.maxConnections) {
        Server.logInfoMessage("POOL " + this.name + ": Criando conexo por demanda.");
        return newConnection();
      }

      if (this.pool.isEmpty() && (this.numConnections > 0)) {
        try {
          Server.logInfoMessage("POOL " + this.name
            + ": Aguardando por conexo (timeout = " + this.connectionTimeout
            + ")");
          this.pool.wait(this.connectionTimeout);
        }
        catch (final InterruptedException e) { /* nada a fazer */
        }
      }

      if (!this.pool.isEmpty()) {
        Server.logInfoMessage("POOL " + this.name + ": Recuperando conexo do pool.");
        return this.pool.removeFirst();
      }

      if (this.numConnections < this.maxConnections) {
        Server.logInfoMessage("POOL " + this.name + ": Criando conexo por demanda.");
        return newConnection();
      }
    }
    return null;
  }

  /**
   * Cria uma conexo para acesso a base de dados.
   * 
   * @return Uma estrutura ConnectionData contendo a conexo criada.
   */
  private ConnectionData newConnection() {
    final long initialTime = System.currentTimeMillis();
    while (true) {
      try {
        final Connection conn =
          DriverManager.getConnection(this.url, this.user, this.password);
        if (conn != null) {
          this.numConnections++;
          return new ConnectionData(conn);
        }
      }
      catch (final SQLException e) {
        Server.logSevereMessage("POOL " + this.name
          + ": Erro na obteno de uma conexo", e);
        if (e.getErrorCode() == this.userPasswordErrorCode) {
          break;
        }
      }
      if (System.currentTimeMillis() - initialTime > this.connectionTimeout) {
        break;
      }
      try {
        Thread.sleep(this.delayBetweenOpen);
      }
      catch (final InterruptedException e) {
        Server.logSevereMessage("POOL " + this.name + ": No deveria ocorrer", e);
      }
    }
    return null;
  }

  /**
   * Retorna uma descrio textual do tempo fornecido. Esse texto usa a grandeza
   * mais adequada ao tempo em questo, entre milisegundos, segundos, minutos,
   * horas e dias. Exemplos de retorno desse mtodo so: "234 milisegundos",
   * "3.5 segundos", "3 segundos", "10 horas", "2.41 dias".
   * 
   * @param t o tempo, em ms, a ser formatado.
   * 
   * @return Um texto que descreve o tempo fornecido.
   */
  private String printTime(final long t) {
    if (t < 1000) {
      return t + " milisegundo" + (t == 1 ? "" : "s");
    }
    float time = t / 1000F;
    if (time < 60) {
      return time + " segundo" + (time == 1 ? "" : "s");
    }
    time /= 60;
    if (time < 60) {
      return time + " minuto" + (time == 1 ? "" : "s");
    }
    time /= 60;
    if (time < 24) {
      return time + " hora" + (time == 1 ? "" : "s");
    }
    time /= 24;
    return time + " dia" + (time == 1 ? "" : "s");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void releaseConnection(final Connection conn, final Statement st,
    final ResultSet rs) {
    releaseConnection(conn, st, rs, false);
  }

  /**
   * Fecha um ResultSet e um Statement e devolve a conexo para o pool.
   * 
   * @param conn Conexo a ser devolvida para o pool.
   * @param st Statement a ser fechado (pode ser nulo).
   * @param rs ResultSet a ser fechado (pode ser nulo).
   * @param dispose descarta a conexo do pool
   */
  @Override
  public void releaseConnection(final Connection conn, final Statement st,
    final ResultSet rs, final boolean dispose) {
    boolean ok = false;
    try {
      if (rs != null) {
        try {
          rs.close();
        }
        catch (final SQLException e) {
          Server.logSevereMessage("POOL " + this.name + ": close - rs", e);
        }
      }
      if (st != null) {
        try {
          st.close();
        }
        catch (final SQLException e) {
          Server.logSevereMessage("POOL " + this.name + ": close - st", e);
        }
      }
      if (conn != null) {
        try {
          conn.commit();
          conn.setAutoCommit(true);
          ok = true;
        }
        catch (final SQLException e) {
          Server.logSevereMessage("POOL " + this.name + ": close - conn", e);
        }
      }
    }
    catch (final Throwable t) {
      Server.logSevereMessage("POOL " + this.name + ": close", t);
    }
    finally {
      synchronized (this.pool) {

        if (conn == null) {
          Server.logInfoMessage("POOL " + this.name + ": Conexo nula devolvida!");
          return;
        }

        final ConnectionData data = this.using.get(conn);
        if (data != null) {
          data.invalid = dispose;
          data.usedTimes++;
          long t = System.currentTimeMillis() - data.inUseTime;
          // Server.logInfoMessage("POOL: Conexo utilizada por
          // "+printTime(t)+".");
          this.using.remove(data);
          if (!ok) {
            Server.logInfoMessage("POOL " + this.name
              + ": Fechando conexo por erro (usada " + data.usedTimes
              + " vezes).");
            disposeConnection(conn);
          }
          else if (data.usedTimes >= this.maxUseTimes) {
            t = System.currentTimeMillis() - data.creationTime;
            Server.logInfoMessage("POOL " + this.name
              + ": Fechando conexo j utilizada " + data.usedTimes
              + " vezes, ao longo de " + printTime(t) + ".");
            disposeConnection(conn);
          }
          else {
            if (!data.invalid) {
              this.pool.add(data);
            }
            else {
              Server.logInfoMessage("Conexo foi invalidada. Descartada.");
              disposeConnection(data.conn);
            }
          }
        }
        else {
          Server.logInfoMessage("POOL " + this.name + ": Conexo sem registro!");
        }
        this.pool.notifyAll();
      }
    }
  }

  /**
   * Tempo mximo, em ms, de espera na criao de uma conexo com o banco.
   * 
   * @param connectionTimeout .
   */
  public void setConnectionTimeout(final long connectionTimeout) {
    this.connectionTimeout = connectionTimeout;
    Server.logInfoMessage("POOL " + this.name + ": connectionTimeout: "
      + connectionTimeout);
  }

  /**
   * Tempo, em ms, de decaimento para fechar metade das conexes livres.
   * 
   * @param decayTime .
   */
  public void setDecayTime(final long decayTime) {
    this.decayTime = decayTime;
    Server.logInfoMessage("POOL " + this.name + ": decayTime: " + decayTime);
  }

  /**
   * Tempo de espera, em ms, entre tentativas de criar conexo com o banco.
   * 
   * @param delayBetweenOpen .
   */
  public void setDelayBetweenOpen(final long delayBetweenOpen) {
    this.delayBetweenOpen = delayBetweenOpen;
    Server.logInfoMessage("POOL " + this.name + ": delayBetweenOpen: "
      + delayBetweenOpen);
  }

  /**
   * O driver da base de dados.
   * 
   * @param driver driver.
   */
  @Override
  public void setDriver(final String driver) {
    this.driver = driver;
    Server.logInfoMessage("POOL " + this.name + ": driver: " + driver);
  }

  /**
   * Nmero mnimo de conexes livres abertas no pool.
   * 
   * @param freeConnections .
   */
  public void setFreeConnections(final int freeConnections) {
    this.freeConnections = freeConnections;
    Server.logInfoMessage("POOL " + this.name + ": freeConnections: "
      + freeConnections);
  }

  /**
   * Nmero mximo de conexes abertas no pool.
   * 
   * @param maxConnections .
   */
  public void setMaxConnections(final int maxConnections) {
    this.maxConnections = maxConnections;
    Server.logInfoMessage("POOL " + this.name + ": maxConnections: " + maxConnections);
  }

  /**
   * Nmero mximo de vezes que uma conexo pode ser utilizada.
   * 
   * @param maxUseTimes .
   */
  public void setMaxUseTimes(final int maxUseTimes) {
    this.maxUseTimes = maxUseTimes;
    Server.logInfoMessage("POOL " + this.name + ": maxUseTimes: " + maxUseTimes);
  }

  /**
   * Define o cdigo de erro utilizado para indicar que o nome de usurio e
   * senha so invlidos para obteno de uma conexo.
   * 
   * @param userPasswordErrorCode cdigo.
   */
  public void setUserPasswordErrorCode(final int userPasswordErrorCode) {
    this.userPasswordErrorCode = userPasswordErrorCode;
  }

  /**
   * Retorna o nmero de cursores abertos em uma conexo.
   * 
   * @param conn conexo a ter o nmero de cursores obtidos.
   * 
   * @return o nmero de cursores abertos.
   */
  public static int getOpenCursors(final Connection conn) {
    PreparedStatement psQuery = null;
    ResultSet rs = null;
    int openCursors = 0;
    try {
      final String sqlQuery =
        "SELECT a.value FROM v$mystat a, v$statname b "
          + "WHERE a.statistic# = b.statistic# "
          + "AND b.name = 'opened cursors current'";
      psQuery = conn.prepareStatement(sqlQuery);
      rs = psQuery.executeQuery();
      if (rs.next()) {
        final Object cursor = rs.getObject(1);
        openCursors = Integer.parseInt(cursor.toString()) - 1;
      }
    }
    catch (final SQLException e) {
      e.printStackTrace();
    }
    finally {
      try {
        if (rs != null) {
          rs.close();
        }
        if (psQuery != null) {
          psQuery.close();
        }
      }
      catch (final SQLException ex) {
        ex.printStackTrace();
      }
    }
    return openCursors;
  }

  /**
   * Construtor da classe, instancia o pool e inicializa.
   * 
   * @param name nome que identifica o pool.
   * @throws ServerException em caso de falha no servidor.
   */
  public DBPool(final String name) throws ServerException {
    super(name);
    setFreeConnections(getIntProperty("freeConnections"));
    setMaxConnections(getIntProperty("maxConnections"));
    setMaxUseTimes(getIntProperty("maxUseTimes"));
    setDecayTime(getLongProperty("decayTime"));
    setDelayBetweenOpen(getLongProperty("delayBetweenOpen"));
    setConnectionTimeout(getLongProperty("connectionTimeout"));
    setUserPasswordErrorCode(getIntProperty("userPasswordErrorCode"));
  }

}
