/*
 * $Id$
 */

package csbase.server.services.notificationservice;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.InvalidClassException;
import java.io.ObjectInputStream;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import csbase.logic.Notification;
import csbase.logic.User;
import csbase.logic.UserNotification;
import csbase.remote.NotificationServiceInterface;
import csbase.remote.RemoteEvent;
import csbase.remote.RemoteObserver;
import csbase.remote.RemoteObserverNotifierInterface;
import csbase.remote.RemoteObserversNotificationManager;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;

/**
 * Fbrica de threads para o executor de updates do servio de notificao
 * 
 * @author Tecgraf/PUC-Rio
 */
class NotificationServiceThreadFactory implements ThreadFactory {
  /**
   * Constri a nova thread
   * 
   * @param r o runnable
   * @return a thread.
   */
  @Override
  public Thread newThread(Runnable r) {
    final Thread thread = new Thread(r);
    thread.setName("NotificationService::executor::" + thread.hashCode());
    return thread;
  }
}

/**
 * Executor de updates do servio de notificao
 * 
 * @author Tecgraf/PUC-Rio
 */
class NotificationServiceExecutor extends ThreadPoolExecutor {
  /**
   * Indicativo de coleta de estatsticas
   */
  private final boolean gatherStats;

  /**
   * Mapa de durao de execuo dos runnables.
   */
  private Map<Runnable, Long> durations;

  /**
   * Tempo mximo de execuo de um runnable (ponderado por seu peso).
   */
  private long weightedMaxRuntime;

  /**
   * Tempo mnimo de execuo de um runnable (ponderado por seu peso).
   */
  private long weightedMinRuntime;

  /**
   * Tempo mximo absoluto de execuo de um runnable.
   */
  private long absoluteMaxRuntime;

  /**
   * Tempo mnimo absoluto de execuo de um runnable.
   */
  private long absoluteMinRuntime;

  /**
   * Nmero de execues
   */
  private AtomicInteger executions;

  /**
   * Nmero de execues com falha.
   */
  private AtomicInteger faults;

  /**
   * Campo que representa o peso da execuo.
   */
  static private ThreadLocal<Integer> executionWeight =
    new ThreadLocal<Integer>();

  /**
   * Ajuste do peso da execuo.
   * 
   * @param weight o peso
   */
  void setExecutionWeight(int weight) {
    executionWeight.set(weight);
  }

  /**
   * Consulta do peso mximo ponderado das execues
   * 
   * @return o peso.
   */
  synchronized long getWeightedMaxRuntime() {
    return weightedMaxRuntime;
  }

  /**
   * Consulta do peso mnimo ponderado das execues
   * 
   * @return o peso.
   */
  synchronized long getWeightedMinRuntime() {
    return weightedMinRuntime;
  }

  /**
   * Consulta do peso mximo absoluto das execues
   * 
   * @return o peso.
   */
  synchronized long getAbsoluteMaxRuntime() {
    return absoluteMaxRuntime;
  }

  /**
   * Consulta do peso mnimo absoluto das execues
   * 
   * @return o peso.
   */
  synchronized long getAbsoluteMinRuntime() {
    return absoluteMinRuntime;
  }

  /**
   * Consulta o nmero de execues da estatstica
   * 
   * @return a quantidade de execues.
   */
  int getExecutions() {
    return executions.intValue();
  }

  /**
   * Consulta o nmero de falhas da estatstica.
   * 
   * @return o nmero de falhas
   */
  int getFaults() {
    return faults.intValue();
  }

  /**
   * Construtor
   * 
   * @see ThreadPoolExecutor#ThreadPoolExecutor(int, int, long, TimeUnit,
   *      BlockingQueue, ThreadFactory)
   * 
   * @param gatherStats indicativo de coleta de estatsca
   * @param corePoolSize tamanho padro do pool de threads
   * @param maximumPoolSize tamanho mximo do pool de threads
   * @param keepAliveTime tempo de vida idle de uma thread
   * @param unit unidade de tempo de {@code keepAliveTime}
   * @param workQueue fila de tarefas.
   * @param threadFactory fbrica de threads.
   */
  NotificationServiceExecutor(boolean gatherStats, int corePoolSize,
    int maximumPoolSize, long keepAliveTime, TimeUnit unit,
    BlockingQueue<Runnable> workQueue, ThreadFactory threadFactory) {
    super(corePoolSize, maximumPoolSize, keepAliveTime, unit, workQueue,
      threadFactory);
    this.gatherStats = gatherStats;
    reset();
  }

  /**
   * Reset dos valores de estatstica
   */
  void reset() {
    durations = new Hashtable<Runnable, Long>();
    weightedMaxRuntime = 0;
    weightedMinRuntime = Long.MAX_VALUE;
    absoluteMaxRuntime = 0;
    absoluteMinRuntime = Long.MAX_VALUE;
    executions = new AtomicInteger(0);
    faults = new AtomicInteger(0);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void beforeExecute(Thread t, Runnable r) {
    if (gatherStats) {
      executionWeight.set(1);
      durations.put(r, System.currentTimeMillis());
    }
    super.beforeExecute(t, r);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void afterExecute(Runnable r, Throwable t) {
    if (gatherStats) {
      Long t0 = durations.remove(r);
      if (t0 == null) {
        faults.incrementAndGet();
      }
      else {
        long dt = System.currentTimeMillis() - t0;
        int weight = executionWeight.get();
        synchronized (this) {
          weightedMaxRuntime = Math.max(weightedMaxRuntime, dt / weight);
          weightedMinRuntime = Math.min(weightedMinRuntime, dt / weight);
          absoluteMaxRuntime = Math.max(absoluteMaxRuntime, dt);
          absoluteMinRuntime = Math.min(absoluteMinRuntime, dt);
        }
      }
      executions.incrementAndGet();
    }
    super.afterExecute(r, t);
  }
}

/**
 * A classe <code>NotificationService</code> implementa o servio de notificao
 * de mensagens.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class NotificationService extends Service implements
  NotificationServiceInterface, RemoteObserverNotifierInterface {

  /**
   * Tabela de mensagens armazenadas.
   */
  private NotificationContainer container = null;

  /**
   * Flag indicativo de sada da thread de limpeza de notificaes velhas do
   * servidor.
   */
  private boolean exitCleanThread = false;

  /**
   * Thread que faz a limpeza das notificaes velhas
   */
  private Thread cleanThread = null;

  /**
   * Flag indicativo de sada da thread de gravao de backup de notificaes do
   * servidor.
   */
  private boolean exitBackupThread = false;

  /**
   * Thread que faz a gravao dos backups das notificaes
   */
  private Thread backupThread = null;

  /**
   * Flag indicativo de sada da thread de gerao de estatsticas de uso das
   * threads de notificao.
   */
  private boolean exitStatsThread = false;

  /**
   * Thread que gera estatsticas de uso das threads de notificao.
   */
  private Thread statsThread = null;

  /**
   * Executor (pool de threads) de notificaes para evitar DoS quando ocorrer
   * avalanche de notificaes.
   */
  private NotificationServiceExecutor executor = null;

  /**
   * Hash de observadores de notificaes de usurio indexada pelo seu id.
   */
  final private Hashtable<Object, RemoteObserversNotificationManager> usersObservers =
    new Hashtable<Object, RemoteObserversNotificationManager>();

  /**
   * Busca da instncia de <code>NotificationService</code>.
   * 
   * @return o servio
   * @see NotificationService
   * @see NotificationServiceInterface#SERVICE_NAME
   */
  public static NotificationService getInstance() {
    return (NotificationService) getInstance(SERVICE_NAME);
  }

  /**
   * Registra um observador remoto de notificao.
   * 
   * @param observer O observador remoto que registra este interesse.
   * @param userId identificador do usurio de interesse.
   */
  @Override
  final public void addObserver(final RemoteObserver observer,
    final Object userId) {
    if ((observer == null) || (userId == null)) {
      return;
    }

    RemoteObserversNotificationManager uobs;
    synchronized (usersObservers) {
      uobs = usersObservers.get(userId);
      if (uobs == null) {
        uobs =
          new RemoteObserversNotificationManager(this, Server.getInstance()
            .getDefaultLocale());
        usersObservers.put(userId, uobs);
      }
    }
    uobs.addObserver(observer);
    notifyUserObservers(userId);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final protected boolean has2Update(final Object arg, final Object event) {
    return true;
  }

  /**
   * Deregistra um observador
   * 
   * @param observer O observador remoto a ser deregistrado.
   * @param userId identificador do usurio de interesse.
   * 
   * @return um indicativo de sucesso.
   */
  @Override
  final public boolean deleteObserver(final RemoteObserver observer,
    final Object userId) {
    if ((observer == null) || (userId == null)) {
      return false;
    }
    synchronized (usersObservers) {
      final RemoteObserversNotificationManager uobs =
        usersObservers.get(userId);
      if (uobs != null) {
        boolean result = uobs.deleteObserver(observer);
        if (uobs.isEmpty()) {
          usersObservers.remove(userId);
        }
        return result;
      }
    }
    return false;
  }

  /**
   * Construo e inicializao do executor.
   */
  private void setupExecutor() {
    boolean gatherStats = getBooleanProperty("generateStats");
    int maxThreads = getIntProperty("maxThreads");
    long keepAlive = getIntProperty("keepAliveMinutes");
    keepAlive *= 60 * 1000;
    Server.logInfoMessage("[gatherStats]: " + gatherStats);
    Server.logInfoMessage("[maxThreads]: " + maxThreads);
    Server.logInfoMessage("[keepAlive]: " + keepAlive + "ms");
    executor =
      new NotificationServiceExecutor(gatherStats, maxThreads, maxThreads,
        keepAlive, TimeUnit.SECONDS, new LinkedBlockingDeque<Runnable>(),
        new NotificationServiceThreadFactory());
    executor.allowCoreThreadTimeOut(true);
    if (gatherStats) {
      createStatsThread();
    }
  }

  /**
   * Inicializao do servio.
   * 
   * @throws ServerException se ocorrer um erro na inicializao.
   */
  @Override
  final public void initService() throws ServerException {
    setupExecutor();
    readBackup();
    createBackupThread();
    createCleanThread();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  final public void notifyTo(Object[] userIds, String content,
    boolean mustPopUp, boolean volFlag) {
    if (content == null) {
      throw new IllegalArgumentException("content == null");
    }
    String sender = getUser().getLogin();
    notifyTo(userIds, new UserNotification(sender, content, mustPopUp, volFlag));
  }

  /**
   * Mtodo para notificao que recebe o Notification pronto.  usado pelo
   * prprio servio e pelos demais servios.
   * 
   * @param userIds Lista de usurios destinatrios, ou null para todos os
   *        usurios.
   * @param notification Informaes da notificao a ser enviada.
   */
  public void notifyTo(Object[] userIds, Notification notification) {
    if (userIds == null) {
      notifyAllUsers(notification);
    }
    else {
      for (int i = 0; i < userIds.length; i++) {
        notifyUser(userIds[i], notification);
      }
    }
  }

  /**
   * Notifica todos os usurios. A notificao ser exibida no painel de
   * notificaes.
   * 
   * @param notification Informaes da notificao a ser enviada.
   */
  private void notifyAllUsers(final Notification notification) {
    if (!isActive()) {
      return;
    }
    try {
      final List<User> users = User.getAllUsers();
      for (User user : users) {
        notifyUser(user.getId(), notification);
      }
    }
    catch (Exception e) {
      Server.logSevereMessage("Falha de notificao all-users.", e);
    }
  }

  /**
   * Envia uma notificao a um usurio.
   * 
   * @param userId o identificador do destinatrio.
   * @param notification A notificao a ser enviada.
   */
  private void notifyUser(Object userId, Notification notification) {
    if (!isActive()) {
      return;
    }
    try {
      insertUserNotification(userId, notification);
      notifyUserObservers(userId);
    }
    catch (Exception e) {
      Server.logSevereMessage("Falha de notificao. De: "
        + notification.getSender() + " Para: " + userId + " - Falha: "
        + e.getMessage());
    }
  }

  /**
   * Trmino do servio.
   */
  @Override
  final public void shutdownService() {
    Server.logInfoMessage("Finalizando executor de updates...");
    if (executor != null) {
      executor.shutdownNow();
    }
    Server.logInfoMessage("Finalizando threads...");
    exitStatsThread = true;
    exitBackupThread = true;
    exitCleanThread = true;
    if (statsThread != null) {
      statsThread.interrupt();
    }
    if (backupThread != null) {
      backupThread.interrupt();
    }
    if (cleanThread != null) {
      cleanThread.interrupt();
    }
    statsThread = null;
    backupThread = null;
    cleanThread = null;
    // a mensagem de log seguinte no faz sentido...
    // melhor remover, ou mudar de lugar
    Server.logInfoMessage("Gravando mensagens no entregues em backup...");
  }

  /**
   * Consulta ao nome do arquivo de backup.
   * 
   * @return o nome do arquivo.
   */
  private String getBackupFileName() {
    final String BACKUP_FILENAME = "backup.dat";
    try {
      final Server server = Server.getInstance();
      final String sep = File.separator;
      final String pName = server.getPersistencyRootDirectoryName();
      final String dName = pName + sep + "notifications";
      server.checkDirectory(dName);
      return dName + sep + BACKUP_FILENAME;
    }
    catch (Throwable t) {
      Server.logSevereMessage("Falha de aquisio de backup:" + t.getMessage());
      return BACKUP_FILENAME;
    }
  }

  /**
   * Limpeza de notificaes velhas.
   * 
   * @param days nmero de dias (de antigidade) para limpeza.
   */
  private void cleanNotifications(final long days) {
    if (container == null) {
      return;
    }
    if (container.clean(days)) {
      Server.logFineMessage("Limpeza de mensagens realiza com sucesso.");
    }
    else {
      Server.logSevereMessage("Falha de limpeza de mensagens");
    }
  }

  /**
   * Gravao de estatsticas de uso das threads de notificao.
   */
  private void writeStats() {
    long wmaxrt = executor.getWeightedMaxRuntime();
    Server.logFineMessage("Tempo mximo por notificao: " + wmaxrt + "ms");
    long wminrt = executor.getWeightedMinRuntime();
    Server.logFineMessage("Tempo mnimo por notificao: " + wminrt + "ms");
    long amaxrt = executor.getAbsoluteMaxRuntime();
    Server.logFineMessage("Tempo mximo por thread de notificaes: " + amaxrt
      + "ms");
    long aminrt = executor.getAbsoluteMinRuntime();
    Server.logFineMessage("Tempo mnimo por thread de notificaes: " + aminrt
      + "ms");
    int execs = executor.getExecutions();
    Server.logFineMessage("Nmero de notificaes enviadas: " + execs);
    int faults = executor.getFaults();
    Server.logFineMessage("Nmero de falhas na estatstica acima: " + faults);
    int active = executor.getActiveCount();
    Server.logFineMessage("Nmero de notificaes sendo enviadas agora: "
      + active);
    int maxsize = executor.getLargestPoolSize();
    Server.logFineMessage("Tamanho mximo atingido pelo pool: " + maxsize);
    int size = executor.getPoolSize();
    Server.logFineMessage("Tamanho atual do pool: " + size);
  }

  /**
   * Criao da thread que gera estatsticas de uso das threads de notificao
   * 
   * @see #initService()
   */
  private void createStatsThread() {
    long statsIntervalMinutes = getIntProperty("statsIntervalMinutes");
    Server.logInfoMessage("[statsIntervalMinutes]: " + statsIntervalMinutes
      + " min.");
    final long sleepTime = statsIntervalMinutes * 60 * 1000;
    statsThread = new Thread(new Runnable() {
      @Override
      public void run() {
        while (!exitStatsThread) {
          try {
            writeStats();
            executor.reset();
            Thread.sleep(sleepTime);
          }
          catch (InterruptedException ie) {
            Server.logInfoMessage("Gerao de estatsticas interrompida!");
          }
        }
        Server
          .logInfoMessage("Finalizando thread de gravao de estatsticas...");
      }
    });
    exitStatsThread = false;
    statsThread.setName(this.getClass().getSimpleName() + "::" + "StatsThread");
    statsThread.start();
  }

  /**
   * Criao da thread que faz a gravao dos backups
   * 
   * @see #initService()
   */
  private void createBackupThread() {
    /*
     * Busca das propriedades que identificam os tempos para gravao
     * (amostragem) dos dados dos SGAs em disco
     */
    final int TEN_MINUTES = 10;
    long backupIntervalMinutes = getIntProperty("backupIntervalMinutes");
    if (backupIntervalMinutes < TEN_MINUTES) {
      backupIntervalMinutes = TEN_MINUTES;
    }
    Server.logInfoMessage("[backupIntervalMinutes]: " + backupIntervalMinutes
      + " min.");
    final long sleepTime = backupIntervalMinutes * 60 * 1000;
    backupThread = new Thread(new Runnable() {
      @Override
      public void run() {
        while (!exitBackupThread) {
          try {
            writeBackup();
            Thread.sleep(sleepTime);
          }
          catch (InterruptedException ie) {
            Server.logWarningMessage("Gravao de backups interrompido!");
            // caso o loop seja interrompido,  possvel que tenha
            // deixado o arquivo de backup
            // corrompido. Devemos ento reescrev-lo
            writeBackup();
          }
        }
        Server
          .logWarningMessage("Finalizando thread de gravao de backups...");
      }
    });
    exitBackupThread = false;
    backupThread.setName(this.getClass().getSimpleName() + "::"
      + "BackupThread");
    backupThread.start();
  }

  /**
   * Criao da thread que faz a limpeza de mensagens velhas
   * 
   * @see #initService()
   */
  private void createCleanThread() {
    /*
     * Busca das propriedades que identificam os tempos para gravao
     * (amostragem) dos dados dos SGAs em disco
     */
    final int TEN_DAYS = 10;
    int cleanDays = getIntProperty("cleanIntervalDays");
    if (cleanDays < TEN_DAYS) {
      cleanDays = TEN_DAYS;
    }
    Server.logFineMessage("[cleanIntervalDays] = " + cleanDays);

    /*
     * Busca das propriedades que identificam o tempo para volatilidade de uma
     * notificao
     */
    final int TEN_SECONDS = 10;
    int volSeconds = getIntProperty("volatileIntervalSeconds");
    if (volSeconds <= 0) {
      volSeconds = TEN_SECONDS;
    }
    Server.logFineMessage("[volatileIntervalSeconds] = " + volSeconds);

    long daysToMili = (long) 24 * (long) 60 * 60 * 1000;
    long secToMili = 1000;
    final long sleepTime = volSeconds * secToMili;
    final long delta = cleanDays * daysToMili;
    Server.logInfoMessage("Intervalo de limpeza: " + cleanDays + " dias.");
    cleanThread = new Thread(new Runnable() {
      @Override
      public void run() {
        while (!exitCleanThread) {
          try {
            cleanNotifications(delta);
            Thread.sleep(sleepTime);
          }
          catch (InterruptedException ie) {
            Server.logWarningMessage("Limpeza de mensagens interrompida!");
          }
        }
        Server.logFineMessage("Finalizando thread de limpeza...");
      }
    });
    exitCleanThread = false;
    cleanThread.setName(this.getClass().getSimpleName() + "::" + "CleanThread");
    cleanThread.start();
  }

  /**
   * Notifica em thread os observadores do usurio.
   * 
   * @param userId identificador do usurio.
   * @param uobs lista de observadores remotos
   * @param notifications lista de notificaes.
   */
  private void notifyRemoteObservers(Object userId,
    RemoteObserversNotificationManager uobs,
    ArrayList<Notification> notifications) {
    executor.setExecutionWeight(uobs.numObservers());
    uobs.notifyObservers(this, notifications.toArray(new RemoteEvent[0]));
  }

  /**
   * Insero de uma notificao de usurio
   * 
   * @param usrId o usurio destinatrio
   * @param notif objeto de notificao
   */
  private void insertUserNotification(final Object usrId,
    final Notification notif) {
    if (container != null) {
      container.insertUserNotification(usrId, notif);
    }
  }

  /**
   * Notifica os observadores do usurio.
   * 
   * @param userId identificador
   */
  private void notifyUserObservers(final Object userId) {
    if (container == null) {
      return;
    }
    if (!container.userHasNotifications(userId)) {
      /*
       * no h notificaes para o usurio em questo
       */
      return;
    }
    final Runnable runnable = new Runnable() {
      @Override
      public void run() {
        RemoteObserversNotificationManager uobs;
        synchronized (usersObservers) {
          uobs = usersObservers.get(userId);
          if (uobs == null || uobs.isEmpty()) {
            return;
          }
        }
        ArrayList<Notification> notifs =
          container.retrieveUserNotifications(userId);
        if (notifs != null && !notifs.isEmpty()) {
          notifyRemoteObservers(userId, uobs, notifs);
        }
      }
    };
    executor.execute(runnable);
  }

  /**
   * Leitura de um backup de notificaes
   */
  private void readBackup() {
    final String backupFileName = getBackupFileName();
    final File backupFile = new File(backupFileName);
    final String backupFilePath = backupFile.getAbsolutePath();
    Server.logInfoMessage("Iniciando recuperao de backup de: "
      + backupFilePath);

    if (!backupFile.exists()) {
      final String msg =
        "No foi encontrado arquivo de backup de notificaes: ";
      Server.logWarningMessage(msg + backupFilePath);
      container = new NotificationContainer();
      return;
    }

    try {
      final ObjectInputStream in =
        new ObjectInputStream(new BufferedInputStream(new FileInputStream(
          backupFile)));
      container = (NotificationContainer) in.readObject();
      in.close();
      Server.logInfoMessage("Notificaes recuperadas de: " + backupFilePath);
    }
    catch (InvalidClassException ce) {
      container = new NotificationContainer();
      final String err = "No foi possvel fazer recuperao de backup. ";
      final String cause = "Verso de container no compatvel com arquivo: ";
      Server.logWarningMessage(err + cause + backupFilePath);
    }
    catch (Exception e) {
      container = new NotificationContainer();
      Server.logSevereMessage("Falha de recuperao de backup: "
        + backupFilePath, e);
    }
  }

  /**
   * Gravao de um backup de notificaes.
   */
  private void writeBackup() {
    final String bkp = getBackupFileName();
    if (container != null) {
      final File dataFile = new File(bkp);
      if (container.dumpContainer(dataFile)) {
        Server.logFineMessage("Feito backup de notificaes em: " + bkp);
      }
      else {
        Server.logSevereMessage("Falha de gravao de backup de notificaes.");
      }
    }
  }

  /**
   * Constri a instncia do servio.
   * 
   * @throws ServerException se houver falha no servidor.
   * @see NotificationService
   */
  public static void createService() throws ServerException {
    new NotificationService();
  }

  /**
   * Construtor do servio.
   * 
   * @throws ServerException em caso de falha na construo do servio.
   */
  protected NotificationService() throws ServerException {
    super(SERVICE_NAME);
  }
}
