/**
 * $Id$
 */

package csbase.server.services.projectservice;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.RandomAccessFile;
import java.io.Reader;
import java.lang.ref.SoftReference;
import java.nio.channels.FileChannel;
import java.nio.charset.Charset;
import java.rmi.RemoteException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.Stack;

import csbase.exception.InfoException;
import csbase.exception.PermissionException;
import csbase.exception.ServiceFailureException;
import csbase.logic.ClientProjectFile;
import csbase.logic.ClientProjectFileInfo;
import csbase.logic.CommonProjectInfo;
import csbase.logic.FileLockEvent;
import csbase.logic.FileLockListenerInterface;
import csbase.logic.ProjectEvent;
import csbase.logic.ProjectFileInfo;
import csbase.logic.ProjectFileType;
import csbase.logic.ProjectFileTypeInfo;
import csbase.logic.SecureKey;
import csbase.logic.User;
import csbase.logic.Utilities;
import csbase.server.FileSystem;
import csbase.server.Server;
import csbase.server.Service;
import csbase.server.services.ftcservice.FTCRequester;
import csbase.server.services.ftcservice.FTCService;
import csbase.server.services.loginservice.LoginService;
import csbase.server.services.messageservice.MessageService;
import csbase.server.services.projectservice.FileLockInterface.LockStatus;
import csbase.util.messages.Message;
import tecgraf.ftc.common.logic.RemoteFileChannelInfo;
import tecgraf.javautils.core.io.FileUtils;

/**
 * <p>
 * Modela um arquivo ou diretrio de um projeto no servidor. Essa classe far os
 * acessos ao sistema de arquivos para manipular as rvores dos projetos.
 * </p>
 * <p>
 * Obs.: Apenas quem tem permisso de leitura pode instanciar esta classe e por
 * isso no  preciso fazer verificaes de permisso de leitura nos mtodos.
 * <br>
 * Os fatos que garantem o descrito acima so:
 * <ol>
 * <li>O construtor desta classe e privado.</li>
 * <li>A nica maneira de criar uma instncia desta classe  atravs do mtodo
 * {@link #buildFromCache(File, Object, ServerProjectFile)}.</li>
 * <li>Instncias geradas por outras instncias tem permisso de leitura, ento
 * no precisamos nos preocupar com os mtodos no estticos que chamam
 * {@link #buildFromCache(File, Object, ServerProjectFile)}.</li>
 * <li>O nico mtodo esttico que chama o
 * {@link #buildFromCache(File, Object, ServerProjectFile)}  o
 * {@link #createRoot(File, Object)}.</li>
 * <li>O mtodo {@link #createRoot(File, Object)}  chamado apenas pelos
 * construtores de {@link ServerProject}.</li>
 * <li>Os construtores de {@link ServerProject} so chamados apenas na
 * {@link ServerProject#createProject(CommonProjectInfo) criao} ou na
 * {@link ServerProject#openProject(Object, boolean) abertura} de um projeto.
 * </li>
 * <li>Quem cria um projeto tem direito de leitura dele.</li>
 * <li>O usurio precisa ter permisso de leitura para abrir um projeto e isso 
 * checado no mtodo {@link ServerProject#openProject(Object, boolean)}.</li>
 * </ol>
 * </p>
 *
 * @author Tecgraf/PUC-Rio
 */
class ServerProjectFile extends FileConfig {
  /**
   * Representa o identificador do projeto ao qual esse arquivo (ou diretrio)
   * pertence.
   */
  private Object projectId;

  /**
   * Cache de objetos do tipo ServerProjectFile que garante a unicidade dos
   * ServerProjectFile criados no lado servidor. As referncias so weak para
   * que no consuma muita mamria (referncias presas no GC).
   */
  final private static HashMap<String, SoftReference<ServerProjectFile>> filesCache =
    new HashMap<String, SoftReference<ServerProjectFile>>();

  /**
   * Comparador para comparaes por nome.
   */
  private static final Comparator<ServerProjectFile> NAME_COMPARATOR =
    new Comparator<ServerProjectFile>() {
      @Override
      public int compare(ServerProjectFile o1, ServerProjectFile o2) {
        return o1.getName().compareTo(o2.getName());
      }
    };

  /**
   * Modela uma requisio FTC. Guarda o usurio e a sesso para poder fazer a
   * autorizao, e o RandomAccessFile correspondente para poder fechar ao
   * trmino do uso.
   *
   * @author Tecgraf
   */
  private class Requester implements FTCRequester {
    /**
     * Usurio que pediu acesso ao arquivo.
     */
    private String userId = null;
    /**
     * Sesso a partir da qual foi feita a requisio de acesso ao arquivo.
     */
    private String sessionKey = null;
    /**
     * Arquivo sendo acessado.
     */
    private RandomAccessFile raf = null;
    /**
     * Indica se o acesso  somente leitura ou leitura e escrita.
     */
    private boolean readOnly;

    /**
     * Cronstri uma requisio de acesso FTC.
     *
     * @param userId Usurio que pediu acesso ao arquivo.
     * @param sessionKey Sesso a partir da qual foi feita a requisio de
     *        acesso ao arquivo.
     */
    Requester(String userId, String sessionKey) {
      this.userId = userId;
      this.sessionKey = sessionKey;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public FileChannel createFileChannel(File file, boolean readOnly)
      throws Exception {
      try {
        this.readOnly = readOnly;
        Service.setUserId(userId);
        String mode;
        if (readOnly) {
          mode = "r";
        }
        else {
          try {
            checkWriteAccess();
          }
          catch (PermissionException e) {
            Server.logSevereMessage("Sem permisso para " + userId
              + " acessar o arquivo: " + getAbsolutePath(), e);
            return null;
          }
          if (sessionKey != null) {
            LockStatus lockStatus = FileLockManager.getInstance().isLocked(
              ServerProjectFile.this, sessionKey);
            if (lockStatus == LockStatus.LOCKED_BY_OTHERS) {
              Server.logSevereMessage("Arquivo bloqueado: "
                + getAbsolutePath());
              return null;
            }
          }
          else if (FileLockManager.getInstance().isLocked(
            ServerProjectFile.this)) {
            Server.logSevereMessage("Arquivo bloqueado: " + getAbsolutePath());
            return null;
          }
          mode = "rw";
        }
        try {
          raf = new RandomAccessFile(file, mode);
          if (!readOnly) {
            setUnderConstruction(true);
          }
          return raf.getChannel();
        }
        catch (FileNotFoundException e) {
          Server.logSevereMessage("Arquivo no encontrado: "
            + getAbsolutePath(), e);
          return null;
        }
      }
      finally {
        Service.setUserId(null);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void fileChannelClosed(File file) throws Exception {
      try {
        Service.setUserId(userId);
        if (raf != null) {
          try {
            raf.close();
            Server.logInfoMessage(String.format(
              "fileChannelClosed: ServerFileProject, path=%s", file
                .getAbsolutePath()));
          }
          catch (IOException e) {
            Server.logSevereMessage(
              "Erro ao tentar fechar arquivo utilizado pelo servidor de "
                + "arquivos: " + getAbsolutePath(), e);
          }
          if (!readOnly) {
            setUnderConstruction(false);
          }
        }
      }
      finally {
        Service.setUserId(null);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isLocked(File file) throws Exception {
      try {
        Service.setUserId(userId);
        if (sessionKey == null) {
          return FileLockManager.getInstance().isLocked(ServerProjectFile.this);
        }
        LockStatus lockStatus = FileLockManager.getInstance().isLocked(
          ServerProjectFile.this, sessionKey);
        return lockStatus == LockStatus.LOCKED_BY_OTHERS;
      }
      finally {
        Service.setUserId(null);
      }
    }
  }

  /**
   * Mtodo que garante a unicidade dos objetos do tipo
   * <code>ServerProjectFile</code> no servidor atravs de uma cache local de
   * controle. Os parmetros so sos mesmos do construtor desta classe, s que
   * constri o objeto de acordo com a real necessidade.
   *
   * @param file arquivo local.
   * @param projectId id do projeto.
   * @param parent diretrio-pai.
   * @return o arquivo
   */
  final static synchronized ServerProjectFile buildFromCache(final File file,
    final Object projectId, final ServerProjectFile parent) {
    final String key = file.getAbsolutePath();
    if (filesCache.containsKey(key)) {
      final ServerProjectFile spf = filesCache.get(key).get();
      if (spf != null) {
        return spf;
      }
    }
    final ServerProjectFile srvFile = new ServerProjectFile(file, projectId,
      parent);
    filesCache.put(key, new SoftReference<ServerProjectFile>(srvFile));
    return srvFile;
  }

  /**
   * Remove da cache a entrada referente ao serverProjectFile e se este for um
   * diretrio, tambm remove todas as entradas de seus filhos, recursivamente,
   * uma vez que elas no sero mais vlidas
   *
   * @param serverProjectFile
   */
  final static synchronized void deleteInCache(
    final ServerProjectFile serverProjectFile) {
    final String key = serverProjectFile.getAbsolutePath();
    filesCache.remove(key);
    /*
     * se for um diretrio, remove do cache todas chaves de filhos dele tambm,
     * recursivamente
     */
    if (serverProjectFile.isDirectory()) {
      ServerProjectFile[] children = serverProjectFile.getChildren();
      for (ServerProjectFile child : children) {
        deleteInCache(child);
      }
    }
  }

  /**
   * Limpeza total da cache.
   */
  final static synchronized void clearAllCache() {
    filesCache.clear();
  }

  /**
   * Abre um stream para leitura do arquivo.
   *
   * @return .
   */
  public InputStream openInputStream() {
    try {
      return new FileInputStream(file);
    }
    catch (FileNotFoundException e) {
      throw new ServiceFailureException(e.getMessage(), e);
    }
  }

  /**
   * Abre um stream para escrita do arquivo. Sobrescreve os dados caso o arquivo
   * j exista.
   *
   * @return stream para escrita do arquivo (sobrescreve dados j existentes)
   * @see #openOutputStream(boolean)
   */
  public OutputStream openOutputStream() {
    return openOutputStream(false);
  }

  /**
   * Abre um stream para escrita do arquivo, com possibilidade de comear a
   * escrita a partir do final do arquivo.
   *
   * @param append <code>true</code> se os dados devem ser escritos a partir do
   *        final, <code>false</code> para escrev-los a partir do incio
   * @return stream para escrita no arquivo
   */
  public OutputStream openOutputStream(boolean append) {
    try {
      return new FileOutputStream(file, append);
    }
    catch (FileNotFoundException e) {
      throw new ServiceFailureException(e.getMessage(), e);
    }
  }

  /**
   * Cria um objeto do tipo ServerProjectFile que representa o arquivo dado. Se
   * o tipo do arquivo for especificado, o arquivo de configurao  criado para
   * guardar esse tipo. O objeto criado  includo como filho do seu pai.
   *
   * @param name nome do arquivo a ser representado pelo ServerProjectFile.
   * @param type O tipo do arquivo ou <code>null</code>.
   * @param userId identificador do usurio que cria o arquivo.
   * @return O ServerProjectFile criado que representa o arquivo dado.
   */
  private ServerProjectFile _createFile(String name, String type,
    Object userId) {
    if (isControlledFileName(name)) {
      String infoMsg = getFormattedString("ProjectService.info.reserved.name",
        new Object[] { name });
      throw new InfoException(infoMsg);
    }

    checkTemplateForCreation(this, name, type);

    ProjectFileTypeInfo typeInfo = ProjectService.getInstance().getFileType(
      type);
    if (typeInfo.getCode().equals(ProjectFileType.UNKNOWN)) {
      typeInfo = ProjectService.getInstance().getTypeFromExtension(name, false);
    }
    File newFile = new File(file, name);
    if (newFile.exists()) {
      String infoMsg = getFormattedString("ServerProjectFile.info.file.exists",
        new Object[] { name, getName() });
      throw new InfoException(infoMsg);
    }
    try {
      if (typeInfo.isDirectory()) {
        if (!newFile.mkdir()) {
          String errMsg = "ServerProjectFile:createDirectory: falha ao "
            + "criar " + newFile.getAbsolutePath();
          throw new ServiceFailureException(errMsg);
        }
      }
      else {
        newFile.createNewFile();
      }
      ServerProjectFile spf = ServerProjectFile.buildFromCache(newFile,
        projectId, this);

      spf.configure(typeInfo.getCode(), userId);
      return spf;
    }
    catch (IOException e) {
      String errMsg = "ServerProjectFile:create: erro ao criar " + newFile
        .getAbsolutePath();
      throw (new ServiceFailureException(errMsg, e));
    }
  }

  /**
   * Atribui as configuraes de criao de um arquivo.
   *
   * @param type .
   * @param userId .
   */
  private synchronized void configure(String type, Object userId) {
    try {
      this.type = type;
      this.createdBy = userId;
      long date = Calendar.getInstance().getTimeInMillis();
      this.creationDate = date;
      writeConfiguration();
      User user = User.getUser(userId);
      String userName = user.getName();
      String userLogin = user.getLogin();
      if (userName == null) {
        userName = (String) userId;
      }
      String msg = getString("ServerProjectFile.file.created.by");
      msg = MessageFormat.format(msg, new Object[] { Utilities.getFormattedDate(
        date), userLogin, userName });
      this.appendDescription(msg);
    }
    catch (RemoteException e) {
      String errMsg = "ServerProjectFile:configure: ";
      throw new ServiceFailureException(errMsg, e);
    }
  }

  /**
   * Cria um objeto do tipo ServerProjectFile que representa a raiz da rvore de
   * projetos.
   *
   * @param rootDir diretrio-raiz, a ser representado pelo ServerProjectFile.
   * @param projectId O identificador do projeto para o qual est sendo criada a
   *        raiz.
   * @return O ServerProjectFile representando a raiz da rvore de projetos.
   */
  static ServerProjectFile createRoot(File rootDir, Object projectId) {
    return ServerProjectFile.buildFromCache(rootDir, projectId, null);
  }

  /**
   * Confirma a existncia do arquivo representado por este objeto no sistema de
   * arquivos.
   */
  synchronized void makeSureExists() {
    if (!file.exists()) {
      String infoMsg = getFormattedString(
        "ServerProjectFile.info.file.not.exists", new Object[] { file
          .getAbsolutePath() });
      throw new InfoException(infoMsg);
    }
  }

  /**
   * Atualiza o diretrio recursivamente, refletindo mudanas no sistema de
   * arquivos, notificando os observadores em seguida. Este mtodo no deve ser
   * invocado para arquivos.
   * <p>
   * <b>IMPORTANTE:</b> este mtodo pode ser custoso se a subrvore possuir
   * muitos elementos
   *
   * @see #refreshDir()
   */
  void rebuildDir() {
    makeSureExists();
    ProjectService projectService = ProjectService.getInstance();
    ClientProjectFile clientDir = projectService.buildClientProjectSubtree(
      this);
    // FIXME provavelmente deveramos lanar um dirRebuiltEvent
    fireDirRefreshedEvent(projectId, getPath(), clientDir);
  }

  /**
   * Garante que o diretrio existe no servidor, e notifica os observadores. A
   * notificao conter apenas o {@link ClientProjectFile} correspondente, os
   * filhos <b>no</b> so percorridos. Este mtodo deve ser executado apenas
   * para diretrios.
   *
   * @see #rebuildDir()
   */
  void refreshDir() {
    makeSureExists();
    ProjectService projectService = ProjectService.getInstance();
    ClientProjectFile clientDir = projectService.buildSingleClientProjectFile(
      this);
    fireDirRefreshedEvent(projectId, getPath(), clientDir);
  }

  /**
   * Altera o estado do arquivo para arquivo em construo, ou no.
   *
   * @param isUnderConstruction Verdadeiro para arquivo em construo e falso
   *        caso contrrio.
   */
  public synchronized void setUnderConstruction(boolean isUnderConstruction) {
    if (this.isUnderConstruction != isUnderConstruction) {
      this.isUnderConstruction = isUnderConstruction;
      writeConfiguration();
      fireStateChangedEvent(projectId, file.getName(), getPath(),
        isUnderConstruction, new Long(getModificationDate()), new Long(size()),
        FileLockManager.getInstance().isLocked(this));
    }
  }

  /**
   * Altera o tipo do arquivo.
   *
   * @param type O tipo do arquivo.
   */
  private synchronized void setType(String type) {
    ProjectFileType pft = ProjectFileType.getFileType(type);
    if (isDirectory() && !pft.isDirectory()) {
      if (!ProjectFileType.UNKNOWN.equals(pft.getCode())) {
        String errMsg = getFormattedString(
          "ServerProjectFile.error.change.type.wrong.dir", getName(), pft
            .getDescription(), pft.getCode());
        throw new ServiceFailureException(errMsg);
      }
      else {
        pft = ProjectFileType.getFileType(ProjectFileType.DIRECTORY_TYPE);
      }
    }
    if (!isDirectory() && pft.isDirectory()) {
      String errMsg = getFormattedString(
        "ServerProjectFile.error.change.type.wrong.file", getName(), pft
          .getDescription(), pft.getCode());
      throw new ServiceFailureException(errMsg);
    }
    this.type = pft.getCode();
    writeConfiguration();
  }

  /**
   * Altera o tipo do arquivo .
   *
   * @param type O tipo do arquivo. Se o tipo for {@code null}, se for um
   *        diretrio o tipo ser {@link ProjectFileType#DIRECTORY_TYPE} ou se
   *        for um arquivo o tipo ser {@link ProjectFileType#UNKNOWN}.
   */
  public synchronized void changeType(String type) {
    checkTemplateForTypeChange(type);
    setType(type);
    /* Notifica os observadores */
    fireNewFileNameEvent(projectId, getPath(), getName(), this.type);
  }

  /**
   * Retorna o nome do arquivo.
   *
   * @return Nome do arquivo.
   */
  public String getName() {
    return file.getName();
  }

  /**
   * Retorna o caminho para o arquivo, sem incluir o diretrio do projeto.
   *
   * @return Caminho at o arquivo, sem incluir o diretrio do projeto.
   */
  public String[] getPath() {
    Stack<String> pathStack = new Stack<String>();
    for (ServerProjectFile spf = this; spf.parent != null; spf = spf.parent) {
      pathStack.push(spf.getName());
    }
    String[] pathArray = new String[pathStack.size()];
    for (int i = 0; i < pathArray.length; i++) {
      pathArray[i] = pathStack.pop();
    }
    return pathArray;
  }

  /**
   * Retorna o caminho para o arquivo. Esse caminho  real, ou seja, o caminho
   * absoluto de onde o arquivo se encontra.
   *
   * @return Caminho absoluto at o arquivo, inclusive.
   */
  public String getAbsolutePath() {
    return file.getAbsolutePath();
  }

  /**
   * Retorna o canocico caminho para o arquivo.
   *
   * @return Caminho canonico at o arquivo, inclusive.
   */
  public String getCanonicalPath() {
    try {
      return file.getCanonicalPath();
    }
    catch (IOException e) {
      throw new ServiceFailureException(
        "ServerProjectFile: Caminho cannico do " + "arquivo no pde ser "
          + "obtido", e);
    }
  }

  /**
   * Informa o tipo do arquivo.
   *
   * @return O tipo do arquivo.
   */
  public String getType() {
    return type;
  }

  /**
   * Retorna o diretrio pai deste arquivo.
   *
   * @return O diretrio pai deste arquivo ou <code>null</code> caso este
   *         arquivo seja o diretrio raiz.
   */
  public ServerProjectFile getParent() {
    return parent;
  }

  /**
   * @return O identificador do projeto.
   */
  public Object getProjectId() {
    return projectId;
  }

  /**
   * Retorna a lista de arquivos deste diretrio.
   * <p>
   * IMPORTANTE: o array  sempre recriado a partir do
   * {@link ServerProjectFile#filesCache cache} ou do filesystem
   *
   * @return a lista de arquivos deste diretrio, ou <code>null</code> caso o
   *         arquivo no seja um diretrio.
   * @see #getNewTree()
   */
  public ServerProjectFile[] getChildren() {
    return getNewTree();
  }

  /**
   * Retorna a lista de arquivos deste diretrio do tipo especificado.
   *
   * @param fileType Tipo dos arquivos que devem ser retornados.
   * @return A lista de arquivos deste diretrio que sejam do tipo especificado,
   *         ou <code>null</code> caso o arquivo no seja um diretrio.
   */
  public ServerProjectFile[] getChildren(String fileType) {
    return getNewTree(fileType);
  }

  /**
   * Retorna um determinado arquivo contido neste diretrio.
   *
   * @param name Nome do arquivo a ser recuperado.
   * @return O arquivo desejado ou null se ele no existir.
   */
  public synchronized ServerProjectFile getChild(final String name) {
    if (!file.isDirectory()) {
      String errMsg = "ServerProjectFile:getChild: " + file.getAbsolutePath()
        + " no  diretrio";
      throw new ServiceFailureException(errMsg);
    }
    File child = new File(file, name);
    if (!child.exists()) {
      return null;
    }
    return ServerProjectFile.buildFromCache(child, projectId, this);
  }

  /**
   * Indica se o diretrio possui filhos.
   *
   * @return <code>true</code> se  um diretrio e se possui filhos
   */
  public synchronized boolean hasChildren() {
    if (!file.isDirectory()) {
      return false;
    }
    /*
     * como no temos um File.isEmpty() ou algo parecido, somos obrigados a
     * pegar todos os filhos de um File apenas para saber se ele tem filhos...
     * =/ Para minimizar os desperdcios, usamos um filtro que retorna um array
     * com apenas um elemento se o diretrio possui um ou mais filhos, e 0 se
     * no possui nenhum. Assim pelo menos evitamos a alocao de um array
     * potencialmente grande...
     */
    String[] childrenList = file.list(new FilenameFilter() {

      boolean foundOne = false;

      @Override
      public boolean accept(File dir, String childName) {
        if (foundOne || FileConfig.isControlledFileName(childName)) {
          return false;
        }
        foundOne = true;
        return true;
      }
    });
    return childrenList != null && childrenList.length > 0;
  }

  /**
   * Cria um arquivo dentro deste diretrio.
   *
   * @param name Nome do arquivo a ser criado.
   * @param type Tipo do arquivo a ser criado.
   * @param userId Identificador do usurio que cria o arquivo.
   * @return arquivo criado
   */
  public synchronized ServerProjectFile createFile(String name, String type,
    Object userId) {
    if (userId == null) {
      String errMsg = "ServerProjectFile:createFile: userId == null";
      throw new ServiceFailureException(errMsg);
    }
    if (!isDirectory()) {
      String infoMsg = getFormattedString(
        "ServerProjectFile.info.create.file.not.directory", new Object[] { name,
            getAbsolutePath() });
      throw new ServiceFailureException(infoMsg);
    }
    ServerProjectFile spf = _createFile(name, type, userId);
    fireNewFileEvent(projectId, spf);
    return spf;
  }

  /**
   * Cria um ou mais arquivos dentro deste diretrio.
   *
   * @param fileInfoList lista de dados dos arquivos a serem criados.
   * @param userId identificador do usurio que solicitou a criao.
   * @param notify notifica a criao da lista (para permitir a criao de
   *        arquivos de controle sem notificao para o cliente).
   */
  public synchronized void createFiles(List<ProjectFileInfo> fileInfoList,
    Object userId, boolean notify) {
    // #TODO Possvel otimizao para os caminhos dos arquivos: repassar somente
    // "a/b/c" elimina a necessidade de se passar "a" ou "a/b", pois os subdirs
    // no encontrados so criados
    if (fileInfoList == null) {
      String errMsg = "ServerProjectFile:createFiles: fileInfoList == null";
      throw new ServiceFailureException(errMsg);
    }
    if (fileInfoList.size() == 0) {
      String errMsg = "ServerProjectFile:createFiles: fileInfoList.size() == 0";
      throw new ServiceFailureException(errMsg);
    }
    if (userId == null) {
      String errMsg = "ServerProjectFile:createFiles: userId == null";
      throw new ServiceFailureException(errMsg);
    }
    if (!isDirectory()) {
      String infoMsg = getFormattedString(
        "ServerProjectFile.info.create.files.not.directory", new Object[] {
            getAbsolutePath() });
      throw new InfoException(infoMsg);
    }
    Set<ServerProjectFile> newFiles = new HashSet<ServerProjectFile>();
    for (int i = 0; i < fileInfoList.size(); i++) {
      ServerProjectFile father = this;
      ProjectFileInfo childInfo = fileInfoList.get(i);
      if (childInfo == null) {
        String errMsg = "ServerProjectFile:createFiles: childInfo == null";
        throw new ServiceFailureException(errMsg);
      }
      String[] childPath = childInfo.getPath();
      if (childPath == null) {
        String errMsg = "ServerProjectFile:createFiles: childPath == null";
        throw new ServiceFailureException(errMsg);
      }
      if (childPath.length == 0) {
        String errMsg = "ServerProjectFile:createFiles: childPath.length == 0";
        throw new ServiceFailureException(errMsg);
      }

      boolean hasParentInEvent = false;
      String childType = childInfo.getType();
      for (int j = 0; j < childPath.length; j++) {
        ServerProjectFile child = father.getChild(childPath[j]);
        if (child == null) {
          /*
           * childPath[j]  criado como diretrio se  um subdiretrio
           * intermedirio
           */
          if (j < childPath.length - 1) {
            child = father._createFile(childPath[j],
              ProjectFileType.DIRECTORY_TYPE, userId);
          }
          else {
            child = father._createFile(childPath[j], childType, userId);
          }

          // Somente os arquivos/diretrios mais externos so notificados.
          // Eventuais arqs/dirs aninhados j esto includos nas
          // referncias <code>children</code> de cada diretrio.
          if (!hasParentInEvent) {
            hasParentInEvent = true;
            newFiles.add(child);
          }
        }
        else {
          if (newFiles.contains(child)) {
            hasParentInEvent = true;
          }
        }
        father = child;
      }
    }

    if (notify) {
      ServerProjectFile[] filesArray = newFiles.toArray(
        new ServerProjectFile[newFiles.size()]);
      fireNewFilesEvent(projectId, filesArray);
    }
  }

  /**
   * Tenta obter um lock para o arquivo corrente. Se o arquivo  um diretrio,
   * tenta obter um lock compartilhado; neste caso, se o lock for obtido mas o
   * contador de referncias for > 1 significa que a operao no pode ser
   * realizada, pois outro lock compartilhado existe para o mesmo dir.
   * <p>
   * Se no  um diretrio, tenta obter um lock exclusivo para o mesmo.
   *
   * @return identificador do lock se o lock foi obtido com sucesso
   */
  private Object acquireLock() {
    FileLockManager lockManager = FileLockManager.getInstance();
    SecureKey key = null;
    if (!ProjectService.isInternalServerRequest()) {
      key = (SecureKey) ProjectService.getKey();
    }
    Object lockId = null;
    if (isDirectory()) {
      /*
       *  um diretrio, precisamos de um lock compartilhado e temos que ser os
       * nicos a faz-lo
       */
      lockId = lockManager.acquireSharedLock(this, key);
      if (lockId == null) {
        String infoMsg = getFormattedString("ServerProjectFile.info.dir.locked",
          new Object[] { getName() });
        throw new InfoException(infoMsg);
      }
      if (lockManager.getLockRefCount(this) > 1) {
        /*
         * conseguimos o lock compartilhado mas no fomos os nicos a faz-lo,
         * no podemos prosseguir. Removemos o lock recm-adquirido e abortamos.
         */
        lockManager.releaseLock(this, lockId);
        String infoMsg = getFormattedString("ServerProjectFile.info.dir.locked",
          new Object[] { getName() });
        throw new InfoException(infoMsg);
      }
    }
    else {
      /*
       *  um arquivo, precisamos de um lock exclusivo
       */
      lockId = lockManager.acquireExclusiveLock(this, key);
      if (lockId == null) {
        String infoMsg = getFormattedString(
          "ServerProjectFile.info.file.locked", new Object[] { getName() });
        throw new InfoException(infoMsg);
      }
    }
    return lockId;
  }

  /**
   * Troca o nome deste arquivo.
   *
   * @param name Novo nome do arquivo.
   */
  public synchronized void rename(String name) {

    checkTemplateForRename(name);
    /*
     * se file-locking est habilitado, s podemos seguir se obtivermos um lock
     * exclusivo sobre o arquivo
     */
    Object lockId = null;
    ServerProjectFile oldFile = this;
    boolean success = false;
    String oldAbsolutePath = getAbsolutePath();
    lockId = acquireLock();
    try {
      /* No pode renomear o diretrio raiz do Projeto */
      if (this.parent == null) {
        String infoMsg = getString("ServerProjectFile.info.rename.root");
        throw (new InfoException(infoMsg));
      }
      /* Valida o novo nome */
      if (FileConfig.isControlledFileName(name)) {
        String infoMsg = getFormattedString(
          "ServerProjectFile.info.reserved.name", new Object[] { name });
        throw (new InfoException(infoMsg));
      }
      /* Verifica se j existe um outro arquivo no diretrio com o mesmo nome */
      File newFile = renamedFile(file, name);
      if (newFile.exists()) {
        String infoMsg = getFormattedString(
          "ServerProjectFile.info.file.exists", new Object[] { name,
              getName() });
        throw new InfoException(infoMsg);
      }

      /* Renomeia o arquivo */
      String[] oldPath = getPath();

      /* Retira o arquivo da cache pois ele vai ser renomeado */
      ServerProjectFile.deleteInCache(this);
      success = rename(newFile);
      if (!success) {
        String errMsg = "ServerProjectFile:rename: "
          + " no foi possvel renomear o arquivo" + " " + getAbsolutePath()
          + " para " + newFile.getAbsolutePath();
        throw (new ServiceFailureException(errMsg));
      }

      /* Notifica os observadores */
      fireNewFileNameEvent(projectId, oldPath, name, type);
    }
    finally {
      if (success) {
        FileLockManager.getInstance().releaseLock(oldAbsolutePath, lockId);
      }
      else {
        FileLockManager.getInstance().releaseLock(oldFile, lockId);
      }
    }
  }

  /**
   * Copia este arquivo para um diretrio, que pode pertencer a outro projeto.
   * Se o diretrio for o mesmo diretrio que este arquivo/diretrio, ele altera
   * o nome da cpia. Ao final, notifica os observadores no novo arquivo.
   *
   * @param directory Diretrio destino.
   */
  public synchronized void copy(ServerProjectFile directory) {
    checkTemplateForCreation(directory, getName(), getType());

    // Verificando se a cpia est sendo feita no mesmo diretrio.
    // Se isto estiver ocorrendo, o nome do arquivo dever ser modificado.
    String copyFileName;
    if (!isDirectory() && parent.getAbsolutePath().equals(directory
      .getAbsolutePath())) {
      copyFileName = getCopyName();
    }
    else {
      copyFileName = file.getName();
    }
    File newFile = new File(directory.file, copyFileName);
    if (newFile.exists()) {
      String infoMsg = getFormattedString("ServerProjectFile.info.file.exists",
        new Object[] { copyFileName, directory.getName() });
      throw new InfoException(infoMsg);
    }
    if (isDirectory()) {
      if (equals(directory)) {
        String infoMsg = getFormattedString(
          "ServerProjectFile.info.copy.into.equal", new Object[] { getName() });
        throw new InfoException(infoMsg);
      }
      ServerProjectFile parent = directory.getParent();
      while (parent != null) {
        if (equals(parent)) {
          String infoMsg = getFormattedString(
            "ServerProjectFile.info.copy.into.descendant", new Object[] {
                getName(), directory.getName() });
          throw new InfoException(infoMsg);
        }
        parent = parent.getParent();
      }
    }
    if (!copyFile(newFile)) {
      String errMsg = String.format(getString("ServerProjectFile.error.copy"),
        copyFileName, directory.getName());
      throw new ServiceFailureException(errMsg);
    }

    /* Obtm o novo filho do diretrio */
    ServerProjectFile spf = directory.getChild(copyFileName);

    /* Notifica os observadores que um filho foi adicionado */
    Object TargetProjectId = spf.getProjectId();
    fireNewFileEvent(TargetProjectId, spf);
  }

  /**
   * Obtm o nome da cpia. O nome da cpia  formado por
   * [prefixo][sequencial].[extenso]. O prefixo  o nome do arquivo original
   * sem extenso. O sequencial  formado por um nmero entre parnteses. Esse
   * nmero  o nmero seguinte da ltima cpia existente.
   *
   * @return O nome da cpia.
   */
  private String getCopyName() {
    int maxCopyNumber = 0;
    final ServerProjectFile father = getParent();
    ServerProjectFile[] brothers = father.getChildren();
    String[] parts = splitFileName();
    String preffix = parts[0];
    String extension = parts[2];

    // Obtendo o maior nmero de cpia.
    for (int i = 0; i < brothers.length; i++) {
      ServerProjectFile brother = brothers[i];
      String[] brotherParts = brother.splitFileName();
      if (brotherParts[0].equals(preffix) && brotherParts[2].equals(extension)
        && (brotherParts[1].length() != 0)) {
        int copyNumber;
        try {
          copyNumber = Integer.parseInt(brotherParts[1]);
          if (copyNumber > maxCopyNumber) {
            maxCopyNumber = copyNumber;
          }
        }
        catch (NumberFormatException ex) {
          throw new ServiceFailureException(
            "Erro ao construir o nome do arquivo.", ex);
        }
      }
    }
    int copyNumber = maxCopyNumber + 1;
    String copyName = preffix + "_" + copyNumber;
    if (extension.length() != 0) {
      copyName += ("." + extension);
    }
    return copyName;
  }

  /**
   * Obtm os nomes e ndices que delimitam o prefixo, o sequencial e a extenso
   * deste arquivo.
   *
   * @return O array dos textos.
   *         <ul>
   *         <li>ndice 0: Ser utilizado para o prefixo;
   *         <li>ndice 1: Ser utilizado para o sequencial (sem os "()");
   *         <li>ndice 2: Ser utilizado para a extenso.
   *         </ul>
   *         Se no houver uma parte, a posio dela no array ser uma string
   *         vazia.
   */
  private String[] splitFileName() {
    String name = getName();
    int ixLastUnderscore = name.lastIndexOf("_");
    int ixDot = name.lastIndexOf(".");
    int ixEndPreffix = name.length();
    String preffix = "";
    String sequencial = "";
    String extension = "";
    if (ixDot != -1) {
      extension = name.substring(ixDot + 1);
      // removendo a extenso do arquivo.
      name = name.substring(0, ixDot);
      ixEndPreffix = ixDot;
    }
    ixLastUnderscore = name.lastIndexOf("_");
    if (ixLastUnderscore != -1) {
      String sequencialCandidate;
      sequencialCandidate = name.substring(ixLastUnderscore + 1);
      if (sequencialCandidate.matches("\\d+")) {
        sequencial = sequencialCandidate;
        ixEndPreffix = ixLastUnderscore;
      }
    }
    preffix = name.substring(0, ixEndPreffix);
    return new String[] { preffix, sequencial, extension };
  }

  /**
   * Indica se este ServerProjectFile  ancestral de outro.
   *
   * @param otherFile .
   * @return .
   */
  public boolean isAncestorOf(ServerProjectFile otherFile) {
    return ServerProjectFile.isAncestor(getPath(), otherFile.getPath());
  }

  /**
   * Indica se um caminho  ancestral do outro.
   *
   * @param ancestorPath O caminho potencialmente ancestral.
   * @param descendantPath O caminho potencialmente descendente.
   * @return Verdadeiro se ancestorPath for ancestral de descendantPath, falso
   *         caso contrrio.
   */
  public static boolean isAncestor(String[] ancestorPath,
    String[] descendantPath) {
    if (ancestorPath == null || descendantPath == null) {
      return false;
    }
    if (ancestorPath.length >= descendantPath.length) {
      return false;
    }
    for (int i = 0; i < ancestorPath.length; i++) {
      if (!ancestorPath[i].equals(descendantPath[i])) {
        return false;
      }
    }
    return true;
  }

  /**
   * Move este arquivo para outro diretrio. O outro diretrio pode pertencer a
   * outro projeto.
   *
   * @param directory Diretrio destino.
   */
  public synchronized void move(ServerProjectFile directory) {

    checkTemplateForRemoval();
    checkTemplateForCreation(directory, getName(), getType());
    /*
     * se file-locking est habilitado, s podemos seguir se obtivermos um lock
     * exclusivo sobre o arquivo
     */
    Object lockId = null;
    lockId = acquireLock();
    boolean success = false;
    File newFile = null;
    String oldAbsolutePath = getAbsolutePath();
    try {
      Object SourceProjectId = getProjectId();
      Object TargetProjectId = directory.getProjectId();

      if (SourceProjectId.equals(TargetProjectId) && isDirectory()
        && isAncestorOf(directory)) {
        String infoMsg = getFormattedString(
          "ServerProjectFile.info.move.into.descendant", new Object[] {
              getName(), directory.getName(), getName() });
        throw new InfoException(infoMsg);
      }
      if (SourceProjectId.equals(TargetProjectId) && isDirectory() && this
        .equals(directory)) {
        String infoMsg = getFormattedString(
          "ServerProjectFile.info.move.into.equal", new Object[] { getName(),
              getName() });
        throw new InfoException(infoMsg);
      }
      String[] oldPath = getPath();
      newFile = new File(directory.file, file.getName());
      if (newFile.exists()) {
        String infoMsg = getFormattedString(
          "ServerProjectFile.info.file.exists", new Object[] { getName(),
              directory.getName() });
        throw new InfoException(infoMsg);
      }
      /* Retira o arquivo da cache */
      ServerProjectFile.deleteInCache(this);

      success = rename(newFile);
      if (!success) {
        String errMsg = "ServerProjectFile:move: falha ao mover "
          + getAbsolutePath() + " " + "para " + directory.file
            .getAbsolutePath();
        throw (new ServiceFailureException(errMsg));
      }
      /* Remove o filho do diretrio onde ele estava */
      parent = directory;

      /* Alterando o projeto, caso esteja sendo movido para um diferente */
      projectId = TargetProjectId;

      /* Notifica os observadores */
      fireNewFileEvent(TargetProjectId, this);
      fireFileDeletedEvent(SourceProjectId, oldPath);
    }
    finally {
      if (success) {
        FileLockManager.getInstance().releaseLock(oldAbsolutePath, lockId);
      }
      else {
        FileLockManager.getInstance().releaseLock(this, lockId);
      }
    }
  }

  /**
   * Remove o arquivo que esse <code>ServerProjectFile</code> representa. Pode
   * ser um diretrio ou um arquivo. Os arquivos de configurao e descrio so
   * removidos tambm. Aps chamado este mtodo, o objeto que representa o
   * arquivo no poder mais ser utilizado e deve ser coletado.
   *
   * @param checkTemplate determina se os templates de projeto devem ser
   * consultados no processo de remoo.
   */
  public synchronized void remove(boolean checkTemplate) {
    String[] path = getPath();
    if (!removeFile(checkTemplate)) {
      String errMsg = "ServerProjectFile:remove: falha ao remover "
        + getAbsolutePath();
      throw new ServiceFailureException(errMsg);
    }

    // O camimho com tamnho 0 representa a raiz do projeto, e neste caso a
    // notificao enviada ser a de remoo de projeto e no a de remoo de
    // arquivo
    if (path.length > 0) {
      /* Notifica os observadores */
      fireFileDeletedEvent(projectId, path);
    }
  }

  /**
   * Retorna as informaes atualizadas desse arquivo.
   *
   * @return .
   */
  public ClientProjectFileInfo getUpdatedInfo() {
    return new ClientProjectFileInfo(getType(), size(), getModificationDate(),
      isUnderConstruction(), isLocked());
  }

  /**
   * Indica se este arquivo  um diretrio.
   *
   * @return Verdadeiro caso seja um diretrio e falso caso contrrio.
   */
  public boolean isDirectory() {
    return file.isDirectory();
  }

  /**
   * Indica se este arquivo  um arquivo em construo.
   *
   * @return Verdadeiro caso seja um arquivo em construo e falso caso
   *         contrrio.
   */
  public boolean isUnderConstruction() {
    return isUnderConstruction;
  }

  /**
   * Retorna boolean indicando se este arquivo est bloqueado.
   *
   * @return Boolean indicando se este arquivo est bloqueado.
   */
  public boolean isLocked() {
    return FileLockManager.getInstance().isLocked(this);
  }

  /**
   * Obtm a descrio atual deste arquivo.
   *
   * @return O texto com a descrio do arquivo.
   */
  public synchronized String getDescription() {
    return readDescription();
  }

  /**
   * Muda a descrio atual deste arquivo.
   *
   * @param text novo texto da descrio do arquivo.
   */
  public synchronized void setDescription(String text) {
    writeDescription(text, false);
  }

  /**
   * Anexa um texto  descrio atual deste arquivo. Permite a gerao de
   * histricos de modificao.
   *
   * @param text texto a ser anexado.
   */
  public synchronized void appendDescription(String text) {
    writeDescription(text, true);
  }

  /**
   * Obtm lock de escrita exclusivo para um arquivo. Locks exclusivos so
   * obtidos apenas uma vez.
   *
   * @return identificador do lock se o lock foi obtido, ou null caso contrrio
   */
  public Object acquireExclusiveLock() {
    SecureKey key = null;
    if (!ProjectService.isInternalServerRequest()) {
      key = (SecureKey) ProjectService.getKey();
    }
    Object lockId = FileLockManager.getInstance().acquireExclusiveLock(this,
      key);
    if (lockId != null) {
      notifyLockStatusChanged();
    }
    return lockId;
  }

  /**
   * Tenta obter um lock de escrita exclusivo para um arquivo. Locks exclusivos
   * so obtidos apenas uma vez. Este mtodo cadastra uma solicitao de lock
   * numa fila priorizada pela ordem de chegada dos pedidos de lock. Quando o
   * lock  obtido, o mtodo FileLockListenerInterface.fileLocked  chamado.
   * Quando o tempo mximo de espera do lock  atingido e o lock no foi obtido
   * o mtodo FileLockListenerInterface.fileLockExpired  chamado.
   *
   * @param listener observador do lock cadastrado
   * @param timeout tempo mximo de espera pelo lock em milisegundos. A
   *        constante INFINITE_TIMOUT foi criada para timout infinito.
   * @return identificador do pedido de lock
   */
  public Object acquireExclusiveLock(FileLockListenerInterface listener,
    long timeout) {
    SecureKey key = null;
    if (!ProjectService.isInternalServerRequest()) {
      key = (SecureKey) ProjectService.getKey();
    }
    InternalFileLockListener internalListener = new InternalFileLockListener(
      listener, key);
    return FileLockManager.getInstance().acquireExclusiveLock(this, key,
      internalListener, timeout);
  }

  /**
   * Obtm lock de escrita compartilhado para um arquivo. Locks compartilhados
   * podem ser obtidos mltiplas vezes; cada novo lock para o arquivo incrementa
   * o contador de referncias para o lock. Locks compartilhados s podem ser
   * removidos se o contador de referncias for igual a zero.
   *
   * @return identificador do lock se o lock foi obtido, ou null caso contrrio
   */
  public Object acquireSharedLock() {
    SecureKey key = null;
    if (!ProjectService.isInternalServerRequest()) {
      key = (SecureKey) ProjectService.getKey();
    }
    Object lockId = FileLockManager.getInstance().acquireSharedLock(this, key);
    if (lockId != null) {
      notifyLockStatusChanged();
    }
    return lockId;
  }

  /**
   * Tenta obter lock de escrita compartilhado para um arquivo. Locks
   * compartilhados podem ser obtidos mltiplas vezes; cada novo lock para o
   * arquivo incrementa o contador de referncias para o lock. Locks
   * compartilhados s podem ser removidos se o contador de referncias for
   * igual a zero. Este mtodo cadastra uma solicitao de lock compartilhado em
   * uma fila priorizada pela ordem de chegada dos pedidos de lock. Quando o
   * lock  obtido, o mtodo FileLockListenerInterface.fileLocked  chamado.
   * Quando o tempo mximo de espera do lock  atingido e o lock no foi obtido
   * o mtodo FileLockListenerInterface.fileLockExpired  chamado.
   *
   * @param listener observador do lock cadastrado
   * @param timeout tempo mximo de espera pelo lock em milisegundos. A
   *        constante INFINITE_TIMOUT foi criada para timout infinito.
   * @return identificador do pedido de lock
   */
  public Object acquireSharedLock(FileLockListenerInterface listener,
    long timeout) {
    SecureKey key = null;
    if (!ProjectService.isInternalServerRequest()) {
      key = (SecureKey) ProjectService.getKey();
    }
    InternalFileLockListener internalListener = new InternalFileLockListener(
      listener, key);
    return FileLockManager.getInstance().acquireSharedLock(this, key,
      internalListener, timeout);
  }

  /**
   * Libera lock de um arquivo. Se o lock  um lock exclusivo, este s 
   * efetivamente liberado se o identificador  o mesmo do lock obtido. Se o
   * lock  compartilhado, a contagem de referncias para o lock  decrementada;
   * se chega a zero, o lock  efetivamente removido.
   *
   * @param lockId identificador do lock
   * @return contagem de referncias para o lock
   */
  public int releaseLock(Object lockId) {
    int lockRefCount = FileLockManager.getInstance().releaseLock(this, lockId);
    if (lockRefCount == 0) {
      notifyLockStatusChanged();
    }
    return lockRefCount;
  }

  /**
   * Fora a remoo do lock do arquivo. S pode ser usado pelo administrador ou
   * pelo usurio dono do projeto, se acontencer alguma situao de ter um lock
   * "preso". Situaes onde as sesso do usurio que foi finalizadas por falha
   * na comunicao, ou usurios que terminam o cliente abruptamente j esto
   * previstas e no devem precisar deste mecanismo de segurana.
   *
   * @return contagem de referncias para o lock
   */
  public int forceReleaseLock() {
    int lockRefCount = FileLockManager.getInstance().forceReleaseLock(this);
    if (lockRefCount == 0) {
      notifyLockStatusChanged();
    }
    return lockRefCount;
  }

  /**
   * Retorna o identificador do usurio que criou esse arquivo.
   *
   * @return O identificador do usurio que criou esse arquivo.
   */
  public Object whoCreated() {
    return createdBy;
  }

  /**
   * Retorna o identificador do projeto no qual o arquivo existe.
   *
   * @param file .
   * @return O identificador do projeto ou null, caso esse arquivo no esteja
   *         no rea de projeto de usurio algum.
   */
  public static String findFileProjectId(File file) {
    ProjectService projectService = ProjectService.getInstance();
    final File rootDir = new File(projectService.getProjectRepositoryPath());
    try {
      final String path = rootDir.getCanonicalPath();
      return findFileProjectIdFromDir(file, path);
    }
    catch (IOException ioe) {
      String err = "Erro ao obter o caminho absoluto para o rep. de projetos";
      Server.logSevereMessage(err, ioe);
      return null;
    }
  }

  /**
   * Retorna o identificador do projeto no qual o arquivo existe.
   *
   * @param file .
   * @param dir .
   * @return O identificador do projeto ou null, caso esse arquivo no esteja
   *         no rea de projeto de usurio algum.
   */
  private static String findFileProjectIdFromDir(File file, String dir) {
    String relativePath = findFilePath(file, dir);
    if (relativePath == null) {
      return null;
    }
    String[] path = FileUtils.splitPath(relativePath);
    return path[0] + File.separator + path[1];
  }

  /**
   * Retorna o caminho relativo do arquivo a partir do seu projeto. Exemplo:
   * /home/.../project/user/proj/file = proj/file
   *
   * @param file .
   * @return .
   */
  public static String[] findFileRelativePath(File file) {
    ProjectService projectService = ProjectService.getInstance();
    String path = findFilePath(file, projectService.getProjectRepositoryPath());
    if (path == null) {
      return null;
    }
    String[] totalPath = FileUtils.splitPath(path);
    String[] relPath = new String[totalPath.length - 1];
    for (int i = 0; i < relPath.length; i++) {
      relPath[i] = totalPath[i + 1];
    }
    return relPath;
  }

  /**
   * Retorna o caminho do arquivo excluindo o diretrio especificado.
   *
   * @param file .
   * @param dir .
   * @return .
   */
  private static String findFilePath(File file, String dir) {
    String path = file.getPath();
    if (dir.charAt(dir.length() - 1) != File.separatorChar) {
      dir += File.separator;
    }
    int index = path.indexOf(dir);
    if (index == -1) {
      return null;
    }
    return path.substring(index + dir.length());
  }

  /**
   * Informa o tamanho corrente deste arquivo. Reflete o nmero corrente de
   * bytes do arquivo.
   *
   * @return Tamanho corrente do arquivo.
   */
  public long size() {
    return file.length();
  }

  /**
   * Informa a data da ltima modificao deste arquivo (formato numrico).
   *
   * @return data da ltima modificao deste arquivo (formato numrico)
   */
  public long getModificationDate() {
    return file.lastModified();
  }

  /**
   * Cria informao para abertura deste arquivo para leitura ou leitura e
   * escrita.
   *
   * @param isReadOnly Indica se o canal deve ser aberto apenas para leitura.
   * @return O ProjectFileChannel que deve ser enviado para o cliente e que se
   *         conecta ao arquivo, permitindo sua leitura e escrita.
   */
  public RemoteFileChannelInfo openChannel(boolean isReadOnly) {
    try {
      String userId = Service.getUser().getLogin();
      Object key = Service.getKey();
      String sessionKey = (key == null ? null : key.toString());
      Requester req = new Requester(userId, sessionKey);
      File f = new File(file.getAbsolutePath());
      Server.logInfoMessage("Criando FileChannelInfo: ServerProjectFile");
      return FTCService.getInstance().createFileChannelInfo(req, f, isReadOnly);
    }
    catch (Exception e) {
      String errMsg = "Falha ao construir informao de canal para " + file
        .getAbsolutePath();
      throw new ServiceFailureException(errMsg, e);
    }
  }

  /**
   * Retorna um texto descritivo do arquivo.
   *
   * @return .
   */
  @Override
  public String toString() {
    String text = "";
    text = text + "File: " + file.getAbsolutePath() + "\n";
    if (parent != null) {
      text = text + "Parent: " + parent.getAbsolutePath() + "\n";
    }
    text = text + "UnderConstruction: " + isUnderConstruction + "\n";
    text = text + "Type : " + type + "\n";
    text = text + "CreatedBy:" + createdBy + "\n";
    text = text + "Project :" + projectId + "\n";
    return text;
  }

  /**
   * Verifica se um determinado arquivo existe. Se o arquivo existir, ele 
   * retornado como resultado do mtodo. Caso contrrio,  retornado null.
   *
   * @param projectId Identificador do projeto.
   * @param path Caminho para o arquivo, <b>no</b> incluindo o nome do projeto.
   * @return <code>ServerProjectFile</code> representando o arquivo, ou null
   *         caso no exista.
   */
  public static ServerProjectFile testFile(Object projectId, String[] path) {
    ServerProject sp = ServerProject.getProject(projectId);
    if (sp == null) {
      String infoMsg = ProjectService.getInstance().getFormattedString(
        "ServerProjectFile.info.project.not.open", new Object[] { ServerProject
          .getProjectName(projectId) });
      throw new InfoException(infoMsg);
    }
    ServerProjectFile spf = sp.getTree();
    for (int i = 0; spf != null && i < path.length; i++) {
      if (path[i] == null || path[i].trim().equals("")) {
        continue;
      }
      ServerProjectFile parent = spf;
      spf = parent.getChild(path[i]);
      /*
       * Se no achou, mas chegou no ltimo nvel, aproveita e verifica se o
       * arquivo tem arquivos de controle. Se tiver, os remove para que no
       * sobrar lixo.
       */
      if (spf == null && i == path.length - 1) {
        String parentPath = parent.getCanonicalPath();
        String filePath = parentPath + File.separator + path[path.length - 1];
        File file = new File(filePath);
        File configFile = renamedConfigFile(file);
        if (configFile.exists()) {
          configFile.delete();
        }
        File descriptionFile = FileConfig.renamedDescriptionFile(file);
        if (descriptionFile.exists()) {
          descriptionFile.delete();
        }
        return null;
      }
    }
    return spf;
  }

  /**
   * Obtm a referncia para um arquivo/diretrio no servidor, pertencente ao
   * projeto especificado.
   * <p>
   * Lana uma {@link ServiceFailureException} se no encontra o arquivo.
   *
   * @param projectId Identificador do projeto.
   * @param path Caminho para o arquivo/diretrio, <b>sem</b> incluir o nome do
   *        projeto.
   * @return <code>ServerProjectFile</code> representando o arquivo/diretrio.
   */
  public static ServerProjectFile findFile(Object projectId, String[] path) {
    ServerProjectFile spf = testFile(projectId, path);
    if (spf == null) {
      ProjectService service = ProjectService.getInstance();
      String prjName = ServerProject.getProjectName(projectId);
      String dirName = FileUtils.joinPath("/", Arrays.copyOf(path, path.length
        - 1));
      Object user = ServerProject.getOwnerId(projectId);
      String infoMsg = service.getFormattedString(
        "ServerProjectFile.info.file.not.found", new Object[] { path[path.length
          - 1], dirName.isEmpty() ? "/" : dirName, prjName, user });
      throw new InfoException(infoMsg);
    }
    return spf;
  }

  /**
   * Notifica aos observadores que o estado do arquivo mudou.
   *
   * @param projectId identificador do projeto
   * @param fileName nome do arquivo
   * @param path caminho do arquivo
   * @param isUnderConstruction se o arquivo est em construo
   * @param modificationDate a data da ltima modificao
   * @param size o tamanho do arquivo
   * @param locked se o arquivo est bloqueado
   */
  private static void fireStateChangedEvent(Object projectId, String fileName,
    String[] path, boolean isUnderConstruction, Long modificationDate,
    Long size, boolean locked) {
    Object[] arg = { path, fileName, isUnderConstruction, modificationDate,
        size, locked };
    try {
      String[] users = ProjectService.getInstance().getUserToNotify(projectId);
      MessageService.getInstance().send(new Message(ProjectEvent.makeEvent(
        projectId, ProjectEvent.FILE_STATE_CHANGED, arg)), users);
    }
    catch (RemoteException e) {
      Server.logSevereMessage("Erro ao notificar os usurios do projeto "
        + projectId + ".", e);
    }
  }

  /**
   * Notifica aos observadores que um arquivo foi criado.
   *
   * @param projectId identificador do projeto
   * @param file novo arquivo criado
   */
  private static void fireNewFileEvent(Object projectId,
    ServerProjectFile file) {
    Object[] args = new Object[2];
    if (null == file.getParent()) {
      args[0] = new String[0];
    }
    else {
      args[0] = file.getParent().getPath();
    }
    ProjectService projectService = ProjectService.getInstance();
    args[1] = projectService.buildClientProjectSubtree(file);
    try {
      String[] users = ProjectService.getInstance().getUserToNotify(projectId);
      MessageService.getInstance().send(new Message(ProjectEvent.makeEvent(
        projectId, ProjectEvent.NEW_FILE, args)), users);
    }
    catch (RemoteException e) {
      Server.logSevereMessage("Erro ao notificar os usurios do projeto "
        + projectId + ".", e);
    }
  }

  /**
   * Notifica aos observadores da criao simultnea de um ou mais arquivos.
   *
   * @param projectId identificador do projeto.
   * @param files referncias para os arquivos recm-criados.
   */
  private static void fireNewFilesEvent(Object projectId,
    ServerProjectFile[] files) {
    String[][] paths = new String[files.length][];
    ClientProjectFile[] clientProjectFiles =
      new ClientProjectFile[files.length];
    for (int fileInx = 0; fileInx < files.length; fileInx++) {
      // O arquivo que foi criado.
      ServerProjectFile file = files[fileInx];
      // Cria um ClientProjectFile representando o caminho at o novo arquivo.
      if (null == file.getParent()) {
        paths[fileInx] = new String[0];
      }
      else {
        paths[fileInx] = file.getParent().getPath();
      }
      ProjectService projectService = ProjectService.getInstance();
      clientProjectFiles[fileInx] = projectService.buildClientProjectSubtree(
        file);
    }
    Object[] args = { paths, clientProjectFiles };
    try {
      String[] users = ProjectService.getInstance().getUserToNotify(projectId);
      MessageService.getInstance().send(new Message(ProjectEvent.makeEvent(
        projectId, ProjectEvent.NEW_FILES, args)), users);
    }
    catch (RemoteException e) {
      Server.logSevereMessage("Erro ao notificar os usurios do projeto "
        + projectId + ".", e);
    }
  }

  /**
   * Notifica aos observadores que um arquivo foi removido.
   *
   * @param projectId .
   * @param path .
   */
  private static void fireFileDeletedEvent(Object projectId, String[] path) {
    Object[] arg = { path };
    try {
      String[] users = ProjectService.getInstance().getUserToNotify(projectId);
      MessageService.getInstance().send(new Message(ProjectEvent.makeEvent(
        projectId, ProjectEvent.FILE_DELETED, arg)), users);
    }
    catch (RemoteException e) {
      Server.logSevereMessage("Erro ao notificar os usurios do projeto "
        + projectId + ".", e);
    }
  }

  /**
   * Notifica aos observadores que um arquivo mudou de nome e/ou tipo.
   *
   * @param projectId .
   * @param path .
   * @param newName .
   * @param newType .
   */
  private static void fireNewFileNameEvent(Object projectId, String[] path,
    String newName, String newType) {
    Object[] arg = { path, newName, newType };
    try {
      String[] users = ProjectService.getInstance().getUserToNotify(projectId);
      MessageService.getInstance().send(new Message(ProjectEvent.makeEvent(
        projectId, ProjectEvent.NEW_FILE_NAME, arg)), users);
    }
    catch (RemoteException e) {
      Server.logSevereMessage("Erro ao notificar os usurios do projeto "
        + projectId + ".", e);
    }
  }

  /**
   * Notifica aos observadores que um diretrio foi atualizado com os dados do
   * servidor.
   *
   * @param projectId identificador do projeto.
   * @param path path at a raiz do projeto
   * @param dir referncia para o objeto no servidor representando o diretrio.
   */
  private static void fireDirRefreshedEvent(Object projectId, String[] path,
    ClientProjectFile dir) {
    Object[] arg = { path, dir };
    try {
      String[] users = ProjectService.getInstance().getUserToNotify(projectId);
      MessageService.getInstance().send(new Message(ProjectEvent.makeEvent(
        projectId, ProjectEvent.DIR_REFRESHED, arg)), users);
    }
    catch (RemoteException e) {
      Server.logSevereMessage("Erro ao notificar os usurios do projeto "
        + projectId + ".", e);
    }
  }

  /**
   * Notifica que o arquivo foi bloqueado ou desbloqueado.
   */
  private void notifyLockStatusChanged() {
    try {
      boolean isLocked = FileLockManager.getInstance().isLocked(this);
      fireStateChangedEvent(getProjectId(), getName(), getPath(),
        isUnderConstruction(), getModificationDate(), size(), isLocked);
    }
    catch (Exception e) {
      // No interrompe o mecanimo de bloqueio/desbloquio se a notificao
      // falhar.
      Server.logWarningMessage(
        "Falha na notificao de bloqueio/desbloqueio de lock do arquivo "
          + file.getAbsolutePath());
    }
  }

  /**
   * Faz leitura crua da descrio.
   *
   * @return texto
   * @throws IOException
   */
  private String rawGetDescription() throws IOException {
    final File descFile = renamedDescriptionFile(file);
    if (!descFile.exists()) {
      final String fileName = getName();
      String fmt = "%s: No existe arquivo de descrio '%s'.";
      final String realFileName = file.getName();
      final String descFileName = createDescriptionFileName(realFileName);
      String msg = String.format(fmt, fileName, descFileName);
      Server.logFineMessage(msg);
      return new String();
    }
    final Server server = Server.getInstance();
    final Charset charset = server.getSystemDefaultCharset();
    try (FileInputStream fileStream = new FileInputStream(descFile);
         Reader reader = new InputStreamReader(fileStream, charset)) {
      char[] buffer = new char[(int) descFile.length()];
      reader.read(buffer);
      final String desc = new String(buffer);
      return desc;
    }
  }

  /**
   * L e devolve o contedo do arquivo de descrio associado a um
   * <code>Project File</code>. Se o arquivo no existir, devolve "".
   *
   * @return .
   */
  private String readDescription() {
    final String fmt = getString("ProjectService.error.read.description");
    try {
      final String desc = rawGetDescription();
      return desc == null ? "" : desc;
    }
    catch (IOException ioe) {
      final String fileName = getName();
      final File descFile = renamedDescriptionFile(file);
      final String path = descFile.getAbsolutePath();
      final String exMsg = ioe.getMessage();
      String err = String.format(fmt, fileName, path, exMsg);
      return err;
    }
  }

  /**
   * Grava um texto no arquivo de descrio.
   *
   * @param text O texto a ser gravado.
   * @param append Se  para ser gravado como apndice ou no.
   */
  private void writeDescription(String text, boolean append) {
    final String prefix = "----------------------------------\n";
    final String sufix = "\n\n";
    final String newText = prefix + text + sufix;
    try {
      if (append) {
        final String desc = rawGetDescription();
        final String newDescription = desc + newText;
        rawWriteDescription(newDescription);
      }
      else {
        rawWriteDescription(newText);
      }
    }
    catch (IOException e) {
      final String fileName = getName();
      final String fmt = "writeDescription: falha ao escrever em '%s'!";
      final String err = String.format(fmt, fileName);
      throw new ServiceFailureException(err, e);
    }
  }

  /**
   * Faz a gravao de uma descrio <b>completa</b> de arquivo.
   *
   * @param newDescription
   * @throws IOException em caso de erro de I/O.
   */
  private void rawWriteDescription(final String newDescription)
    throws IOException {
    final File descFile = renamedDescriptionFile(file);
    final Server server = Server.getInstance();
    final Charset charset = server.getSystemDefaultCharset();
    try (FileOutputStream fileStream = new FileOutputStream(descFile);
         OutputStreamWriter writer = new OutputStreamWriter(fileStream,
      charset)) {
      writer.write(newDescription);
      writer.flush();
    }
  }

  /**
   * Atualiza a descrio. O novo texto pode ser acrescentado  descrio
   * existente ou pode substitui-la.
   *
   * @param text texto a ser anexado
   * @param descFile arquivo com a descrio
   * @throws IOException em caso de erro de I/O.
   */
  private void appendDirectlyToDescriptionFile(String text, File descFile)
    throws IOException {
      final Server server = Server.getInstance();
      final Charset charset = server.getSystemDefaultCharset();
    try (FileInputStream inStream = new FileInputStream(descFile);
         InputStreamReader inReader = new InputStreamReader(inStream, charset);
         BufferedReader inBuffReader = new BufferedReader(inReader);
         FileOutputStream outStream = new FileOutputStream(descFile);
         OutputStreamWriter outWriter = new OutputStreamWriter(outStream,
      charset)) {

      StringBuffer contentBuffer = new StringBuffer();
      String line = inBuffReader.readLine();
      while (line != null) {
        contentBuffer = contentBuffer.append(line + "\n");
        line = inBuffReader.readLine();
      }
      contentBuffer = contentBuffer.append(text + "\n");

      final String content = contentBuffer.toString();
      outWriter.write(content);
      outWriter.flush();
    }
  }

  /**
   * Ordena os filhos desse <code>ServerProjectFile</code> que representa um
   * diretrio.
   *
   * @param children - lista de filhos.
   */
  private void sortChildren(final List<ServerProjectFile> children) {
    Collections.sort(children, NAME_COMPARATOR);
  }

  /**
   * A partir do sistema de arquivos, reconstri os filhos desse
   * <code>ServerProjectFile</code> que representa um diretrio. So
   * instanciados novos <code>ServerProjectFiles</code> que passam a ser seus
   * filhos.
   *
   * @return array com os filhos do diretrio
   * @see #getNewTree(String)
   */
  private ServerProjectFile[] getNewTree() {
    return getNewTree(null);
  }

  /**
   * Gera um array com os filhos diretos deste diretrio, filtrando pelo tipo.
   *
   * @param type tipo dos arquivos (para filtragem). Pode ser <code>null</code>,
   *        o que significa que todos os arquivos/diretrios sero considerados.
   * @return array com os filhos do diretrio, filtrados pelo tipo
   * @see #getNewTree()
   */
  private ServerProjectFile[] getNewTree(String type) {
    if (!file.isDirectory()) {
      String errMsg = "ServerProjectFile:buildTree: " + file.getAbsolutePath()
        + " no  diretrio";
      throw new ServiceFailureException(errMsg);
    }
    File[] files = getFileListWithoutControlFiles();

    final List<ServerProjectFile> children = new ArrayList<ServerProjectFile>();
    for (final File child : files) {
      ServerProjectFile spf = ServerProjectFile.buildFromCache(child, projectId,
        this);

      /*
       * Tenta inferir o tipo pela extensao, caso o arquivo nao tenha o arquivo
       * de configuracao
       */
      File cfg = spf.getConfigFile();
      if ((cfg == null) || (!cfg.exists())) {
        ProjectFileTypeInfo typeInfo = ProjectService.getInstance()
          .getTypeFromExtension(spf.getName(), spf.isDirectory());
        spf.setType(typeInfo.getCode());
      }
      /*
       * apenas registra como filhos os arquivos com o tipo correto
       */
      if (type == null || type.equals(spf.getType())) {
        children.add(spf);
      }
    }
    sortChildren(children);
    return children.toArray(new ServerProjectFile[children.size()]);
  }

  /**
   * @return lista de filhos do diretrio corrente, desconsiderando os arquivos
   *         de controle; retorna <code>null</code> se o arquivo no  um
   *         diretrio
   */
  private File[] getFileListWithoutControlFiles() {
    return file.listFiles(new FilenameFilter() {

      @Override
      public boolean accept(File dir, String childName) {
        return !isControlledFileName(childName);
      }
    });
  }

  /**
   * Move o arquivo, o arquivo de configurao e o arquivo de descrio.
   *
   * @param newFile .
   * @return .
   */
  private boolean rename(File newFile) {
    ProjectService srv = ProjectService.getInstance();
    /* Tenta renomear o arquivo */
    File parentDir = new File(newFile.getParent());
    parentDir.mkdirs();
    if (!file.exists()) {
      Server.logSevereMessage("Impossvel localizar [" + file.getAbsolutePath()
        + "] para fazer o rename!");
      return false;
    }

    // Tenta mover o arquivo.
    if (!move(file, newFile)) {
      if (srv != null) {
        Server.logSevereMessage("Impossvel renomear de [" + file
          .getAbsolutePath() + "] para [" + newFile.getAbsolutePath() + "]");
        return false;
      }
    }
    /*
     * Tenta renomear o arquivo de configurao, se existir. Se no conseguir,
     * desfaz o anterior
     */
    File configFile = getConfigFile();
    File newConfigFile = renamedConfigFile(newFile);
    if (configFile.exists()) {
      if (!move(configFile, newConfigFile)) {
        String errMsg = "ServerProjectFile:rename: falha ao mover " + configFile
          .getAbsolutePath() + " para " + newConfigFile.getAbsolutePath();
        if (!move(newFile, file)) {
          errMsg = errMsg + "\nServerProjectFile:rename: falha ao mover "
            + newFile.getAbsolutePath() + " de volta para " + getAbsolutePath();
          configFile.delete();
          configFile = newConfigFile;
          writeConfiguration();
        }
        if (srv != null) {
          Server.logSevereMessage(errMsg);
        }
        return false;
      }
    }
    /*
     * Tenta renomear o arquivo de descrio, se existir. Se no conseguir,
     * abandona a descrio.
     */
    File descrFile = renamedDescriptionFile(file);
    File newDescrFile = renamedDescriptionFile(newFile);
    if (descrFile.exists()) {
      if (!move(descrFile, newDescrFile)) {
        String errMsg = "ServerProjectFile:rename: falha ao mover " + descrFile
          .getAbsolutePath() + " de volta para " + newDescrFile
            .getAbsolutePath();
        if (srv != null) {
          Server.logSevereMessage(errMsg);
        }
        descrFile.delete();
        return false;
      }
      descrFile = newDescrFile;
    }
    /* Atualiza a referncia para o arquivo. */
    file = newFile;
    return true;
  }

  /**
   * Tenta mover o arquivo com o mtodo renameTo da classe File. Caso no
   * consiga, usa o mtodo move da classe FileSystem.
   *
   * @param from .
   * @param to .
   * @return .
   */
  private boolean move(File from, File to) {
    if (!from.renameTo(to)) {
      return FileSystem.move(from, to);
    }
    return true;
  }

  /**
   * Copia o arquivo representado por este <code>ServerProjectFile</code> para
   * um outro arquivo passado como parmetro. Se for um arquivo comum, o
   * contedo  copiado juntamente com os arquivos de configurao e descrio,
   * se esses existirem. Se for um diretrio, esse mtodo  chamado
   * recursivamente para cada um de seus filhos. No caso de erro, a operao 
   * desfeita, removendo os arquivos que tiverem sido copiados.
   *
   * @param newFile O arquivo destino da cpia.
   * @return true se a cpia teve sucesso, ou false caso contrrio.
   */
  private boolean copyFile(File newFile) {
    if (file.isDirectory()) {
      return copyDirectory(newFile);
    }
    else {
      return copyRegularFile(newFile);
    }
  }

  private boolean copyDirectory(File newFile) {
    /* Se for diretrio, copia recursivamente seus filhos */
    if (!newFile.mkdirs()) {
      String errMsg =
        "ServerProjectFile:copyFile: erro na criao de diretrio " + newFile
          .getAbsolutePath();
      Server.logWarningMessage(errMsg);
      return false;
    }
    ServerProjectFile[] children = getChildren();
    boolean success = true;
    for (int i = 0, n = children.length; i < n; i++) {
      File newChildFile = new File(newFile, children[i].file.getName());
      success = children[i].copyFile(newChildFile);
      if (!success) {
        String errMsg = "ServerProjectFile:copyFile: erro na cpia do filho "
          + children[i].getAbsolutePath() + " para " + newChildFile
            .getAbsolutePath();
        Server.logWarningMessage(errMsg);
        break;
      }
    }

    if (success) {
      return copyConfigurationFiles(newFile);
    }
    return false;
  }

  /**
   * Copia um arquivo regular. Faz a cpia do arquivo, do arquivo de
   * configurao e do arquivo do descrio.
   *
   * @param newFile .
   * @return .
   */
  private boolean copyRegularFile(File newFile) {
    /* Copia o arquivo */
    if (!FileSystem.copyFile(file, newFile)) {
      String errMsg = "ServerProjectFile:copyFile: erro na cpia fsica de "
        + file.getAbsolutePath() + " para " + newFile.getAbsolutePath();
      Server.logWarningMessage(errMsg);
      return false;
    }
    /* Copia o arquivo de configurao */
    return copyConfigurationFiles(newFile);
  }

  /**
   * Copia do arquivo de configurao e do arquivo do descrio.
   *
   * @param newFile .
   * @return .
   */
  private boolean copyConfigurationFiles(File newFile) {
    final File parentFile = newFile.getParentFile();
    final String newFileName = newFile.getName();
    File configFile = getConfigFile();
    File newConfigFile = new File(parentFile, createConfigFileName(
      newFileName));
    if (configFile.exists() && !FileSystem.copyFile(configFile,
      newConfigFile)) {
      String errMsg = "ServerProjectFile:copyFile: erro na cpia de "
        + configFile.getAbsolutePath() + " para " + newConfigFile
          .getAbsolutePath();
      Server.logWarningMessage(errMsg);
      newConfigFile.delete();
      newFile.delete();
      return false;
    }
    /* Copia o arquivo de descrio. */
    return copyDescriptionFile(file, newFile);
  }

  /**
   * Copia o arquivo de descrio do arquivo de origem para o destino. O arquivo
   * de descrio da origem ser copiado para o arquivo de descrio do destino.
   *
   * @param from Arquivo de origem do qual ser obtido o arquivo de descrio.
   * @param to Arquivo de destino.
   * @return .
   */
  private boolean copyDescriptionFile(File from, File to) {
    File descrFile = renamedDescriptionFile(from);
    File newDescrFile = renamedDescriptionFile(to);
    if (descrFile.exists() && !FileSystem.copyFile(descrFile, newDescrFile)) {
      String errMsg = "ServerProjectFile:copyDescriptionFile: erro na cpia de "
        + descrFile.getAbsolutePath() + " para " + newDescrFile
          .getAbsolutePath();
      Server.logSevereMessage(errMsg);
      newDescrFile.delete();
      return false;
    }
    try {
      if (!newDescrFile.exists()) {
        newDescrFile.createNewFile();
      }
      User user = Service.getUser();
      String userName = "";
      if (user != null) {
        userName = user.getName();
      }
      String message = "Em " + Utilities.getFormattedDate((new Date())
        .getTime()) + "\nArquivo copiado por " + userName + ".\n";
      appendDirectlyToDescriptionFile(message, newDescrFile);
    }
    catch (Exception e) {
      String errMsg =
        "ServerProjectFile:copyDescriptionFile: erro ao escrever no arquivo "
          + "de configurao: " + newDescrFile.getAbsolutePath();
      Server.logSevereMessage(errMsg);
    }
    return true;
  }

  /**
   * Tenta remover o diretrio, removendo recursivamente todos os arquivos e
   * diretrios dentro deste.
   *
   * @param file Diretrio a ser removido (o mtodo no verifica null).
   * @return true caso o diretrio tenha sido removido com sucesso ou no
   *         exista.
   */
  private static boolean pruneDirectory(File file) {
    if (!file.exists()) {
      return true;
    }
    if (file.isDirectory()) {
      File[] fileList = file.listFiles();
      if (fileList.length > 0) {
        for (int i = 0; i < fileList.length; i++) {
          if (fileList[i].isDirectory()) {
            pruneDirectory(fileList[i]);
          }
          else {
            fileList[i].delete();
          }
        }
      }
      return file.delete();
    }
    return false;
  }

  /**
   * Remove o arquivo representado por este <code>ServerProjectFile</code>. Se
   * for um diretrio, esse mtodo  chamado recursivamente para cada um de seus
   * filhos. Se for um arquivo comum, apaga o arquivo. Em ambos os casos, o
   * arquivo  removido fisicamente do sistema de arquivos e, junto com ele, so
   * removidos tambm os arquivos de configurao e de descrio, se existirem.
   * Aps a remoo, esse objeto  removido da lista de filhos do seu pai.
   *
   * @param checkTemplate determina se os templates de projeto devem ser
   * consultados no processo de remoo.
   * @return flag indicando se o arquivo foi removido com sucesso
   */
  public boolean removeFile(boolean checkTemplate) {

    if (checkTemplate) {
      checkTemplateForRemoval();
    }
    /*
     * se file-locking est habilitado, s podemos seguir se obtivermos um lock
     * exclusivo sobre o arquivo
     */
    Object lockId = null;
    lockId = acquireLock();
    try {
      if (file.isFile()) {
        if (!file.delete()) {
          String errMsg = "ServerProjectFile:remove: " + getAbsolutePath()
            + " no pode ser " + "removido devido a "
            + "falha ao apagar o arquivo " + file.getAbsolutePath();
          Server.logWarningMessage(errMsg);
          return false;
        }
      }
      else if (file.isDirectory()) {
        if (!pruneDirectory(file)) {
          String errMsg = "ServerProjectFile:remove: " + getAbsolutePath()
            + " no pode ser " + "removido devido a "
            + "falha ao apagar o arquivo " + file.getAbsolutePath();
          Server.logWarningMessage(errMsg);
          return false;
        }
      }
      /* Remove o arquivo de configurao, se houver. */
      File configFile = getConfigFile();
      if (configFile != null) {
        configFile.delete();
      }
      /* Remove o arquivo de descrio, se houver */
      File descrFile = renamedDescriptionFile(file);
      if (descrFile.exists()) {
        descrFile.delete();
      }
      /* Retira o arquivo da cache */
      ServerProjectFile.deleteInCache(this);
      return true;
    }
    catch (Exception e) {
      Server.logSevereMessage(e.getMessage(), e);
      return false;
    }
    finally {
      if (lockId != null) {
        FileLockManager.getInstance().releaseLock(this, lockId);
      }
    }
  }

  /**
   * Testa a igualdade de dois arquivos.
   *
   * @param o Objeto a ser comparado com o atual
   * @return true se ambos os objetos forem considerados iguais
   */
  @Override
  public boolean equals(Object o) {
    if (!(o instanceof ServerProjectFile)) {
      return false;
    }
    ServerProjectFile other = (ServerProjectFile) o;
    return this.file.equals(other.file);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    return file.hashCode();
  }

  /**
   * Constri um <code>ServerProjectFile</code>. <b>ATENO:</b> Este mtodo
   * <u>no</u> pode ser chamado diretamente (somente atravs do mtodo esttico
   * <code>newServerProjectFile</code>) pois  fundamental garantir a unicidade
   * destes objetos no servidor.
   *
   * @param file arquivo local.
   * @param projectId id do projeto.
   * @param parent diretrio-pai.
   * @see ServerProjectFile#buildFromCache(File, Object, ServerProjectFile)
   */
  private ServerProjectFile(File file, Object projectId,
    ServerProjectFile parent) {
    super(file);
    this.projectId = projectId;
    this.parent = parent;
    readConfiguration();
  }

  /**
   * Muda a data de alterao deste arquivo
   *
   * @param date Data em long
   */
  public synchronized void setModificationDate(long date) {
    file.setLastModified(date);
  }

  /**
   * Traduz uma determinada chave.
   *
   * @param key chave a ser traduzida
   * @return chave traduzida
   */
  private String getString(String key) {
    ProjectService projectService = ProjectService.getInstance();
    return projectService.getString(key);
  }

  /**
   * Traduz uma determinada chave na sua string formatada inferindo o locale do
   * usurio (thread) que originou a chamada (ser usado um locale padro se
   * houver falha nesta identificao).
   *
   * @param key chave do bundle.
   * @param objects array de objetos de formatao.
   * @return string com o texto respectivo a chave.
   */
  private String getFormattedString(final String key, final Object... objects) {
    ProjectService projectService = ProjectService.getInstance();
    return projectService.getFormattedString(key, objects);
  }

  /**
   * Obtm as informaes sobre um arquivo atualizvel.
   *
   * @return As informaes sobre um arquivo atualizvel.
   */
  public UpdatableFileInfo getUpdatableFileInfo() {
    return this.updatableFileInfo;
  }

  /**
   * Define as informaes sobre um arquivo atualizvel.
   *
   * @param updatableFileInfo As informaes sobre um arquivo atualizvel.
   */
  public synchronized void setUpdatableFileInfo(
    UpdatableFileInfo updatableFileInfo) {
    if (updatableFileInfo == null) {
      throw new IllegalArgumentException(
        "As informaes sobre o arquivo no podem ter valor null.");
    }
    if (this.updatableFileInfo != null) {
      if (this.updatableFileInfo.equals(updatableFileInfo)) {
        return;
      }
    }
    this.updatableFileInfo = updatableFileInfo;
    writeConfiguration();
  }

  /**
   * Esta classe encapsula um listener remoto de lock de arquivo. O objetivo
   * dela  saber quando um lock de arquivo  obtido para notificar os projetos
   * abertos. No queremos que o FileLockManager acesse classe especficas do
   * servio de projetos para que ele possa ser migrado para outro pacote e seja
   * usado por outros servios.
   */
  class InternalFileLockListener implements FileLockListenerInterface {
    /**
     * Observador do lock cadastrado pela api do servio de projetos.
     */
    FileLockListenerInterface externalListener;
    SecureKey sessionKey;

    /**
     * Constri um observador interno de lock. Encapsula o observador externo.
     *
     * @param externalListener
     */
    InternalFileLockListener(FileLockListenerInterface externalListener,
      SecureKey sessionKey) {
      this.externalListener = externalListener;
      this.sessionKey = sessionKey;
    }

    /**
     * {@inheritDoc} Este mtodo j chamado em uma <code>Thread</code> separada
     * pelo FileLockManager. Portanto, se a notificao ficar presa em uma
     * chamada rmi, no interrompe a execuo do servio.
     */
    @Override
    public void fileLockExpired(Object lockId) throws RemoteException {
      if (externalListener != null) {
        externalListener.fileLockExpired(lockId);
      }
      if (this.sessionKey != null) {
        User user = LoginService.getInstance().getUserByKey(this.sessionKey);

        MessageService.getInstance().send(new Message(new FileLockEvent(lockId,
          false)), new String[] { (String) user.getId() });
      }
    }

    /**
     * {@inheritDoc} Este mtodo j chamado em uma <code>Thread</code> separada
     * pelo FileLockManager. Portanto, se a notificao ficar presa em uma
     * chamada rmi, no interrompe a execuo do servio.
     */
    @Override
    public void fileLocked(Object lockId) throws RemoteException {
      try {
        if (externalListener != null) {
          externalListener.fileLocked(lockId);
        }
        if (this.sessionKey != null) {
          User user = LoginService.getInstance().getUserByKey(this.sessionKey);

          MessageService.getInstance().send(new Message(new FileLockEvent(
            lockId, true)), new String[] { (String) user.getId() });
        }

        ServerProjectFile.this.notifyLockStatusChanged();
      }
      catch (RemoteException e) {
        Server.logSevereMessage(
          "Falha na notificao de lock obtido. Pedido de lock: "
            + ServerProjectFile.this.getName(), e);
        releaseLock(lockId);
      }
    }
  }

  /**
   * Indica se este arquivo  o root da arvore.
   *
   * @return Retorna boolean indicando se este arquivo  o root da arvore.
   */
  public boolean isRoot() {
    return parent == null;
  }

  /**
   * Lana uma {@link PermissionException exceo} caso o usurio corrente no
   * tenha permisso de escrita no arquivo. Esse mtodo s existe aqui para ser
   * testado na abertura de um canal, pois  o nico ponto onde o acesso  feito
   * diretamente a partir do cliente, via FTC. A classe ProjectService faz todos
   * os testes de permissionamento necessrios e como ela  o nico ponto de
   * acesso ao servio, nenhum teste aqui  necessrio.
   *
   * @throws PermissionException se o usurio logado no possuir permisso de
   *         escrita neste arquivo.
   */
  private void checkWriteAccess() {
    ServerProject project = ServerProject.getProject(projectId);
    if (!project.userHasAccessRW(Service.getUser().getId())) {
      String templateMsg;
      if (isDirectory()) {
        templateMsg = getString(
          "ServerProjectFile.error.access.write.directory");
      }
      else {
        templateMsg = getString("ServerProjectFile.error.access.write.file");
      }
      String message = String.format(templateMsg, FileUtils.joinPath(getPath()),
        project.getName());
      throw new PermissionException(message);
    }
  }

  /**
   * Obtm as informaes bsicas deste arquivo.
   *
   * @return as informaes.
   */
  private ProjectFileInfo getProjectFileInfo() {
    if (isRoot()) {
      return new ProjectFileInfo(new String[] { "" }, getType());
    }
    return new ProjectFileInfo(getPath(), getType());
  }

  /**
   * Verifica se o template permite a troca de tipo do arquivo.
   *
   * @param newType o novo tipo.
   */
  private void checkTemplateForTypeChange(String newType) {
    ServerProject project = ServerProject.getProject(projectId);
    for (ProjectTemplate template : project.getProjectTemplates()) {
      if (!template.canChangeType(this, newType)) {
        String key = "ServerProjectFile.error.template.newType";
        String path = FileUtils.joinPath(true, File.separatorChar, getPath());
        String infoMsg = getFormattedString(key, new Object[] { getName(),
            newType, path });
        throw (new PermissionException(infoMsg));
      }
    }
  }

  /**
   * Verifica se o template permite a criao do arquivo.
   *
   * @param directory o diretrio-pai do novo arquivo.
   * @param name o nome do arquivo a ser criado.
   * @param type o tipo do arquivo a ser criado.
   */
  private void checkTemplateForCreation(ServerProjectFile directory,
    String name, String type) {
    ServerProject project = ServerProject.getProject(projectId);
    for (ProjectTemplate template : project.getProjectTemplates()) {
      if (!template.canCreate(this, name, type)) {
        String key = "ServerProjectFile.error.template.create";
        String path = FileUtils.joinPath(true, File.separatorChar, directory
          .getPath());
        String infoMsg = getFormattedString(key, new Object[] { name, type,
            path });
        throw (new PermissionException(infoMsg));
      }
    }
  }

  /**
   * Verifica se o template permite a troca de nome do arquivo.
   *
   * @param newName o novo nome.
   */
  private void checkTemplateForRename(String newName) {
    ServerProject project = ServerProject.getProject(projectId);
    for (ProjectTemplate template : project.getProjectTemplates()) {
      if (!template.canRename(this, newName)) {
        String key = "ServerProjectFile.error.template.rename";
        String path = FileUtils.joinPath(true, File.separatorChar, getPath());
        String infoMsg = getFormattedString(key, new Object[] { getName(),
            newName, path });
        throw (new PermissionException(infoMsg));
      }
    }
  }

  /**
   * Verifica se o template permite a remoo do arquivo.
   */
  private void checkTemplateForRemoval() {
    ServerProject project = ServerProject.getProject(projectId);
    for (ProjectTemplate template : project.getProjectTemplates()) {
      if (!template.canDelete(this)) {
        String key = "ServerProjectFile.error.template.delete";
        String path = FileUtils.joinPath(true, File.separatorChar, getPath());
        String infoMsg = getFormattedString(key, new Object[] { getName(),
            path });
        throw (new PermissionException(infoMsg));
      }
    }
  }

}
