/**
 * $Id$
 */
package csbase.server.services.projectservice;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.List;

import csbase.logic.SecureKey;
import csbase.server.services.loginservice.LoginService;

/**
 * Esta classe representa o lock compartilhado de um arquivo. Possui uma lista
 * de <code>SessionReference</code> que indica se o lock foi obtido por uma
 * chamada do cliente, ou se o lock foi obtido por uma <code>Thread</code> que
 * no veio diretamente do cliente.
 */
class SharedFileLock implements FileLockInterface {

  /**
   * Lista de <code>SessionReference</code> para a sesso do usurio que
   * solicitou o lock.
   */
  private final List<SessionReference> lockList;

  /**
   * Indica se o lock possui uma sesso de usurio associada, ou se o lock foi
   * obtido por uma <code>Thread</code> que no veio diretamente do cliente.
   */
  private class SessionReference {
    /**
     * Identificador do lock.  usado para identificar que o desbloqueio  feito
     * por quem pediu o lock.
     */
    Object id;
    /**
     * Indica se o lock foi solicitado por um servio, a partir de uma
     * <code>Thread</code> do servidor. Quando o lock  obtido por uma
     * <code>Thread</code> que no vem diretamente do cliente, no h sesso
     * associada.
     */
    boolean serverRequested;
    /**
     * <code>WeakReference</code> para a sesso do usurio que obteve o lock.
     * Quando a sesso expira (o usurio deslogou), a refncia fica com null.
     */
    WeakReference<SecureKey> sessionKeyRef;

    /**
     * Constri uma <code>SessionReference</code> com uma
     * <code>WeakReference</code> para a sesso do usurio que obteve o lock.
     *
     * @param sessionKeyRef <code>WeakReference</code> para a sesso do
     *        usurio que obteve o lock
     * @param id identificador do pedido de lock
     */
    SessionReference(WeakReference<SecureKey> sessionKeyRef, Object id) {
      this.id = id;
      this.sessionKeyRef = sessionKeyRef;
      this.serverRequested = false;
    }

    /**
     * Constri uma <code>SessionReference</code>. Usado quando o pedido de
     * lock veio de uma chamada em que no h sesso do usurio associada.
     *
     * @param id identificador do pedido de lock
     */
    SessionReference(Object id) {
      this.id = id;
      this.serverRequested = true;
    }
  }

  /**
   * Constri um lock compartilhado
   *
   * @param sessionKey sesso do usurio que pediu o lock. Permite
   *        <code>null</code> quando no houver sesso associada.
   * @param id identificador do pedido de lock
   */
  SharedFileLock(SecureKey sessionKey, Object id) {
    lockList = new ArrayList<SessionReference>();
    newLocker(sessionKey, id);
  }

  /**
   * {@inheritDoc}
   */
  public int getLockRefCount(SecureKey sessionKey) {
    int refCount = 0;
    /*
     * fazemos o mesmo percorrimento feito em removeExpired(), mas aproveitamos
     * para contabilizar as referncias ao longo do caminho
     */
    List<SessionReference> cleanupList = new ArrayList<SessionReference>();
    for (SessionReference sessionReference : lockList) {
      if (sessionReference.serverRequested) {
        if (sessionKey == null) {
          refCount++;
        }
      }
      else {
        SecureKey skey = sessionReference.sessionKeyRef.get();
        if (skey == null) {
          // o lock expirou, separamos ele para ser removido
          cleanupList.add(sessionReference);
        }
        else if (skey.equals(sessionKey)) {
          refCount++;
        }
      }
    }
    // removemos os locks expirados
    lockList.removeAll(cleanupList);
    return refCount;
  }

  /**
   * {@inheritDoc}
   */
  public int getLockRefCount() {
    removeExpired();
    return lockList.size();
  }

  /**
   * {@inheritDoc}
   */
  public boolean hasExpired() {
    removeExpired();
    return lockList.isEmpty();
  }

  /**
   * Remove todas as referncias a sesses que j tenham expirado.
   */
  private void removeExpired() {
    List<SessionReference> cleanupList = new ArrayList<SessionReference>();
    for (SessionReference sessionRef : lockList) {
      if (!sessionRef.serverRequested) {
        SecureKey ref = sessionRef.sessionKeyRef.get();
        if (ref == null ||
          LoginService.getInstance().getUserByKey(ref) == null) {
          // o lock expirou, separamos ele para ser removido
          cleanupList.add(sessionRef);
        }
      }
    }
    // removemos os locks expirados
    lockList.removeAll(cleanupList);
  }

  /**
   * {@inheritDoc}
   */
  public LockStatus checkLockStatus(SecureKey sessionKey) {
    return checkLockStatus(sessionKey.toString());
  }

  /**
   * Procura por uma sesso especfica na lista de locks.
   *
   * @param lockId identificador do lock
   * @return a referncia fraca para a sesso procurada ou null caso esta no
   *         exista na lista
   */
  private SessionReference findSessionRef(Object lockId) {
    for (SessionReference sessionRef : lockList) {
      if (sessionRef.id.equals(lockId)) {
        return sessionRef;
      }
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  public boolean newLocker(SecureKey sessionKey, Object lockId) {
    removeExpired();
    if (sessionKey == null) {
      lockList.add(new SessionReference(lockId));
    }
    else {
      lockList.add(new SessionReference(
        new WeakReference<SecureKey>(sessionKey), lockId));
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  public boolean removeLocker(Object lockId) {
    SessionReference ref = findSessionRef(lockId);
    if (ref == null) {
      return false;
    }
    lockList.remove(ref);
    return true;
  }

  /**
   * {@inheritDoc}
   */
  public boolean isShared() {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  public LockStatus checkLockStatus(String sessionKey) {
    removeExpired();
    for (SessionReference sessionRef : lockList) {
      if (sessionRef.serverRequested) {
        if (sessionKey == null) {
          return LockStatus.LOCKED_BY_USER;
        }
      }
      else {
        SecureKey ref = sessionRef.sessionKeyRef.get();
        if (ref.toString().equals(sessionKey)) {
          return LockStatus.LOCKED_BY_USER;
        }
      }
    }
    if (lockList.isEmpty()) {
      return LockStatus.UNLOCKED;
    }
    return LockStatus.LOCKED_BY_OTHERS;
  }

  /**
   * Obtm os identificadores das referncias para este lock
   * @return array com os identificadores das referncias para este lock.
   */
  Object[] getIds() {
    Object[] ids = new Object[lockList.size()];
    for (int i=0; i<lockList.size(); i++) {
      ids[i] = lockList.get(i).id;
    }
    return ids;
  }

}
