/*
 * $Id: ApplicationService.java 175084 2016-07-18 17:29:34Z isabella $
 */
package csbase.server.services.applicationservice;

import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Hashtable;
import java.util.List;
import java.util.Properties;
import java.util.regex.Pattern;

import org.apache.commons.io.filefilter.FileFilterUtils;
import org.apache.commons.io.filefilter.IOFileFilter;
import org.apache.commons.io.monitor.FileAlterationListenerAdaptor;
import org.apache.commons.io.monitor.FileAlterationMonitor;
import org.apache.commons.io.monitor.FileAlterationObserver;

import tecgraf.ftc.common.logic.RemoteFileChannelInfo;
import tecgraf.javautils.core.io.FileUtils;
import csbase.exception.PermissionException;
import csbase.logic.User;
import csbase.logic.applicationservice.ApplicationCategory;
import csbase.logic.applicationservice.ApplicationRegistry;
import csbase.logic.applicationservice.ApplicationRegistryException;
import csbase.logic.applicationservice.ApplicationsReloadNotification;
import csbase.logic.applicationservice.ReloadApplicationsEvent;
import csbase.remote.ApplicationServiceInterface;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.TransactionCallbackInterface;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;
import csbase.server.TransactionManager;
import csbase.server.services.ftcservice.FTCService;
import csbase.server.services.notificationservice.NotificationService;

/**
 * Servio de gerncia de aplicaes.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class ApplicationService extends Service implements
  ApplicationServiceInterface {

  /**
   * Chave (propriedade) para indicar os diretrios de aplicaes.
   */
  private static final String APPLICATIONS_DIR_PROPERTY =
    "applications.directory";

  /**
   * Chave (propriedade) para indicar o diretrio de categorias
   */
  private static final String CATEGORIES_DIR_PROPERTY = "categories.directory";

  /**
   * Chave (propriedade) para indicar o intervalo de monitorao dos diretrios
   * de aplicaes.
   */
  private static final String REPOSITORY_MONITOR_INTERVAL_PROPERTY =
    "applications.directory.monitor.interval";

  /**
   * Registro de aplicaes
   */
  final private Hashtable<String, ApplicationRegistry> registries =
    new Hashtable<String, ApplicationRegistry>();

  /**
   * Registro de aplicaes
   */
  final private Hashtable<String, ApplicationCategory> categories =
    new Hashtable<String, ApplicationCategory>();

  /**
   * Transao que garante a execuo da gerncia do repositrio em modo
   * exclusivo.
   */
  private TransactionManager transaction;

  /**
   * Monitor de alteraes de arquivos dentro do repositrio de aplicaes.
   */
  private FileAlterationMonitor monitor;

  /**
   * Lista com os padres de nome que determina que uma classe pode ser
   * carregada no classloader da aplicao (alm do classpath da aplicao).
   */
  private final List<Pattern> classLoaderWhiteList = new ArrayList<Pattern>();

  /**
   * Lista com os padres de nome que determina que uma classe *no* pode ser
   * carregada no classloader da aplicao (alm do classpath da aplicao).
   */
  private final List<Pattern> classLoaderBlackList = new ArrayList<Pattern>();

  /**
   * Filtro para pegar diretrios vlidos nos repositrios.
   */
  final private FileFilter directoryFilter = new FileFilter() {
    @Override
    public boolean accept(final File file) {
      final boolean isDir = file.isDirectory();
      final String name = file.getName();
      final boolean nameOk = !name.startsWith(".");
      return isDir && nameOk;
    }
  };

  /**
   * Cria a imagem da aplicao.
   * 
   * @param id id da aplicao.
   * @param directory o diretrio da aplicao.
   * @param size tamanho.
   * @return a imagem <code>Icon</code>.
   * @throws ServerException em caso de falha na carga.
   */
  private byte[] buildImage(final String id, final File directory,
    final int size) throws ServerException {
    final List<String> extensions = getStringListProperty("image.extension");
    File file = null;
    for (final String ext : extensions) {
      final String fileName = id + "." + size + "." + ext;
      final File f = new File(directory, fileName);
      if (f.exists()) {
        file = f;
        break;
      }
    }

    if (file == null) {
      final String path = directory.getAbsolutePath();
      final String prefix = "No foi localizado cone de apl./cat. em: ";
      final String err = prefix + path + " - tamanho: " + size;
      Server.logWarningMessage(err);
      return null;
    }

    if (!file.canRead()) {
      final String path = directory.getAbsolutePath();
      final String prefix = "Sem permisso de leitura de: ";
      final String err = prefix + path + " - tamanho: " + size;
      throw new ServerException(err);
    }

    final String path = file.getAbsolutePath();
    final long length = file.length();
    if (length >= Integer.MAX_VALUE) {
      final String err = "cone muito grande para apl./cat.: " + path;
      throw new ServerException(err);
    }

    try {
      final int sz = (int) length;
      final byte[] array = new byte[sz];
      final FileInputStream fStream = new FileInputStream(file);
      fStream.read(array);
      fStream.close();
      return array;
    }
    catch (final IOException ioe) {
      final String fmt = "Falha de I/O para cone de aplicao/categoria: %s";
      final String err = String.format(fmt, path);
      throw new ServerException(err, ioe);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Hashtable<String, ApplicationCategory> getApplicationCategories() {
    return new Hashtable<String, ApplicationCategory>(categories);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Hashtable<String, ApplicationRegistry> getApplicationRegistries() {
    return new Hashtable<String, ApplicationRegistry>(registries);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ApplicationRegistry getApplicationRegistry(String id) {
    return registries.get(id);
  }

  /**
   * Obtm o repositrio em que o recurso se encontra.
   * 
   * @param appId identificador da aplicao.
   * @param resourcePath caminho do recurso a partir de um dos repositrios da
   *        aplicao.
   * 
   * @return o repositrio em que o recurso se encontra.
   */
  private File getApplicationDirectoryForResource(final String appId,
    final String[] resourcePath) {
    if (resourcePath == null || resourcePath.length < 1) {
      return null;
    }

    // Verificao que previne uma aplicao acessar o espao de outra.
    final String path = FileUtils.joinPath(File.separator, resourcePath);
    for (String p : resourcePath) {
      if (p.contains("..")) {
        final String fmt = "Recusado resource de aplicao %s com path %s!";
        final String err = String.format(fmt, appId, path);
        Server.logSevereMessage(err);
        return null;
      }
    }

    final List<String> repPaths = getApplicationRepositories();
    if (repPaths.size() == 0) {
      return null;
    }

    // Percorre a lista de diretrios, retornando o ltimo
    // que contm o recurso.
    File lastDir = null;
    for (final String repPath : repPaths) {
      final File repDir = new File(repPath);
      final File appDir = new File(repDir, appId);
      final File resource = new File(appDir, path);
      if (resource.exists()) {
        lastDir = appDir;
      }
    }
    return lastDir;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public RemoteFileChannelInfo getApplicationResource(final String appId,
    final String[] resourcePath) throws RemoteException {
    final File file = getApplicationInternalResource(appId, resourcePath);
    if (file == null) {
      return null;
    }

    try {
      final FTCService ftcService = FTCService.getInstance();
      return ftcService.createFileChannelInfo(file, true);
    }
    catch (Exception e) {
      throw new RemoteException(e.getMessage(), e);
    }
  }

  /**
   * Obtm um arquivo relativo ao caminho da aplicao.
   * 
   * @param appId o identificador da aplicao.
   * @param resourcePath o caminho para o arquivo.
   * @return o arquivo ou nulo, se este no for encontrado.
   */
  private File getApplicationInternalResource(final String appId,
    final String[] resourcePath) {
    final File directory =
      getApplicationDirectoryForResource(appId, resourcePath);
    if (directory == null || !directory.exists()) {
      return null;
    }

    final String path = FileUtils.joinPath(File.separator, resourcePath);
    final File file = new File(directory, path);
    if (!file.exists()) {
      return null;
    }

    File parent = file.getParentFile();
    while (parent != null && !parent.equals(directory)) {
      parent = parent.getParentFile();
    }
    if (parent == null) {
      return null;
    }
    return file;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean has2Update(final Object arg, final Object event) {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void initService() throws ServerException {
    transaction = new TransactionManager();
    loadClassLoaderConfiguration();
    loadRepositories();
    createRepositoryWatchers();
  }

  /**
   * Carrega as configuraes usadas para definir as classes disponveis no
   * {@link ClassLoader} das aplicaes carregadas dinamicamente, alm das que
   * foram definidas no classpath.
   */
  private void loadClassLoaderConfiguration() {
    List<String> whiteListProperties =
      getStringListProperty("classloader.whitelist");
    if (whiteListProperties != null) {
      for (String string : whiteListProperties) {
        Pattern pattern = Pattern.compile(string.trim());
        this.classLoaderWhiteList.add(pattern);
      }
    }
    List<String> blackListProperties =
      getStringListProperty("classloader.blacklist");
    if (blackListProperties != null) {
      for (String string : blackListProperties) {
        Pattern pattern = Pattern.compile(string.trim());
        this.classLoaderBlackList.add(pattern);
      }
    }
  }

  /**
   * Cria o monitor de alteraes no repositrio de aplicaes.
   * 
   * @throws ServerException em caso de erro.
   */
  private void createRepositoryWatchers() throws ServerException {
    final List<String> appPaths = getApplicationRepositories();
    long pollingInterval = getIntProperty(REPOSITORY_MONITOR_INTERVAL_PROPERTY);
    if (pollingInterval >= 0) {
      monitor = new FileAlterationMonitor(1000 * pollingInterval);
      for (final String appPath : appPaths) {
        IOFileFilter hiddenFilesFilter =
          FileFilterUtils.notFileFilter(FileFilterUtils.prefixFileFilter("."));
        FileAlterationObserver dirObserver =
          new FileAlterationObserver(new File(appPath), hiddenFilesFilter);
        dirObserver.addListener(new ReloadApplicationListener(appPath));
        monitor.addObserver(dirObserver);
      }
      try {
        monitor.start();
        Server
          .logInfoMessage("Iniciada a monitorao de arquivos no repositrio de aplicaes com intervalo de "
            + pollingInterval + " segundos");
      }
      catch (Exception e) {
        throw new ServerException(
          "Could not start file monitoring on applications repository", e);
      }
    }
  }

  /**
   * Obtm a lista de repositrios de algoritmos.
   * 
   * @return a lista com os caminhos para os repositrios.
   */
  private List<String> getApplicationRepositories() {
    final List<String> appPaths =
      getStringListProperty(APPLICATIONS_DIR_PROPERTY);
    return appPaths;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean reloadApplications() {
    checkApplicationsAdminPermission();
    TransactionCallbackInterface cb = new TransactionCallbackInterface() {
      @Override
      public boolean isAlive() {
        return true;
      }
    };
    final String uName = getUser().getName();
    Server.logInfoMessage("Pedido de recarga de aplicaes vinda de " + uName);
    try {
      if (!lock(cb)) {
        Server
          .logSevereMessage(getString("ApplicationService.error.applications.block_reload_failed"));
        return false;
      }
      loadRepositories();
      unlock(cb);
      Server.logInfoMessage("Recarga de aplicativos efetuada!");
      notifyObservers(new ReloadApplicationsEvent());
      final NotificationService notService = NotificationService.getInstance();
      ApplicationsReloadNotification notification =
        new ApplicationsReloadNotification(getSenderName());
      notService.notifyTo(null, notification);
      return true;
    }
    catch (ServerException e) {
      Server.logSevereMessage("Erro ao recarregar aplicaes", e);
      return false;
    }
  }

  /**
   * Carrega os dados dos repositrios de aplicaes.
   * 
   * @throws ServerException em caso de erro ao ler os repositrios.
   */
  private synchronized void loadRepositories() throws ServerException {
    registries.clear();
    categories.clear();
    loadApplicationsFromRepositories();
    loadCategoryFromRepository();
    checkConfiguration();
  }

  /**
   * Faz a anlise da configurao das aplicaes para detectar algum erro.
   */
  private void checkConfiguration() {
    final Collection<ApplicationRegistry> regs = registries.values();
    final List<ApplicationRegistry> invalidRegistries =
      new ArrayList<ApplicationRegistry>();
    for (ApplicationRegistry reg : regs) {
      if (!isValidRegistry(reg)) {
        invalidRegistries.add(reg);
      }
    }
    for (ApplicationRegistry applicationRegistry : invalidRegistries) {
      regs.remove(applicationRegistry);
    }
  }

  /**
   * Verifica a configurao da aplicao para detectar algum erro.
   * 
   * @param reg o registro da aplicao.
   * @return <code>true</code> se a configurao est correta ou
   *         <code>false</code> caso contrrio.
   */
  private boolean isValidRegistry(ApplicationRegistry reg) {
    try {
      reg.closeConfiguration();
      return true;
    }
    catch (ApplicationRegistryException e) {
      Server
        .logSevereMessage("Erro no registro da aplicao " + reg.getId(), e);
      return false;
    }
  }

  /**
   * Faz a carga de um registro de aplicao.
   * 
   * @param appDirectory o diretrio que representa a aplicao.
   * @throws ServerException em caso de falha.
   */
  private void loadApplication(final File appDirectory) throws ServerException {
    final String appPath = appDirectory.getAbsolutePath();
    final String id = appDirectory.getName();
    final ApplicationRegistry preExistant = registries.get(id);
    final ApplicationRegistry reg;
    if (preExistant != null) {
      final String err = "Detectada redefinio para aplicao: " + id;
      Server.logInfoMessage(err);
      reg = preExistant;
    }
    else {
      reg = new ApplicationRegistry(id, System.currentTimeMillis());
      registries.put(id, reg);
    }

    final String fileName = id + ".properties";
    final File propertyFile = new File(appDirectory, fileName);
    if (!propertyFile.exists()) {
      final String fmt = "No foram localizadas novas propriedades em %s";
      final String err = String.format(fmt, appPath);
      Server.logInfoMessage(err);
    }
    else {
      final Properties preLoaded;
      if (preExistant != null) {
        preLoaded = preExistant.getSpecificProperties();
      }
      else {
        preLoaded = new Properties();
      }
      final Properties properties = new Properties(preLoaded);
      try (InputStream in = new FileInputStream(propertyFile)) {
        properties.load(in);
        reg.setSpecificProperties(properties);
      }
      catch (final IOException e) {
        final String path = propertyFile.getAbsolutePath();
        final String err = "Falha de I/O na definio de aplicao: " + path;
        throw new ServerException(err, e);
      }
    }

    final byte[] img16 = buildImage(id, appDirectory, 16);
    if (img16 != null) {
      reg.setIconDefinition(img16);
    }

    final byte[] img32 = buildImage(id, appDirectory, 32);
    if (img32 != null) {
      reg.setImageDefinition(img32);
    }

    List<String> classPath = reg.getClasspath();
    if (classPath != null && !classPath.isEmpty()) {
      reg.setClasspathBaseDir(appDirectory);
      reg.setClassLoaderWhiteList(classLoaderWhiteList);
      reg.setClassLoaderBlackList(classLoaderBlackList);
    }
  }

  /**
   * Faz a carga do repositrio de aplicaes.
   * 
   * @param appDirectory repositrio
   * @throws ServerException em caso de falha.
   */
  private void loadApplicationRepository(final File appDirectory)
    throws ServerException {
    if (!appDirectory.exists()) {
      final String err = "No foi encontrado o diretrio de aplicaes: ";
      Server.logWarningMessage(err + appDirectory.getAbsolutePath());
      return;
    }

    if (!appDirectory.isDirectory()) {
      final String err = "No  diretrio (aplicaes): ";
      throw new ServerException(err + appDirectory.getAbsolutePath());
    }

    final File[] directories = appDirectory.listFiles(directoryFilter);
    for (final File directory : directories) {
      try {
        loadApplication(directory);
      }
      catch (ServerException e) {
        Server.logSevereMessage(
          "Erro ao carregar a aplicao " + directory.getName(), e);
      }
    }
  }

  /**
   * Adiciona uma novo registro de aplicao do repositrio especificado.
   * 
   * @param appId identificador da aplicao.
   */
  private synchronized void reloadApplication(String appId) {
    Server.logInfoMessage("Recarregando aplicao " + appId);
    try {
      if (registries.containsKey(appId)) {
        registries.remove(appId);
      }
      List<String> repositories = getApplicationRepositories();
      for (String repository : repositories) {
        File appDirectory = new File(repository, appId);
        if (appDirectory.exists()) {
          loadApplication(appDirectory);
        }
      }
      ApplicationRegistry registry = registries.get(appId);
      if (registry != null && !isValidRegistry(registry)) {
        registries.remove(appId);
      }
    }
    catch (ServerException e) {
      Server.logSevereMessage("Erro ao carregar a aplicao " + appId, e);
    }
  }

  /**
   * Faz a carga de ids de aplicao.
   * 
   * @param category categoria
   * @param properties propriedades
   * @throws ServerException em caso de categoria vazia.
   */
  private final void loadApplicationsIdsForCategory(
    final ApplicationCategory category, final Properties properties)
    throws ServerException {
    final String catId = category.getId();
    boolean noApplicationInside = true;
    for (int i = 1; true; i++) {
      final String key = catId + ".application." + i;
      final String v = properties.getProperty(key);
      if (v == null) {
        break;
      }
      final String appId = v.trim();
      Hashtable<String, ApplicationRegistry> regs = getApplicationRegistries();
      if (!regs.containsKey(appId)) {
        final String fmt =
          "Cat. de aplicaes <%s> no pode conter "
            + "aplicao cujo id  <%s> (inexistente). Descartando...";
        final String err = String.format(fmt, category.getId(), appId);
        Server.logSevereMessage(err);
      }
      else {
        category.addApplicationId(appId);
        noApplicationInside = false;
      }
    }
    if (noApplicationInside) {
      final String err = "Categoria de aplicaes vazia: " + catId;
      Server.logSevereMessage(err);
    }
  }

  /**
   * Faz a carga de um diretrio de aplicaes.
   * 
   * @param catDirectory o diretrio que representa a aplicao.
   * @throws ServerException em caso de falha.
   */
  private void loadCategory(final File catDirectory) throws ServerException {
    final String id = catDirectory.getName();
    final ApplicationCategory existant = categories.get(id);
    if (existant != null) {
      final String err = "Dupla de definio de grupo de applicao: " + id;
      throw new ServerException(err);
    }

    final ApplicationCategory category;
    final String fileName = id + ".properties";
    final File file = new File(catDirectory, fileName);
    final Properties properties = new Properties();
    try (InputStream in = new FileInputStream(file)) {
      properties.load(in);

      final byte[] img16 = buildImage(id, catDirectory, 16);
      final byte[] img32 = buildImage(id, catDirectory, 32);
      category = new ApplicationCategory(id, properties, img16, img32);
      loadApplicationsIdsForCategory(category, properties);
    }
    catch (final IOException e) {
      final String path = file.getAbsolutePath();
      final String err = "Falha de definio de aplicao: " + path;
      throw new ServerException(err, e);
    }

    categories.put(id, category);
    Server.logInfoMessage("Registrado categoria de aplicao: " + id + ".");
  }

  /**
   * Faz a carga de aplicaes dos repositrios
   * 
   * @throws ServerException em caso de falha.
   */
  private void loadApplicationsFromRepositories() throws ServerException {
    final List<String> appPaths = getApplicationRepositories();
    if (appPaths.size() == 0) {
      Server.logSevereMessage("No application repositories defined!");
    }

    for (final String appPath : appPaths) {
      loadApplicationsFromRepository(appPath);
    }

    if (registries.size() == 0) {
      Server.logWarningMessage("No applications defined!");
    }
  }

  /**
   * Carrega as aplicaes do repositrio especificado.
   * 
   * @param appPath path do repositrio
   * @throws ServerException em caso de erro ao ler o repositrio.
   */
  private void loadApplicationsFromRepository(final String appPath)
    throws ServerException {
    final File repository = new File(appPath);
    final String path = repository.getAbsolutePath();
    Server.logInfoMessage("Carregando repositrio de aplicaes: " + path);

    if (!repository.exists()) {
      final String err = "No foi encontrado o repositrio de aplicaes: ";
      throw new ServerException(err + path);
    }

    if (!repository.isDirectory()) {
      final String err = "Repositrio de apps: " + path + " no  diretrio!";
      throw new ServerException(err);
    }

    loadApplicationRepository(repository);
  }

  /**
   * Faz a carga dos repositrios de aplicaes.
   * 
   * @throws ServerException em caso de falha.
   */
  private void loadCategoryFromRepository() throws ServerException {
    final String catPath = getStringProperty(CATEGORIES_DIR_PROPERTY);

    final File repository = new File(catPath);
    final String path = repository.getAbsolutePath();
    Server.logInfoMessage("Carregando repositrio de categorias: " + path);

    if (!repository.exists()) {
      final String err = "No foi encontrado o diretrio de categorias: ";
      Server.logWarningMessage(err + repository.getAbsolutePath());
      return;
    }
    if (!repository.isDirectory()) {
      final String err = "No  diretrio (categorias): ";
      throw new ServerException(err + repository.getAbsolutePath());
    }

    final File[] directories = repository.listFiles(directoryFilter);
    for (final File directory : directories) {
      loadCategory(directory);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void shutdownService() {
    if (monitor != null) {
      try {
        monitor.stop();
      }
      catch (Exception e) {
        Server
          .logSevereMessage(
            "Erro ao interromper a monitorao de arquivos no repositrio de aplicaes",
            e);
      }
    }
    registries.clear();
    categories.clear();
  }

  /**
   * Constri a instncia do servio de gerncia de algoritmos
   * 
   * @throws ServerException caso ocorra algum problema durante a instanciao.
   */
  @SuppressWarnings("unused")
  public static void createService() throws ServerException {
    new ApplicationService();
  }

  /**
   * Obtm uma referncia para a instncia nica (singleton) do servio de
   * algoritmos.
   * 
   * @return referncia para a instncia nica do servio de algoritmos.
   */
  public static ApplicationService getInstance() {
    final String serviceName = ApplicationServiceInterface.SERVICE_NAME;
    return (ApplicationService) Service.getInstance(serviceName);
  }

  /**
   * Construtor
   * 
   * @throws ServerException em caso de falha.
   */
  protected ApplicationService() throws ServerException {
    super(ApplicationServiceInterface.SERVICE_NAME);
    ClientRemoteLocator.applicationService = this;
  }

  /**
   * Construtor
   * 
   * @param srvName nome do servip
   * @throws ServerException exceo.
   */
  protected ApplicationService(final String srvName) throws ServerException {
    super(srvName);
    ClientRemoteLocator.applicationService = this;
  }

  /**
   * Verifica se h transao ativa de gerncia do repositrio.
   * 
   * @return true caso haja transao ativa, false caso contrrio.
   */
  @Override
  public boolean isLocked() {
    return transaction.isLocked();
  }

  /**
   * Inicia uma transao de gerncia do repositrio de algoritmos (equivale 
   * obteno de um "lock").
   * 
   * @param cb Callback que  utilizada pelo servio para saber se quem efetuou
   *        o lock ainda est respondendo.
   * 
   * @return true caso o lock tenha sido feito com sucesso, false caso
   *         contrrio.
   */
  @Override
  public boolean lock(TransactionCallbackInterface cb) {
    checkApplicationsAdminPermission();
    return transaction.lock(cb);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void unlock(TransactionCallbackInterface cb) {
    checkApplicationsAdminPermission();
    transaction.unlock(cb);
  }

  /**
   * Verifica se  o administrador.
   */
  private void checkApplicationsAdminPermission() {
    User user = Service.getUser();
    if (!user.isAdmin()) {
      throw new PermissionException(
        getString("ApplicationService.error.applications.no_permission"));
    }
  }

  /**
   * Observador de recarga de aplicaes em um repositrio.
   */
  private final class ReloadApplicationListener extends
    FileAlterationListenerAdaptor {

    /**
     * Caminho do repositrio.
     */
    private final String repositoryPath;

    /**
     * Construtor.
     * 
     * @param repositoryPath caminho do repositrio.
     */
    private ReloadApplicationListener(String repositoryPath) {
      this.repositoryPath = repositoryPath;
    }

    /**
     * Obtm o caminho interno da aplicao.
     * 
     * @param file o diretrio modificado.
     * @return o caminho para dentro da aplicao.
     */
    public String[] getApplicationPath(File file) {
      String path = file.getPath();
      String newPath = path.replace(repositoryPath, "");
      String[] splitPath = FileUtils.splitPath(newPath);
      return splitPath;
    }

    /**
     * Obtm o identificador da aplicao.
     * 
     * @param file o diretrio modificado.
     * @return o identificador.
     */
    private String getApplicationId(File file) {
      String[] applicationPath = getApplicationPath(file);
      if (applicationPath != null && applicationPath.length > 0) {
        return applicationPath[0];
      }
      else {
        return null;
      }
    }

    /**
     * Determina se o diretrio especificado  a raiz da aplicao.
     * 
     * @param file o diretrio.
     * @return <code>true</code> se o diretrio for a raiz ou <code>false</code>
     *         caso contrrio.
     */
    private boolean isInApplicationDirectory(File file) {
      String[] applicationPath = getApplicationPath(file);
      if (applicationPath.length > 1) {
        return true;
      }
      return false;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void onFileChange(File file) {
      /*
       * Verifica se o arquivo modificado est dentro de um diretrio de
       * aplicao. Os casos de arquivos adicionados/removidos j so tratados
       * pelo mtodo #onDirectoryChange.
       */
      Server.logFineMessage("Arquivo modificado: " + file);
      if (isInApplicationDirectory(file)) {
        applicationWasUpdated(file);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void onDirectoryCreate(File directory) {
      /*
       * Se o diretrio criado for a raiz de uma aplicao,  porque o evento
       * realmente representa a adio de um novo diretrio de aplicao. Caso
       * contrrio, o evento pode ser ignorado pois o mtodo #onDirectoryChange
       * j ir tratar os demais casos.
       */
      Server.logFineMessage("Diretrio adicionado: " + directory);
      if (isInApplicationDirectory(directory)) {
        applicationWasUpdated(directory);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void onDirectoryDelete(File directory) {
      /*
       * Se o diretrio deletado for a raiz de uma aplicao,  porque o evento
       * realmente representa a remoo de um diretrio de aplicao. Caso
       * contrrio, o evento pode ser ignorado pois o mtodo #onDirectoryChange
       * j ir tratar os demais casos.
       */
      Server.logFineMessage("Diretrio removido: " + directory);
      if (isInApplicationDirectory(directory)) {
        applicationWasUpdated(directory);
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void onDirectoryChange(File directory) {
      Server.logInfoMessage("Modificado diretrio " + directory);
      applicationWasUpdated(directory);
    }

    /**
     * Lida com a atualizao do arquivo/diretrio da aplicao.
     * 
     * @param file o arquivo/diretrio atualizado.
     */
    private void applicationWasUpdated(File file) {
      String appId = getApplicationId(file);
      if (appId != null) {
        reloadApplication(appId);
      }
    }
  }
}
