/*
 * $Id: LocalAdministrationService.java 175407 2016-08-10 20:31:03Z clinio $
 */
package csbase.server.services.administrationservice;

import java.rmi.RemoteException;
import java.util.HashSet;
import java.util.List;

import javax.swing.ImageIcon;

import csbase.exception.PermissionException;
import csbase.exception.ServiceFailureException;
import csbase.exception.administration.AdministrationDeleteException;
import csbase.logic.AdministrationEvent;
import csbase.logic.MonitoredServerListener;
import csbase.logic.Permission;
import csbase.logic.Platform;
import csbase.logic.PlatformInfo;
import csbase.logic.Role;
import csbase.logic.RoleInfo;
import csbase.logic.ServerURI;
import csbase.logic.User;
import csbase.logic.UserGroup;
import csbase.logic.UserGroupInfo;
import csbase.logic.UserInfo;
import csbase.logic.UserPasswordPermission;
import csbase.logic.UserProjectInfo;
import csbase.logic.UserUpdate;
import csbase.remote.AdministrationServiceInterface;
import csbase.remote.ServerRemoteLocator;
import csbase.server.LocalServerRemoteMonitor;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.services.projectservice.ProjectService;

/**
 * A classe <code>LocalAdministrationService</code> implementa o servio de
 * administrao especfico para servidores locais, responsvel pela gerncia de
 * usurios, grupos de usurios, perfis e plataformas. Todas as alteraes so
 * delegadas para o servidor central. Alm disso, este servio observa mudanas
 * no servidor central para reflet-las localmente.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class LocalAdministrationService extends AdministrationService implements
  AdministrationServiceInterface, MonitoredServerListener {
  /**
   * Tempo de espera entre tentativas de lock do servidor central
   */
  private static final long LOCK_SLEEP_TIME = 1000; // 1 seg

  /**
   * Nmero mximo de tentativas de lock do servidor central
   */
  private static final long MAX_LOCK_TRIES = 50;

  /**
   * Observador remoto do AdministrationService do servidor central.
   */
  CentralAdministrationObserver observer;

  /**
   * Construtor
   * 
   * @throws ServerException
   */
  LocalAdministrationService() throws ServerException {
    super();
  }

  /**
   * Inicializao do servio. Coloca um observador remoto no
   * AdministrationService do servidor central para que as modificaes no
   * servidor central sejam refletidas neste servidor local. Tambm se insere
   * como observador do RemoteMonitor para poder sincronizar todos os dados
   * sempre que a conexo  reestabelecida.
   * 
   * @throws ServerException em caso de falha de inicializao do servio.
   */
  @Override
  public void initService() throws ServerException {
    Server.logInfoMessage("Iniciando servio local de administrao...");
    LocalServerRemoteMonitor monitor = LocalServerRemoteMonitor.getInstance();
    monitor.addListener(this);
    try {
      observer = new CentralAdministrationObserver(this);
      observer.install();
    }
    catch (RemoteException e) {
      throw new ServerException("Falha ao ativar AdministrationObserver", e);
    }
  }

  /**
   * Trmino do servio. Remove o observador remoto do servidor central e o
   * observador local do RemoteMonitor.
   */
  @Override
  public void shutdownService() {
    Server.logInfoMessage("Terminando servio local de administrao...");
    LocalServerRemoteMonitor monitor = LocalServerRemoteMonitor.getInstance();
    monitor.stop();
  }

  /**
   * Remoo de usurios de seus projetos
   * 
   * @param id id do usurio
   */
  private void removeUserFromOwnProjects(Object id) {
    ProjectService projService = ProjectService.getInstance();
    List<UserProjectInfo> projects = projService.getProjectsFromUser(id);
    if (projects != null) {
      for (UserProjectInfo info : projects) {
        projService.removeProject(info.getProjectId());
      }
    }
  }

  /**
   * Criao de usurio local.
   * 
   * @param user usurio
   */
  private void localCreateUser(User user) {
    try {
      user = userDAO.createUser(user.getUserInfo());
      notifyObservers(AdministrationEvent.CREATE, user);
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro na criao local do usurio " + user, e);
    }
  }

  /**
   * Modificao de usurio local.
   * 
   * @param user usurio
   */
  private void localModifyUser(User user) {
    try {
      user = userDAO.modifyUser(user.getId(), user.getUserInfo());
      notifyObservers(AdministrationEvent.MODIFY, user);
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro na alterao local do usurio " + user, e);
    }
  }

  /**
   * Remoo de usurio.
   * 
   * @param user usurio
   */
  private void localDeleteUser(User user) {
    try {
      Object id = user.getId();
      removeUserFromSharedProjects(id);
      removeUserFromOwnProjects(id);
      userDAO.deleteUser(id);
      notifyObservers(AdministrationEvent.DELETE, user);
    }
    catch (Exception e) {
      Server.logSevereMessage(
        "Erro na remoo local do usurio " + user + ": ", e);
    }
  }

  private void localCreateUserGroup(UserGroup userGroup) {
    try {
      userGroup = userGroupsDAO.createUserGroup(userGroup.getUserGroupInfo());
      notifyObservers(AdministrationEvent.CREATE, userGroup);
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro na criao local de UserGroup " + userGroup
        + ": ", e);
    }
  }

  private void localModifyUserGroup(UserGroup userGroup) {
    try {
      userGroup =
        userGroupsDAO.modifyUserGroup(userGroup.getId(), userGroup
          .getUserGroupInfo());
      notifyObservers(AdministrationEvent.MODIFY, userGroup);
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro na alterao local de UserGroup "
        + userGroup + ": ", e);
    }
  }

  private void localDeleteUserGroup(UserGroup userGroup) {
    try {
      userGroupsDAO.deleteUserGroup(userGroup.getId());
      notifyObservers(AdministrationEvent.DELETE, userGroup);
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro na remoo local de UserGroup " + userGroup
        + ": ", e);
    }
  }

  private void localCreateRole(Role role) {
    try {
      role = roleDAO.createRole(role.getRoleInfo());
      notifyObservers(AdministrationEvent.CREATE, role);
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro na criao local de perfil " + role + ": ",
        e);
    }
  }

  private void localModifyRole(Role role) {
    try {
      role = roleDAO.modifyRole(role.getId(), role.getRoleInfo());
      notifyObservers(AdministrationEvent.MODIFY, role);
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro na alterao local de perfil " + role
        + ": ", e);
    }
  }

  private void localDeleteRole(Role role) {
    try {
      roleDAO.deleteRole(role.getId());
      notifyObservers(AdministrationEvent.DELETE, role);
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro na remoo local de perfil " + role + ": ",
        e);
    }
  }

  private void localCreatePermission(Permission permission) {
    try {
      permissionDAO.createPermission(permission);
      notifyObservers(AdministrationEvent.CREATE, permission);
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro na criao local de permisso "
        + permission + ": ", e);
    }
  }

  private void localModifyPermission(Permission permission) {
    try {
      permission =
        permissionDAO.modifyPermission(permission.getId(), permission);
      notifyObservers(AdministrationEvent.MODIFY, permission);
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro na alterao local de permisso "
        + permission + ": ", e);
    }
  }

  private void localDeletePermission(Permission permission) {
    try {
      permissionDAO.deletePermission(permission.getId());
      notifyObservers(AdministrationEvent.DELETE, permission);
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro na remoo local de permisso "
        + permission + ": ", e);
    }
  }

  private void localCreatePlatform(Platform platform) {
    try {
      platform = platformDAO.createPlatform(platform.getPlatformInfo());
      notifyObservers(AdministrationEvent.CREATE, platform);
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro na criao local de plataforma " + platform
        + ": ", e);
    }
  }

  private void localModifyPlatform(Platform platform) {
    try {
      platform =
        platformDAO
          .modifyPlatform(platform.getId(), platform.getPlatformInfo());
      notifyObservers(AdministrationEvent.MODIFY, platform);
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro na alterao local de plataforma "
        + platform + ": ", e);
    }
  }

  private void localDeletePlatform(Platform platform) {
    try {
      platformDAO.deletePlatform(platform.getId());
      notifyObservers(AdministrationEvent.DELETE, platform);
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro na remoo local de plataforma " + platform
        + ": ", e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void notifyConnectionLost(ServerURI serverURI) {
    Server
      .logInfoMessage("A comunicao com o servidor central foi interrompida");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void notifyLoggedIn(ServerURI serverURI) {
    /** nada a fazer */
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void notifyLoggedOut(ServerURI serverURI) {
    /** nada a fazer */
  }

  /**
   * {@inheritDoc} Inicia uma thread de sincronizao de bases com o servidor
   * central quando a conexo com o servidor  restabelecida.
   */
  @Override
  public void notifyConnectionReestablished(ServerURI serverURI) {
    Server
      .logInfoMessage("A comunicao com o servidor central foi restabelecida");
    (new Thread() {
      @Override
      public void run() {
        Server
          .logInfoMessage("Iniciada sincronizao de bases com o servidor central.");
        try {
          // Indica ao observador para entrar em lock
          observer.setLockMode(true);

          // Obtm o servio central de administrao
          AdministrationServiceInterface central =
            ServerRemoteLocator.administrationService;

          // Trava o servio central de administrao
          int numTries = 0;
          while (!central.lock(observer)) {
            if (++numTries > MAX_LOCK_TRIES) {
              Server
                .logSevereMessage("Falha no lock do servidor central na sincronizao de bases.");
              return;
            }
            try {
              Thread.sleep(LOCK_SLEEP_TIME);
            }
            catch (InterruptedException e) {
            }
          }

          // Trava o servio local de administrao
          numTries = 0;
          while (!lock(observer)) {
            if (++numTries > MAX_LOCK_TRIES) {
              central.unlock(observer);
              Server
                .logSevereMessage("Falha no lock do servidor local na sincronizao de bases.");
              return;
            }
            try {
              Thread.sleep(LOCK_SLEEP_TIME);
            }
            catch (InterruptedException e) {
            }
          }

          // Obtm todos os dados do servidor central, para evitar que uma
          // queda na comunicao, durante a sincronizao, deixe o servidor
          // com as bases inconsistentes. A atualizao de usurios pode ir
          // ao servidor buscar usurios especficos (novos ou alterados).
          // Nesse caso, se houver erro, os usurios sero atualizados na
          // prxima sincronizao, mas as bases ficam, de certa forma,
          // consistentes: usurio que sobrem com referncias a objetos
          // inexistentes podero enfrentar dificuldades mas isso  razovel.
          List<Platform> newPlatforms = central.getAllPlatforms();
          List<Permission> newPermissions =
            central.getAllPermissionsWithPasswords();
          List<Role> newRoles = central.getAllRoles();
          List<UserGroup> newUserGroups = central.getAllUserGroups();
          List<UserUpdate> userUpdates = central.getAllUserUpdates();

          // Sincroniza os bancos de dados. A ordem de execuo  importante
          // para garantir que no haja inconsistncias.
          Server.logInfoMessage("Sincronizando base de plataformas...");
          synchronizePlatformDatabase(newPlatforms);
          Server.logInfoMessage("Sincronizando base de permisses...");
          synchronizePermissionDatabase(newPermissions);
          Server.logInfoMessage("Sincronizando base de perfis...");
          synchronizeRoleDatabase(newRoles);
          Server.logInfoMessage("Sincronizando base de grupos de usurios...");
          synchronizeUserGroupDatabase(newUserGroups);
          Server.logInfoMessage("Sincronizando base de usurios...");
          synchronizeUserDatabase(userUpdates, central);

          // Destrava ambos os servios de administrao
          unlock(observer);
          central.unlock(observer);
        }
        catch (Exception e) {
          Server.logSevereMessage("Erro na sincronizao de bases: ", e);
          return;
        }
        finally {
          // Indica ao observador para sair em lock
          observer.setLockMode(false);
        }
        Server
          .logInfoMessage("Terminada sincronizao de bases com o servidor central.");
      }
    }).start();
  }

  private void synchronizePlatformDatabase(List<Platform> newPlatforms) {
    try {
      HashSet<Object> allPlatformIds = new HashSet<Object>(newPlatforms.size());
      for (int i = 0; i < newPlatforms.size(); i++) {
        Platform newPlatform = newPlatforms.get(i);
        Object id = newPlatform.getId();
        Platform oldPlatform = getPlatform(id);
        if (oldPlatform == null) {
          localCreatePlatform(newPlatform);
        }
        else if (!newPlatform.equalContents(oldPlatform)) {
          localModifyPlatform(newPlatform);
        }
        allPlatformIds.add(id);
      }
      List localPlatforms = getAllPlatforms();
      for (int i = 0; i < localPlatforms.size(); i++) {
        Platform platform = (Platform) localPlatforms.get(i);
        if (!allPlatformIds.contains(platform.getId())) {
          localDeletePlatform(platform);
        }
      }
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro na sincronizao de plataformas: ", e);
    }
  }

  private void synchronizePermissionDatabase(List<Permission> newPermissions) {
    try {
      HashSet<Object> allPermissionIds =
        new HashSet<Object>(newPermissions.size());
      for (int i = 0; i < newPermissions.size(); i++) {
        Permission newPermission = newPermissions.get(i);
        Object id = newPermission.getId();
        if (newPermission instanceof UserPasswordPermission) {
          UserPasswordPermission upp = (UserPasswordPermission) newPermission;
          upp.setLocalPassword(upp.getPassword());
        }
        Permission oldPermission = getPermission(id);
        if (oldPermission == null) {
          localCreatePermission(newPermission);
        }
        else if (!newPermission.equalContents(oldPermission)) {
          localModifyPermission(newPermission);
        }
        allPermissionIds.add(id);
      }
      List localPermissions = getAllPermissions();
      for (int i = 0; i < localPermissions.size(); i++) {
        Permission permission = (Permission) localPermissions.get(i);
        if (!allPermissionIds.contains(permission.getId())) {
          localDeletePermission(permission);
        }
      }
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro na sincronizao de permisses: ", e);
    }
  }

  private void synchronizeRoleDatabase(List<Role> newRoles) {
    try {
      HashSet<Object> allRoleIds = new HashSet<Object>(newRoles.size());
      for (int i = 0; i < newRoles.size(); i++) {
        Role newRole = newRoles.get(i);
        Object id = newRole.getId();
        Role oldRole = getRole(id);
        if (oldRole == null) {
          localCreateRole(newRole);
        }
        else if (!newRole.equalContents(oldRole)) {
          localModifyRole(newRole);
        }
        allRoleIds.add(id);
      }
      List localRoles = getAllRoles();
      for (int i = 0; i < localRoles.size(); i++) {
        Role role = (Role) localRoles.get(i);
        if (!allRoleIds.contains(role.getId())) {
          localDeleteRole(role);
        }
      }
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro na sincronizao de perfis: ", e);
    }
  }

  private void synchronizeUserGroupDatabase(List<UserGroup> newUserGroups) {
    try {
      HashSet<Object> allUserGroupIds =
        new HashSet<Object>(newUserGroups.size());
      for (int i = 0; i < newUserGroups.size(); i++) {
        UserGroup newUserGroup = newUserGroups.get(i);
        Object id = newUserGroup.getId();
        UserGroup oldUserGroup = getUserGroup(id);
        if (oldUserGroup == null) {
          localCreateUserGroup(newUserGroup);
        }
        else if (!newUserGroup.equalContents(oldUserGroup)) {
          localModifyUserGroup(newUserGroup);
        }
        allUserGroupIds.add(id);
      }
      List localUserGroups = getAllUserGroups();
      for (int i = 0; i < localUserGroups.size(); i++) {
        UserGroup userGroup = (UserGroup) localUserGroups.get(i);
        if (!allUserGroupIds.contains(userGroup.getId())) {
          localDeleteUserGroup(userGroup);
        }
      }
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro na sincronizao de UserGroups: ", e);
    }
  }

  private void synchronizeUserDatabase(List<UserUpdate> userUpdates,
    AdministrationServiceInterface central) {
    try {
      HashSet<Object> allUserIds = new HashSet<Object>(userUpdates.size());
      for (int i = 0; i < userUpdates.size(); i++) {
        UserUpdate update = userUpdates.get(i);
        Object id = update.getId();
        User localUser = getUser(id);
        if (localUser == null) {
          localCreateUser(central.getUser(id));
        }
        else if (localUser.isAdmin()) {
          // No caso do administrador, fora a atualizao pois ele pode ter
          // sido modificado localmente para permitir o boostrap da conexo
          // com o servidor central.
          localModifyUser(central.getUser(id));
        }
        else if (localUser.getLastUpdate() != update.getLastUpdate()) {
          // Teoricamente, a atualizao s precisaria ser feita caso a data
          // de modificao local fosse menor que a data de modificao no
          // servidor central. Porm, ao usarmos o !=, conseguimos obter um
          // resultado mais correto no caso de uso indevido: quando a ligao
          // com o servidor central  desativada, o usurio  modificado no
          // servidor local, apenas, e ento a relao de dependncia 
          // reestabelecida. Note que isso nunca deveria ser feito...
          localModifyUser(central.getUser(id));
        }
        allUserIds.add(id);
      }
      List localUsers = getAllUsers();
      for (int i = 0; i < localUsers.size(); i++) {
        User user = (User) localUsers.get(i);
        if (!allUserIds.contains(user.getId())) {
          localDeleteUser(user);
        }
      }
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro na sincronizao de usurios: ", e);
    }
  }

  //
  // Atualizaes de usurios por notificaes vindas do servidor central.
  //
  void updateUser(AdministrationEvent event) throws Exception {
    User user = (User) event.item;
    switch (event.type) {
      case AdministrationEvent.CREATE:
        localCreateUser(user);
        break;
      case AdministrationEvent.MODIFY:
        localModifyUser(user);
        break;
      case AdministrationEvent.DELETE:
        localDeleteUser(user);
        break;
      default:
        Server.logSevereMessage("Evento no reconhecido: " + event + " - type="
          + event.type);
        break;
    }
  }

  void updateUserGroup(AdministrationEvent event) throws Exception {
    UserGroup userGroup = (UserGroup) event.item;
    switch (event.type) {
      case AdministrationEvent.CREATE:
        localCreateUserGroup(userGroup);
        break;
      case AdministrationEvent.MODIFY:
        localModifyUserGroup(userGroup);
        break;
      case AdministrationEvent.DELETE:
        localDeleteUserGroup(userGroup);
        break;
      default:
        Server.logSevereMessage("Evento no reconhecido: " + event + " - type="
          + event.type);
        break;
    }
  }

  void updateRole(AdministrationEvent event) throws Exception {
    Role role = (Role) event.item;
    switch (event.type) {
      case AdministrationEvent.CREATE:
        localCreateRole(role);
        break;
      case AdministrationEvent.MODIFY:
        localModifyRole(role);
        break;
      case AdministrationEvent.DELETE:
        localDeleteRole(role);
        break;
      default:
        Server.logSevereMessage("Evento no reconhecido: " + event + " - type="
          + event.type);
        break;
    }
  }

  void updatePermission(AdministrationEvent event) throws Exception {
    Permission permission = (Permission) event.item;
    if (permission instanceof UserPasswordPermission) {
      permission =
        ServerRemoteLocator.administrationService
          .getPermissionWithPassword(permission.getId());
    }
    switch (event.type) {
      case AdministrationEvent.CREATE:
        localCreatePermission(permission);
        break;
      case AdministrationEvent.MODIFY:
        localModifyPermission(permission);
        break;
      case AdministrationEvent.DELETE:
        localDeletePermission(permission);
        break;
      default:
        Server.logSevereMessage("Evento no reconhecido: " + event + " - type="
          + event.type);
        break;
    }
  }

  void updatePlatform(AdministrationEvent event) throws Exception {
    Platform platform = (Platform) event.item;
    switch (event.type) {
      case AdministrationEvent.CREATE:
        localCreatePlatform(platform);
        break;
      case AdministrationEvent.MODIFY:
        localModifyPlatform(platform);
        break;
      case AdministrationEvent.DELETE:
        localDeletePlatform(platform);
        break;
      default:
        Server.logSevereMessage("Evento no reconhecido: " + event + " - type="
          + event.type);
        break;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public User createUser(UserInfo info) {
    checkCreateUserPermission();
    try {
      return ServerRemoteLocator.administrationService.createUser(info);
    }
    catch (RemoteException e) {
      throw new ServiceFailureException(
        getString("LocalAdministrationService.error.conecting.central.server"));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ImageIcon getPhoto(Object id) {
    try {
      return ServerRemoteLocator.administrationService.getPhoto(id);
    }
    catch (RemoteException e) {
      throw new ServiceFailureException(
        getString("LocalAdministrationService.error.conecting.central.server"));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public User modifyUser(Object id, UserInfo info) throws PermissionException,
    RemoteException {
    checkModifyUserPermission(id, info);
    try {
      return ServerRemoteLocator.administrationService.modifyUser(id, info);
    }
    catch (RemoteException e) {
      throw new ServiceFailureException(
        getString("LocalAdministrationService.error.conecting.central.server"));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void deleteUser(Object id) throws AdministrationDeleteException {
    checkDeleteUserPermission();
    checkNoProjects(id);
    try {
      ServerRemoteLocator.administrationService.deleteUser(id);
    }
    catch (RemoteException e) {
      throw new ServiceFailureException(
        getString("LocalAdministrationService.error.conecting.central.server"));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public UserGroup createUserGroup(UserGroupInfo info) {
    checkCreateUserGroupPermission();
    try {
      return ServerRemoteLocator.administrationService.createUserGroup(info);
    }
    catch (RemoteException e) {
      throw new ServiceFailureException(
        getString("LocalAdministrationService.error.conecting.central.server"));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public UserGroup modifyUserGroup(Object id, UserGroupInfo info) {
    checkModifyUserGroupPermission();
    try {
      return ServerRemoteLocator.administrationService
        .modifyUserGroup(id, info);
    }
    catch (RemoteException e) {
      throw new ServiceFailureException(
        getString("LocalAdministrationService.error.conecting.central.server"));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void deleteUserGroup(Object id) throws AdministrationDeleteException {
    checkDeleteUserGroupPermission();
    try {
      ServerRemoteLocator.administrationService.deleteUserGroup(id);
    }
    catch (RemoteException e) {
      throw new ServiceFailureException(
        getString("LocalAdministrationService.error.conecting.central.server"));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Role createRole(RoleInfo info) {
    checkCreateRolePermission();
    try {
      return ServerRemoteLocator.administrationService.createRole(info);
    }
    catch (RemoteException e) {
      throw new ServiceFailureException(
        getString("LocalAdministrationService.error.conecting.central.server"));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Role modifyRole(Object id, RoleInfo info) {
    checkModifyRolePermission();
    try {
      return ServerRemoteLocator.administrationService.modifyRole(id, info);
    }
    catch (RemoteException e) {
      throw new ServiceFailureException(
        getString("LocalAdministrationService.error.conecting.central.server"));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void deleteRole(Object id) throws AdministrationDeleteException {
    checkDeleteRolePermission();
    try {
      ServerRemoteLocator.administrationService.deleteRole(id);
    }
    catch (RemoteException e) {
      throw new ServiceFailureException(
        getString("LocalAdministrationService.error.conecting.central.server"));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Permission createPermission(Permission permission) {
    checkCreatePermissionPermission();
    try {
      return ServerRemoteLocator.administrationService
        .createPermission(permission);
    }
    catch (RemoteException e) {
      throw new ServiceFailureException(
        getString("LocalAdministrationService.error.conecting.central.server"));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Permission modifyPermission(Object id, Permission permission) {
    checkModifyPermissionPermission();
    try {
      return ServerRemoteLocator.administrationService.modifyPermission(id,
        permission);
    }
    catch (RemoteException e) {
      throw new ServiceFailureException(
        getString("LocalAdministrationService.error.conecting.central.server"));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void deletePermission(Object id) throws AdministrationDeleteException {
    checkDeletePermissionPermission();
    try {
      ServerRemoteLocator.administrationService.deletePermission(id);
    }
    catch (RemoteException e) {
      throw new ServiceFailureException(
        getString("LocalAdministrationService.error.conecting.central.server"));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Platform createPlatform(PlatformInfo info) {
    checkCreatePlatformPermission();
    try {
      return ServerRemoteLocator.administrationService.createPlatform(info);
    }
    catch (RemoteException e) {
      throw new ServiceFailureException(
        getString("LocalAdministrationService.error.conecting.central.server"));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Platform modifyPlatform(Object id, PlatformInfo info) {
    checkModifyPlatformPermission();
    try {
      return ServerRemoteLocator.administrationService.modifyPlatform(id, info);
    }
    catch (RemoteException e) {
      throw new ServiceFailureException(
        getString("LocalAdministrationService.error.conecting.central.server"));
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void deletePlatform(Object id) {
    checkDeletePlatformPermission();
    try {
      ServerRemoteLocator.administrationService.deletePlatform(id);
    }
    catch (RemoteException e) {
      throw new ServiceFailureException(
        getString("LocalAdministrationService.error.conecting.central.server"));
    }
  }
}
