/**
 * $Id$
 */
package csbase.server.services.openbusservice;

import static java.lang.Byte.parseByte;

import java.io.IOException;
import java.security.GeneralSecurityException;
import java.security.interfaces.RSAPrivateKey;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import csbase.logic.ExternalUser;
import org.omg.CORBA.COMM_FAILURE;
import org.omg.CORBA.NO_PERMISSION;
import org.omg.CORBA.OBJECT_NOT_EXIST;
import org.omg.CORBA.ORB;
import org.omg.CORBA.TRANSIENT;
import org.omg.CORBA.ORBPackage.InvalidName;
import org.omg.PortableServer.POA;
import org.omg.PortableServer.POAHelper;
import org.omg.PortableServer.Servant;
import org.omg.PortableServer.POAManagerPackage.AdapterInactive;
import org.omg.PortableServer.POAManagerPackage.State;

import scs.core.ComponentContext;
import scs.core.ComponentId;
import scs.core.IComponent;
import scs.core.exception.SCSException;
import tecgraf.openbus.CallerChain;
import tecgraf.openbus.Connection;
import tecgraf.openbus.InvalidLoginCallback;
import tecgraf.openbus.OpenBusContext;
import tecgraf.openbus.SharedAuthSecret;
import tecgraf.openbus.core.ORBInitializer;
import tecgraf.openbus.core.v2_0.services.ServiceFailure;
import tecgraf.openbus.core.v2_0.services.access_control.AccessDenied;
import tecgraf.openbus.core.v2_0.services.access_control.LoginInfo;
import tecgraf.openbus.core.v2_0.services.offer_registry.InvalidProperties;
import tecgraf.openbus.core.v2_0.services.offer_registry.InvalidService;
import tecgraf.openbus.core.v2_0.services.offer_registry.OfferRegistry;
import tecgraf.openbus.core.v2_0.services.offer_registry.ServiceOffer;
import tecgraf.openbus.core.v2_0.services.offer_registry.ServiceOfferDesc;
import tecgraf.openbus.core.v2_0.services.offer_registry.ServiceProperty;
import tecgraf.openbus.core.v2_0.services.offer_registry.UnauthorizedFacets;
import tecgraf.openbus.exception.AlreadyLoggedIn;
import tecgraf.openbus.exception.CryptographyException;
import tecgraf.openbus.exception.InvalidEncodedStream;
import tecgraf.openbus.exception.InvalidLoginProcess;
import tecgraf.openbus.exception.InvalidPropertyValue;
import tecgraf.openbus.security.Cryptography;
import csbase.exception.OperationFailureException;
import csbase.exception.PermissionException;
import csbase.exception.ServiceFailureException;
import csbase.logic.BusInfo;
import csbase.logic.User;
import csbase.logic.openbus.OpenBusLoginToken;
import csbase.remote.OpenBusServiceInterface;
import csbase.server.Server;
import csbase.server.ServerException;
import csbase.server.Service;
import csbase.server.services.openbusservice.OpenbusStatus.ConnectionState;

// TODO Documentar melhor o uso do OpenBus Service

/**
 * Representa o servio que trata da conexo do CSBase com o OpenBus.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class OpenBusService extends Service implements
  OpenBusServiceInterface {
  /**
   * Chave da propriedade de busca por ID da oferta de servio
   */
  private static final String OPENBUS_OFFER_ID_KEY = "openbus.offer.id";

  /**
   * Enumerao com os critrios possveis para agrupar servios no retorno do
   * mtodo <code>findServiceList</code>
   */
  private static enum FIND {
    ALL,
    FIRST
  };

  /**
   * Tempo de validade da conexo usada para gerar um token do cliente (em
   * milis).
   */
  private final long TOKEN_EXPIRATION_TIME = 3 * 60 * 1000;
  /**
   * A instncia nica do servio do OpenBus.
   */
  private static OpenBusService instance;
  /**
   * Os componentes cadastrados no servio.
   */
  private final Set<ComponentEntry> components;
  /**
   * Os componentes que no foram registrado devido a falha.
   */
  private final Set<ComponentEntry> failedComponents;
  /**
   * As facetas dos servios que publicam ofertas no OpenBus.
   */
  private final Hashtable<String, OpenBusServiceFacet> facets;
  /**
   * <i>Thread</i> responsvel pelo ORB.
   */
  private ORBThread orbThread;
  /**
   * Informaes sobre a localizao do OpenBus.
   */
  private BusInfo busInfo;
  /**
   * O nome da entidade representada por este servidor no OpenBus.
   */
  private String entityName;
  /**
   * A chave privada deste servidor.
   */
  private RSAPrivateKey privateKey;
  /**
   * Flag que indica se houve falha na inicializao do ACS.
   */
  private boolean acsInitFailed;
  /**
   * Contexto do OpenBus
   */
  private OpenBusContext context;
  /**
   * Propriedade das conexes
   */
  private Properties connectionProps;
  /**
   * A ltima mensagem enviada para o log, para evitar duplicao.
   */
  private String lastLogMessage;
  /**
   * Nmero de vezes que a ltima mensagem seria repetida no log.
   */
  private int numLastLogMessage;
  /**
   * Indica o status corrente da conexo com o barramento
   */
  private static OpenbusStatus openbusStatus;

  /**
   * Cria o servio do OpenBus.
   * 
   * @throws ServerException Caso ocorra algum erro na criao do servio.
   */
  private OpenBusService() throws ServerException {
    super(SERVICE_NAME);

    this.components = new HashSet<ComponentEntry>();
    this.failedComponents =
      Collections.synchronizedSet(new HashSet<ComponentEntry>());
    this.facets = new Hashtable<String, OpenBusServiceFacet>();

    this.setEnabled(this.getBooleanProperty("enabled"));

    if (this.isEnabled()) {
      openbusStatus =
        new OpenbusStatus(ConnectionState.NOT_INITIALIZED, new Date());

      configureOpenbusLogger();

      this.entityName = this.getEntityName();
      String accessControlServiceHost = this.getStringProperty("ACS.hostAddr");
      int accessControlServicePort = this.getIntProperty("ACS.port");
      final Properties orbProps = getExternalPropertyFile("ORB.file");
      
      Properties clientORBProps = null;
      if (hasProperty("Client.ORB.file"))
    	  clientORBProps = getExternalPropertyFile("Client.ORB.file");

      this.busInfo =
        new BusInfo(accessControlServiceHost, accessControlServicePort,
          orbProps, clientORBProps);

      String privateKeyFilePath = this.getPrivateKeyFilePath();

      try {
        this.privateKey =
          Cryptography.getInstance().readKeyFromFile(privateKeyFilePath);
      }
      catch (GeneralSecurityException e) {
        throw new ServerException(e);
      }
      catch (IOException e) {
        throw new ServerException(e);
      }
      catch (CryptographyException e) {
        throw new ServerException(e);
      }

      final Properties props = (Properties) busInfo.getOrbProperties().clone();

      /*
       * As propriedades que definem o ip e porta so definidas nas propriedades
       * do servio.
       */
      if (!isPropertyNull("ORB.port")) {
        int port = getIntProperty("ORB.port");
        String portStr = Integer.toString(port);
        props.setProperty("OAPort", portStr);
        final String fmt = "Porta para o ORB: %s";
        final String message = String.format(fmt, portStr);
        Server.logInfoMessage(message);
      }
      if (!isPropertyNull("ORB.hostAddr")) {
        String addr = getStringProperty("ORB.hostAddr");
        props.setProperty("OAIAddr", addr);
        final String fmt = "IP para o ORB: %s";
        final String message = String.format(fmt, addr);
        Server.logInfoMessage(message);
      }

      //Para resolver o bug da renovao do login aps queda do barramento.
      props.setProperty(
        "jacorb.connection.client.disconnect_after_systemexception", "false");

      this.orbThread = new ORBThread(ORBInitializer.initORB(null, props));
      orbThread.start();
      try {
        this.context =
          (OpenBusContext) orbThread.orb
            .resolve_initial_references("OpenBusContext");
      }
      catch (InvalidName e) {
        throw new ServerException("Falha ao obter o OpenBusContext.");
      }

      this.connectionProps = new Properties();
      connectionProps.put("legacy.disable", "false");
      connectionProps.put("legacy.delegate", "originator");
      Connection conn = null;
      try {
        conn =
          this.context.createConnection(busInfo.getHost(), this.busInfo
            .getPort(), this.connectionProps);
      }
      catch (InvalidPropertyValue e) {
        throw new ServerException(e);
      }

      this.context.setDefaultConnection(conn);
      conn.onInvalidLoginCallback(new OnInvalidLogin());

      String openBusMessage =
        String.format("Parmetros de conexo com o Openbus:\n"
          + "1 - Endereo: %s.\n2 - Porta: %s\n3 - Nome da entidade: %s\n"
          + "4 - Chave privada: %s", this.busInfo.getHost(), this.busInfo
          .getPort(), entityName, privateKeyFilePath);

      Server.logInfoMessage(openBusMessage);
    }
  }

  /**
   * Obtm o ORB utilizado pelo servio.
   * 
   * @return o ORB
   */
  public ORB getORB() {
    return orbThread.orb;
  }

  /**
   * Obtm o POA raiz.
   * 
   * @return o POA raiz
   * 
   * @throws OperationFailureException caso ocorra algum erro na obteno do POA
   */
  public POA getRootPOA() throws OperationFailureException {
    try {
      POA poa =
        POAHelper.narrow(getORB().resolve_initial_references("RootPOA"));
      if (!poa.the_POAManager().get_state().equals(State.ACTIVE)) {
        poa.the_POAManager().activate();
      }
      return poa;
    }
    catch (InvalidName e) {
      String msg = "Erro ao obter o POA.";
      Server.logSevereMessage(msg);
      throw new OperationFailureException(msg);
    }
    catch (AdapterInactive e) {
      String msg = "Erro ao obter o POA.";
      Server.logSevereMessage(msg);
      throw new OperationFailureException(msg);
    }
  }

  /**
   * Une-se  cadeia de chamadas.
   */
  public void joinChain() {
    this.context.joinChain();
  }

  /**
   * Une-se  cadeia de chamadas.
   * 
   * @param chain a representao de cadeia de chamandas em array de bytes
   * 
   * @throws OperationFailureException caso ocorra algum erro na unio  cadeia
   *         de chamadas.
   */
  public void joinChain(byte[] chain) throws OperationFailureException {
    try {
      CallerChain callerChain = this.context.decodeChain(chain);
      this.context.joinChain(callerChain);
    }
    catch (InvalidEncodedStream e) {
      String msg = "Erro ao unir-se  cadeia de chamadas.";
      Server.logSevereMessage(msg, e);
      throw new OperationFailureException(msg);
    }
  }

  /**
   * Sai da cadeia de chamadas.
   */
  public void exitChain() {
    this.context.exitChain();
  }

  /**
   * Direciona os registros de logs do SDK Openbus para os mesmos handlers
   * usados no Logger do servio.
   */
  private void configureOpenbusLogger() {
    Logger thisLogger = Logger.getLogger(this.getClass().getName());

    Logger SDKLogger = Logger.getLogger("tecgraf.openbus");
    SDKLogger.setParent(thisLogger);
    SDKLogger.setUseParentHandlers(true);
    SDKLogger.setLevel(Level.parse(getStringProperty("log.level")));

    Logger jacorbLogger = Logger.getLogger("jacorb");
    SDKLogger.setParent(thisLogger);
    SDKLogger.setUseParentHandlers(true);
    jacorbLogger.setLevel(Level.parse(getStringProperty("log.level")));
  }

  /**
   * Obtm a instncia nica do servio.
   * 
   * @return A instncia nica do servio.
   */
  public static OpenBusService getInstance() {
    return instance;
  }

  /**
   * Cria a instncia nica do servio.
   * 
   * @throws ServerException Se houver erro na inicializao.
   */
  public static void createService() throws ServerException {
    instance = new OpenBusService();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void initService() {
    initComponents();
    final Connection conn = this.context.getDefaultConnection();
    Thread t = new Thread() {
      @Override
      public void run() {
        conn.onInvalidLoginCallback().invalidLogin(conn, null);
      };
    };
    t.start();
  }

  /**
   * Obtm as ofertas das facetas dos componentes.
   * 
   * @param componentName o nome do componente
   * @param facetsFactories os nomes das classes das fbricas de facetas de
   *        componentes
   * @return a lista das instncias das fbricas
   */
  private List<OpenBusServiceOffer> getFacetOffers(String componentName,
    List<String> facetsFactories) {
    List<OpenBusServiceOffer> facetOffers =
      new ArrayList<OpenBusServiceOffer>();
    for (int i = 0; i < facetsFactories.size(); i++) {
      String facetFactoryClass = facetsFactories.get(i);
      try {
        Class<?> facetsFactory = Class.forName(facetFactoryClass);
        OpenBusServiceOffer factory =
          OpenBusServiceOffer.class.cast(facetsFactory.newInstance());
        facetOffers.add(factory);
      }
      catch (ClassNotFoundException e) {
        Server.logSevereMessage(
          "No existe uma classe para criao das facetas do componente "
            + componentName, e);
      }
      catch (ClassCastException e) {
        Server.logSevereMessage("Classe configurada para o componente "
          + componentName + " no implementa a interface "
          + OpenBusServiceOffer.class.getName(), e);
      }
      catch (InstantiationException e) {
        Server.logSevereMessage(
          "Erro na criao do objeto que cria as facetas do componente "
            + componentName, e);
      }
      catch (IllegalAccessException e) {
        Server.logSevereMessage(
          "Acesso ilegal a classe que cria as facetas do componente "
            + componentName, e);
      }
    }
    return facetOffers;
  }

  /**
   * Inicializa as informaes sobre os componentes que devem ser registrados no
   * Openbus, verificando possveis falhas na configurao desses componentes.
   */
  private void initComponents() {
    List<String> componentsNames = getStringListProperty("component.name");
    List<String> componentsVersions =
      getStringListProperty("component.version");
    for (int i = 0; i < componentsNames.size(); i++) {
      String componentName = componentsNames.get(i);
      String componentVersion = componentsVersions.get(i);
      if (componentVersion == null) {
        Server.logWarningMessage("O componente " + componentName
          + " no foi adicionado na lista para registro porque no possui uma "
          + "verso configurada.");
        continue;
      }
      final ComponentId componentId =
        makeComponentId(componentName, componentVersion);
      List<OpenBusServiceOffer> factories =
        getFacetOffers(componentName, getStringListProperty("component."
          + (i + 1) + ".facets"));
      if (factories.size() == 0) {
        Server
          .logWarningMessage("No existem facetas configuradas para o componente "
            + componentName);
        continue;
      }
      List<String> facetList = new ArrayList<String>();
      Map<String, ServiceProperty> propertyMap =
        new HashMap<String, ServiceProperty>();
      for (OpenBusServiceOffer factory : factories) {
        for (OpenBusServiceFacet facet : factory.getFacets()) {
          if (this.facets.containsKey(facet.interfaceName)) {
            Server.logWarningMessage("A faceta  " + facet.interfaceName
              + " est configurada para ser publicada mais de uma vez.");
          }
          this.facets.put(facet.interfaceName, facet);
          facetList.add(facet.interfaceName);
        }
        factory.registerFactories();

        Properties props = factory.getProperties();
        if (props != null && !props.isEmpty()) {
          for (Object key : props.keySet()) {
            if (!propertyMap.containsKey(key)) {
              propertyMap.put((String) key, new ServiceProperty((String) key,
                (String) props.get(key)));
            }
          }
        }
      }
      List<String> propNames =
        getStringListProperty("component." + (i + 1) + ".property.name");
      int sizePropNames = propNames.size();
      if (sizePropNames > 0) {
        List<String> propValues =
          getStringListProperty("component." + (i + 1) + ".property.value");
        int sizePropValues = propValues.size();
        if (sizePropNames != sizePropValues) {
          Server
            .logSevereMessage("Quantidade de nomes de propriedades e de valores de propriedades do compomente "
              + componentName + " no correspondem.");
          continue;
        }
        for (int j = 0; j < propNames.size(); j++) {
          String name = propNames.get(j);
          String value = propValues.get(j);
          ServiceProperty property = new ServiceProperty(name, value);
          if (!propertyMap.containsKey(property.name)) {
            propertyMap.put(property.name, property);
          }
        }
      }

      this.components.add(new ComponentEntry(componentId, facetList,
        propertyMap.size() == 0 ? new ServiceProperty[0] : propertyMap.values()
          .toArray(new ServiceProperty[0])));
    }
  }

  /**
   * Cria um identificador de um componente.
   * 
   * @param componentName nome do componente
   * @param componentVersion verso do componente
   * @return o identificador do componente
   */
  private ComponentId makeComponentId(String componentName,
    String componentVersion) {
    final String[] tmp = componentVersion.split("[\\.]");
    byte major = parseByte(tmp[0]);
    byte minor = tmp.length >= 2 ? parseByte(tmp[1]) : 0;
    byte patch = tmp.length >= 3 ? parseByte(tmp[2]) : 0;
    return new ComponentId(componentName, major, minor, patch, "Java");
  }

  /**
   * Cria um componente <code>IComponent</code>.
   * 
   * @param componentId o identificador do componente
   * @param facetList lista de facetas do componente
   * @return o componente
   * @throws OperationFailureException se houver uma falha na criao do
   *         componente
   */
  private IComponent makeComponent(ComponentId componentId,
    List<String> facetList) throws OperationFailureException {
    try {
      ORB orb = getORB();
      POA poa = getRootPOA();

      final ComponentContext context =
        new ComponentContext(orb, poa, componentId);

      for (String facetId : facetList) {
        OpenBusServiceFacet facet = facets.get(facetId);
        if (facet == null) {
          throw new OperationFailureException(
            "O componente {0} no adicionou a faceta {1} porque essa faceta no foi instanciada.",
            new Object[] { componentId.name, facetId });
        }
        if (facet.context == null) {
          context.addFacet(facet.name, facet.interfaceName, facet.servant);
          facet.setComponentContext(context);
        }
        else {
          Server
            .logWarningMessage(MessageFormat
              .format(
                "O componente {0} no adicionou a faceta {1} porque essa faceta j est publicada no componente {2}.",
                new Object[] { getComponentIdAsString(componentId), facetId,
                    getComponentIdAsString(context.getComponentId()) }));
        }
      }

      return context.getIComponent();
    }
    catch (SCSException e) {
      Server.logSevereMessage("Erro na criao do componente SCS", e);
      throw new OperationFailureException("Erro na criao do componente {0}",
        new Object[] { componentId.name });
    }
  }

  /**
   * Procura no barramento um servio com as propriedades especificadas. Os
   * servios que estiverem indisponveis sero desconsiderados.
   * 
   * @param properties as propriedades usadas como critrio de busca
   * 
   * @return um servio qualquer que atenda os critrios de busca ou nulo caso
   *         no exista um servio com os critrios especificados
   * 
   * @throws OperationFailureException caso ocorra algum erro na busca do
   *         servio
   */
  public IComponent findService(Properties properties)
    throws OperationFailureException {
    IComponent[] services = findServiceListByFilter(properties, FIND.FIRST);
    if (services.length > 0) {
      return services[0];
    }

    return null;
  }

  /**
   * Procura no barramento todos servios com as propriedades especificadas. Os
   * servios que estiverem indisponveis sero desconsiderados.
   * 
   * @param properties as propriedades usadas como critrios de busca
   * 
   * @return a lista de servios que atendem os critrios de busca
   * 
   * @throws OperationFailureException caso ocorra algum erro na busca do
   *         servio
   */
  public IComponent[] findServiceList(Properties properties)
    throws OperationFailureException {
    return findServiceListByFilter(properties, FIND.ALL);
  }

  /**
   * Procura no barramento todos servios com as propriedades especificadas. Os
   * servios que estiverem indisponveis sero desconsiderados.
   * 
   * @param properties as propriedades usadas como critrios de busca
   * @param filter agrupador dos servios encontrados
   * 
   * @return a lista de servios que atendem os critrios de busca
   * 
   * @throws OperationFailureException caso ocorra algum erro na busca do
   *         servio
   */
  private IComponent[] findServiceListByFilter(Properties properties,
    FIND filter) throws OperationFailureException {

    ServiceProperty[] serviceProps = new ServiceProperty[properties.size()];
    int i = 0;
    for (String key : properties.stringPropertyNames()) {
      serviceProps[i] = new ServiceProperty(key, properties.getProperty(key));
      i++;
    }
    ServiceOfferDesc[] services;
    try {
      services = this.context.getOfferRegistry().findServices(serviceProps);
    }
    catch (Exception e) {
      Server.logSevereMessage(
        "Falha ao buscar servios com as seguintes propriedades: \n"
          + properties.toString(), e);
      throw new OperationFailureException("Falha na busca de servios.", e);
    }

    ArrayList<IComponent> components = new ArrayList<IComponent>();

    for (int j = 0; j < services.length; j++) {
      String offerId = "";
      for (ServiceProperty prop : services[j].properties) {
        if (prop.name.equals(OPENBUS_OFFER_ID_KEY)) {
          offerId = prop.value;
          break;
        }
      }
      try {
        IComponent service = services[j].service_ref;
        if (!service._non_existent()) {
          components.add(service);
          if (filter.equals(FIND.FIRST)) {
            break;
          }
        }
      }
      catch (TRANSIENT e) {
        Server.logWarningMessage("Oferta de servio encontrada (" + offerId
          + ") mas servio est inalcanvel.");
      }
      catch (COMM_FAILURE e) {
        Server.logWarningMessage("Oferta de servio encontrada (" + offerId
          + ") mas houve falha na comunicao com servio encontrado.");
      }
      catch (Exception e) {
        Server.logSevereMessage("Oferta de servio encontrada (" + offerId
          + ") mas houve falha ao testar a comunicao.", e);
      }
    }
    return components.toArray(new IComponent[components.size()]);
  }

  public IComponent findServiceById(String id) throws OperationFailureException {
    Properties props = new Properties();

    props.put(OPENBUS_OFFER_ID_KEY, id);
    return findService(props);
  }

  /**
   * Obtm o identificador do usurio que est executando uma requisio atravs
   * do OpenBus.
   * 
   * @return O identificador do usurio ou {@code null}, caso no haja
   *         requisio em execuo.
   */
  public String getRequesterUserId() {
    CallerChain chain = this.context.getCallerChain();

    if (chain == null) {
      return null;
    }

    String entity;
    if (chain.originators().length == 0) {
      if (chain.caller().entity.equals(User.getAdminId())) {
        throw new ServiceFailureException(
          "No  permitido na credencial delegar para um usurio 'admin'.");
      }
      entity = chain.caller().entity;
    }
    else {
      entity = chain.originators()[0].entity;
    }

    try {
      return User.getUserByLogin(entity).getLogin();
    }
    catch (Exception e) {
      throw new ServiceFailureException(
        "No foi possvel obter o identificador do usurio '" + entity + "'.");
    }
  }

  /**
   * Obtm o identificador do sistema em que o usurio est executando uma
   * requisio atravs do OpenBus.
   * 
   * @return o identificador do sistema ou {@code null}, caso no haja
   *         requisio em execuo
   */
  public String getRequesterSystemId() {
    CallerChain chain = this.context.getCallerChain();

    if (chain == null) {
      return null;
    }

    return chain.caller().entity;
  }

  /**
   * Inicia o usurio do servio CSBase e tambm o sistema onde esse usurio
   * est executando uma requisio atravs do OpenBus. Para isso, considera-se
   * como base a credencial usada para acesso no OpenBus.
   * 
   * @return retorna o identificador do usurio
   */
  public String initCSBaseAccess() {
    String userId = getRequesterUserId();
    String systemId = getRequesterSystemId();
    Service.setUserId(new ExternalUser(userId, ExternalUser
      .ExternalAuthMechanism.OPENBUS));
    Service.setSystemId(systemId);
    return userId;
  }

  /**
   * Finaliza o acesso do usurio do servio CSBase na sesso em que foi feita
   * uma requisio atravs do OpenBus. O sistema em que esse usurio estava
   * executando a requisio tambm  finalizado na sesso.
   * 
   */
  public void finishCSBaseAccess() {
    Service.setUserId(null);
    Service.setSystemId(null);
  }

  /**
   * Retorna o identificador de um componente no formato
   * <nome_componente>:<verso do comonente>.
   * 
   * @param componentId identificador do componente
   * 
   * @return representao do identificador como string de caracteres
   */
  private String getComponentIdAsString(ComponentId componentId) {
    StringBuffer stringComponentId = new StringBuffer(componentId.name);
    stringComponentId.append(':');
    stringComponentId.append(componentId.major_version);
    stringComponentId.append(".");
    stringComponentId.append(componentId.minor_version);
    stringComponentId.append(".");
    stringComponentId.append(componentId.patch_version);

    return stringComponentId.toString();
  }

  /**
   * Indica se houve falha na inicializao do ACS.
   * 
   * @param failed <code>true</code> se houve falha na inicializao do ACS
   */
  private void acsInitFailed(boolean failed) {
    // a rigor este mtodo deveria ser sincronizado, e deveramos ter um getter
    // tambm sincronizado...
    this.acsInitFailed = failed;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void shutdownService() {
    this.unregisterComponents();
    try {
      this.context.getDefaultConnection().logout();
    }
    catch (Exception e) {
      Server.logSevereMessage("Erro no barramento.", e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isActive() {

    // Logar, em nvel verboso, o motivo para a constatao da inatividade do servio.
    if (!super.isActive()) {
      Server
        .logFineMessage("OpenbusService no est ativo porque a propriedade enabled do servio est false.");
    }
    else if (acsInitFailed) {
      Server
        .logFineMessage("OpenbusService no est ativo porque a flag acsInitFailed est true.");
    }
    //    else if (!allComponentsRegistered) {
    //      Server
    //        .logFineMessage("OpenbusService no est ativo porque a flag allComponentsRegistered est false.");
    //    }

    return super.isActive() && !acsInitFailed;
    //    && allComponentsRegistered;
  }

  /**
   * Utiliza o Servio de Registro para retirar as ofertas de servio dos
   * componentes cadastrados.
   * 
   * @return {@code true}, se todas as ofertas foram descadastradadas, ou
   *         {@code false}, caso contrrio.
   */
  private boolean unregisterComponents() {
    boolean status = true;
    for (ComponentEntry entry : this.components) {
      try {
        if (entry.serviceOffer != null) {
          entry.serviceOffer.remove();
          Server.logInfoMessage("O componente "
            + entry.component.getComponentId().name + " teve sua oferta "
            + entry.offerId + " removida do barramento.");
        }
      }
      catch (Exception e) {
        Server.logSevereMessage("Erro ao remover a oferta " + entry.offerId, e);
        status = false;
      }
    }

    return status;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean has2Update(Object arg, Object event) {
    return false;
  }

  /**
   * Adiciona uma faceta que ser ofertada no OpenBus.
   * 
   * @param name da faceta
   * @param interfaceName identificador da interface da faceta
   * @param servant servant que implementa a interface da faceta
   */
  public void addFacet(String name, String interfaceName, Servant servant) {
    this.facets.put(interfaceName, new OpenBusServiceFacet(name, interfaceName,
      servant));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public BusInfo getBusInfo() {
    LoginInfo loginInfo = this.context.getCurrentConnection().login();
    if (loginInfo != null) {
      this.busInfo.setLoginId(loginInfo.id);
    }
    return this.busInfo;
  }

  /**
   * Obtm o estado da conexo com o barramento Openbus.
   * 
   * @return estado da conexo
   */
  public OpenbusStatus getOpenbusStatus() {
    return openbusStatus;
  }

  public boolean isConnected() {
    Connection conn = this.context.getDefaultConnection();
    if (conn == null) {
      return false;
    }

    if (conn.login() == null) {
      return false;
    }
    else {
      return true;
    }
  }

  public boolean isRegistryServerActive() {
    try {
      return !this.context.getOfferRegistry()._non_existent();
    }
    catch (org.omg.CORBA.TRANSIENT e) {
      return false;
    }
  }

  /**
   * Obtm o nome de entidade cadastrado no OpenBus.
   * 
   * @return O nome de entidade.
   */
  public String getEntityName() {
    return this.getStringProperty("entity.name");
  }

  /**
   * Obtm o caminho do arquivo da chave privada para acesso ao barramento
   * Openbus.
   * 
   * @return o caminho do arquivo da chave privada.
   */
  public String getPrivateKeyFilePath() {
    return this.getStringProperty("private.key.file");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<ComponentId> getRegisteredComponents() {
    if (!Service.getUser().isAdmin()) {
      throw new PermissionException();
    }

    try {
      Set<ComponentId> result = new HashSet<ComponentId>();
      for (ComponentEntry entry : components) {
        result.add(entry.componentId);
      }
      return result;
    }
    catch (Exception e) {
      return null;
    }
  }

  /**
   * Obtm as facetas de um componente registrado como oferta de servio pelo
   * CSBase.
   * 
   * @param id o identificador do componente publicado no barramento
   * @return o array com as facetas (repId da IDL) ou null caso no exista um
   *         componente publicado com o identificador fornecido.
   */
  public String[] getRegisteredComponentFacets(ComponentId id) {
    for (ComponentEntry component : components) {
      if (compareComponentId(component.componentId, id)) {
        return component.facets.toArray(new String[0]);
      }
    }
    return null;
  }

  /**
   * Obtm as propriedades de um componente registrado como oferta de servio
   * pelo CSBase.
   * 
   * @param id o identificador do componente publicado no barramento
   * @return o array com as propriedades ou null caso no exista um componente
   *         publicado com o identificador fornecido.
   */
  public Properties getRegisteredComponentProperties(ComponentId id) {
    Properties props = new Properties();
    for (ComponentEntry component : components) {
      if (compareComponentId(component.componentId, id)) {
        //        return component.properties.clone();
        for (ServiceProperty servProp : component.properties) {
          props.put(servProp.name, servProp.value);
        }
        return props;
      }
    }
    return null;
  }

  /**
   * Obtm o identificador da publicao do componente no barramento.
   * 
   * @param id o identificador do componente
   * 
   * @return o identificador da publicao do componente no barramento
   */
  public String getRegisteredComponentBusId(ComponentId id) {
    for (ComponentEntry component : components) {
      if (compareComponentId(component.componentId, id)) {
        return component.offerId;
      }
    }
    return null;
  }

  /**
   * Obtm um conjunto com os componentes com falha no registro. S pode ser
   * executado pelo admin.
   * 
   * @return conjunto com os componentes com falha no registro
   */
  @Override
  public Set<ComponentId> getFailedComponents() {
    //    if (!Service.getUser().isAdmin()) { // TODO mjulia VERIFICAR QUEM USA
    //      throw new PermissionException();
    //    }
    try {
      Set<ComponentId> result = new HashSet<ComponentId>();
      for (ComponentEntry component : failedComponents) {
        result.add(component.componentId);
      }
      return result;
    }
    catch (Exception e) {
      return null;
    }
  }

  /**
   * Verifica se dois identificadores de componentes so iguais.
   * 
   * @param id1 identificador do primeiro componente
   * @param id2 identificador do segundo componente
   * @return {@code true} se os dois identificadores forem iguais ou
   *         {@code false} caso contrrio.
   */
  private boolean compareComponentId(ComponentId id1, ComponentId id2) {
    return id1.major_version == id2.major_version
      && id1.minor_version == id2.minor_version && id1.name == id2.name
      && id1.patch_version == id2.patch_version
      && id1.platform_spec == id2.platform_spec;
  }

  /**
   * Autentica uma conexo usando um token e retorna um novo token para ser
   * utilizado pelo cliente.
   * 
   * @param token o token usado para autenticar uma conexo
   * 
   * @return o token que ser enviado para o cliente para autenticar sua conexo
   *         com o barramento.
   */
  public OpenBusLoginToken doTokenLogin(final OpenBusLoginToken token) {
    Properties props = new Properties();
    props.put("legacy.disable", "false");
    props.put("legacy.delegate", "originator");
    final Connection tokenConn;
    try {
      tokenConn =
        this.context.createConnection(busInfo.getHost(), busInfo.getPort(),
          props);
    }
    catch (InvalidPropertyValue e) {
      Server
        .logSevereMessage(
          "Erro ao criar OpenBusToken: falha na criao de conexo com o barramento.",
          e);
      return null;
    }
    try {
      SharedAuthSecret oldSecret = context.decodeSharedAuth(token.secret);
      tokenConn.loginBySharedAuth(oldSecret);
    }
    catch (InvalidEncodedStream e) {
      Server.logSevereMessage(
        "Erro ao criar OpenBusToken: falha na decodificao do segredo.", e);
      return null;
    }
    catch (AlreadyLoggedIn e) {
      // Nada a fazer, conexo j autenticada
    }
    catch (InvalidLoginProcess e) {
      Server
        .logSevereMessage(
          "Erro ao criar OpenBusToken: falha na autenticao de conexo com o barramento.",
          e);
      return null;
    }
    catch (AccessDenied e) {
      Server
        .logSevereMessage(
          "Erro ao criar OpenBusToken: falha na autenticao de conexo com o barramento.",
          e);
      return null;
    }
    catch (ServiceFailure e) {
      Server
        .logSevereMessage(
          "Erro ao criar OpenBusToken: falha na autenticao de conexo com o barramento.",
          e);
      return null;
    }
    if (tokenConn.login() == null) {
      return null;
    }
    else {
      SharedAuthSecret newSecret;
      try {
        newSecret = tokenConn.startSharedAuth();
      }
      catch (ServiceFailure e) {
        Server
          .logSevereMessage(
            "Erro ao criar OpenBusToken: falha na criao da conexo compartilhada.",
            e);
        return null;
      }

      Thread closeConnTask = new Thread() {

        @Override
        public void run() {
          try {
            Thread.sleep(TOKEN_EXPIRATION_TIME);
            tokenConn.logout();
          }
          catch (InterruptedException e) {
            // Nada a fazer
          }
          catch (ServiceFailure e) {
            Server.logSevereMessage(
              "Falha ao fechar a conexo usada para autenticao compartilhada do usurio "
                + token.user + ".", e);
          }
        }
      };
      closeConnTask.start();

      return new OpenBusLoginToken(token.user, context
        .encodeSharedAuth(newSecret));
    }
  }

  /**
   * Obtm o contexto do componente que publicou uma determinada faceta.
   * 
   * @param facetId identificador da faceta
   * @return o contexto do componente
   */
  public ComponentContext getComponentContext(String facetId) {
    return facets.get(facetId).context;
  }

  /**
   * Envia uma mensagem para o log, usando nvel SEVERE, se no for repetida.
   * 
   * @param msg a mensagem a ser registrada
   * 
   * @return true se a mensagem foi gravada, false caso seja repetida
   */
  private boolean logOnce(String msg) {
    if (lastLogMessage == null || !lastLogMessage.equals(msg)) {
      if (numLastLogMessage > 1) {
        Server.logWarningMessage("A mensagem '" + lastLogMessage
          + "' foi repetida " + numLastLogMessage + " vezes");
      }
      lastLogMessage = msg;
      numLastLogMessage = 1;
      Server.logSevereMessage(msg);
      return true;
    }
    numLastLogMessage++;
    return false;
  }

  // -------------------------------------------------------------------------
  // inner classes

  /**
   * Callback chamada quando o login  invalidado.
   * 
   * @author Tecgraf
   */
  class OnInvalidLogin implements InvalidLoginCallback {
    /**
     * {@inheritDoc}
     */
    @Override
    public void invalidLogin(Connection conn, LoginInfo loginInfo) {
      for (int i = 1; i < Integer.MAX_VALUE; ++i) {
        try {
          conn.loginByCertificate(entityName, privateKey);
          openbusStatus =
            new OpenbusStatus(ConnectionState.CONNECTED, new Date());
          break;
        }
        catch (AlreadyLoggedIn e) {
          break;
        }
        catch (org.omg.CORBA.SystemException e) {
          logOnce("Erro inesperado ao acessar o barramento para realizar o login."
            + e);
          acsInitFailed(true);
          openbusStatus =
            new OpenbusStatus(ConnectionState.ACS_UNAVAILABLE, e, new Date());
          try {
            Thread.sleep(i * 1000);
          }
          catch (InterruptedException ie) {
          }
        }
        catch (Exception e) {
          logOnce("Erro ao fazer o login no barramento." + e);
          acsInitFailed(true);
          openbusStatus =
            new OpenbusStatus(ConnectionState.ACS_LOGIN_FAILURE, e, new Date());
          break;
        }
      }

      acsInitFailed(false);

      // Garante que no h registro de contexto de publicao das facetas
      for (OpenBusServiceFacet facet : facets.values()) {
        facet.setComponentContext(null);
      }

      for (ComponentEntry entry : components) {
        entry.notifyRelogin();
      }
    }
  }

  /**
   * Representa uma entrada de um componente cadastrado.
   * 
   * @author Tecgraf/PUC-Rio
   * 
   */
  private class ComponentEntry {
    /**
     * Identificador do componente.
     */
    ComponentId componentId;
    /**
     * Lista com os identificadores das interfaces das facetas que este
     * componente oferece.
     */
    List<String> facets;
    /**
     * Componente.
     */
    IComponent component;
    /**
     * Propriedades do componente.
     */
    ServiceProperty[] properties;
    /**
     * Referncia remota da oferta.
     */
    ServiceOffer serviceOffer;
    /**
     * Identificador da offerta.
     */
    String offerId;
    /**
     * Task para republicar a oferta aps um re-login.
     */
    ReloginTask offerRegisterTask;

    /**
     * Cria uma entrada de um componente cadastrado.
     * 
     * @param id identificador do componente
     * @param facetList lista com os identificadores das interfaces das facetas
     * @param props propriedades do componente
     */
    ComponentEntry(ComponentId id, List<String> facetList,
      ServiceProperty[] props) {
      this.componentId = id;
      this.facets = facetList;
      this.properties = props;
      this.offerRegisterTask =
        new ReloginTask(context.getDefaultConnection(), new ReloginAllocator() {
          @Override
          public String reallocate(LoginInfo login) throws Exception {
            OfferRegistry offerRegistry = context.getOfferRegistry();
            try {
              component = makeComponent(componentId, facets);
            }
            catch (OperationFailureException e) {
              String log =
                MessageFormat.format(
                  "Erro ao criar o componente SCS {0}:{1}.{2}.{3}.",
                  new Object[] { entityName, componentId.name,
                      componentId.major_version, componentId.major_version,
                      componentId.patch_version });
              Server.logSevereMessage(log, e);
            }

            String loginId = null;
            do {
              ServiceOffer offer = null;
              try {
                offer = offerRegistry.registerService(component, properties);
                setServiceOffer(offer);
              }
              catch (UnauthorizedFacets e) {
                StringBuffer facetNames = new StringBuffer();
                for (int i = 0; i < e.facets.length; i++) {
                  if (i > 0) {
                    facetNames.append(", ");
                  }
                  facetNames.append(e.facets[i]);
                }
                String log =
                  MessageFormat
                    .format(
                      "A entidade {0} no possui permisso para publicar as seguintes facetas do componente {1}:{2}.{3}.{4}: {5}",
                      new Object[] { entityName, componentId.name,
                          componentId.major_version, componentId.major_version,
                          componentId.patch_version, facetNames.toString() });
                Server.logSevereMessage(log, e);
              }
              catch (InvalidService e) {
                String log =
                  MessageFormat
                    .format(
                      "O componente {0}:{1}.{2}.{3}  invlido e no ser publicado no barramento.",
                      new Object[] { componentId.name,
                          componentId.major_version, componentId.major_version,
                          componentId.patch_version });
                Server.logSevereMessage(log, e);
                components.remove(getEntry());
              }
              catch (InvalidProperties e) {
                String log =
                  MessageFormat
                    .format(
                      "O componente {0}:{1}.{2}.{3} contem propriedades invlidas e no ser publicado no barramento.",
                      new Object[] { componentId.name,
                          componentId.major_version, componentId.major_version,
                          componentId.patch_version });
                Server.logSevereMessage(log, e);
                components.remove(getEntry());
              }
              catch (ServiceFailure e) {
                String log =
                  MessageFormat.format(
                    "Erro ao registrar o componente {0}:{1}.{2}.{3}.",
                    new Object[] { entityName, componentId.name,
                        componentId.major_version, componentId.major_version,
                        componentId.patch_version });
                Server.logSevereMessage(log, e);
              }
              catch (TRANSIENT e) {
                String log =
                  MessageFormat.format(
                    "Erro ao publicar o componente SCS {0}:{1}.{2}.{3}.",
                    new Object[] { entityName, componentId.name,
                        componentId.major_version, componentId.major_version,
                        componentId.patch_version });
                Server.logSevereMessage(log, e);
              }
              catch (COMM_FAILURE e) {
                String log =
                  MessageFormat.format(
                    "Erro ao publicar o componente SCS {0}:{1}.{2}.{3}.",
                    new Object[] { entityName, componentId.name,
                        componentId.major_version, componentId.major_version,
                        componentId.patch_version });
                Server.logSevereMessage(log, e);
              }
              catch (NO_PERMISSION e) {
                String log =
                  MessageFormat.format(
                    "Erro ao publicar o componente SCS {0}:{1}.{2}.{3}.",
                    new Object[] { entityName, componentId.name,
                        componentId.major_version, componentId.major_version,
                        componentId.patch_version });
                Server.logSevereMessage(log, e);
              }
              catch (Throwable e) {
                String log =
                  MessageFormat.format(
                    "Erro ao publicar o componente SCS {0}:{1}.{2}.{3}.",
                    new Object[] { entityName, componentId.name,
                        componentId.major_version, componentId.major_version,
                        componentId.patch_version });
                Server.logSevereMessage(log, e);
              }
              try {
                ServiceOfferDesc desc = offer.describe();
                for (ServiceProperty prop : desc.properties) {
                  if (prop.name.equals("openbus.offer.login")) {
                    loginId = prop.value;
                  }
                }
              }
              catch (OBJECT_NOT_EXIST e) {
                System.err.println("offer was removed");
              }
            } while (loginId == null);
            Server
              .logInfoMessage("O componente "
                + componentId.name
                + " foi registrado com sucesso no Openbus e sua oferta tem o identificador "
                + offerId);

            return loginId;
          }
        });
    }

    /**
     * Atribui a referncia remota da oferta
     * 
     * @param serviceOffer a referncia remota da oferta
     */
    void setServiceOffer(ServiceOffer serviceOffer) {
      this.serviceOffer = serviceOffer;
      String id = "";
      for (ServiceProperty prop : serviceOffer.properties()) {
        if (prop.name.equals(OPENBUS_OFFER_ID_KEY)) {
          id = prop.value;
        }
      }
      this.offerId = id;
    }

    /**
     * Obtm o ComponentEntry
     * 
     * @return o ComponentEntry
     */
    ComponentEntry getEntry() {
      return this;
    }

    /**
     * Notifica o ComponentEntry do evento de relogin.
     */
    public void notifyRelogin() {
      this.offerRegisterTask.notifyRelogin();
    }
  }

  /**
   * Responsvel por executar o ORB.
   * 
   * @author Tecgraf/PUC-Rio
   */
  private static class ORBThread extends Thread {
    /**
     * O ORB.
     */
    private final ORB orb;

    /**
     * Cria a <i>thread</i> para execuo do ORB.
     * 
     * @param orb O ORB.
     */
    ORBThread(ORB orb) {
      this.orb = orb;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void run() {
      this.orb.run();
    }
  }
}
