package csbase.server.services.opendreamsservice.opendreams.v1_9;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Set;

import csbase.logic.CommandFinalizationInfo;
import csbase.logic.CommandFinalizationInfo.FinalizationInfoType;
import csbase.logic.CommandFinalizationType;
import csbase.logic.CommandNotification;
import csbase.logic.CommandSuccessNotification;
import csbase.logic.ExtendedCommandFinalizationInfo;
import csbase.logic.FailureFinalizationType;
import csbase.server.Server;
import tecgraf.openbus.DRMAA.v1_9.ExitTimeoutException;
import tecgraf.openbus.DRMAA.v1_9.InvalidJobException;
import tecgraf.openbus.DRMAA.v1_9.JobInfo;
import tecgraf.openbus.opendreams.v1_9.FinalizationType;
import tecgraf.openbus.opendreams.v1_9.OpenDreamsJobInfo;
import tecgraf.openbus.opendreams.v1_9.OpenDreamsJobInfoImpl;

/**
 * Controle dos jobs submetidos pelo OpenDreams.
 * 
 * @author Tecgraf PUC-Rio
 */
public class JobInfoQueue {

  /**
   * O Job terminou mas no tem um codigo de sada
   */
  public final static int NO_EXIT_CODE = 256; // Acima de 255 no  um exit code vlido

  /**
   * Tempo em milisegundos para reavaliar se a fila j possui informao sobre o
   * trmino do job.
   */
  public static final long WAITING_TIMEOUT = 100;

  /**
   * Informao de trmino para cada job submetido. Diferentes chamadas podem
   * aguardar a espera por um mesmo comando, porm apenas uma delas recebe a
   * informao de trmino; as outras chamadas recebem uma exceo. Esse mapa
   * relaciona o identificador do comando submetido com a infornao de trmino
   * de job.
   */
  private Map<String, OpenDreamsJobInfo> jobsSubmited =
    new Hashtable<String, OpenDreamsJobInfo>();

  /**
   * Indica que um job foi submetido
   * 
   * @param jobId o identificador do job submetido
   */
  public synchronized void jobSubmited(String jobId) {
    OpenDreamsJobInfo jobInfo = jobsSubmited.get(jobId);
    if (jobInfo == null) {
      jobInfo = new OpenDreamsJobInfoImpl();
      jobInfo.jobId = jobId;
      jobsSubmited.put(jobId, jobInfo);
    }
  }

  /**
   * Indica que um job finalizou
   * 
   * @param cmdEvent evento com as informaes de trmino do job
   */
  public synchronized void jobExit(CommandNotification cmdEvent) {
    String jobId = cmdEvent.getCommandId().toString();
    OpenDreamsJobInfo jobInfo = jobsSubmited.get(jobId);
    if (jobInfo == null) {
      jobInfo = new OpenDreamsJobInfoImpl();
      jobInfo.jobId = jobId;
      jobsSubmited.put(jobId, jobInfo);
    }
    List<String[]> resourceUsage = new ArrayList<String[]>();
    if (cmdEvent.getFinalizationInfo()
      .getFinalizationType() == CommandFinalizationType.SUCCESS) {
      CommandSuccessNotification endNotification =
        (CommandSuccessNotification) cmdEvent;
      if (endNotification.getCPUTime() != null) {
        resourceUsage.add(new String[] { "cpu_time", endNotification
          .getCPUTime().toString() });
      }
      if (endNotification.getElapsedTime() != null) {
        resourceUsage.add(new String[] { "elapsed_time", endNotification
          .getElapsedTime().toString() });
      }
      if (endNotification.getUserTime() != null) {
        resourceUsage.add(new String[] { "user_time", endNotification
          .getUserTime().toString() });
      }
    }
    if (resourceUsage.size() > 0) {
      jobInfo.resourceUsage = resourceUsage.toArray(new String[resourceUsage
        .size()][]);
    }
    if (cmdEvent.getFinalizationInfo()
      .getInfoType() == FinalizationInfoType.EXTENDED) {
      ExtendedCommandFinalizationInfo finalizationInfo =
        (ExtendedCommandFinalizationInfo) cmdEvent.getFinalizationInfo();
      jobInfo.guiltyNodeId = String.valueOf(finalizationInfo.getGuiltyNodeId());
    }
    jobInfo.hasExited = true;
    CommandFinalizationInfo finalizationInfo = cmdEvent.getFinalizationInfo();
    if (finalizationInfo == null) {
      Server.logSevereMessage("A informao de finalizacao do comando " + jobId
        + " no existe (null)");
      jobInfo.exitStatus = NO_EXIT_CODE;
    }
    else {
      Integer exitCode = finalizationInfo.getExitCode();
      Server.logInfoMessage("A informao de finalizacao do comando " + jobId
        + " possui exitCode=" + exitCode);
      jobInfo.exitStatus = exitCode == null ? NO_EXIT_CODE : exitCode;
      jobInfo.finalizationType = getFinalizationType(finalizationInfo
        .getFinalizationType(), finalizationInfo.getFailureCause());
      jobInfo.description = finalizationInfo.getFailureCause() == null
        ? finalizationInfo.getFinalizationType().getDescription()
        : finalizationInfo.getFailureCause().getDescription();
    }
    jobInfo.wasAborted = cmdEvent.getFinalizationInfo().getFinalizationType()
      .equals(FinalizationType.KILLED);
    jobInfo.hasSignaled = false;
    jobInfo.terminatingSignal = "";
    jobInfo.hasCoreDump = false;
    notifyAll();
  }

  /**
   * Mapeia o tipo de finalizao do comando do CSBase para o tipo de
   * finalizao de comando definido pela IDL do OpenDreams.
   * 
   * @param type o tipo de finalizao do comando no CSBase
   * @param cause a causa da falha no caso do comando ou null caso o comando no
   *        tenha falhado.
   * @return o tipo de finalizao do comando mapeado para o OpenDreams
   */
  public static FinalizationType getFinalizationType(
    CommandFinalizationType type, FailureFinalizationType cause) {
    switch (type) {
      case END:
        return FinalizationType.EXIT_CODE_IGNORED;
      case SUCCESS:
        return FinalizationType.EXIT_CODE_SUCCESS;
      case EXECUTION_ERROR:
        return FinalizationType.EXIT_CODE_ERROR;
      case NO_EXIT_CODE:
        return FinalizationType.EXIT_CODE_FAILED;
      case FAILED:
        switch (cause) {
          case UNKNOWN:
            return FinalizationType.UNKNOWN;
          case COMMAND_IDENTIFIER_NOT_FOUND:
            return FinalizationType.COMMAND_IDENTIFIER_NOT_FOUND;
          case SGA_EXECUTION_ERROR:
            return FinalizationType.SGA_EXECUTION_ERROR;
          case CSFS_SERVICE_UNAVAILABLE:
            return FinalizationType.CSFS_SERVICE_UNAVAILABLE;
          case FAILED_SETUP_EXECUTION_ENVIRONMENT:
            return FinalizationType.FAILED_SETUP_EXECUTION_ENVIRONMENT;
          case NO_SGA_AVAILABLE_TO_ROOT_COMMAND:
            return FinalizationType.NO_SGA_AVAILABLE_TO_ROOT_COMMAND;
          case SGA_IS_NOT_AVAILABLE:
            return FinalizationType.SGA_IS_NOT_AVAILABLE;
          case PROJECT_NOT_FOUND:
            return FinalizationType.PROJECT_NOT_FOUND;
          case USER_WITHOUT_PERMISSION_FOR_EXECUTION:
            return FinalizationType.USER_WITHOUT_PERMISSION_FOR_EXECUTION;
        }
      case KILLED:
        return FinalizationType.KILLED;
      case LOST:
        return FinalizationType.LOST;
      default:
        return FinalizationType.UNDEFINED;
    }
  }

  /**
   * Aguarda que um determinado job submetido termine sua execuo.
   * 
   * Se mais de uma chamada estiver aguardando pelo mesmo job, apenas uma delas
   * recebe a informao de trmino do job. As outras chamadas recebem e exceo
   * <code>InvalidJobException</code>. Essa mesma exceo  tambm lanada se o
   * identificador do job no for encontrado entre os jobs submetidos nessa
   * sesso.
   * 
   * No caso do timeout ter esgotado sem que o job tenha terminado, a exceo
   * <code>ExitTimeoutException</code>  lanada. Nesse caso, a informao sobre
   * o job permanece disponvel para uma outra requisio.
   * 
   * @param jobId o nome do job monitorado
   * @param timeout o tempo de espera limite em segundos
   * @return a informao sobre o job submetido
   * @throws InvalidJobException se o identificador do job no foi encontrado ou
   *         se outra requisio j recebeu a informao de trmino do job.
   * @throws ExitTimeoutException se o timeout esgotou e o job no terminou.
   */
  public JobInfo waitFor(String jobId, long timeout) throws InvalidJobException,
    ExitTimeoutException {
    synchronized (this) {
      /* Verifica se existe um job submetido com o identificador especificado */
      JobInfo info = jobsSubmited.get(jobId);
      if (info == null) {
        throw new InvalidJobException(
          "No foi encontrado um job com o nome especificado: " + jobId);
      }
      /* Se o tempo especificado foi TIMEOUT_NO_WAIT retorna o jobInfo */
      if (timeout == Session.TIMEOUT_NO_WAIT) {
        jobsSubmited.remove(jobId);
        if (!info.hasExited) {
          //        	OpenDreamsJobInfo odInfo = (OpenDreamsJobInfo) info;
          //        	//odInfo.progressData = 
          //        	info.
          throw new ExitTimeoutException(
            "O timeout esgotou e o job no terminou: " + jobId);
        }
        return info;
      }
    }
    final long start = System.currentTimeMillis();
    long limit = timeout * 1000; // tempo mximo para timeout em ms
    boolean waitForever = (timeout == Session.TIMEOUT_WAIT_FOREVER);
    while (waitForever || System.currentTimeMillis() - start < limit) {
      try {
        synchronized (this) {
          wait(WAITING_TIMEOUT);
        }
        synchronized (this) {
          JobInfo info = jobsSubmited.get(jobId);
          if (info == null) {
            throw new InvalidJobException(
              "Outra requisio j recebeu a informao de trmino desse"
                + "mesmo job: " + jobId);
          }
          if (info.hasExited) {
            jobsSubmited.remove(jobId);
            return info;
          }
        }
      }
      catch (InterruptedException e) {
      }
    }
    /* Quando o tempo esgotou, a informao sobre o job permanece guardada */
    throw new ExitTimeoutException("O timeout esgotou e o job no terminou: "
      + jobId);
  }

  /**
   * Aguarda que qualquer um dos jobs submetidos termine sua execuo. Quando um
   * deles terminar, retorna a informao sobre esse job.
   * 
   * Apenas uma chamada recebe a informao de um determinado job. Se outras
   * chamadas estiverem aguardando e no houver mais jobs submetidos (no momento
   * em que a requisio de "wait" for feita), as outras recebem a exceo
   * <code>InvalidJobException</code>.
   * 
   * No caso do timeout ter esgotado sem que algum job tenha terminado, a
   * exceo <code>ExitTimeoutException</code>  lanada.
   * 
   * @param timeout o tempo de espera limite em segundos
   * @return a informao sobre o primeiro job que terminou
   * @throws InvalidJobException se no h mais jobs submetidos na sesso que
   *         ainda no tenham terminado sua execuo
   * @throws ExitTimeoutException se o timeout esgotou e nenhum job terminou.
   */
  public JobInfo waitForAnyJob(long timeout) throws ExitTimeoutException,
    InvalidJobException {
    final Set<String> jobsAvailable = jobsSubmited.keySet();
    synchronized (this) {
      if (jobsSubmited.size() == 0) {
        throw new InvalidJobException("No h mais jobs submetidos na sesso "
          + "que ainda no tenham terminado sua execuo");
      }
      /* Se o tempo especificado foi TIMEOUT_NO_WAIT retorna o jobInfo */
      if (timeout == Session.TIMEOUT_NO_WAIT) {
        for (String jobId : jobsAvailable) {
          JobInfo info = jobsSubmited.get(jobId);
          if (info.hasExited) {
            jobsSubmited.remove(jobId);
            return info;
          }
        }
        /* O tempo NO_WAIT esgotou e o job no terminou */
        throw new ExitTimeoutException(
          "O timeout esgotou e o job no terminou");
      }
    }
    final long start = System.currentTimeMillis();
    long limit = timeout * 1000; // tempo mximo para timeout em ms
    boolean waitForever = (timeout == Session.TIMEOUT_WAIT_FOREVER);
    while (waitForever || System.currentTimeMillis() - start < limit) {
      try {
        synchronized (this) {
          wait(WAITING_TIMEOUT);
        }
        synchronized (this) {
          JobInfo info = null;
          for (String jobId : jobsAvailable) {
            info = jobsSubmited.get(jobId);
            if (info.hasExited) {
              jobsSubmited.remove(jobId);
              return info;
            }
          }
          if (info == null) {
            throw new InvalidJobException("No h mais jobs submetidos na "
              + "sesso que ainda no tenham terminado sua execuo");
          }
        }
      }
      catch (InterruptedException e) {
      }
    }
    throw new ExitTimeoutException("O timeout esgotou e o job no terminou");
  }

  /**
   * Obtm as informaes sobre se o job submetido j finalizou sua execuo.
   * 
   * @param jobId o nome do job
   * @return o objeto {@link JobInfo} que possui as informaes sobre o job
   *         submetido
   * @throws InvalidJobException se o identificador do job no foi encontrado
   */
  public JobInfo getJobInfo(String jobId) throws InvalidJobException {
    synchronized (this) {
      JobInfo info = jobsSubmited.get(jobId);
      if (info == null) {
        throw new InvalidJobException(
          "No foi encontrado um job com o nome especificado: " + jobId);
      }
      return info;
    }
  }

  /**
   * Agurada que uma lista de jobs submetidos teminem suas execues.
   * 
   * Se mais de uma chamada estiver aguardando pelo mesmo job, apenas uma delas
   * recebe a informao de trmino do job. As outras chamadas recebem e exceo
   * <code>InvalidJobException</code>. Essa mesma exceo  tambm lanada se o
   * identificador do job no for encontrado entre os jobs submetidos nessa
   * sesso.
   * 
   * Se o parmetro dispose tiver o valor <code>true</code> as informaes de
   * trmino dos jobs da lista sero descartadas.
   * 
   * No caso do timeout ter esgotado sem que todos os jobs da lista tenham
   * terminado, a exceo <code>ExitTimeoutException</code>  lanada. Nesse
   * caso, as informaes sobre os jobs permanecem disponveis para uma outra
   * requisio.
   * 
   * @param jobIdList lista de identificadores de jobs para esperar
   * @param timeout o tempo de espera limite em segundos
   * @param dispose indica se as informao sobre a execuo de um job dever ser
   *        apagadas ou no
   * 
   * @throws InvalidJobException se um identificador de um job da lista no foi
   *         encontrado ou se outra requisio j recebeu a informao de
   *         trmino deste job.
   * @throws ExitTimeoutException se o timeout esgotou e nenhum job terminou.
   */
  public void waitFor(String[] jobIdList, long timeout, boolean dispose)
    throws InvalidJobException, ExitTimeoutException {
    if (jobIdList.length == 0) {
      return;
    }

    synchronized (this) {
      if (timeout == Session.TIMEOUT_NO_WAIT) {
        if (dispose) {
          for (String jobId : jobIdList) {
            jobsSubmited.remove(jobId);
          }
        }
        return;
      }
    }

    final long start = System.currentTimeMillis();
    long limit = timeout * 1000; // tempo mximo para timeout em ms
    boolean waitForever = (timeout == Session.TIMEOUT_WAIT_FOREVER);
    while (waitForever || System.currentTimeMillis() - start < limit) {
      try {
        synchronized (this) {
          wait(WAITING_TIMEOUT);
        }
        synchronized (this) {
          // Verifica se todos os jobs da lista finalizaram
          boolean allHasExited = true;
          for (String jobId : jobIdList) {
            JobInfo info = jobsSubmited.get(jobId);

            if (info == null) {
              throw new InvalidJobException(
                "Outra requisio j recebeu a informao de trmino desse"
                  + "mesmo job: " + jobId);
            }

            allHasExited = allHasExited && info.hasExited;
          }

          if (allHasExited) {
            for (String jobId : jobIdList) {
              JobInfo info = jobsSubmited.get(jobId);
              if (info == null) {
                throw new InvalidJobException(
                  "Outra requisio j recebeu a informao de trmino desse"
                    + "mesmo job: " + jobId);
              }
              if (dispose) {
                jobsSubmited.remove(jobId);
              }
            }
            return;
          }
        }
      }
      catch (InterruptedException e) {
      }
    }
    /* Quando o tempo esgotou, a informao sobre o job permanece guardada */
    throw new ExitTimeoutException("O timeout esgotou e o job no terminou: "
      + jobIdList);
  }

  /**
   * Aguarda que todos os jobs ativos na sesso, no momento da chamada do
   * mtodo, terminem suas execues.
   * 
   * Se alguma outra chamada estiver aguardando o trmino de algum job ativo na
   * sesso e receber sua informao de trmino, no  lanada exceo,
   * diferentemente com o comportamento dos outros mtodos de espera.
   * 
   * Se o parmetro dispose tiver o valor <code>true</code> as informaes de
   * trmino dos jobs ativos na sesso sero descartadas.
   * 
   * No caso do timeout ter esgotado sem que todos os jobs ativos na sesso
   * tenham terminado, a exceo <code>ExitTimeoutException</code>  lanada.
   * Nesse caso, as informaes sobre os jobs permanecem disponveis para uma
   * outra requisio.
   * 
   * @param timeout o tempo de espera limite em segundos
   * @param dispose indica se as informao sobre a execuo de um job dever ser
   *        apagadas ou no
   * 
   * @throws ExitTimeoutException se o timeout esgotou e nenhum job terminou.
   */
  public void waitForAllJobs(long timeout, boolean dispose)
    throws ExitTimeoutException {
    String[] jobIdList = getAllJobs();
    if (jobIdList.length == 0) {
      return;
    }

    synchronized (this) {
      if (timeout == Session.TIMEOUT_NO_WAIT) {
        if (dispose) {
          for (String jobId : jobIdList) {
            jobsSubmited.remove(jobId);
          }
        }
        return;
      }
    }

    final long start = System.currentTimeMillis();
    long limit = timeout * 1000; // tempo mximo para timeout em ms
    boolean waitForever = (timeout == Session.TIMEOUT_WAIT_FOREVER);
    while (waitForever || System.currentTimeMillis() - start < limit) {
      try {
        synchronized (this) {
          wait(WAITING_TIMEOUT);
        }
        synchronized (this) {
          // Verifica se todos os jobs finalizaram
          boolean allHasExited = true;
          for (String jobId : jobIdList) {
            JobInfo info = jobsSubmited.get(jobId);
            if (info != null) {
              allHasExited = allHasExited && info.hasExited;
            }
          }

          if (allHasExited) {
            for (String jobId : jobIdList) {
              JobInfo info = jobsSubmited.get(jobId);
              if (info != null) {
                if (dispose) {
                  jobsSubmited.remove(jobId);
                }
              }
            }
            return;
          }
        }
      }
      catch (InterruptedException e) {
      }
    }
    /* Quando o tempo esgotou, a informao sobre o job permanece guardada */
    throw new ExitTimeoutException("O timeout esgotou e o job no terminou: "
      + jobIdList);
  }

  /**
   * Obtem todos os jobs submetidos na sesso
   * 
   * @return lista com os identificadores dos jobs
   */
  public String[] getAllJobs() {
    String[] jobs = new String[jobsSubmited.size()];
    return jobsSubmited.keySet().toArray(jobs);
  }
}
