package csbase.server.services.schedulerservice.classadpolicies;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;
import java.util.Vector;

import condor.classad.AttrName;
import condor.classad.ClassAd;
import condor.classad.ClassAdParser;
import condor.classad.Constant;
import condor.classad.RecordExpr;
import csbase.logic.CommandInfo;
import csbase.logic.SGASet;
import csbase.logic.algorithms.AlgorithmConfigurator;

public class ClassAdManager {
	private HashMap<String, RecordExpr> sgasAds = new HashMap<String, RecordExpr>();
	private HashMap<String, RecordExpr> cmdsAds = new HashMap<String, RecordExpr>();

	private static final String defaultSGAAdDesc = "Requirements = true; Rank = int(4-other.priority); ";
	private static final String defaultCmdAdDesc_Rank = "Rank = int(other.freeRAMMemory); ";

	private ClassAdParser cap = new ClassAdParser();

	public ClassAdMatchInfo match(CommandInfo cmd, SGASet server) {
		RecordExpr cmdAd = getCmdAd(cmd);
		RecordExpr sgaAd = getSGAAd(server);
		
		if (cmdAd == null || sgaAd == null)
			return new ClassAdMatchInfo(server, cmd, false, -1, -1);

		int[] ranks = ClassAd.match(sgaAd, cmdAd);
		if (ranks != null)
			return new ClassAdMatchInfo(server, cmd, true, ranks[0], ranks[1]);
		else
			return new ClassAdMatchInfo(server, cmd, false, -1, -1);
	}

	private RecordExpr getCmdAd(CommandInfo cmd) {
		if (cmdsAds.containsKey(cmd.getId()))
			return cmdsAds.get(cmd.getId());
		RecordExpr ad = createCmdAd(cmd);
		if (ad != null)
			cmdsAds.put(cmd.getId(), ad);
		return ad;
	}

	private RecordExpr getSGAAd(SGASet server) {
		if (sgasAds.containsKey(server.getName()))
			return sgasAds.get(server.getName());
		RecordExpr ad = createSGAAd(server);
		if (ad != null)
			sgasAds.put(server.getName(), ad);
		return ad;
	}

	public RecordExpr alterSGAAd(SGASet server) {
		if (!sgasAds.containsKey(server.getName()))
			return null;
		RecordExpr ad = getSGAAd(server);
		ad.removeAttribute(AttrName.fromString("freeRAMMemory"));
		ad.insertAttribute("freeRAMMemory",
				Constant.getInstance(server.getRAMFreeMemoryMb()));
		return ad;
	}

	private RecordExpr createSGAAd(SGASet server) {
		cap.reset(getSGAAdDesc(server));
		try {
			return (RecordExpr) cap.parse();
		} catch (IllegalStateException e) {
			e.printStackTrace();
		}
		return null;
	}

	private RecordExpr createCmdAd(CommandInfo cmd) {
		cap.reset(getCommandAdDesc(cmd));
		try {
			return (RecordExpr) cap.parse();
		} catch (IllegalStateException e) {
			e.printStackTrace();
		}
		return null;
	}

	private String getSGAAdDesc(SGASet sgaSet) {
		StringBuilder descBuilder = new StringBuilder("[");
		descBuilder.append(defaultSGAAdDesc);
		appendAdInfo(descBuilder, "cpuLoad", sgaSet.getCPULoad1());
		appendAdInfo(descBuilder, "numProcessors", sgaSet.getNumProcessors());
		appendAdInfo(descBuilder, "name", sgaSet.getName());
		appendAdInfo(descBuilder, "platform", sgaSet.getPlatformId());
		appendAdInfo(descBuilder, "ramMemory", sgaSet.getRAMMemoryInfoMb());
		appendAdInfo(descBuilder, "swapMemory", sgaSet.getSwapMemoryInfoMb());
		appendAdInfo(descBuilder, "freeRAMMemory", sgaSet.getRAMFreeMemoryMb());
		appendAdInfo(descBuilder, "freeSwapMem", sgaSet.getSwapFreeMemory());

		appendSGAResources(sgaSet, descBuilder);
		descBuilder.append("]");
		return descBuilder.toString();
	}

	private void appendSGAResources(SGASet sgaSet, StringBuilder builder) {
		builder.append("resources = {");
		Set<String> resources = sgaSet.getAllInfo()[0].getRequirements();
		for (String res : resources) {
			builder.append("\"");
			builder.append(res);
			builder.append("\", ");
		}
		builder.append("}; ");
	}

	private String getCommandAdDesc(CommandInfo cmd) {
		String ad = cmd.getExtraPropertiesMap().get("ClassAdDesc");
		if (ad != null)
			return ad;
		CommandRequirements cmdReqs = getCommandRequirements(cmd);
		StringBuilder descBuilder = new StringBuilder("[");
		descBuilder.append(defaultCmdAdDesc_Rank);
		appendAdInfo(descBuilder, "id", cmd.getId());
		appendAdInfo(descBuilder, "priority", cmd.getPriority().ordinal());

		appendCommandSupportedPlatforms(cmd, cmdReqs.platforms, descBuilder);
		appendCommandRequirements(cmd, cmdReqs.memoryAmount, cmdReqs.cpuAmount,
				cmdReqs.requirements, descBuilder);
		descBuilder.append("]");
		return descBuilder.toString();
	}

	private void appendCommandSupportedPlatforms(CommandInfo cmd,
			Vector<String> platforms, StringBuilder builder) {
		builder.append("supportedPlatforms = {");
		Set<String> supportedPlatforms = new HashSet<String>();
		if (cmd.getPlatformFilter() != null) {
			supportedPlatforms.add(cmd.getPlatformFilter());
		} else {
			supportedPlatforms.addAll(platforms);
		}
		for (String platform : supportedPlatforms) {
			builder.append("\"");
			builder.append(platform);
			builder.append("\", ");
		}
		builder.append("}; ");
	}

	private CommandRequirements getCommandRequirements(CommandInfo cmd) {
		CommandRequirements cmdReqs = new CommandRequirements();
		AlgorithmConfigurator configurator;
		try {
			configurator = cmd.getConfigurator();
			// Atualiza informaes sobre o algoritmo.
			configurator.updateAlgorithmVersion();
			Set<String> platformsSet = configurator.getPlatforms();
			String platformRestriction = cmd.getPlatformFilter();
			cmdReqs.platforms = new Vector<String>();
			if (platformRestriction == null) {
				cmdReqs.platforms.addAll(platformsSet);
			} else {
				if (platformsSet.contains(platformRestriction)) {
					cmdReqs.platforms.add(platformRestriction);
				} else {
					cmdReqs.platforms = new Vector<String>();
				}
			}

			cmdReqs.cpuAmount = configurator.getCpuAmount();
			cmdReqs.memoryAmount = configurator.getMemoryAmount();
			cmdReqs.requirements = configurator.getRequirements();
		} catch (Exception e) {
			e.printStackTrace();
		}

		String reqMemStr = cmd.getExtraPropertiesMap().get("RequiredMemory");
		if (reqMemStr != null) {
			try {
				cmdReqs.memoryAmount = Integer.parseInt(reqMemStr);
			} catch (NumberFormatException e) {
			}
		}
		return cmdReqs;
	}

	private void appendCommandRequirements(CommandInfo cmd, float memoryAmount,
			float cpuAmount, Set<String> requirements, StringBuilder builder) {
		// Sempre vai haver o requisito de plataforma compatvel
		builder.append("Requirements = member(other.platform, self.supportedPlatforms)");
		if (memoryAmount > 0) {
			builder.append("&& other.freeRAMMemory>");
			builder.append(memoryAmount);
		}
		if (cpuAmount > 0) {
			builder.append("&& other.numProcessors>");
			builder.append(cpuAmount);
		}
		if (!requirements.isEmpty()) {
			for (String req : requirements) {
				builder.append("&& member(\"");
				builder.append(req.toUpperCase());
				builder.append("\", other.resources)");
			}
		}
		String demandedSGA = cmd.getSGAName();
		if (demandedSGA != null && !demandedSGA.isEmpty()) {
			builder.append("&& other.name==\"");
			builder.append(demandedSGA);
			builder.append("\"");
		}
		builder.append(";");
	}

	private void appendAdInfo(StringBuilder builder, String infoKey,
			String infoValue) {
		builder.append(infoKey);
		builder.append(" = \"");
		builder.append(infoValue);
		builder.append("\"; ");
	}

	private void appendAdInfo(StringBuilder builder, String infoKey,
			double infoValue) {
		builder.append(infoKey);
		builder.append(" = ");
		builder.append(infoValue);
		builder.append("; ");
	}

}

class CommandRequirements {
	public float memoryAmount;
	public float cpuAmount;
	public Vector<String> platforms;
	public Set<String> requirements;
}
