package csbase.server.services.schedulerservice;

import csbase.logic.CapacityType;
import csbase.logic.CommandInfo;
import csbase.logic.SGAInfo;
import csbase.logic.SGASet;
import csbase.server.services.schedulerservice.filter.SGAFilterByCommandRequirements;

import java.util.*;

/**
 * Poltica de escalonamento que privilegia os SGAs com a melhor combinao
 * entre as capacidades atuais de processamento, leitura/escrita em disco e
 * transferncia de dados na rede. Os melhores SGAs so definidos com base em
 * uma equao com variveis alimentadas pelos resultados dos benchmarks
 * aplicados aos SGAs.
 * 
 * @author valeria
 *
 */
public class CPUIOCapacityPolicy implements SchedulerPolicyInterface {
	private long minWritingCap;
	private long maxWritingCap;
	private long minSendCap;
	private long maxSendCap;
	private double minProcCap;
	private double maxProcCap;

	/**
	 * Inicializa os mnimos e mximos das taxas de processamento, leitura/
	 * escrita em disco e transferncia em rede dos SGAs.
	 * 
	 * @param servers
	 *            Servidores disponveis para execuo
	 */
	private void initializeExtremes(List<SGASet> servers) {
		long writeCap;
		long cpuCap;
		long transferCap;
		if (servers.size() > 0) {
			SGASet sga = servers.get(0);
			writeCap = sga.getCapacity(CapacityType.DISK_WRITE);
			if (writeCap == SGAInfo.NO_CAPACITY)
				writeCap = 1;
			minWritingCap = writeCap;
			maxWritingCap = writeCap;
			cpuCap = sga.getCapacity(CapacityType.CPU);
			if (cpuCap == SGAInfo.NO_CAPACITY)
				cpuCap = 1;
			minProcCap = (1 / (sga.getCPULoad1() + 1)) * cpuCap;
			maxProcCap = (1 / (sga.getCPULoad1() + 1)) * cpuCap;
			// A taxa de transferncia apresenta tratamento diferenciado j
			// que pode assumir dois valores: SGASet.NO_CAPACITY e
			// SGASet.ALL_CAPACITY.
			transferCap = sga.getCapacity(CapacityType.NET);
			if (transferCap == SGAInfo.NO_CAPACITY) {
				transferCap = 1;
			} else {
				if (transferCap == SGAInfo.ALL_CAPACITY) {
					transferCap = Long.MAX_VALUE;
				}
			}
			minSendCap = transferCap;
			maxSendCap = transferCap;
		}
		for (int i = 1; i < servers.size(); i++) {
			SGASet sga = servers.get(i);
			writeCap = sga.getCapacity(CapacityType.DISK_WRITE);
			if (writeCap == SGAInfo.NO_CAPACITY)
				writeCap = 1;
			if (writeCap < minWritingCap)
				minWritingCap = writeCap;
			else if (writeCap > maxWritingCap)
				maxWritingCap = writeCap;
			cpuCap = sga.getCapacity(CapacityType.CPU);
			if (cpuCap == SGAInfo.NO_CAPACITY)
				cpuCap = 1;
			if (((1 / (sga.getCPULoad1() + 1)) * cpuCap) < minProcCap)
				minProcCap = (1 / (sga.getCPULoad1() + 1)) * cpuCap;
			else if (((1 / (sga.getCPULoad1() + 1)) * cpuCap) > maxProcCap)
				maxProcCap = (1 / (sga.getCPULoad1() + 1)) * cpuCap;
			transferCap = sga.getCapacity(CapacityType.NET);
			if (transferCap == SGAInfo.NO_CAPACITY)
				transferCap = 1;
			else if (transferCap == SGAInfo.ALL_CAPACITY)
				transferCap = Long.MAX_VALUE;
			if (transferCap < minSendCap)
				minSendCap = transferCap;
			else if (transferCap > maxSendCap)
				maxSendCap = transferCap;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public Map<CommandInfo, SGASet> chooseServer(List<CommandInfo> commands,
			List<SGASet> servers) {
		BalancedCommandDistribution bdc = new BalancedCommandDistribution(servers);
		
		Map<CommandInfo, SGASet> allocation = new HashMap<CommandInfo, SGASet>();
		for (CommandInfo cmd : commands) {

			List<SGASet> filteresSGAs = SGAFilterByCommandRequirements.filter(cmd,
					servers);
			if (filteresSGAs == null) {
				continue;
			}

			initializeExtremes(filteresSGAs);

			List<SGASet> sgas = getSortedSGAByCPUAndIOCapacity(filteresSGAs);

			if (sgas.size() != 0) {
				SGASet choosedServer = bdc.chooseLightest(sgas);
				allocation.put(cmd, choosedServer);
			}
		}
		return null;
	}

	private List<SGASet> getSortedSGAByCPUAndIOCapacity(List<SGASet> servers) {
		List<SGASet> sgasSets = new ArrayList<SGASet>();
		sgasSets.addAll(servers);
		Collections.sort(sgasSets, new SGASetCPUAndIOComparator(minWritingCap,
				maxWritingCap, minProcCap, maxProcCap, minSendCap, maxSendCap));
		return sgasSets;
	}
}
